<?php
/**
 * FCA Settings Class
 * 
 * Handles all settings functionality for the FCA Pages plugin
 */
class FCA_Settings {
    /**
     * Default settings
     */
    private static $defaults = [
        'menu_enabled' => true,
        'menu_title' => 'Custom Pages',
        'menu_icon' => '📄',
        'include_settings_link' => true,
        'selected_pages' => [],
        'menu_priority' => 50,
        'page_order' => []
    ];
    
    /**
     * Current settings
     */
    private static $settings = [];
    
    /**
     * Initialize the settings
     */
    public static function init() {
        // Add the settings submenu
        add_action('admin_menu', [self::class, 'add_settings_menu']);
        
        // Register settings
        add_action('admin_init', [self::class, 'register_settings']);
        
        // Load current settings
        self::load_settings();
    }
    
    /**
     * Add the settings menu
     */
    public static function add_settings_menu() {
        add_submenu_page(
            'fca-pages-admin',                  // Parent slug
            __('Menu Settings', 'fca-pages'),   // Page title
            __('Menu Settings', 'fca-pages'),   // Menu title
            'manage_options',                   // Capability
            'fca-pages-settings',               // Menu slug
            [self::class, 'render_settings_page'] // Callback
        );
    }
    
    /**
     * Register the settings
     */
    public static function register_settings() {
        // Register a new setting
        register_setting(
            'fca_pages_settings',          // Option group
            'fca_pages_menu_settings',     // Option name
            [
                'sanitize_callback' => [self::class, 'sanitize_settings'],
                'default' => self::$defaults
            ]
        );
        
        // Add settings section for menu options
        add_settings_section(
            'fca_pages_menu_section',      // ID
            __('Menu Settings', 'fca-pages'), // Title
            [self::class, 'render_section_description'], // Callback
            'fca_pages_settings'           // Page
        );
        
        // Add settings fields
        add_settings_field(
            'menu_enabled',                // ID
            __('Enable Menu Group', 'fca-pages'), // Title
            [self::class, 'render_menu_enabled_field'], // Callback
            'fca_pages_settings',          // Page
            'fca_pages_menu_section'       // Section
        );
        
        add_settings_field(
            'menu_title',                  // ID
            __('Menu Group Title', 'fca-pages'), // Title
            [self::class, 'render_menu_title_field'], // Callback
            'fca_pages_settings',          // Page
            'fca_pages_menu_section'       // Section
        );
        
        add_settings_field(
            'menu_icon',                   // ID
            __('Menu Group Icon', 'fca-pages'), // Title
            [self::class, 'render_menu_icon_field'], // Callback
            'fca_pages_settings',          // Page
            'fca_pages_menu_section'       // Section
        );
        
        add_settings_field(
            'include_settings_link',       // ID
            __('Include Settings Link', 'fca-pages'), // Title
            [self::class, 'render_settings_link_field'], // Callback
            'fca_pages_settings',          // Page
            'fca_pages_menu_section'       // Section
        );
        
        add_settings_field(
            'selected_pages',              // ID
            __('Pages to Include', 'fca-pages'), // Title
            [self::class, 'render_selected_pages_field'], // Callback
            'fca_pages_settings',          // Page
            'fca_pages_menu_section'       // Section
        );
        
        add_settings_field(
            'menu_priority',               // ID
            __('Menu Priority', 'fca-pages'), // Title
            [self::class, 'render_menu_priority_field'], // Callback
            'fca_pages_settings',          // Page
            'fca_pages_menu_section'       // Section
        );
        
        // Add a new section for page ordering
        add_settings_section(
            'fca_pages_order_section',       // ID
            __('Page Order', 'fca-pages'),   // Title
            [self::class, 'render_order_section_description'], // Callback
            'fca_pages_settings'             // Page
        );
        
        // Add field for page ordering
        add_settings_field(
            'page_order',                    // ID
            __('Arrange Pages', 'fca-pages'), // Title
            [self::class, 'render_page_order_field'], // Callback
            'fca_pages_settings',            // Page
            'fca_pages_order_section'        // Section
        );
        
        // Enqueue necessary scripts
        add_action('admin_enqueue_scripts', [self::class, 'enqueue_admin_scripts']);
    }
    
    /**
     * Enqueue admin scripts
     */
    public static function enqueue_admin_scripts($hook) {
        // Only load on our settings page
        if (strpos($hook, 'fca-pages-settings') === false) {
            return;
        }
        
        // Enqueue jQuery UI sortable
        wp_enqueue_script('jquery-ui-sortable');
        
        // Add custom styles
        $style = '
            .fca-sortable-pages {
                background: #fff;
                border: 1px solid #ccd0d4;
                border-radius: 4px;
                padding: 10px;
                margin-bottom: 15px;
                max-width: 500px;
            }
            .fca-sortable-pages li {
                cursor: move;
                background: #f8f9fa;
                margin: 0 0 5px;
                padding: 10px 15px;
                border: 1px solid #e2e4e7;
                border-radius: 3px;
                font-weight: 500;
                display: flex;
                align-items: center;
            }
            .fca-sortable-pages li .dashicons {
                margin-right: 8px;
                color: #72777c;
            }
            .fca-sortable-pages li:hover {
                background: #f1f1f1;
            }
            .fca-sortable-placeholder {
                border: 1px dashed #b4b9be;
                background: #f7fcfe;
                height: 40px;
                margin-bottom: 5px;
            }
            .fca-drag-handle {
                cursor: move;
                color: #72777c;
                margin-right: 10px;
            }
        ';
        wp_add_inline_style('wp-admin', $style);
    }
    
    /**
     * Sanitize settings
     */
    public static function sanitize_settings($input) {
        // --- ORIGINAL CODE ---
        $sanitized = [];
        
        // Menu enabled (boolean)
        $sanitized['menu_enabled'] = isset($input['menu_enabled']) ? true : false;
        
        // Menu title (text)
        $sanitized['menu_title'] = sanitize_text_field($input['menu_title'] ?? self::$defaults['menu_title']);
        
        // Menu icon (text)
        $sanitized['menu_icon'] = sanitize_text_field($input['menu_icon'] ?? self::$defaults['menu_icon']);
        
        // Include settings link (boolean)
        $sanitized['include_settings_link'] = isset($input['include_settings_link']) ? true : false;
        
        // Selected pages (array of ids)
        $sanitized['selected_pages'] = [];
        if (isset($input['selected_pages']) && is_array($input['selected_pages'])) {
            foreach ($input['selected_pages'] as $page_id) {
                $sanitized['selected_pages'][] = absint($page_id);
            }
        }
        
        // Menu priority (integer)
        $sanitized['menu_priority'] = absint($input['menu_priority'] ?? self::$defaults['menu_priority']);
        
        // Page order (array of ids)
        $sanitized['page_order'] = [];
        if (isset($input['page_order']) && is_array($input['page_order'])) {
            foreach ($input['page_order'] as $page_id) {
                $sanitized['page_order'][] = absint($page_id);
            }
        }
        
        return $sanitized;
        // --- END ORIGINAL CODE ---
    }
    
    /**
     * Get the path to settings file for file-based storage
     */
    private static function get_settings_file_path() {
        $upload_dir = wp_upload_dir();
        $base_dir = $upload_dir['basedir'];
        $fca_dir = $base_dir . '/fca-pages';
        
        // Create directory if it doesn't exist
        if (!is_dir($fca_dir)) {
            wp_mkdir_p($fca_dir);
        }
        
        return $fca_dir . '/settings.json';
    }
    
    /**
     * Save settings to file
     */
    private static function save_settings_to_file($settings) {
        $file_path = self::get_settings_file_path();
        $json_data = wp_json_encode($settings);
        return file_put_contents($file_path, $json_data);
    }
    
    /**
     * Load settings from file
     */
    private static function load_settings_from_file() {
        $file_path = self::get_settings_file_path();
        if (!file_exists($file_path)) {
            return false;
        }
        
        $json_data = file_get_contents($file_path);
        if (empty($json_data)) {
            return false;
        }
        
        $settings = json_decode($json_data, true);
        return is_array($settings) ? $settings : false;
    }
    
    /**
     * Load settings from database or file
     */
    public static function load_settings() {
        $saved = false;
        
        // Try file-based storage first (most reliable)
        $file_data = self::load_settings_from_file();
        if ($file_data !== false) {
            $saved = $file_data;
        }
        
        // Fallback to database settings if file not found
        if (empty($saved) || !is_array($saved)) {
        $saved = get_option('fca_pages_menu_settings', []);
        }
        
        // Make sure we have the expected structure with fallbacks
        $settings = wp_parse_args($saved, self::$defaults);
        
        // Ensure menu_enabled is boolean (might be saved as 1/0)
        $settings['menu_enabled'] = (bool)$settings['menu_enabled'];
        $settings['include_settings_link'] = (bool)$settings['include_settings_link'];
        
        // Ensure arrays are arrays
        $settings['selected_pages'] = is_array($settings['selected_pages']) ? $settings['selected_pages'] : [];
        $settings['page_order'] = is_array($settings['page_order']) ? $settings['page_order'] : [];
        
        self::$settings = $settings;
    }
    
    /**
     * Get all settings or a specific setting
     */
    public static function get_settings($key = null) {
        if ($key === null) {
            return self::$settings;
        }
        
        return self::$settings[$key] ?? self::$defaults[$key] ?? null;
    }
    
    /**
     * Render the settings page
     */
    public static function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Process form submission - File-based storage
        if (isset($_POST['fca_save_settings']) && check_admin_referer('fca_settings_nonce', 'fca_settings_nonce')) {
            // Get raw submitted data
            $input = isset($_POST['settings']) ? $_POST['settings'] : [];
            
            // Sanitize the data
            $sanitized = [
                'menu_enabled' => isset($input['menu_enabled']) ? 1 : 0,
                'menu_title' => sanitize_text_field($input['menu_title'] ?? 'Custom Pages'),
                'menu_icon' => sanitize_text_field($input['menu_icon'] ?? '📄'),
                'include_settings_link' => isset($input['include_settings_link']) ? 1 : 0,
                'menu_priority' => absint($input['menu_priority'] ?? 50),
                'selected_pages' => isset($input['selected_pages']) && is_array($input['selected_pages']) ? 
                    array_map('absint', $input['selected_pages']) : [],
                'page_order' => isset($input['page_order']) && is_array($input['page_order']) ? 
                    array_map('absint', $input['page_order']) : []
            ];
            
            // Save to file storage primarily
            $file_result = self::save_settings_to_file($sanitized);
            
            // As a fallback, also try regular WP option
            update_option('fca_pages_menu_settings', $sanitized);
            
            // Force refresh the settings array with our changes
            self::$settings = $sanitized;
            
            // Show success message
            if ($file_result !== false) {
                echo '<div class="notice notice-success is-dismissible"><p>' . __('Settings saved successfully.', 'fca-pages') . '</p></div>';
            } else {
                echo '<div class="notice notice-error is-dismissible"><p>' . __('Error saving settings. Please check file permissions.', 'fca-pages') . '</p></div>';
            }
        }
        
        // Get the current settings to display
        $settings = self::get_settings();
        
        // Get all published pages for selection
        $pages = get_posts([
            'post_type' => 'fca_page',
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
        ]);
        
        // Prepare ordered page list for the UI
        $ordered_pages = [];
        $selected_pages = $settings['selected_pages'];
        $page_order = $settings['page_order'];

        // If we have both selected pages and a custom order
        if (!empty($page_order) && !empty($selected_pages) && !empty($pages)) {
            // First, add pages that are in the saved order
            foreach ($page_order as $id) {
                foreach ($pages as $key => $page) {
                    if ($page->ID == $id && in_array($page->ID, $selected_pages)) {
                        $ordered_pages[] = $page;
                        unset($pages[$key]);
                        break;
                    }
                }
            }
            
            // Then, add any selected pages that weren't in the order
            foreach ($pages as $key => $page) {
                if (in_array($page->ID, $selected_pages)) {
                    $ordered_pages[] = $page;
                    unset($pages[$key]);
                }
            }
        } 
        // If we have selected pages but no custom order
        elseif (!empty($selected_pages) && !empty($pages)) {
            foreach ($pages as $key => $page) {
                if (in_array($page->ID, $selected_pages)) {
                    $ordered_pages[] = $page;
                    unset($pages[$key]);
                }
            }
        }
        // If user hasn't selected specific pages (show all pages in our ordering UI)
        elseif (empty($selected_pages) && !empty($pages)) {
            $ordered_pages = $pages;
            $pages = []; // Clear the original array since we've moved all pages
        }
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <p><?php _e('Configure how custom pages appear in the Fluent Community menu.', 'fca-pages'); ?></p>
            
            <form method="post" action="">
                <?php wp_nonce_field('fca_settings_nonce', 'fca_settings_nonce'); ?>
                
                <h2 class="title"><?php _e('Menu Options', 'fca-pages'); ?></h2>
                
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php _e('Enable Menu Group', 'fca-pages'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="settings[menu_enabled]" value="1" <?php checked(1, $settings['menu_enabled']); ?>>
                                <?php _e('Show custom pages in the Fluent Community sidebar menu', 'fca-pages'); ?>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php _e('Menu Group Title', 'fca-pages'); ?></th>
                        <td>
                            <input type="text" name="settings[menu_title]" value="<?php echo esc_attr($settings['menu_title']); ?>" class="regular-text">
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php _e('Menu Icon', 'fca-pages'); ?></th>
                        <td>
                            <input type="text" name="settings[menu_icon]" value="<?php echo esc_attr($settings['menu_icon']); ?>" class="regular-text">
                            <p class="description"><?php _e('Use emoji or leave blank. Examples: 📄 📑 📋 📃 📜', 'fca-pages'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php _e('Include Settings Link', 'fca-pages'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="settings[include_settings_link]" value="1" <?php checked(1, $settings['include_settings_link']); ?>>
                                <?php _e('Include "Page Settings" link in the menu group', 'fca-pages'); ?>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php _e('Menu Priority', 'fca-pages'); ?></th>
                        <td>
                            <input type="number" name="settings[menu_priority]" value="<?php echo esc_attr($settings['menu_priority']); ?>" min="1" max="100" step="1" class="small-text">
                            <p class="description"><?php _e('Lower numbers appear higher in the menu (default: 50)', 'fca-pages'); ?></p>
                        </td>
                    </tr>
                </table>
                
                <h2 class="title"><?php _e('Pages to Include', 'fca-pages'); ?></h2>
                
                <?php if (empty($pages) && empty($ordered_pages)): ?>
                    <p><?php _e('No custom pages found. Create some pages first.', 'fca-pages'); ?></p>
                <?php else: ?>
                    
                    <p><?php _e('Select which pages to include in the menu:', 'fca-pages'); ?></p>
                    
                    <div style="max-height: 300px; overflow-y: auto; padding: 15px; border: 1px solid #ddd; margin-bottom: 20px; background: #fff;">
                        <!-- "Select All" option -->
                        <p>
                            <label>
                                <input type="checkbox" id="fca-select-all-pages">
                                <strong><?php _e('Select All', 'fca-pages'); ?></strong>
                            </label>
                        </p>
                        
                        <hr>
                        
                        <!-- Page checkboxes -->
                        <?php 
                        // First add previously selected pages that were removed from $pages array
                        foreach ($ordered_pages as $page): 
                            $is_selected = in_array($page->ID, $selected_pages) || empty($selected_pages);
                        ?>
                        <p>
                            <label>
                                <input type="checkbox" name="settings[selected_pages][]" value="<?php echo esc_attr($page->ID); ?>" <?php checked($is_selected, true); ?> class="fca-page-checkbox">
                                <?php echo esc_html($page->post_title); ?>
                            </label>
                        </p>
                        <?php endforeach; ?>

                        <!-- Add any pages that weren't previously selected -->
                        <?php foreach ($pages as $page): ?>
                        <p>
                            <label>
                                <input type="checkbox" name="settings[selected_pages][]" value="<?php echo esc_attr($page->ID); ?>" <?php checked(empty($selected_pages), true); ?> class="fca-page-checkbox">
                                <?php echo esc_html($page->post_title); ?>
                            </label>
                        </p>
                        <?php endforeach; ?>
                    </div>
                    
                    <p class="description"><?php _e('Leave all unchecked to include all pages.', 'fca-pages'); ?></p>
                    
                    <!-- JavaScript for "Select All" functionality -->
                    <script>
                        document.addEventListener('DOMContentLoaded', function() {
                            var selectAllCheckbox = document.getElementById('fca-select-all-pages');
                            var pageCheckboxes = document.querySelectorAll('.fca-page-checkbox');
                            
                            // "Select All" changes all checkboxes
                            selectAllCheckbox.addEventListener('change', function() {
                                pageCheckboxes.forEach(function(checkbox) {
                                    checkbox.checked = selectAllCheckbox.checked;
                                });
                            });
                            
                            // Update "Select All" based on individual checkboxes
                            function updateSelectAll() {
                                var allChecked = true;
                                pageCheckboxes.forEach(function(checkbox) {
                                    if (!checkbox.checked) {
                                        allChecked = false;
                                    }
                                });
                                selectAllCheckbox.checked = allChecked;
                            }
                            
                            pageCheckboxes.forEach(function(checkbox) {
                                checkbox.addEventListener('change', updateSelectAll);
                            });
                            
                            // Set initial state
                            updateSelectAll();
                        });
                    </script>
                    
                    <h2 class="title"><?php _e('Page Order', 'fca-pages'); ?></h2>
                    
                    <p><?php _e('Drag and drop to reorder pages in the menu. This order overrides the default order.', 'fca-pages'); ?></p>
                    
                    <div class="fca-page-order-container">
                        <ul id="fca-sortable-pages" class="fca-sortable-pages">
                            <?php foreach ($ordered_pages as $page): ?>
                            <li data-id="<?php echo esc_attr($page->ID); ?>">
                                <span class="dashicons dashicons-menu"></span>
                                <?php echo esc_html($page->post_title); ?>
                                <input type="hidden" name="settings[page_order][]" value="<?php echo esc_attr($page->ID); ?>">
                            </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    
                    <style>
                        .fca-sortable-pages {
                            background: #fff;
                            border: 1px solid #ddd;
                            padding: 10px;
                            max-width: 500px;
                            list-style: none;
                            margin: 0;
                        }
                        .fca-sortable-pages li {
                            cursor: move;
                            padding: 10px;
                            margin-bottom: 5px;
                            background: #f8f9fa;
                            border: 1px solid #e2e4e7;
                            border-radius: 3px;
                            display: flex;
                            align-items: center;
                        }
                        .fca-sortable-pages li:hover {
                            background: #f0f0f0;
                        }
                        .fca-sortable-pages li .dashicons {
                            margin-right: 8px;
                            color: #72777c;
                        }
                        .ui-state-highlight {
                            height: 40px;
                            line-height: 1.2em;
                            background: #f7fcfe;
                            border: 1px dashed #bfe7f3;
                        }
                    </style>
                    
                    <!-- JavaScript for sortable functionality -->
                    <script>
                    jQuery(document).ready(function($) {
                        $('#fca-sortable-pages').sortable({
                            placeholder: 'ui-state-highlight',
                            update: function(event, ui) {
                                // Update hidden inputs after sorting
                                $('#fca-sortable-pages li').each(function(index) {
                                    var pageId = $(this).data('id');
                                    $(this).find('input[type="hidden"]').remove();
                                    $(this).append('<input type="hidden" name="settings[page_order][]" value="' + pageId + '">');
                                });
                            }
                        });
                    });
                    </script>
                <?php endif; ?>
                
                <p class="submit">
                    <input type="hidden" name="fca_save_settings" value="1">
                    <?php submit_button(__('Save Settings', 'fca-pages'), 'primary', 'submit', false); ?>
                </p>
            </form>
            
            <div class="fca-admin-footer" style="margin-top: 20px; padding: 15px; background: #f8f8f8; border: 1px solid #ddd;">
                <h3><?php _e('About FCA Pages', 'fca-pages'); ?></h3>
                <p><?php _e('FCA Pages allows you to create and manage custom pages inside your Fluent Community portal.', 'fca-pages'); ?></p>
                <p><?php _e('Settings are stored securely and persist between page refreshes.', 'fca-pages'); ?></p>
                <p><a href="https://www.fluentcommunityaddons.com" target="_blank"><?php _e('Fluent Community Add-ons', 'fca-pages'); ?></a></p>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render section description
     */
    public static function render_section_description() {
        echo '<p>' . __('These settings control how your custom pages are displayed in the Fluent Community sidebar menu.', 'fca-pages') . '</p>';
    }
    
    /**
     * Render menu enabled field
     */
    public static function render_menu_enabled_field() {
        $menu_enabled = self::get_settings('menu_enabled');
        ?>
        <label>
            <input type="checkbox" name="fca_pages_menu_settings[menu_enabled]" <?php checked($menu_enabled); ?>>
            <?php _e('Show custom pages in the Fluent Community sidebar menu', 'fca-pages'); ?>
        </label>
        <?php
    }
    
    /**
     * Render menu title field
     */
    public static function render_menu_title_field() {
        $menu_title = self::get_settings('menu_title');
        ?>
        <input type="text" name="fca_pages_menu_settings[menu_title]" value="<?php echo esc_attr($menu_title); ?>" class="regular-text">
        <p class="description"><?php _e('The title of the menu group in the sidebar', 'fca-pages'); ?></p>
        <?php
    }
    
    /**
     * Render menu icon field
     */
    public static function render_menu_icon_field() {
        $menu_icon = self::get_settings('menu_icon');
        ?>
        <input type="text" name="fca_pages_menu_settings[menu_icon]" value="<?php echo esc_attr($menu_icon); ?>" class="regular-text">
        <p class="description"><?php _e('Menu icon (emoji or blank)', 'fca-pages'); ?></p>
        <?php
    }
    
    /**
     * Render settings link field
     */
    public static function render_settings_link_field() {
        $include_settings_link = self::get_settings('include_settings_link');
        ?>
        <label>
            <input type="checkbox" name="fca_pages_menu_settings[include_settings_link]" <?php checked($include_settings_link); ?>>
            <?php _e('Include "Page Settings" link in the menu group', 'fca-pages'); ?>
        </label>
        <?php
    }
    
    /**
     * Render selected pages field
     */
    public static function render_selected_pages_field() {
        $selected_pages = self::get_settings('selected_pages');
        
        // Get all published pages
        $pages = get_posts([
            'post_type' => 'fca_page',
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
        ]);
        
        if (empty($pages)) {
            echo '<p>' . __('No custom pages found. Create some pages first.', 'fca-pages') . '</p>';
            return;
        }
        
        echo '<p>' . __('Select which pages to include in the menu:', 'fca-pages') . '</p>';
        
        echo '<div style="max-height: 200px; overflow-y: auto; padding: 10px; border: 1px solid #ddd; margin-bottom: 10px;">';
        
        // Show "Select All" option
        echo '<p>';
        echo '<label>';
        echo '<input type="checkbox" id="fca-select-all-pages">';
        echo ' ' . __('Select All', 'fca-pages');
        echo '</label>';
        echo '</p>';
        
        // Show checkbox for each page
        foreach ($pages as $page) {
            $is_selected = in_array($page->ID, $selected_pages);
            echo '<p>';
            echo '<label>';
            echo '<input type="checkbox" name="fca_pages_menu_settings[selected_pages][]" value="' . esc_attr($page->ID) . '" ' . checked($is_selected, true, false) . ' class="fca-page-checkbox">';
            echo ' ' . esc_html($page->post_title);
            echo '</label>';
            echo '</p>';
        }
        
        echo '</div>';
        
        // Add JavaScript for "Select All" functionality
        ?>
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                var selectAllCheckbox = document.getElementById('fca-select-all-pages');
                var pageCheckboxes = document.querySelectorAll('.fca-page-checkbox');
                
                // "Select All" changes all checkboxes
                selectAllCheckbox.addEventListener('change', function() {
                    pageCheckboxes.forEach(function(checkbox) {
                        checkbox.checked = selectAllCheckbox.checked;
                    });
                });
                
                // Update "Select All" based on individual checkboxes
                function updateSelectAll() {
                    var allChecked = true;
                    pageCheckboxes.forEach(function(checkbox) {
                        if (!checkbox.checked) {
                            allChecked = false;
                        }
                    });
                    selectAllCheckbox.checked = allChecked;
                }
                
                pageCheckboxes.forEach(function(checkbox) {
                    checkbox.addEventListener('change', updateSelectAll);
                });
                
                // Set initial state
                updateSelectAll();
            });
        </script>
        <?php
        
        echo '<p class="description">' . __('Leave all unchecked to include all pages.', 'fca-pages') . '</p>';
    }
    
    /**
     * Render menu priority field
     */
    public static function render_menu_priority_field() {
        $menu_priority = self::get_settings('menu_priority');
        ?>
        <input type="number" name="fca_pages_menu_settings[menu_priority]" value="<?php echo esc_attr($menu_priority); ?>" min="1" max="100" step="1" class="small-text">
        <p class="description"><?php _e('Lower numbers appear higher in the menu (default: 50)', 'fca-pages'); ?></p>
        <?php
    }
    
    /**
     * Render order section description
     */
    public static function render_order_section_description() {
        echo '<p>' . __('Drag and drop to reorder pages in the menu. This order overrides the default ordering.', 'fca-pages') . '</p>';
    }
    
    /**
     * Render page order field with drag and drop interface
     */
    public static function render_page_order_field() {
        $page_order = self::get_settings('page_order');
        $selected_pages = self::get_settings('selected_pages');
        
        // Get all published pages
        $pages = get_posts([
            'post_type' => 'fca_page',
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
        ]);
        
        if (empty($pages)) {
            echo '<p>' . __('No custom pages found. Create some pages first.', 'fca-pages') . '</p>';
            return;
        }
        
        // Filter pages if selected pages is set
        if (!empty($selected_pages)) {
            $filtered_pages = [];
            foreach ($pages as $page) {
                if (in_array($page->ID, $selected_pages)) {
                    $filtered_pages[] = $page;
                }
            }
            $pages = $filtered_pages;
        }
        
        // Order pages based on saved order
        if (!empty($page_order) && !empty($pages)) {
            $ordered_pages = [];
            
            // First, add pages that are in the saved order
            foreach ($page_order as $id) {
                foreach ($pages as $key => $page) {
                    if ($page->ID == $id) {
                        $ordered_pages[] = $page;
                        unset($pages[$key]);
                        break;
                    }
                }
            }
            
            // Then, add any remaining pages
            $pages = array_values($pages); // Re-index array
            foreach ($pages as $page) {
                $ordered_pages[] = $page;
            }
            
            $pages = $ordered_pages;
        }
        
        echo '<div class="fca-page-order-container">';
        echo '<ul id="fca-sortable-pages" class="fca-sortable-pages">';
        
        foreach ($pages as $page) {
            echo '<li data-id="' . esc_attr($page->ID) . '">';
            echo '<span class="fca-drag-handle dashicons dashicons-menu"></span>';
            echo esc_html($page->post_title);
            echo '<input type="hidden" name="fca_pages_menu_settings[page_order][]" value="' . esc_attr($page->ID) . '">';
            echo '</li>';
        }
        
        echo '</ul>';
        
        // Add JavaScript for sortable functionality
        ?>
        <script>
        jQuery(document).ready(function($) {
            $('#fca-sortable-pages').sortable({
                placeholder: 'fca-sortable-placeholder',
                update: function(event, ui) {
                    // Update all hidden inputs with the correct name after sorting
                    console.log('Sortable updated - updating hidden inputs');
                    $('#fca-sortable-pages li input[type="hidden"]').each(function() {
                        var pageId = $(this).data('id');
                        // Recreate the input to avoid any potential issues
                        $(this).find('input[type="hidden"]').remove();
                        $(this).append('<input type="hidden" name="fca_pages_menu_settings[page_order][]" value="' + pageId + '">');
                    });
                }
            });
            
            // Debug form submission
            $('form').on('submit', function() {
                var values = [];
                $('#fca-sortable-pages li input[type="hidden"]').each(function() {
                    values.push($(this).val());
                });
                console.log('Form submission - page_order values: ', values);
            });
        });
        </script>
        <?php
        
        echo '</div>';
        
        echo '<p class="description">' . __('Drag and drop pages to set their order in the menu.', 'fca-pages') . '</p>';
    }
} 