<?php
/*
Plugin Name: FCA Pages - Custom Pages for Fluent Community
Description: Adds custom, routable pages within the Fluent Community SPA.
Version: 1.0.2
Author: Fluent Community Add-ons
Author URI: https://www.fluentcommunityaddons.com
Text Domain: fca-pages
*/

defined('ABSPATH') or die;

// Add use statement for cleaner access if class exists
if (class_exists('\FluentCommunity\App\Helpers\Util')) {
    require_once ABSPATH . WPINC . '/pluggable.php'; // Ensure pluggable functions are available if needed early
    class_alias('\FluentCommunity\App\Helpers\Util', 'FcaFcUtil'); // Create an alias
}

class FCA_Pages {
    private static $instance = null;
    private $dummy_post_id = null; // Add property declaration here
    
    public static function instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // Register Custom Post Type 'fca_page'
        add_action('init', [$this, 'register_post_types'], 5); // Register CPTs earlier
        
        // Add the top-level admin menu
        add_action('admin_menu', [$this, 'add_admin_menu']);

        // Add pages to the Fluent Community sidebar menu
        add_filter('fluent_community/menu_groups_for_user', [$this, 'add_menu_items']);
        
        // Inject page data and routing logic into the Fluent Community portal head
        add_action('fluent_community/portal_head', [$this, 'inject_page_router']);
        
        // Inject custom styles for our elements AND block styles
        add_action('fluent_community/portal_head', [$this, 'inject_custom_styles']);
        add_action('fluent_community/portal_head', [$this, 'enqueue_portal_block_styles']);

        // Add meta boxes to the edit screen
        add_action('add_meta_boxes', [$this, 'add_meta_boxes']);

        // Save the meta box data
        add_action('save_post', [$this, 'save_meta_box_data']);

        // Force the block editor for our post type
        add_filter('use_block_editor_for_post_type', [$this, 'use_block_editor'], 10, 2);
        
        // Add back to Fluent Community link in admin bar
        add_action('admin_bar_menu', [$this, 'add_back_to_fc_link'], 100);
        
        // Add post-save notice on admin screen
        add_action('admin_notices', [$this, 'admin_notices']);
        
        // Initialize settings
        $this->init_settings();
        
        // Register Custom Blocks
        add_action('init', [$this, 'register_custom_blocks']);
        
        // Add custom block category
        add_filter('block_categories_all', [$this, 'add_block_category'], 10, 2);
        
        // Enqueue editor-specific styles (like the main FC theme styles)
        add_action('enqueue_block_editor_assets', [$this, 'enqueue_block_editor_styles']);
    }
    
    /**
     * Initialize plugin settings
     */
    private function init_settings() {
        // Load settings class
        require_once plugin_dir_path(__FILE__) . 'includes/class-fca-settings.php';
        
        // Initialize settings
        FCA_Settings::init();
    }

    /**
     * Register BOTH 'fca_page' and the dummy 'fca_editor_dummy' post types.
     */
    public function register_post_types() {
        // Args for the main fca_page CPT (Keep as is)
        $fca_page_args = [
            'labels' => [
                'name'               => _x('Custom Pages', 'post type general name', 'fca-pages'),
                'singular_name'      => _x('Custom Page', 'post type singular name', 'fca-pages'),
                'menu_name'          => _x('Custom Pages', 'admin menu', 'fca-pages'),
                'name_admin_bar'     => _x('Custom Page', 'add new on admin bar', 'fca-pages'),
                'add_new'            => _x('Add New', 'fca_page', 'fca-pages'),
                'add_new_item'       => __('Add New Custom Page', 'fca-pages'),
                'new_item'           => __('New Custom Page', 'fca-pages'),
                'edit_item'          => __('Edit Custom Page', 'fca-pages'),
                'view_item'          => __('View Custom Page', 'fca-pages'),
                'all_items'          => __('All Custom Pages', 'fca-pages'),
                'search_items'       => __('Search Custom Pages', 'fca-pages'),
                'parent_item_colon'  => __('Parent Custom Pages:', 'fca-pages'),
                'not_found'          => __('No custom pages found.', 'fca-pages'),
                'not_found_in_trash' => __('No custom pages found in Trash.', 'fca-pages')
            ],
            'description'       => __('Custom pages for display within Fluent Community.', 'fca-pages'),
            'public'            => false, // Keep false if accessed only via SPA/iframe
            'show_ui'           => true, // Needed for capabilities, but hidden via show_in_menu
            'show_in_menu'      => false,
            'show_in_nav_menus' => false,
            'show_in_rest'      => true, // Essential for block editor
            'query_var'         => true,
            'rewrite'           => ['slug' => 'fca-page-view', 'with_front' => false], // May not be needed if never viewed directly
            'capability_type'   => 'page',
            'has_archive'       => false,
            'hierarchical'      => false,
            'menu_position'     => null,
            'supports'          => ['title', 'editor', 'author', 'revisions', 'custom-fields', 'thumbnail'], // Added 'thumbnail'
            'menu_icon'         => 'dashicons-admin-page',
        ];
        register_post_type('fca_page', $fca_page_args);

        // Args for the dummy editor post type
        $dummy_args = [
            'label'        => 'FCA Editor Dummy',
            'public'       => false, // Not publicly visible
            'publicly_queryable' => false,
            'show_ui'      => false, // Hide from admin UI
            'show_in_menu' => false,
            'show_in_nav_menus' => false,
            'show_in_admin_bar' => false,
            'show_in_rest' => true, // MUST be true for editor
            'supports'     => ['title', 'editor'], // Only need title and editor support
            'capabilities' => [ // Use page capabilities so admins can interact via REST
                'edit_post'          => 'edit_pages',
                'read_post'          => 'read_pages',
                'delete_post'        => 'delete_pages',
                'edit_posts'         => 'edit_pages',
                'edit_others_posts'  => 'edit_others_pages',
                'publish_posts'      => 'publish_pages',
                'read_private_posts' => 'read_private_pages',
                'create_posts'       => 'edit_pages',
            ],
        ];
        register_post_type('fca_editor_dummy', $dummy_args);

        // Ensure our dummy post exists
        $this->ensure_dummy_post_exists();
    }

    /**
     * Find or create the single dummy post used for the editor.
     */
    private function ensure_dummy_post_exists() {
        $args = [
            'post_type'      => 'fca_editor_dummy',
            'post_status'    => 'draft', // Keep it as a draft
            'posts_per_page' => 1,
            'fields'         => 'ids',
        ];
        $existing = get_posts($args);

        if (!empty($existing)) {
            $this->dummy_post_id = $existing[0];
        } else {
            $post_data = [
                'post_title'   => 'FCA Editor Dummy Post',
                'post_content' => '<!-- wp:paragraph --><p>Dummy content holder.</p><!-- /wp:paragraph -->',
                'post_type'    => 'fca_editor_dummy',
                'post_status'  => 'draft',
                'comment_status' => 'closed',
                'ping_status'    => 'closed',
            ];
            // Important: Temporarily remove save_post hook to prevent infinite loop during dummy creation
            remove_action('save_post', [$this, 'save_meta_box_data']); 
            $new_id = wp_insert_post($post_data, true);
            // Re-add the hook immediately after
            add_action('save_post', [$this, 'save_meta_box_data']); 
            
            if (!is_wp_error($new_id)) {
                $this->dummy_post_id = $new_id;
            } else {
                 error_log('FCA Pages Error: Failed to create dummy post - ' . $new_id->get_error_message());
            }
        }
        // Store it as an option for persistence? Maybe not needed if checked on init.
    }

    /**
     * Inject page data and routing logic into the Fluent Community portal head
     */
    public function inject_page_router() {
        // Fetch PUBLISHED pages for the dynamic viewer component
        $published_pages = get_posts([
            'post_type' => 'fca_page',
            'post_status' => 'publish',
            'numberposts' => -1,
        ]);
        
        $pages_data = [];
        
        foreach ($published_pages as $page) {
            // Apply filters to the content to ensure shortcodes are processed
            $content = apply_filters('the_content', $page->post_content);
            
            // Get layout settings (Desktop)
            $content_width = get_post_meta($page->ID, '_fca_content_width', true) ?: '100';
            $content_width_unit = get_post_meta($page->ID, '_fca_content_width_unit', true) ?: '%';
            $content_alignment = get_post_meta($page->ID, '_fca_content_alignment', true) ?: 'center';

            // Get layout settings (Tablet)
            $content_width_tablet = get_post_meta($page->ID, '_fca_content_width_tablet', true) ?: '100';
            $content_width_unit_tablet = get_post_meta($page->ID, '_fca_content_width_unit_tablet', true) ?: '%';
            
            // Get layout settings (Mobile)
            $content_width_mobile = get_post_meta($page->ID, '_fca_content_width_mobile', true) ?: '100';
            $content_width_unit_mobile = get_post_meta($page->ID, '_fca_content_width_unit_mobile', true) ?: '%';
            
            // Wrap content in the container div (style will be applied via JS/CSS variables)
            $content = '<div class="fcom_page_content fcom_md_content">' . $content . '</div>';
            
            $pages_data[$page->post_name] = [
                'id' => $page->ID,
                'title' => $page->post_title,
                'slug' => $page->post_name,
                'content' => $content, 
                'layout' => [
                    'desktop' => [
                        'width' => $content_width,
                        'unit' => $content_width_unit,
                        'align' => $content_alignment
                    ],
                    'tablet' => [
                        'width' => $content_width_tablet,
                        'unit' => $content_width_unit_tablet
                    ],
                    'mobile' => [
                        'width' => $content_width_mobile,
                        'unit' => $content_width_unit_mobile
                    ]
                    // Note: Alignment is assumed to be the same across breakpoints for now
                ]
                // ... other data if needed by viewer component ...
            ];
        }
        
        // Get REST API base URL and Nonce
        $rest_api_url = esc_url_raw(rest_url('/wp/v2/fca_page'));
        $rest_nonce = wp_create_nonce('wp_rest'); 
        
        // Get Editor Endpoint URL
        $editor_endpoint_url = esc_url_raw(home_url('/?fca_page_editor=1')); 

        // Get Correct Home URL (already present)
        $correct_home_url = esc_url_raw(home_url('/')); 
        
        // Use Util::baseUrl for JS if available, otherwise construct manually
        $js_base_url = home_url('/');
        if (class_exists('FcaFcUtil')) {
            $js_base_url = FcaFcUtil::baseUrl(); // Get base without path
        }
        
        ?>
        <style>
        /* ... existing styles ... */

        /* Styles for the Editor Component */
        .fca-page-editor-container {
            display: flex;
            flex-direction: column;
            height: calc(100vh - var(--fcom-header-height, 65px));
            overflow: hidden;
        }
        .fca-page-editor-header {
             padding: 10px 20px;
             border-bottom: 1px solid var(--fcom-primary-border, #eee);
             display: flex;
             justify-content: space-between;
             align-items: center;
             background-color: var(--fcom-secondary-bg, #f9f9f9);
             flex-shrink: 0; /* Prevent header from shrinking */
        }
        .fca-page-editor-header h2 {
             margin: 0;
             font-size: 1.1em;
        }
        .fca-page-editor-redirect {
            flex-grow: 1; 
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 40px;
            font-size: 1.2em;
            color: #555;
            background-color: #f9f9f9;
        }
        </style>
        <script id="fca-pages-router-script">
        (function() {
            // Store published page data for the viewer component
            window.fcaPagesData = <?php echo wp_json_encode($pages_data); ?>;
            // Store correct base URL for constructing view links
            window.fcaBaseUrl = <?php echo wp_json_encode($js_base_url); ?>; 
            // Store portal slug for dynamic URL construction
            window.fcaPortalSlug = <?php echo wp_json_encode(apply_filters('fluent_community/portal_slug', 'portal')); ?>;
            
            // Add function to reload page data from server
            window.fcaReloadPageData = function(slug) {
                console.log('FCA Pages: Reloading page data for:', slug);
                const apiUrl = '<?php echo esc_url_raw(rest_url('/wp/v2/fca_page')); ?>';
                
                // Clear any cache
                fetch(`${apiUrl}?slug=${slug}&_=${Date.now()}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data && data.length > 0) {
                            const page = data[0];
                            
                            // Create a temporary div to hold the content for processing
                            const tempDiv = document.createElement('div');
                            tempDiv.innerHTML = page.content.rendered;
                            
                            // Process the content - add any styles or processing here
                            const processedContent = tempDiv.innerHTML;
                            
                            // Update the global page data
                            if (!window.fcaPagesData) window.fcaPagesData = {};
                            
                            // Create or update page data with proper styling
                            window.fcaPagesData[slug] = {
                                id: page.id,
                                title: page.title.rendered,
                                slug: page.slug,
                                content: processedContent
                            };
                            
                            console.log('FCA Pages: Page data reloaded from API:', page.title.rendered);
                            
                            // Reload the page to show updated content
                            window.location.reload();
                        }
                    })
                    .catch(error => {
                        console.error('FCA Pages: Error reloading page data:', error);
                    });
            };
            
            // Helper function to update menu active state
            function updateFcaMenuActiveState(activeSlug) {
                const menuLinks = document.querySelectorAll('.fca-page-link, .fca-page-settings-link');
                menuLinks.forEach(link => {
                    link.classList.remove('fca-menu-is-active');
                    if (activeSlug && link.href) {
                         const linkPath = new URL(link.href).pathname;
                         let expectedPath;
                         if (activeSlug === 'settings') {
                            expectedPath = '/' + window.fcaPortalSlug + '/pages/settings'; // Use stored dynamic portal slug
                         } else {
                            expectedPath = '/' + window.fcaPortalSlug + '/page/' + activeSlug; // Use stored dynamic portal slug
                         }
                         // Check if href ends with the expected path segment
                         if (linkPath.endsWith(expectedPath)) { 
                            link.classList.add('fca-menu-is-active');
                         }
                    }
                });
            }

            // --- Define Vue Components ---

            // Settings Page Component
            const FcaPageSettingsComponent = {
                template: `
                    <div class="el-scrollbar fca-pages-settings-scrollbar"> 
                        <div class="el-scrollbar__wrap" style="height: calc(100vh - var(--fcom-header-height, 65px)); overflow-y: auto;"> 
                            <div class="el-scrollbar__view">  
                                <div class="fca-page-settings-container">
                                    <div class="fca-page-settings-header">
                                         <h2>Custom Pages</h2>
                                         <button @click="addNewPage" class="el-button el-button--primary el-button--small">Add New Page</button> 
                                    </div>

                                    <div v-if="loading" style="text-align: center; padding: 40px; color: #888;">Loading pages...</div>
                                    <div v-else-if="error" style="text-align: center; padding: 40px; color: red;">{{ error }}</div>
                                    <ul v-else-if="pages.length > 0" class="fca-page-settings-list">
                                        <li v-for="page in pages" :key="page.id">
                                            <div class="page-info">
                                                <span class="page-title">{{ page.title.rendered || page.id }}</span>
                                                <span class="page-status">Status: {{ page.status }}</span>
                                            </div>
                                            <div class="page-actions">
                                                <button @click="editPage(page.id)" class="el-button el-button--default el-button--small">Edit</button>
                                                <a :href="getViewUrl(page.slug)" target="_blank" class="el-button el-button--default el-button--small">View</a>
                                                <button v-if="page.status !== 'trash'" @click="deletePage(page.id)" class="el-button el-button--danger el-button--small">Trash</button>
                                                <button v-else @click="deletePage(page.id, true)" class="el-button el-button--danger el-button--small">Delete Permanently</button>
                                                <!-- Add Restore button if needed -->
                                            </div>
                                        </li>
                                    </ul>
                                    <div v-else style="text-align: center; padding: 40px; color: #888;">No custom pages found.</div>
                                </div>
                            </div>
                        </div>
                        <div class="el-scrollbar__bar is-horizontal" style="display: none;"><div class="el-scrollbar__thumb" style="transform: translateX(0%);"></div></div>
                        <div class="el-scrollbar__bar is-vertical"><div class="el-scrollbar__thumb" style="transform: translateY(0%);"></div></div>
                    </div>
                `,
                data: function() {
                    return {
                        pages: [],
                        loading: true,
                        error: null,
                        portalBase: window.FluentCommunityUtil?.appData?.config?.portal_base || 'portal' 
                    };
                },
                methods: {
                    fetchPages: async function() {
                        this.loading = true;
                        this.error = null;
                        try {
                            // Fetch all statuses, only necessary fields
                            const response = await fetch('<?php echo $rest_api_url; ?>?status=any&_fields=id,title,slug,status&context=edit&_wpnonce=<?php echo $rest_nonce; ?>&_=' + Date.now(), { // Add nonce and cache buster
                                headers: {
                                    'Accept': 'application/json',
                                    // Nonce is often needed even for GET in WP REST API for 'edit' context or status=any
                                    // 'X-WP-Nonce': '<?php echo $rest_nonce; ?>' // Included in URL now
                                }
                            });
                            if (!response.ok) {
                                let errorMsg = `Error fetching pages: ${response.statusText}`;
                                try {
                                     const errorData = await response.json();
                                     errorMsg += ` - ${errorData.message || JSON.stringify(errorData)}`;
                                } catch(e) {}
                                throw new Error(errorMsg);
                            }
                            this.pages = await response.json();
                            console.log('FCA Pages Settings: Pages fetched:', this.pages);
                        } catch (err) {
                            console.error('FCA Pages Settings: Error fetching pages:', err);
                            this.error = err.message;
                            this.pages = []; // Clear pages on error
                        } finally {
                            this.loading = false;
                        }
                    },
                    editPage: function(pageId) {
                        console.log('FCA Pages Settings: Navigate to edit page:', pageId);
                        const adminUrl = '<?php echo admin_url('post.php?post='); ?>' + pageId + '&action=edit';
                        window.location.href = adminUrl;
                    },
                    addNewPage: function() {
                        console.log('FCA Pages Settings: Navigate to add new page.');
                        const adminUrl = '<?php echo admin_url('post-new.php?post_type=fca_page'); ?>';
                        window.location.href = adminUrl;
                    },
                     getViewUrl: function(slug) {
                         // Construct the frontend view URL using the dynamic portal slug
                         if (!slug) return '#';
                         const base = window.fcaBaseUrl || ''; 
                         const baseUrl = base.endsWith('/') ? base : base + '/'; 
                         const portalSlug = window.fcaPortalSlug || 'portal'; // Use stored portal slug with fallback
                         const finalUrl = (baseUrl + portalSlug + '/page/' + slug).replace(/([^:]\/)\/+/g, "$1");
                         return finalUrl;
                     },
                    deletePage: async function(pageId, force = false) {
                        console.log('FCA Pages Settings: Delete page requested:', pageId, 'Force:', force);
                        if (!confirm(`Are you sure you want to ${force ? 'permanently delete' : 'trash'} this page?`)) {
                    return;
                }
                
                        try {
                            const deleteUrl = `<?php echo $rest_api_url; ?>/${pageId}?_wpnonce=<?php echo $rest_nonce; ?>${force ? '&force=true' : ''}`;
                            const response = await fetch(deleteUrl, {
                                method: 'DELETE',
                                headers: {
                                    // 'X-WP-Nonce': '<?php echo $rest_nonce; ?>' // Included in URL now
                                }
                            });

                            if (!response.ok) {
                                let errorMsg = `Error deleting page: ${response.statusText}`;
                                try {
                                     const errorData = await response.json();
                                     errorMsg += ` - ${errorData.message || JSON.stringify(errorData)}`;
                                } catch(e) {}
                                throw new Error(errorMsg);
                            }
                            
                            // Successfully deleted/trashed, refresh list
                            this.fetchPages(); 

                        } catch (err) {
                            console.error('FCA Pages Settings: Error deleting page:', err);
                            alert(`Failed to delete page: ${err.message}`);
                        }
                    }
                },
                mounted: function() {
                    console.log('FCA Pages Settings: Component mounted.');
                    this.fetchPages();
                    updateFcaMenuActiveState('settings'); // Activate menu item
                    document.title = 'Page Settings | ' + (window.fluentFramework?.app?.config?.globalProperties?.$t('Fluent Community') || 'Fluent Community');
                },
                 beforeUnmount: function() {
                    updateFcaMenuActiveState(null); // Deactivate menu item
                 }
            };

            // Page Viewer Component (existing logic)
            const FcaPageViewerComponent = {
                            template: `
                                <div class="el-scrollbar fca-pages-scrollbar"> 
                                    <div class="el-scrollbar__wrap" style="height: calc(100vh - var(--fcom-header-height, 65px)); overflow-y: auto;"> 
                                        <div class="el-scrollbar__view">  
                                <div v-if="page" :key="page.slug" class="fca-page-content-loaded"> 
                                                <div class="fhr_content_layout_header"> 
                                                    <h1 role="region" aria-label="Page Title" class="fhr_page_title">
                                                        {{ page.title }}
                                                    </h1>
                                                    <div role="region" aria-label="Actions" class="fhr_page_actions"></div>
                                                </div>
                                                <div class="fca-page-body entry-content" v-html="page.content" style="line-height: 1.7; color: #444; padding: 25px;"></div>
                                            </div>
                                            <div v-else class="fca-page-loading" style="text-align: center; padding: 40px; color: #888;">
                                                <p>{{ loadingMessage }}</p>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="el-scrollbar__bar is-horizontal" style="display: none;"><div class="el-scrollbar__thumb" style="transform: translateX(0%);"></div></div>
                                    <div class="el-scrollbar__bar is-vertical"><div class="el-scrollbar__thumb" style="transform: translateY(0%);"></div></div>
                                </div>
                            `,
                data: function() {
                    return {
                        page: null,
                        loadingMessage: 'Loading page...',
                        error: null,
                        resizeHandler: null // Store resize handler for removal
                    };
                },
                methods: {
                    loadPageContent: function() {
                        const pageSlug = this.$route.params.pageSlug;
                        console.log('FCA Pages Viewer: Loading page with slug:', pageSlug);
                        
                        if (!pageSlug) {
                            this.loadingMessage = 'Error: No page specified';
                            return;
                        }
                        
                        const forceRefresh = this.shouldForceRefresh();
                        
                        if (!forceRefresh && window.fcaPagesData && window.fcaPagesData[pageSlug]) {
                            this.page = window.fcaPagesData[pageSlug];
                            console.log('FCA Pages Viewer: Page loaded from preloaded data:', this.page.title);
                            
                            this.updateDocumentTitle();
                            updateFcaMenuActiveState(pageSlug);
                            this.updatePageLayoutStyles(); // Apply styles
                        } else {
                            this.loadingMessage = forceRefresh ? 'Refreshing page content...' : 'Loading page content...';
                            console.log('FCA Pages Viewer: ' + (forceRefresh ? 'Force refreshing page:' : 'Fetching page from API:'), pageSlug);
                            this.fetchPageFromAPI(pageSlug);
                        }
                    },
                    
                    shouldForceRefresh: function() {
                        const urlParams = new URLSearchParams(window.location.search);
                        return urlParams.has('nocache');
                    },
                    
                    fetchPageFromAPI: function(pageSlug) {
                        const apiUrl = '<?php echo esc_url_raw(rest_url('/wp/v2/fca_page')); ?>';
                        
                        // Fetch *all* data including meta now, simplify logic
                        fetch(`${apiUrl}?slug=${pageSlug}&_embed&context=view&_wpnonce=<?php echo wp_create_nonce('wp_rest'); ?>&_=${Date.now()}`)
                            .then(response => {
                                if (!response.ok) {
                                    throw new Error(`API error: ${response.status}`);
                                }
                                return response.json();
                            })
                            .then(data => {
                                if (data && data.length > 0) {
                                    const pageApiData = data[0];
                                    this.processPageData(pageApiData);
                                } else {
                                    this.loadingMessage = 'Page not found';
                                    console.error('FCA Pages Viewer: Page not found in API response');
                                }
                            })
                            .catch(error => {
                                this.loadingMessage = 'Error loading page: ' + error.message;
                                console.error('FCA Pages Viewer: Error fetching page from API:', error);
                            });
                    },
                    
                    processPageData: function(pageApiData) {
                         // Get layout settings from meta
                         const meta = pageApiData.meta || {};
                         const layout = {
                             desktop: {
                                 width: meta._fca_content_width || '100',
                                 unit: meta._fca_content_width_unit || '%',
                                 align: meta._fca_content_alignment || 'center'
                             },
                             tablet: {
                                 width: meta._fca_content_width_tablet || '100',
                                 unit: meta._fca_content_width_unit_tablet || '%'
                             },
                             mobile: {
                                 width: meta._fca_content_width_mobile || '100',
                                 unit: meta._fca_content_width_unit_mobile || '%'
                             }
                         };
                         
                         // Wrap content (assuming it's already wrapped from PHP injection)
                         const content = pageApiData.content.rendered;
                         
                         // Create page object with processed content and layout
                         this.page = {
                             id: pageApiData.id,
                             title: pageApiData.title.rendered,
                             slug: pageApiData.slug,
                             content: content,
                             layout: layout
                         };
                         
                         // Store in global cache if needed (might overwrite initial PHP data)
                         if (window.fcaPagesData) {
                             window.fcaPagesData[pageApiData.slug] = this.page;
                         }
                         
                         console.log('FCA Pages Viewer: Page processed with layout data:', this.page.title);
                         
                         this.updateDocumentTitle();
                         updateFcaMenuActiveState(pageApiData.slug);
                         this.updatePageLayoutStyles(); // Apply styles
                    },

                    updateDocumentTitle: function() {
                         if (!this.page) return;
                         document.title = this.page.title + ' | ' + 
                                     (window.fluentFramework?.app?.config?.globalProperties?.$t('Fluent Community') || 'Fluent Community');
                    },

                    // NEW Method: Update Layout Styles via CSS Variables
                    updatePageLayoutStyles: function() {
                        if (!this.page || !this.page.layout) {
                             // Reset to defaults if no layout data
                            document.documentElement.style.setProperty('--fca-current-page-width', '100%');
                            document.documentElement.style.setProperty('--fca-current-page-margin-left', 'auto');
                            document.documentElement.style.setProperty('--fca-current-page-margin-right', 'auto');
                            return; 
                         }

                        const layout = this.page.layout;
                        const screenWidth = window.innerWidth;
                        const tabletBreakpoint = parseInt(getComputedStyle(document.documentElement).getPropertyValue('--fca-tablet-breakpoint') || '992');
                        const mobileBreakpoint = parseInt(getComputedStyle(document.documentElement).getPropertyValue('--fca-mobile-breakpoint') || '768');
                        
                        let effectiveLayout;
                        if (screenWidth < mobileBreakpoint) {
                            effectiveLayout = layout.mobile;
                        } else if (screenWidth < tabletBreakpoint) {
                            effectiveLayout = layout.tablet;
                        } else {
                            effectiveLayout = layout.desktop;
                        }
                        
                        // Use desktop alignment setting for all breakpoints for now
                        const alignment = layout.desktop.align;
                        const width = effectiveLayout.width + effectiveLayout.unit;
                        
                        let marginLeft = 'auto';
                        let marginRight = 'auto';
                        
                        if (alignment === 'left') {
                            marginLeft = '0';
                            marginRight = 'auto';
                        } else if (alignment === 'right') {
                            marginLeft = 'auto';
                            marginRight = '0';
                        } // 'center' uses the defaults (auto, auto)
                        
                        // Set CSS Variables
                        document.documentElement.style.setProperty('--fca-current-page-width', width);
                        document.documentElement.style.setProperty('--fca-current-page-margin-left', marginLeft);
                        document.documentElement.style.setProperty('--fca-current-page-margin-right', marginRight);
                        
                        // console.log(`Applied width: ${width}, align: ${alignment} for screen ${screenWidth}px`);
                    }
                },
                mounted: function() {
                    console.log('FCA Pages Viewer: Component mounted');
                    this.loadPageContent();
                    // Add resize listener
                    this.resizeHandler = () => this.updatePageLayoutStyles();
                    window.addEventListener('resize', this.resizeHandler);
                },
                beforeUnmount: function() {
                    updateFcaMenuActiveState(null); // Deactivate menu item
                    // Remove resize listener
                    if (this.resizeHandler) {
                        window.removeEventListener('resize', this.resizeHandler);
                    }
                    // Optional: Reset styles on unmount?
                    // document.documentElement.style.removeProperty('--fca-current-page-width');
                    // ... etc
                },
                watch: {
                    '$route.params.pageSlug': function(newSlug, oldSlug) {
                        if (newSlug !== oldSlug) {
                            console.log('FCA Pages Viewer: Route changed, loading new page:', newSlug);
                            this.loadPageContent();
                        }
                    }
                }
            };

            // NEW: Editor Component
            const FcaPageEditorComponent = {
                template: `
                    <div class="fca-page-editor-container">
                        <div class="fca-page-editor-header">
                            <h2>{{ pageId == 0 ? 'Add New Page' : 'Edit Page' }}</h2>
                             <div>
                                <button @click="goBack" class="el-button el-button--default el-button--small">Back to Pages</button>
                             </div>
                        </div>
                        <div class="fca-page-editor-redirect">
                            <p>Redirecting to WordPress admin editor...</p>
                        </div>
                    </div>
                `,
                props: ['pageId'], // Get pageId from route params
                            data: function() {
                                return {
                        adminUrl: null
                                };
                            },
                            methods: {
                    goBack: function() {
                        if (this.$router) {
                            this.$router.push({ name: 'fca_page_settings' });
                        }
                    },
                    redirectToAdmin: function() {
                        const postId = parseInt(this.pageId) || 0;
                        console.log('FCA Page Editor: Redirecting to WordPress admin for post ID:', postId);
                        
                        // Construct the URL for WordPress admin editor
                        let adminUrl = '';
                        if (postId > 0) {
                            // Edit existing page
                            adminUrl = '<?php echo admin_url('post.php?post='); ?>' + postId + '&action=edit';
                        } else {
                            // Add new page
                            adminUrl = '<?php echo admin_url('post-new.php?post_type=fca_page'); ?>';
                        }
                        
                        // Redirect to WordPress admin
                        window.location.href = adminUrl;
                                }
                            },
                            mounted: function() {
                     console.log('FCA Page Editor: Component mounted for page ID:', this.pageId);
                    document.title = (this.pageId == 0 ? 'Add New Page' : 'Edit Page') + ' | ' + 
                        (window.fluentFramework?.app?.config?.globalProperties?.$t('Fluent Community') || 'Fluent Community');
                    
                    // Redirect to WordPress admin after a short delay
                    setTimeout(this.redirectToAdmin, 500);
                 }
            };


            // --- Register Routes ---
            document.addEventListener("fluentCommunityUtilReady", function () {
                if (!window.FluentCommunityUtil || !window.FluentCommunityUtil.hooks) {
                    console.error('FCA Pages: FluentCommunityUtil not found.');
                    return;
                }
                
                console.log('FCA Pages: Initializing routes.');

                window.FluentCommunityUtil.hooks.addFilter("fluent_com_portal_routes", "fca_pages_routes", function (routes) {
                    if (!Array.isArray(routes)) {
                         console.error('FCA Pages: Expected routes to be an array.');
                         return routes; 
                    }

                    // Settings List Route
                    routes.push({
                        path: "/pages/settings", 
                        name: "fca_page_settings",
                        component: FcaPageSettingsComponent,
                        meta: { active: "fca_page_settings", parent: "fca_pages_group" }
                    });

                    // Dynamic Page Viewer Route
                    routes.push({
                        path: "/page/:pageSlug", 
                        name: "fca_custom_page",
                        component: FcaPageViewerComponent,
                        meta: { active: "fca_custom_page", parent: "fca_pages_group" }
                    });

                    console.log('FCA Pages: Custom routes successfully added.');
                    return routes; 
                });
            });
        })();
        </script>
        <?php
    }
    
    /**
     * Inject custom styles for our elements 
     */
    public function inject_custom_styles() {
         // You might have specific styles needed for viewer components or settings page
         ?>
         <style id="fca-pages-custom-styles">
             /* Style for content rendered via v-html */
             .fca-page-body .fcom_md_content {
                 /* Add styles if needed */
             }
             
             /* Define CSS Variables for Page Layout */
             :root { /* Apply globally or scope to viewer container if needed */
                 --fca-current-page-width: 100%; 
                 --fca-current-page-margin-left: auto;
                 --fca-current-page-margin-right: auto;
                 --fca-tablet-breakpoint: 992px; /* Example */
                 --fca-mobile-breakpoint: 768px; /* Example */
             }
             
             /* Apply Variables to the Page Content Container */
             .fcom_page_content.fcom_md_content {
                 width: var(--fca-current-page-width);
                 max-width: 100%; /* Prevent overflow */
                 margin-left: var(--fca-current-page-margin-left);
                 margin-right: var(--fca-current-page-margin-right);
                 box-sizing: border-box; /* Include padding/border in width */
             }

             /* Styles for the Settings Page List */
             .fca-page-settings-list {
                 list-style: none;
                 padding: 0;
                 margin: 0;
             }
             .fca-page-settings-list li {
                 display: flex;
                 justify-content: space-between;
                 align-items: center;
                 padding: 15px 20px;
                 border-bottom: 1px solid var(--fcom-primary-border, #eee);
             }
             .fca-page-settings-list li:last-child {
                 border-bottom: none;
             }
             .fca-page-settings-list .page-info {
                 flex-grow: 1;
                 margin-right: 15px;
             }
             .fca-page-settings-list .page-info .page-title {
                font-weight: 600;
                 color: var(--fcom-primary-text);
                 display: block;
             }
             .fca-page-settings-list .page-info .page-status {
                 font-size: 0.9em;
                 color: var(--fcom-secondary-text);
                 text-transform: capitalize;
             }
             .fca-page-settings-list .page-actions button,
             .fca-page-settings-list .page-actions a {
                 margin-left: 8px;
                font-size: 0.9em;
                 padding: 5px 10px;
                 cursor: pointer;
             }
             .fca-page-settings-header {
                 padding: 15px 20px;
                 border-bottom: 1px solid var(--fcom-primary-border, #eee);
                display: flex;
                justify-content: space-between;
                 align-items: center;
            }
             .fca-page-settings-header h2 {
                margin: 0;
                 font-size: 1.2em;
             }
              /* Styles for the Editor Component */
             .fca-page-editor-container {
                 display: flex;
                 flex-direction: column;
                 height: calc(100vh - var(--fcom-header-height, 65px));
                 overflow: hidden;
             }
             .fca-page-editor-header {
                 padding: 10px 20px;
                 border-bottom: 1px solid var(--fcom-primary-border, #eee);
                display: flex;
                 justify-content: space-between;
                 align-items: center;
                 background-color: var(--fcom-secondary-bg, #f9f9f9);
                 flex-shrink: 0;
             }
             .fca-page-editor-header h2 {
                 margin: 0;
                 font-size: 1.1em;
             }
             .fca-page-editor-redirect {
                 flex-grow: 1; 
                 display: flex;
                 justify-content: center;
                 align-items: center;
                 padding: 40px;
                 font-size: 1.2em;
                 color: #555;
                 background-color: #f9f9f9;
             }
         </style>
         <?php
     }

    /**
     * Add the top-level "Custom Pages" admin menu.
     */
    public function add_admin_menu() {
        add_menu_page(
            __('Custom Pages', 'fca-pages'),          // Page title
            __('Custom Pages', 'fca-pages'),          // Menu title
            'edit_pages',                             // Capability required
            'fca-pages-admin',                        // Menu slug
            [$this, 'render_admin_page'],             // Callback function
            'dashicons-admin-page',                   // Icon
            26                                        // Position (adjust as needed)
        );

        // Add submenu link to the standard "All Pages" list for fca_page
        add_submenu_page(
            'fca-pages-admin',                       // Parent slug
            __('All Custom Pages', 'fca-pages'),     // Page title
            __('All Pages', 'fca-pages'),            // Menu title
            'edit_pages',                            // Capability
            'edit.php?post_type=fca_page'            // Standard WP list table URL
        );

        // Add submenu link to the standard "Add New" page for fca_page
        add_submenu_page(
            'fca-pages-admin',                       // Parent slug
            __('Add New Custom Page', 'fca-pages'),  // Page title
            __('Add New', 'fca-pages'),             // Menu title
            'edit_pages',                           // Capability 
            'post-new.php?post_type=fca_page'       // Standard WP add new URL
        );
    }

    /**
     * Render the admin page 
     */
    public function render_admin_page() {
        ?>
        <div class="wrap">
            <h1><?php _e('FCA Custom Pages', 'fca-pages'); ?></h1>
            <p><?php _e('Manage your custom pages for Fluent Community.', 'fca-pages'); ?></p>
             <!-- Removed table, linking to standard WP lists -->
            <p>
                 <a href="<?php echo admin_url('edit.php?post_type=fca_page'); ?>" class="button button-primary"><?php _e('View All Pages', 'fca-pages'); ?></a>
                 <a href="<?php echo admin_url('post-new.php?post_type=fca_page'); ?>" class="button"><?php _e('Add New Page', 'fca-pages'); ?></a>
                 <a href="<?php echo admin_url('admin.php?page=fca-pages-settings'); ?>" class="button"><?php _e('Menu Settings', 'fca-pages'); ?></a>
            </p>
            <p><em><?php _e('Pages created here will be displayed within your Fluent Community portal.', 'fca-pages'); ?></em></p>
             
             <?php
                // Example: Show a quick list with correct view links
                 $pages = get_posts([
                     'post_type' => 'fca_page',
                     'post_status' => 'any',
                     'numberposts' => 5, // Show recent 5
                     'orderby' => 'date',
                     'order' => 'DESC',
                 ]);
                 if (!empty($pages)) {
                    echo '<h2>' . __('Recent Pages', 'fca-pages') . '</h2>';
                    echo '<ul>';
                    // Check using correct namespace
                    $fc_helper_exists = class_exists('\FluentCommunity\App\Services\Helper'); 
                    $portal_slug = $fc_helper_exists ? '' : apply_filters('fluent_community/portal_slug', 'portal'); 
                    foreach ($pages as $page) {
                         $edit_url = get_edit_post_link($page->ID);
                         echo '<li>';
                         echo '<strong>' . esc_html($page->post_title) . '</strong> (' . esc_html(get_post_status_object($page->post_status)->label) . ')';
                         echo ' - <a href="' . esc_url($edit_url) . '">Edit (WP Admin)</a>';
                         if ($page->post_status == 'publish' && !empty($page->post_name)) {
                            $page_path = "/page/{$page->post_name}";
                            $view_url = $fc_helper_exists 
                                // Use correct namespace
                                ? \FluentCommunity\App\Services\Helper::baseUrl($page_path) 
                                : home_url("/{$portal_slug}{$page_path}");
                            echo ' | <a href="' . esc_url($view_url) . '" target="_blank">View (Portal)</a>';
                         }
                         echo '</li>';
                     }
                     echo '</ul>';
                 }
             ?>
        </div>
        <?php
    }

    /**
     * Add meta boxes to the edit screen
     */
    public function add_meta_boxes() {
        add_meta_box(
            'fca_page_settings_meta_box',             // ID
            __('Page Layout Settings', 'fca-pages'), // Title
            [$this, 'render_settings_meta_box'],     // Callback
            'fca_page',                               // Post type
            'side',                                   // Context (normal, side, advanced)
            'default'                                 // Priority (high, core, default, low)
        );
    }

    /**
     * Render the settings meta box
     */
    public function render_settings_meta_box($post) {
        // Add a nonce field for security
        wp_nonce_field('fca_page_save_meta_box', 'fca_page_meta_box_nonce');

        // Retrieve existing values
        $content_width = get_post_meta($post->ID, '_fca_content_width', true) ?: '100';
        $content_width_unit = get_post_meta($post->ID, '_fca_content_width_unit', true) ?: '%';
        $content_alignment = get_post_meta($post->ID, '_fca_content_alignment', true) ?: 'center';
        
        // Retrieve new responsive values
        $content_width_tablet = get_post_meta($post->ID, '_fca_content_width_tablet', true) ?: '100';
        $content_width_unit_tablet = get_post_meta($post->ID, '_fca_content_width_unit_tablet', true) ?: '%';
        $content_width_mobile = get_post_meta($post->ID, '_fca_content_width_mobile', true) ?: '100';
        $content_width_unit_mobile = get_post_meta($post->ID, '_fca_content_width_unit_mobile', true) ?: '%';

        ?>
        <p>
            <strong><?php _e('Desktop', 'fca-pages'); ?></strong><br>
            <label for="fca_content_width" class="screen-reader-text"><?php _e('Content Width (Desktop):', 'fca-pages'); ?></label>
            <input type="number" id="fca_content_width" name="fca_content_width" value="<?php echo esc_attr($content_width); ?>" min="1" style="width: 60px;">
            <select id="fca_content_width_unit" name="fca_content_width_unit">
                    <option value="%" <?php selected($content_width_unit, '%'); ?>>%</option>
                    <option value="px" <?php selected($content_width_unit, 'px'); ?>>px</option>
                </select>
            </p>
            
            <p>
            <strong><?php _e('Tablet', 'fca-pages'); ?></strong> (<?php _e('e.g., < 992px', 'fca-pages'); ?>)<br>
            <label for="fca_content_width_tablet" class="screen-reader-text"><?php _e('Content Width (Tablet):', 'fca-pages'); ?></label>
            <input type="number" id="fca_content_width_tablet" name="fca_content_width_tablet" value="<?php echo esc_attr($content_width_tablet); ?>" min="1" style="width: 60px;">
            <select id="fca_content_width_unit_tablet" name="fca_content_width_unit_tablet">
                    <option value="%" <?php selected($content_width_unit_tablet, '%'); ?>>%</option>
                    <option value="px" <?php selected($content_width_unit_tablet, 'px'); ?>>px</option>
                </select>
            </p>

            <p>
            <strong><?php _e('Mobile', 'fca-pages'); ?></strong> (<?php _e('e.g., < 768px', 'fca-pages'); ?>)<br>
            <label for="fca_content_width_mobile" class="screen-reader-text"><?php _e('Content Width (Mobile):', 'fca-pages'); ?></label>
            <input type="number" id="fca_content_width_mobile" name="fca_content_width_mobile" value="<?php echo esc_attr($content_width_mobile); ?>" min="1" style="width: 60px;">
            <select id="fca_content_width_unit_mobile" name="fca_content_width_unit_mobile">
                    <option value="%" <?php selected($content_width_unit_mobile, '%'); ?>>%</option>
                    <option value="px" <?php selected($content_width_unit_mobile, 'px'); ?>>px</option>
                </select>
            </p>

            <hr style="margin: 1em 0;">

            <p>
            <label for="fca_content_alignment"><strong><?php _e('Content Alignment:', 'fca-pages'); ?></strong></label><br>
            <select id="fca_content_alignment" name="fca_content_alignment">
                    <option value="left" <?php selected($content_alignment, 'left'); ?>><?php _e('Left', 'fca-pages'); ?></option>
                <option value="center" <?php selected($content_alignment, 'center'); ?>><?php _e('Center', 'fca-pages'); ?></option>
                    <option value="right" <?php selected($content_alignment, 'right'); ?>><?php _e('Right', 'fca-pages'); ?></option>
                </select>
            </p>
         <p class="description">
             <?php _e('Adjust the maximum width and alignment for different screen sizes. Widths apply up to the specified breakpoint.', 'fca-pages'); ?>
         </p>
        <?php
    }

    /**
     * Save the meta box data
     */
    public function save_meta_box_data($post_id) {
        // Check if nonce is set and valid
        if (!isset($_POST['fca_page_meta_box_nonce']) || !wp_verify_nonce($_POST['fca_page_meta_box_nonce'], 'fca_page_save_meta_box')) {
            return;
        }
        
        // Ignore autosaves and checks for correct post type
        // Check if the current user has permission to edit the post. 
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        if (!isset($_POST['post_type']) || 'fca_page' !== $_POST['post_type']) return; 
        if (!current_user_can('edit_post', $post_id)) return;

        // Sanitize and save width
        if (isset($_POST['fca_content_width'])) {
            update_post_meta($post_id, '_fca_content_width', sanitize_text_field($_POST['fca_content_width']));
        }
         // Sanitize and save width unit
        if (isset($_POST['fca_content_width_unit'])) {
             $unit = in_array($_POST['fca_content_width_unit'], ['%', 'px']) ? $_POST['fca_content_width_unit'] : '%';
            update_post_meta($post_id, '_fca_content_width_unit', $unit);
        }
         // Sanitize and save alignment
        if (isset($_POST['fca_content_alignment'])) {
             $align = in_array($_POST['fca_content_alignment'], ['left', 'center', 'right']) ? $_POST['fca_content_alignment'] : 'center';
            update_post_meta($post_id, '_fca_content_alignment', $align);
        }
        
        // --- Save Responsive Fields ---
        // Sanitize and save tablet width
        if (isset($_POST['fca_content_width_tablet'])) {
            update_post_meta($post_id, '_fca_content_width_tablet', sanitize_text_field($_POST['fca_content_width_tablet']));
        }
         // Sanitize and save tablet width unit
        if (isset($_POST['fca_content_width_unit_tablet'])) {
             $unit_tablet = in_array($_POST['fca_content_width_unit_tablet'], ['%', 'px']) ? $_POST['fca_content_width_unit_tablet'] : '%';
            update_post_meta($post_id, '_fca_content_width_unit_tablet', $unit_tablet);
        }
        // Sanitize and save mobile width
        if (isset($_POST['fca_content_width_mobile'])) {
            update_post_meta($post_id, '_fca_content_width_mobile', sanitize_text_field($_POST['fca_content_width_mobile']));
        }
         // Sanitize and save mobile width unit
        if (isset($_POST['fca_content_width_unit_mobile'])) {
             $unit_mobile = in_array($_POST['fca_content_width_unit_mobile'], ['%', 'px']) ? $_POST['fca_content_width_unit_mobile'] : '%';
            update_post_meta($post_id, '_fca_content_width_unit_mobile', $unit_mobile);
        }
    }

    /**
     * Add pages to the Fluent Community sidebar menu
     */
    public function add_menu_items($menu_groups) {
        // Make sure the class is loaded
        if (!class_exists('FCA_Settings')) {
            require_once plugin_dir_path(__FILE__) . 'includes/class-fca-settings.php';
        }
        
        // Check if menu is enabled via any available storage method
        $menu_enabled = FCA_Settings::get_settings('menu_enabled');
        if (!$menu_enabled) {
            return $menu_groups;
        }
        
        // Get menu settings
        $menu_title = FCA_Settings::get_settings('menu_title');
        $menu_icon = FCA_Settings::get_settings('menu_icon');
        $include_settings_link = FCA_Settings::get_settings('include_settings_link');
        $selected_pages = FCA_Settings::get_settings('selected_pages');
        $menu_priority = FCA_Settings::get_settings('menu_priority');
        $page_order = FCA_Settings::get_settings('page_order');
        
        // Get all published pages
        $pages = get_posts([
            'post_type' => 'fca_page',
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'menu_order title', 
            'order' => 'ASC',
        ]);
        
        // If no pages and no settings link, don't add menu
        if (empty($pages) && !$include_settings_link) {
            return $menu_groups;
        }
        
        // Create menu group if it doesn't exist
        $target_group_key = 'fca_pages_group';
        if (!isset($menu_groups[$target_group_key])) {
            $menu_groups[$target_group_key] = [
                'title' => $menu_title, 
                'id' => $target_group_key,
                'priority' => $menu_priority, 
                'children' => [],
            ];
        }

        // Check if the correct Helper class exists using the correct namespace
        $fc_helper_exists = class_exists('\FluentCommunity\App\Services\Helper'); // Correct Namespace

        // Function to generate the URL robustly
        $generate_url = function($relative_path) use ($fc_helper_exists) {
            if ($fc_helper_exists) {
                // Use the correct Helper class with the correct namespace
                return \FluentCommunity\App\Services\Helper::baseUrl($relative_path);
            } else {
                // Fallback: Get dynamic slug ONLY if Helper class is missing
        $portal_slug = apply_filters('fluent_community/portal_slug', 'portal'); 
                return home_url("/{$portal_slug}{$relative_path}");
            }
        };

        // Add static Settings link 
        if ($include_settings_link) {
            $settings_permalink = $generate_url('/pages/settings'); 
         $menu_groups[$target_group_key]['children']['settings'] = [ 
             'id'          => 'fca_page_settings', 
             'key'         => 'fca_page_settings', 
             'title'       => __('Page Settings', 'fca-pages'),
             'icon_image'  => '',
             'shape_svg'   => '', 
             'emoji'       => '⚙️', 
             'permalink'   => $settings_permalink, 
             'link_classes'=> 'space_menu_item route_url fca-page-settings-link', 
         ];
        }

        // Filter pages if specific pages are selected
        if (!empty($selected_pages)) {
            $filtered_pages = [];
        foreach ($pages as $page) {
                if (in_array($page->ID, $selected_pages)) {
                    $filtered_pages[] = $page;
                }
            }
            $pages = $filtered_pages;
        }
        
        // Apply custom page order if set
        if (!empty($page_order) && !empty($pages)) {
            $ordered_pages = [];
            foreach ($page_order as $id) {
                foreach ($pages as $key => $page) {
                    if ($page->ID == $id) {
                        $ordered_pages[] = $page;
                        unset($pages[$key]);
                        break;
                    }
                }
            }
            $pages = array_values($pages); 
            foreach ($pages as $page) {
                $ordered_pages[] = $page;
            }
            $pages = $ordered_pages;
        }

        // Add dynamic page links 
        foreach ($pages as $page) {
            if (empty($page->post_name)) {
                continue;
            }
            $permalink = $generate_url("/page/{$page->post_name}"); 
            $menu_groups[$target_group_key]['children']['page_'.$page->ID] = [ 
                'id'          => 'fca_page_' . $page->ID, 
                'key'         => 'fca_page_' . $page->ID, 
                'title'       => $page->post_title,
                'icon_image'  => '',
                'shape_svg'   => '',
                'emoji'       => $menu_icon,
                'permalink'   => $permalink, 
                'link_classes'=> 'space_menu_item route_url fca-page-link', 
            ];
        }
        
        return $menu_groups;
    }

    /**
     * Check if the current request is for the standalone editor and render it.
     * Simplified to work directly with the target post.
     */
    public function maybe_render_fca_editor() {
        // This method is no longer needed as we're using the WordPress admin editor
            return;
        }

    /**
     * Set up the environment and render the minimal HTML page for the editor.
     * Uses WordPress's built-in mechanism to load the block editor.
     */
    private function render_editor_page($dummy_post, $real_post_data) {
        // This method is no longer needed as we're using the WordPress admin editor
        return;
    }

    /**
     * Helper to get available image sizes (similar to FluentBlockEditorHandler)
     */
    private function get_available_image_sizes() {
         $image_sizes = [];
         // Use standard WP function to get registered sizes
         foreach ( wp_get_registered_image_subsizes() as $name => $size ) {
             // Filter out potentially huge sizes if needed?
             // if ($size['width'] < 2600 && $size['height'] < 2600) {}
             $image_sizes[] = [
                 'slug' => $name,
                 'name' => ucwords( str_replace( ['-', '_'], ' ', $name ) ), // Format name nicely
             ];
         }
         // Ensure 'full' size is always present if missing
         if (!wp_list_filter($image_sizes, ['slug' => 'full'])) {
              $image_sizes[] = ['slug' => 'full', 'name' => 'Full Size'];
         }
         return $image_sizes;
    }
    
    /**
     * Force the use of the block editor for our custom post type
     */
    public function use_block_editor($use_block_editor, $post_type) {
        if ($post_type === 'fca_page') {
            return true;
        }
        return $use_block_editor;
    }

    /**
     * Add a link to go back to Fluent Community in the admin bar
     */
    public function add_back_to_fc_link($admin_bar) {
        global $post;
        
        if (is_admin() && $post && $post->post_type === 'fca_page') {
            // Check using correct namespace
            $fc_helper_exists = class_exists('\FluentCommunity\App\Services\Helper'); 
            $portal_slug = $fc_helper_exists ? '' : apply_filters('fluent_community/portal_slug', 'portal'); 
            
            // Back to settings link
            $settings_path = '/pages/settings';
            $settings_link = $fc_helper_exists 
                // Use correct namespace
                ? \FluentCommunity\App\Services\Helper::baseUrl($settings_path) 
                : home_url("/{$portal_slug}{$settings_path}");
            $admin_bar->add_node([
                'id'    => 'back-to-fluent-community',
                'title' => __('Back to Fluent Community', 'fca-pages'),
                'href'  => $settings_link,
                'meta'  => [
                    'class' => 'fca-back-to-fc',
                    'title' => __('Return to Fluent Community', 'fca-pages')
                ]
            ]);
            
            // View link
            if ($post->post_status === 'publish' && !empty($post->post_name)) {
                $page_path = "/page/{$post->post_name}";
                $view_link = $fc_helper_exists 
                    // Use correct namespace
                    ? \FluentCommunity\App\Services\Helper::baseUrl($page_path) 
                    : home_url("/{$portal_slug}{$page_path}");
                $admin_bar->add_node([
                    'id'    => 'view-in-fluent-community',
                    'title' => __('View in Fluent Community', 'fca-pages'),
                    'href'  => $view_link,
                    'meta'  => [
                        'class' => 'fca-view-in-fc',
                        'title' => __('View this page in Fluent Community', 'fca-pages'),
                        'target' => '_blank'
                    ]
                ]);
            }
        }
    }
    
    /**
     * Show a notice after saving a page with a link back to Fluent Community
     */
    public function admin_notices() {
        global $post, $pagenow;
        
        if ($pagenow === 'post.php' && isset($_GET['message']) && $_GET['message'] == 1 && $post && $post->post_type === 'fca_page') {
            // Check using correct namespace
            $fc_helper_exists = class_exists('\FluentCommunity\App\Services\Helper'); 
            $portal_slug = $fc_helper_exists ? '' : apply_filters('fluent_community/portal_slug', 'portal');
            
            // Back link
            $settings_path = '/pages/settings';
            $back_link = $fc_helper_exists 
                 // Use correct namespace
                ? \FluentCommunity\App\Services\Helper::baseUrl($settings_path) 
                : home_url("/{$portal_slug}{$settings_path}");
                
            $view_link = '';
            $view_link_no_cache = '';
            
            if ($post->post_status === 'publish' && !empty($post->post_name)) {
                 $page_path = "/page/{$post->post_name}";
                 $view_link = $fc_helper_exists 
                     // Use correct namespace
                    ? \FluentCommunity\App\Services\Helper::baseUrl($page_path) 
                    : home_url("/{$portal_slug}{$page_path}");
                
                $view_link_no_cache = add_query_arg(['nocache' => time()], $view_link);
            }
            
            echo '<div class="notice notice-success is-dismissible">';
            echo '<p>' . __('Page saved successfully!', 'fca-pages') . ' ';
            echo '<a href="' . esc_url($back_link) . '" class="button button-secondary">' . __('Back to Fluent Community', 'fca-pages') . '</a> ';
            
            if ($view_link_no_cache) {
                echo '<a href="' . esc_url($view_link_no_cache) . '" class="button button-primary" target="_blank">' . __('View Page (Clear Cache)', 'fca-pages') . '</a>';
                
                // Add a JavaScript snippet to reload the page data
                echo '<script>
                    document.addEventListener("DOMContentLoaded", function() {
                        // Add a function to reload page content from parent window
                        window.refreshPageContent = function() {
                            if (confirm("Refresh page cache in Fluent Community?")) {
                                // Create an iframe to execute the reload function
                                var iframe = document.createElement("iframe");
                                iframe.style.display = "none";
                                iframe.src = "' . esc_url($view_link) . '";
                                
                                // Add event listener for when iframe loads
                                iframe.onload = function() {
                                    try {
                                        // Try to call the reload function in the iframe
                                        iframe.contentWindow.fcaReloadPageData("' . esc_js($post->post_name) . '");
                                        alert("Page cache has been refreshed. You may need to reload the Fluent Community page to see your changes.");
                                    } catch(e) {
                                        console.error("Failed to refresh page content:", e);
                                        alert("Failed to refresh page content. Please try viewing the page directly.");
                                    }
                                    
                                    // Remove the iframe after use
                                    document.body.removeChild(iframe);
                                };
                                
                                // Add iframe to document to start loading
                                document.body.appendChild(iframe);
                            }
                        };
                        
                        // Add refresh button
                        var refreshButton = document.createElement("button");
                        refreshButton.className = "button button-secondary";
                        refreshButton.textContent = "Force Refresh Page Cache";
                        refreshButton.onclick = function(e) {
                            e.preventDefault();
                            window.refreshPageContent();
                        };
                        
                        // Find the notice paragraph and append the button
                        var notice = document.querySelector(".notice.notice-success p");
                        if (notice) {
                            notice.appendChild(document.createTextNode(" "));
                            notice.appendChild(refreshButton);
                        }
                    });
                </script>';
            }
            
            echo '</p></div>';
        }
    }

    /**
     * Register custom blocks
     */
    public function register_custom_blocks() {
        // Register Button block using metadata
        register_block_type_from_metadata( plugin_dir_path(__FILE__) . 'blocks/fca-button/' );
        
        // Register Card block using metadata
        register_block_type_from_metadata( plugin_dir_path(__FILE__) . 'blocks/fca-card/' );

        // Register Spaces List block using metadata
        register_block_type_from_metadata( plugin_dir_path(__FILE__) . 'blocks/fca-spaces-list/' );

        // Register Image block using metadata
        register_block_type_from_metadata( plugin_dir_path(__FILE__) . 'blocks/fca-image/' );
        
        // Register Pages List block using metadata
        register_block_type_from_metadata( plugin_dir_path(__FILE__) . 'blocks/fca-pages-list/' );
        
        // Register Courses List block using metadata
        register_block_type_from_metadata( plugin_dir_path(__FILE__) . 'blocks/fca-courses-list/' );
        
        // Enqueue editor-specific styles (like the main FC theme styles)
        add_action('enqueue_block_editor_assets', [$this, 'enqueue_block_editor_styles']);
    }
    
    /**
     * Enqueue styles specifically for the block editor.
     */
    public function enqueue_block_editor_styles() {
        global $post;
        
        // Check if we are on the editor screen for our custom post type
        if ($post && $post->post_type === 'fca_page') {
            $plugin_version = get_plugin_data(__FILE__)['Version'];
            $style_path = 'css/editor-blocks.css';
            $style_url = plugin_dir_url(__FILE__) . $style_path;
            $style_file_path = plugin_dir_path(__FILE__) . $style_path;

            // Enqueue the consolidated editor stylesheet
            if (file_exists($style_file_path)) {
                wp_enqueue_style(
                    'fca-pages-editor-blocks',
                    $style_url,
                    array('wp-edit-blocks'), // Dependency
                    filemtime($style_file_path) // Versioning based on file modification time
                );
                    } else {
                 // Optional: Log an error or show an admin notice if the file is missing
                 error_log('FCA Pages: editor-blocks.css not found at ' . $style_file_path);
            }
        }
    }
    
    /**
     * Add custom block category
     */
    public function add_block_category( $categories, $editor_context ) {
        // Check if context is a post type that uses the editor
        if ( ! empty( $editor_context->post ) ) {
            // Add our category
            array_unshift( $categories, [
                'slug'  => 'fca-blocks',
                'title' => __( 'Fluent Community Blocks', 'fca-pages' ),
                'icon'  => 'dashicons-fluent', // You might need to replace this with an actual Dashicon or SVG
            ] );
        }
        return $categories;
    }

    /**
     * Enqueue block styles specifically for the Fluent Community portal frontend.
     */
    public function enqueue_portal_block_styles() {
        // Core Block Styles (for Columns, etc.)
        $core_style_path = plugin_dir_path(__FILE__) . 'css/core-block-styles.css';
        if (file_exists($core_style_path)) {
            $core_style_url = plugin_dir_url(__FILE__) . 'css/core-block-styles.css';
            echo '<link rel="stylesheet" id="fca-core-block-styles" href="' . esc_url($core_style_url) . '?ver=' . filemtime($core_style_path) . '" media="all">';
        }

        // Card Block Style
        $card_style_path = plugin_dir_path(__FILE__) . 'blocks/fca-card/style.css';
        if (file_exists($card_style_path)) {
            $card_style_url = plugin_dir_url(__FILE__) . 'blocks/fca-card/style.css';
            echo '<link rel="stylesheet" id="fca-card-block-style" href="' . esc_url($card_style_url) . '?ver=' . filemtime($card_style_path) . '" media="all">';
        }
        
        // Button Block Style
        $button_style_path = plugin_dir_path(__FILE__) . 'blocks/fca-button/style.css';
        if (file_exists($button_style_path)) {
            $button_style_url = plugin_dir_url(__FILE__) . 'blocks/fca-button/style.css';
            echo '<link rel="stylesheet" id="fca-button-block-style" href="' . esc_url($button_style_url) . '?ver=' . filemtime($button_style_path) . '" media="all">';
        }

        // Spaces List Block Style
        $spaces_list_style_path = plugin_dir_path(__FILE__) . 'blocks/fca-spaces-list/style.css';
        if (file_exists($spaces_list_style_path)) {
            $spaces_list_style_url = plugin_dir_url(__FILE__) . 'blocks/fca-spaces-list/style.css';
            echo '<link rel="stylesheet" id="fca-spaces-list-block-style" href="' . esc_url($spaces_list_style_url) . '?ver=' . filemtime($spaces_list_style_path) . '" media="all">';
        }

        // Image Block Style
        $image_style_path = plugin_dir_path(__FILE__) . 'blocks/fca-image/style.css';
        if (file_exists($image_style_path)) {
            $image_style_url = plugin_dir_url(__FILE__) . 'blocks/fca-image/style.css';
            echo '<link rel="stylesheet" id="fca-image-block-style" href="' . esc_url($image_style_url) . '?ver=' . filemtime($image_style_path) . '" media="all">';
        }

        // Pages List Block Style
        $pages_list_style_path = plugin_dir_path(__FILE__) . 'blocks/fca-pages-list/style.css';
        if (file_exists($pages_list_style_path)) {
            $pages_list_style_url = plugin_dir_url(__FILE__) . 'blocks/fca-pages-list/style.css';
            echo '<link rel="stylesheet" id="fca-pages-list-block-style" href="' . esc_url($pages_list_style_url) . '?ver=' . filemtime($pages_list_style_path) . '" media="all">';
        }

        // Courses List Block Style
        $courses_list_style_path = plugin_dir_path(__FILE__) . 'blocks/fca-courses-list/style.css';
        if (file_exists($courses_list_style_path)) {
            $courses_list_style_url = plugin_dir_url(__FILE__) . 'blocks/fca-courses-list/style.css';
            echo '<link rel="stylesheet" id="fca-courses-list-block-style" href="' . esc_url($courses_list_style_url) . '?ver=' . filemtime($courses_list_style_path) . '" media="all">';
        }
    }

} // End class FCA_Pages

/**
 * Initialize the FCA_Pages plugin ensuring it runs after FluentCommunity is likely loaded.
 */
function fca_pages_init() {
    // Optional: Add a check here for FluentCommunity constants/classes if needed
    // if (!defined('FLUENT_COMMUNITY_PLUGIN_VERSION')) { return; }
FCA_Pages::instance();
}
add_action('plugins_loaded', 'fca_pages_init', 20); // Use priority 20 to run later
