<?php
/**
 * Server-side rendering for the FCA Spaces List block.
 * --- Simplified for Debugging Critical Error ---
 */

defined('ABSPATH') || exit;

global $wpdb;

// --- Helper function to get Space URL --- 
if (!function_exists('fca_get_space_url')) {
    function fca_get_space_url($space_slug) {
        if (empty($space_slug)) {
            return '#';
        }
        // Define the relative path with /home
        $relative_path = "/space/{$space_slug}/home";

        // Check if the correct Helper class exists using the correct namespace
        $fc_helper_exists = class_exists('\FluentCommunity\App\Services\Helper');

        if ($fc_helper_exists) {
            // Use the correct Helper class
            return \FluentCommunity\App\Services\Helper::baseUrl($relative_path);
        } else {
            // Fallback: Get dynamic slug ONLY if Helper class is missing
            $portal_slug = apply_filters('fluent_community/portal_slug', 'portal'); 
            return home_url("/{$portal_slug}{$relative_path}");
        }
    }
}

// --- Get Attributes ---
$display_mode = $attributes['displayMode'] ?? 'all';
$selected_spaces_str = $attributes['selectedSpaces'] ?? '';
$number_of_spaces = $attributes['numberOfSpaces'] ?? 5;
$show_description = $attributes['showDescription'] ?? true;
$alignment = $attributes['align'] ?? '';
// New attributes
$show_image = $attributes['showImage'] ?? true;
$show_privacy = $attributes['showPrivacy'] ?? true;
$show_member_count = $attributes['showMemberCount'] ?? true;
$show_post_count = $attributes['showPostCount'] ?? false;
$card_bg_color = $attributes['cardBackgroundColor'] ?? '';
$card_border_color = $attributes['cardBorderColor'] ?? '';
$card_border_radius = $attributes['cardBorderRadius'] ?? 8;
$number_of_columns = $attributes['numberOfColumns'] ?? 0;

// --- Fetch Spaces using the confirmed Utility method --- 
$all_spaces = [];
$fetch_error = '';

if (class_exists('\FluentCommunity\App\Functions\Utility') && method_exists('\FluentCommunity\App\Functions\Utility', 'getSpaces')) {
    try {
        $spaces_collection = \FluentCommunity\App\Functions\Utility::getSpaces(false); 
        
        if ($spaces_collection && method_exists($spaces_collection, 'all')) {
            $spaces_array = $spaces_collection->all(); 
            
            foreach ($spaces_array as $space_item) {
                $space_obj = (object) $space_item;
                // Filter primarily by status = published
                if (isset($space_obj->status) && $space_obj->status === 'published') { 
                    $all_spaces[] = $space_obj;
                }
            }
            
            // Apply default sorting (newest first)
            usort($all_spaces, function($a, $b) {
                 $time_a = isset($a->created_at) ? strtotime($a->created_at) : 0;
                 $time_b = isset($b->created_at) ? strtotime($b->created_at) : 0;
                 return $time_b <=> $time_a;
            });

        } elseif ($spaces_collection) {
             $fetch_error = 'Utility::getSpaces() returned unexpected format.';
        } else {
             $fetch_error = 'Utility::getSpaces() returned no results.';
        }
    } catch (\Exception $e) {
        $fetch_error = 'Error calling Utility::getSpaces(): ' . $e->getMessage();
        error_log('FCA Spaces List Block Error: ' . $fetch_error);
    }
} else {
    $fetch_error = 'Fluent Community Utility class or getSpaces method not found.';
}

if (!empty($fetch_error) && empty($all_spaces)) { 
     echo '<p>' . esc_html__('Error fetching spaces:', 'fca-pages') . ' ' . esc_html($fetch_error) . '</p>';
     return;
}

// --- Filter Spaces Based on Attributes ---
$final_spaces = [];
if ($display_mode === 'specific') {
    $space_ids = array_map('intval', array_filter(explode(',', $selected_spaces_str)));
    if (!empty($space_ids)) {
        $keyed_spaces = [];
        foreach ($all_spaces as $space) {
             if (isset($space->id)) { 
                 $keyed_spaces[$space->id] = $space;
             }
        }
        foreach ($space_ids as $id) {
            if (isset($keyed_spaces[$id])) {
                $final_spaces[] = $keyed_spaces[$id];
            }
        }
    } 
} else { // 'all' mode
    $limit = max(1, intval($number_of_spaces));
    $final_spaces = array_slice($all_spaces, 0, $limit);
}

// --- Render Block --- 
if (empty($final_spaces)) {
    if ($display_mode === 'specific' && empty($space_ids)){
         echo '<p>' . esc_html__('Please enter valid Space IDs in the block settings.', 'fca-pages') . '</p>';
    } else {
        echo '<p>' . esc_html__('No published spaces found matching the criteria.', 'fca-pages') . '</p>'; // Updated message
    }
    return;
}

// Define SVGs for icons (copied from example HTML)
$icon_public = '<svg width="14" height="14" viewBox="0 0 14 14" fill="currentColor" xmlns="http://www.w3.org/2000/svg"><path d="M3.1564 3.30236C2.23368 4.26115 1.66634 5.5644 1.66634 7.00016C1.66634 9.9457 4.05415 12.3335 6.99967 12.3335C7.72021 12.3335 8.40741 12.1906 9.03441 11.9316C9.11107 11.4313 8.94287 10.9512 8.87407 10.7832C8.72101 10.4102 8.21574 9.7723 7.37227 8.88736C7.14714 8.6507 7.16161 8.46916 7.24207 7.9297L7.25134 7.86876C7.30601 7.49923 7.39774 7.28076 8.64114 7.08336C9.27281 6.98323 9.43894 7.2357 9.66921 7.5853C9.69467 7.62403 9.72014 7.66203 9.74621 7.7001C9.96494 8.01996 10.127 8.0931 10.3718 8.20443C10.4815 8.25416 10.6183 8.3169 10.8017 8.4211C11.2364 8.6695 11.2364 8.94996 11.2364 9.56496V9.6347C11.2364 9.89576 11.2109 10.125 11.1707 10.3241C11.8982 9.4125 12.333 8.2571 12.333 7.00016C12.333 4.80076 11.0017 2.9123 9.10094 2.09671C8.73214 2.34558 8.22621 2.69834 8.04967 2.94016C7.95941 3.06375 7.83181 3.6946 7.41681 3.74667C7.30807 3.76032 7.16221 3.75076 7.00767 3.74081C6.59334 3.71388 6.02647 3.67754 5.84501 4.17013C5.72981 4.4823 5.70954 5.3298 6.08227 5.7697C6.14194 5.83996 6.15347 5.96996 6.11294 6.11583C6.05967 6.30736 5.95207 6.4239 5.91854 6.44796C5.85427 6.41103 5.72607 6.26223 5.63926 6.16096C5.43003 5.91783 5.16904 5.61505 4.83165 5.52134C4.70897 5.48738 4.57411 5.45928 4.44333 5.43174C4.07727 5.35501 3.66263 5.26772 3.56629 5.06218C3.49595 4.91163 3.49626 4.7043 3.49653 4.48526C3.49653 4.20763 3.49653 3.89369 3.36053 3.58854C3.30798 3.47044 3.2376 3.37647 3.1564 3.30236ZM6.99967 13.6668C3.31777 13.6668 0.333008 10.682 0.333008 7.00016C0.333008 3.31826 3.31777 0.333496 6.99967 0.333496C10.6815 0.333496 13.6663 3.31826 13.6663 7.00016C13.6663 10.682 10.6815 13.6668 6.99967 13.6668Z" fill="currentColor"></path></svg>';
$icon_private = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1024 1024"><path fill="currentColor" d="M224 448a32 32 0 0 0-32 32v384a32 32 0 0 0 32 32h576a32 32 0 0 0 32-32V480a32 32 0 0 0-32-32zm0-64h576a96 96 0 0 1 96 96v384a96 96 0 0 1-96 96H224a96 96 0 0 1-96-96V480a96 96 0 0 1 96-96"></path><path fill="currentColor" d="M512 544a32 32 0 0 1 32 32v192a32 32 0 1 1-64 0V576a32 32 0 0 1 32-32m192-160v-64a192 192 0 1 0-384 0v64zM512 64a256 256 0 0 1 256 256v128H256V320A256 256 0 0 1 512 64"></path></svg>';
$icon_members = '<svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M13.8494 12C14.3489 12 14.7462 11.6856 15.103 11.2461C15.8333 10.3463 14.6342 9.62723 14.1769 9.27507C13.712 8.91708 13.1929 8.71428 12.6667 8.66667M12 7.33333C12.9205 7.33333 13.6667 6.58714 13.6667 5.66667C13.6667 4.74619 12.9205 4 12 4" stroke="currentColor" stroke-linecap="round"></path><path d="M2.15096 12C1.65143 12 1.2541 11.6856 0.897356 11.2461C0.167051 10.3463 1.36611 9.62723 1.82342 9.27507C2.28831 8.91708 2.80738 8.71428 3.33366 8.66667M3.66699 7.33333C2.74652 7.33333 2.00033 6.58714 2.00033 5.66667C2.00033 4.74619 2.74652 4 3.66699 4" stroke="currentColor" stroke-linecap="round"></path><path d="M5.38953 10.0738C4.70834 10.495 2.92232 11.3551 4.01013 12.4313C4.54151 12.957 5.13334 13.333 5.87741 13.333H10.1232C10.8673 13.333 11.4591 12.957 11.9905 12.4313C13.0783 11.3551 11.2923 10.495 10.6111 10.0738C9.01375 9.08608 6.9869 9.08608 5.38953 10.0738Z" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"></path><path d="M10.3337 4.99984C10.3337 6.2885 9.28899 7.33317 8.00033 7.33317C6.71166 7.33317 5.66699 6.2885 5.66699 4.99984C5.66699 3.71117 6.71166 2.6665 8.00033 2.6665C9.28899 2.6665 10.3337 3.71117 10.3337 4.99984Z" stroke="currentColor"></path></svg>';
$icon_posts = '<svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M2.66667 3.99984H13.3333M2.66667 7.99984H13.3333M2.66667 11.9998H8" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path></svg>'; // Example posts icon

// Correct Placeholder Image URL
$placeholder_img = plugins_url('assets/images/space-placeholder.jpg', 'fluent-community/fluent-community.php');
// Add a check in case FC isn't active or path changes
if (!file_exists(WP_PLUGIN_DIR . '/fluent-community/assets/images/space-placeholder.jpg')) {
    $placeholder_img = ''; // Or provide a fallback within fca-pages
}

$wrapper_attributes = get_block_wrapper_attributes(['class' => 'fca-spaces-list-wrapper']);

// --- Generate Grid Class --- 
$grid_classes = 'fca-spaces-list fca-spaces-list-grid'; // Base classes
if ($number_of_columns > 0 && $number_of_columns <= 6) { // Add class only for 1-6 columns
    $grid_classes .= ' fca-grid-cols-' . intval($number_of_columns);
}

// Use fcom_box_card_list for the grid layout
echo '<div ' . $wrapper_attributes . '>'; 
// echo '<!-- Debug: Using Utility::getSpaces() -->'; 
// Apply the generated classes
echo '<div class="' . esc_attr($grid_classes) . '">'; // NO inline style

foreach ($final_spaces as $space) {
    $space_id = $space->id ?? 0;
    $space_title = esc_html($space->name ?? $space->title ?? 'Untitled Space'); 
    $space_slug = $space->slug ?? '';
    $space_url = fca_get_space_url($space_slug);
    $space_description_raw = $space->description ?? ($space->short_description ?? '');
    $space_description = $show_description ? wp_trim_words(wp_kses_post($space_description_raw), 15, '...') : ''; 
    $space_type = $space->type ?? 'public'; 
    
    // --- Get Cover Image (Look in Settings Meta) --- 
    $space_image_url_raw = '';
    if (isset($space->settings)) {
        $settings_data = maybe_unserialize($space->settings);
        if (is_array($settings_data) && !empty($settings_data['og_image'])) {
            $space_image_url_raw = $settings_data['og_image'];
        } elseif (is_object($settings_data) && !empty($settings_data->og_image)) {
             $space_image_url_raw = $settings_data->og_image;
        }
    } 

    // Use placeholder if raw value is empty or not a valid URL
    $display_image = (!empty($space_image_url_raw) && filter_var($space_image_url_raw, FILTER_VALIDATE_URL)) ? $space_image_url_raw : $placeholder_img;
    
    // --- Get Live Member Count --- 
    global $wpdb;
    $member_table = $wpdb->prefix . 'fcom_space_user';
    $post_table = $wpdb->prefix . 'fcom_posts'; // Need this table name again
    $member_count = 0;
    $post_count = 0; // Initialize live post count

    if ($space_id > 0) {
        // Member Count Query (remains the same)
        if ($show_member_count && $wpdb->get_var("SHOW TABLES LIKE '{$member_table}'") == $member_table) {
            $member_count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$member_table} WHERE space_id = %d AND status = %s",
                $space_id,
                'active' 
            ));
        }
        
        // --- Get Live Post Count --- 
        if ($show_post_count && $wpdb->get_var("SHOW TABLES LIKE '{$post_table}'") == $post_table) {
             // Count posts with status = published, regardless of type
            $post_count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$post_table} WHERE space_id = %d AND status = %s",
                $space_id,
                'published' 
            ));
        }
    }
    $member_count = intval($member_count); 
    $post_count = intval($post_count); // Use the live count

    // --- Apply Card Styles --- 
    $card_styles = '';
    if (!empty($card_bg_color)) {
        $card_styles .= 'background-color:' . esc_attr($card_bg_color) . ';';
    }
    if (!empty($card_border_color)) {
        $card_styles .= 'border-color:' . esc_attr($card_border_color) . ';';
    }
    if (isset($card_border_radius) && is_numeric($card_border_radius)) { // Check if set and numeric
        $card_styles .= 'border-radius:' . intval($card_border_radius) . 'px;';
    }
    $card_style_attr = !empty($card_styles) ? ' style="' . $card_styles . '"' : '';

    echo '<div role="region" aria-label="Space Card" class="fca-space-card"' . $card_style_attr . '>';
    
    // Avatar/Image Section (Conditional)
    if ($show_image) {
        echo '<div class="fca-space-card-avatar">'; 
        echo '<a href="' . $space_url . '" aria-label="Go to Space">';
        echo '<img src="' . esc_url($display_image) . '" alt="' . esc_attr($space_title) . ' cover image" loading="lazy">'; 
        echo '</a></div>';
    }
    
    // Info Section
    echo '<div class="fca-space-card-info">'; 
    // -- Name
    echo '<div class="fca-space-card-name">'; 
    echo '<a href="' . $space_url . '" class="">' . $space_title . '</a>'; 
    // TODO: Add admin tag logic if needed
    echo '</div>';
    // -- Details (Conditional)
    if ($show_privacy || $show_member_count || $show_post_count) {
        echo '<div class="fca-space-card-details">'; 
        // ---- Privacy (Conditional)
        if ($show_privacy) {
            echo '<div class="fca-space-card-detail-item">'; 
            echo '<i class="el-icon">' . ($space_type === 'private' ? $icon_private : $icon_public) . '</i>'; 
            echo '<span>' . esc_html(ucfirst($space_type)) . '</span>';
            echo '</div>';
        }
        // ---- Members (Conditional)
        if ($show_member_count) {
            echo '<div class="fca-space-card-detail-item">'; 
            echo '<i class="el-icon">' . $icon_members . '</i>';
            echo '<span>' . esc_html($member_count) . ' ' . ($member_count === 1 ? __('Member', 'fca-pages') : __('Members', 'fca-pages')) . '</span>';
            echo '</div>';
        }
        // ---- Posts (Conditional)
        if ($show_post_count) {
             echo '<div class="fca-space-card-detail-item">'; 
             echo '<i class="el-icon">' . $icon_posts . '</i>';
             // Use the live $post_count value
             echo '<span>' . esc_html($post_count) . ' ' . ($post_count === 1 ? __('Post', 'fca-pages') : __('Posts', 'fca-pages')) . '</span>';
             echo '</div>';
        }
        echo '</div>'; // End details
    }
    
    // -- Description (Conditional)
    if ($show_description && !empty($space_description)) {
        echo '<p class="fca-space-card-description">' . $space_description . '</p>'; 
    }
    echo '</div>'; // End info

    // Action Section
    echo '<div class="fca-space-card-action">'; 
    echo '<a href="' . $space_url . '" class="fca-space-card-button">' . __('View Space', 'fca-pages') . '</a>'; 
    echo '</div>';

    echo '</div>'; // End fca-space-card
}

echo '</div>'; // End fca-spaces-list-grid
echo '</div>'; // End wrapper 