/**
 * Editor Script for FCA Spaces List Block (No Build Step Version)
 */
( function( wp ) {
    const { __ } = wp.i18n;
    const { registerBlockType } = wp.blocks;
    const { InspectorControls, useBlockProps, BlockControls, AlignmentToolbar, ColorPalette } = wp.blockEditor;
    const { PanelBody, SelectControl, TextControl, TextareaControl, ToggleControl, RangeControl } = wp.components;
    const ServerSideRender = wp.serverSideRender;
    const { createElement, Fragment } = wp.element; 
    const metadata = {
        "$schema": "https://schemas.wp.org/trunk/block.json",
        "apiVersion": 3,
        "name": "fca-pages/fca-spaces-list",
        "title": "FCA Spaces List",
        "category": "fca-blocks",
        "description": "Displays a list of Fluent Community spaces.",
        "icon": "groups",
        "keywords": ["fluent", "community", "spaces", "groups", "list"],
        "version": "1.0.0",
        "textdomain": "fca-pages",
        "supports": {
            "html": false,
            "align": true
        },
        "attributes": {
            "displayMode": { "type": "string", "default": "all" },
            "selectedSpaces": { "type": "string", "default": "" },
            "numberOfSpaces": { "type": "number", "default": 5 },
            "showDescription": { "type": "boolean", "default": true },
            "align": { "type": "string", "default": "" },
            "showImage": { "type": "boolean", "default": true },
            "showPrivacy": { "type": "boolean", "default": true },
            "showMemberCount": { "type": "boolean", "default": true },
            "showPostCount": { "type": "boolean", "default": true },
            "cardBackgroundColor": { "type": "string", "default": "#ffffff" },
            "cardBorderColor": { "type": "string", "default": "#000000" },
            "cardBorderRadius": { "type": "number", "default": 0 },
            "numberOfColumns": { "type": "number", "default": 0 }
        },
        // No editorScript needed here as we register directly
        // Style is still needed if you have separate editor-only styles
        // render is handled server-side
    }; // Re-defining metadata here for simplicity, or fetch it if possible

    // --- Edit Component --- 
    const EditComponent = ( props ) => {
        const { attributes, setAttributes } = props;
        const {
            displayMode, selectedSpaces, numberOfSpaces, align, 
            showImage, showDescription, showPrivacy, showMemberCount, showPostCount,
            cardBackgroundColor, cardBorderColor, cardBorderRadius,
            numberOfColumns
        } = attributes;
        const blockProps = useBlockProps();
        
        // Available Colors - you can define your palette or use defaults
        const colorPalette = wp.data.select( "core/block-editor" ).getSettings().colors;

        // Inspector Controls definition using createElement
        const inspectorControls = createElement( InspectorControls, null, 
            createElement( PanelBody, { title: __( 'Display Settings', 'fca-pages' ), initialOpen: true },
                createElement( SelectControl, {
                    label: __( 'Display Mode', 'fca-pages' ),
                    value: displayMode,
                    options: [
                        { label: __( 'Show All Public Spaces', 'fca-pages' ), value: 'all' },
                        { label: __( 'Show Specific Spaces', 'fca-pages' ), value: 'specific' },
                    ],
                    onChange: ( value ) => setAttributes( { displayMode: value } ),
                } ),
                displayMode === 'all' && createElement( TextControl, {
                    label: __( 'Number of Spaces', 'fca-pages' ),
                    type: "number",
                    value: numberOfSpaces,
                    onChange: ( value ) => setAttributes( { numberOfSpaces: parseInt( value, 10 ) || 0 } ),
                    min: "1",
                } ),
                displayMode === 'specific' && createElement( TextareaControl, {
                    label: __( 'Space IDs', 'fca-pages' ),
                    help: __( 'Enter comma-separated IDs of the spaces to display.', 'fca-pages' ),
                    value: selectedSpaces,
                    onChange: ( value ) => setAttributes( { selectedSpaces: value } ),
                } ),
                createElement( RangeControl, {
                    label: __( 'Number of Columns', 'fca-pages' ),
                    value: numberOfColumns,
                    onChange: ( value ) => setAttributes( { numberOfColumns: value } ),
                    min: 0,
                    max: 6, // Allow up to 6 columns, 0 for auto
                    step: 1,
                    help: __( 'Set to 0 for automatic column wrapping.', 'fca-pages' ),
                })
            ),
            createElement( PanelBody, { title: __( 'Visibility Settings', 'fca-pages' ), initialOpen: false },
                 createElement( ToggleControl, {
                    label: __( 'Show Cover Image', 'fca-pages' ),
                    checked: showImage,
                    onChange: ( value ) => setAttributes( { showImage: value } ),
                } ),
                createElement( ToggleControl, {
                    label: __( 'Show Description', 'fca-pages' ),
                    checked: showDescription,
                    onChange: ( value ) => setAttributes( { showDescription: value } ),
                } ),
                createElement( ToggleControl, {
                    label: __( 'Show Privacy (Public/Private)', 'fca-pages' ),
                    checked: showPrivacy,
                    onChange: ( value ) => setAttributes( { showPrivacy: value } ),
                } ),
                createElement( ToggleControl, {
                    label: __( 'Show Member Count', 'fca-pages' ),
                    checked: showMemberCount,
                    onChange: ( value ) => setAttributes( { showMemberCount: value } ),
                } ),
                 createElement( ToggleControl, {
                    label: __( 'Show Post Count', 'fca-pages' ),
                    checked: showPostCount,
                    onChange: ( value ) => setAttributes( { showPostCount: value } ),
                } )
            ),
            createElement( PanelBody, { title: __( 'Card Style Settings', 'fca-pages' ), initialOpen: false },
                 createElement( 'p', null, __( 'Background Color', 'fca-pages' ) ),
                 createElement( ColorPalette, { 
                     colors: colorPalette,
                     value: cardBackgroundColor,
                     onChange: ( color ) => setAttributes( { cardBackgroundColor: color } ),
                     clearable: true,
                     disableCustomColors: false,
                 } ),
                 createElement( 'p', { style: { marginTop: '10px' } }, __( 'Border Color', 'fca-pages' ) ),
                 createElement( ColorPalette, { 
                     colors: colorPalette,
                     value: cardBorderColor,
                     onChange: ( color ) => setAttributes( { cardBorderColor: color } ),
                     clearable: true,
                     disableCustomColors: false,
                 } ),
                 createElement( RangeControl, {
                    label: __( 'Border Radius (px)', 'fca-pages' ),
                    value: cardBorderRadius,
                    onChange: ( value ) => setAttributes( { cardBorderRadius: value } ),
                    min: 0,
                    max: 50, // Adjust max as needed
                    step: 1,
                 })
            )
        );

        // Block Controls (Alignment) using createElement
        const blockControls = createElement( BlockControls, null,
            createElement( AlignmentToolbar, {
                value: align,
                onChange: ( newAlign ) => setAttributes( { align: newAlign } ),
            } )
        );

        // Main Edit structure using createElement
        return createElement( Fragment, null, // Use Fragment instead of <>
            blockControls, // Add Block Controls
            inspectorControls, // Add Inspector Controls
            createElement( 'div', blockProps, // The main block wrapper
                createElement( ServerSideRender, {
                    block: metadata.name,
                    attributes: attributes,
                } )
            )
        );
    };

    // --- Register Block Type --- 
    registerBlockType( metadata.name, {
        edit: EditComponent,
        // No save function needed due to server-side rendering
    } );

} )( window.wp ); 