<?php
/**
 * Server-side rendering for the FCA Pages List block.
 */

defined('ABSPATH') || exit;

// --- Helper function to get Page URL --- 
if (!function_exists('fca_get_page_url')) {
    function fca_get_page_url($page_slug) {
        if (empty($page_slug)) {
            return '#';
        }
        $relative_path = "/page/{$page_slug}"; // Use singular page slug

        $fc_helper_exists = class_exists('\FluentCommunity\App\Services\Helper');

        if ($fc_helper_exists) {
            return \FluentCommunity\App\Services\Helper::baseUrl($relative_path);
        } else {
            $portal_slug = apply_filters('fluent_community/portal_slug', 'portal'); 
            return home_url("/{$portal_slug}{$relative_path}");
        }
    }
}

// --- Get Attributes --- 
$display_mode = $attributes['displayMode'] ?? 'all';
$selected_pages_str = $attributes['selectedPages'] ?? '';
$number_of_pages = $attributes['numberOfPages'] ?? 6; 
$show_image = $attributes['showImage'] ?? true;
$show_description = $attributes['showDescription'] ?? true;
$alignment = $attributes['align'] ?? '';
$card_bg_color = $attributes['cardBackgroundColor'] ?? '';
$card_border_color = $attributes['cardBorderColor'] ?? '';
$card_border_radius = $attributes['cardBorderRadius'] ?? 8;
$number_of_columns = $attributes['numberOfColumns'] ?? 0;

// --- Fetch Pages --- 
$all_pages_query_args = [
    'post_type' => 'fca_page',
    'post_status' => 'publish',
    'numberposts' => -1, // Get all initially, filter later
    'orderby' => 'date', // Or 'title', 'menu_order'
    'order' => 'DESC'
];

$all_pages = get_posts($all_pages_query_args);

// --- Filter Pages Based on Attributes --- 
$final_pages = [];
if ($display_mode === 'specific') {
    $page_ids = array_map('intval', array_filter(explode(',', $selected_pages_str)));
    if (!empty($page_ids)) {
        $keyed_pages = [];
        foreach ($all_pages as $page) {
             $keyed_pages[$page->ID] = $page;
        }
        foreach ($page_ids as $id) {
            if (isset($keyed_pages[$id])) {
                $final_pages[] = $keyed_pages[$id];
            }
        }
    } 
} else { // 'all' mode
    $limit = max(1, intval($number_of_pages));
    $final_pages = array_slice($all_pages, 0, $limit);
}

// --- Render Block --- 
if (empty($final_pages)) {
    if ($display_mode === 'specific' && empty($page_ids)){
         echo '<p>' . esc_html__('Please enter valid Page IDs in the block settings.', 'fca-pages') . '</p>';
    } else {
        echo '<p>' . esc_html__('No published pages found matching the criteria.', 'fca-pages') . '</p>';
    }
    return;
}

// Placeholder image (use a generic one or create one for pages)
// Using the space one for now
$placeholder_img = plugins_url('../fca-spaces-list/assets/images/space-placeholder.jpg', __FILE__); 
if (!file_exists(plugin_dir_path(__FILE__) . '../fca-spaces-list/assets/images/space-placeholder.jpg')) {
    $placeholder_img = ''; 
}

$wrapper_attributes = get_block_wrapper_attributes(['class' => 'fca-pages-list-wrapper']);

// --- Generate Grid Class --- 
$grid_classes = 'fca-pages-list fca-spaces-list-grid'; // Base classes
if ($number_of_columns > 0 && $number_of_columns <= 6) { // Add class only for 1-6 columns
    $grid_classes .= ' fca-grid-cols-' . intval($number_of_columns);
}

// Reuse the grid class from spaces list
echo '<div ' . $wrapper_attributes . '>'; 
// Apply the generated classes
echo '<div class="' . esc_attr($grid_classes) . '">'; // NO inline style

foreach ($final_pages as $page) {
    $page_id = $page->ID;
    $page_title = esc_html($page->post_title);
    $page_slug = $page->post_name;
    $page_url = fca_get_page_url($page_slug);
    
    // Get excerpt or trim content for description
    $page_description = '';
    if ($show_description) {
        if (has_excerpt($page_id)) {
            $page_description = esc_html(get_the_excerpt($page_id));
        } else {
            // Trimmed content as fallback
            $page_description = wp_trim_words(wp_kses_post(strip_shortcodes($page->post_content)), 15, '...');
        }
    }

    // --- Get Featured Image --- 
    $display_image = $placeholder_img; // Default to placeholder
    if ($show_image && has_post_thumbnail($page_id)) {
        $thumbnail_url = get_the_post_thumbnail_url($page_id, 'medium_large'); // Or other size like 'large', 'thumbnail'
        if ($thumbnail_url) {
            $display_image = $thumbnail_url;
        }
    }

    // --- Apply Card Styles --- 
    $card_styles = '';
    if (!empty($card_bg_color)) {
        $card_styles .= 'background-color:' . esc_attr($card_bg_color) . ';';
    }
    if (!empty($card_border_color)) {
        $card_styles .= 'border-color:' . esc_attr($card_border_color) . ';';
    }
    if (isset($card_border_radius) && is_numeric($card_border_radius)) { 
        $card_styles .= 'border-radius:' . intval($card_border_radius) . 'px;';
    }
    $card_style_attr = !empty($card_styles) ? ' style="' . $card_styles . '"' : '';

    // --- Render Card HTML (Similar to Space Card) --- 
    // Reuse fca-space-card classes for styling consistency
    echo '<div role="region" aria-label="Page Card" class="fca-page-card fca-space-card"' . $card_style_attr . '>';
    
    // Avatar/Image Section (Conditional)
    if ($show_image) {
        echo '<div class="fca-space-card-avatar">'; // Reuse class
        echo '<a href="' . esc_url($page_url) . '" aria-label="Go to Page">';
        echo '<img src="' . esc_url($display_image) . '" alt="' . esc_attr($page_title) . ' featured image" loading="lazy">'; 
        echo '</a></div>';
    }
    
    // Info Section
    echo '<div class="fca-space-card-info">'; // Reuse class
    // -- Name
    echo '<div class="fca-space-card-name">'; // Reuse class
    echo '<a href="' . esc_url($page_url) . '" class="">' . $page_title . '</a>'; 
    echo '</div>';
    // -- Details (Removed for pages) --
    
    // -- Description (Conditional)
    if ($show_description && !empty($page_description)) {
        echo '<p class="fca-space-card-description">' . $page_description . '</p>'; // Reuse class
    }
    echo '</div>'; // End info

    // Action Section
    echo '<div class="fca-space-card-action">'; // Reuse class
    echo '<a href="' . esc_url($page_url) . '" class="fca-space-card-button">' . __('View Page', 'fca-pages') . '</a>'; // Reuse class
    echo '</div>';

    echo '</div>'; // End fca-page-card
}

echo '</div>'; // End fca-pages-list grid
echo '</div>'; // End wrapper 