/**
 * Editor Script for FCA Image Block (No Build Step Version)
 */
( function( wp ) {
    const { __ } = wp.i18n;
    const { registerBlockType } = wp.blocks;
    const { 
        InspectorControls, 
        BlockControls, 
        AlignmentToolbar, 
        MediaPlaceholder, 
        BlockIcon,
        useBlockProps
    } = wp.blockEditor;
    const { 
        PanelBody, 
        RangeControl, 
        TextControl, 
        Button, 
        Spinner 
    } = wp.components;
    const { createElement, Fragment } = wp.element; 
    const { useSelect } = wp.data; // Needed to check if image exists
    const metadata = { /* block.json data copy - required for direct registration */ 
        "name": "fca-pages/fca-image",
        "attributes": { 
            "url": { "type": "string", "source": "attribute", "selector": "img", "attribute": "src" },
            "alt": { "type": "string", "source": "attribute", "selector": "img", "attribute": "alt", "default": "" },
            "id": { "type": "number" },
            "width": { "type": "number" },
            "height": { "type": "number" },
            "align": { "type": "string", "default": "none" },
            "borderRadius": { "type": "number", "default": 8 }
        },
        // Other necessary metadata fields can be copied if needed
    };

    // --- Edit Component --- 
    const EditComponent = ( props ) => {
        const { attributes, setAttributes, isSelected, className } = props;
        const { url, alt, id, align, borderRadius, width, height } = attributes;
        const blockProps = useBlockProps({ className: align && align !== 'none' ? `align${align}` : '' });

        const onSelectImage = ( media ) => {
            if ( ! media || ! media.url ) {
                setAttributes( { url: undefined, id: undefined, alt: '' } );
                return;
            }
            setAttributes( { 
                url: media.url, 
                id: media.id, 
                alt: media.alt || '' // Use existing alt or empty string
            } );
        };

        const onRemoveImage = () => {
            setAttributes( { url: undefined, id: undefined, alt: '' } );
        };

        const onUploadError = ( message ) => {
            console.error( 'Image Upload Error:', message );
            // Maybe add a notice?
        };

        // Check if the selected image ID still exists in the media library
        const imageObject = useSelect( ( select ) => {
                const { getMedia } = select( 'core' );
                return id ? getMedia( id ) : null;
            }, [ id ] );

        // If we have an ID but the image object isn't available (still loading or deleted)
        const imageIsMissing = id && !imageObject;
        // If we have a URL but no ID (e.g., external image) or the imageObject is loaded
        const showImagePreview = !!url && (!id || imageObject);

        // Inspector Controls
        const inspectorControls = createElement( InspectorControls, null,
            createElement( PanelBody, { title: __( 'Image Settings', 'fca-pages' ) },
                createElement( TextControl, {
                    label: __( 'Alt Text', 'fca-pages' ),
                    value: alt,
                    onChange: ( value ) => setAttributes( { alt: value } ),
                    help: __( 'Describe the purpose of the image. Leave empty if decorative.', 'fca-pages' ),
                } ),
                createElement( 'div', { style: { display: 'flex', gap: '8px', marginBottom: '10px' } },
                    createElement( TextControl, {
                        label: __( 'Width', 'fca-pages' ),
                        type: 'number',
                        value: width,
                        onChange: ( value ) => setAttributes( { width: parseInt( value, 10 ) || undefined } ),
                        min: 1,
                    } ),
                    createElement( TextControl, {
                        label: __( 'Height', 'fca-pages' ),
                        type: 'number',
                        value: height,
                        onChange: ( value ) => setAttributes( { height: parseInt( value, 10 ) || undefined } ),
                        min: 1,
                    } )
                ),
                createElement( RangeControl, {
                    label: __( 'Border Radius (px)', 'fca-pages' ),
                    value: borderRadius,
                    onChange: ( value ) => setAttributes( { borderRadius: value } ),
                    min: 0,
                    max: 50, 
                    step: 1,
                 })
            )
        );

        // Block Controls
        const blockControls = createElement( BlockControls, null, 
            createElement( AlignmentToolbar, {
                value: align,
                onChange: ( newAlign ) => setAttributes( { align: newAlign === undefined ? 'none' : newAlign } ), // Handle undefined case
            } )
        );

        // --- Image Styles for Preview --- 
        const imgPreviewStyles = {
            borderRadius: borderRadius + 'px',
            maxWidth: '100%',
            height: 'auto',
            width: 'auto'
        };
        if (width) {
            imgPreviewStyles.width = width + 'px';
        }
        if (height) {
            imgPreviewStyles.height = height + 'px';
        }

        return createElement( Fragment, null,
            blockControls,
            inspectorControls,
            createElement( 'div', blockProps, 
                ! url ? (
                    // --- Placeholder --- 
                    createElement( MediaPlaceholder, {
                        icon: createElement( BlockIcon, { icon: metadata.icon || 'format-image' } ),
                        labels: { 
                            title: __( 'FCA Image', 'fca-pages' ), 
                            instructions: __( 'Upload an image file, pick one from your media library, or add one with a URL.', 'fca-pages' ) 
                        },
                        onSelect: onSelectImage,
                        accept: 'image/*',
                        allowedTypes: [ 'image' ],
                        onError: onUploadError,
                    } )
                ) : (
                    // --- Image Preview --- 
                    createElement( Fragment, null, 
                        showImagePreview && !imageIsMissing && createElement( 'img', {
                            src: url,
                            alt: alt,
                            style: imgPreviewStyles
                        } ),
                        imageIsMissing && createElement( 'p', { className: 'fca-image-missing' }, __( 'Image not found in media library.', 'fca-pages' ) ),
                        isSelected && createElement( Button, { 
                            variant: "secondary", 
                            onClick: onRemoveImage, 
                            style: { display: 'block', marginTop: '10px' } 
                        }, __( 'Remove Image', 'fca-pages' ) )
                    )
                )
            )
        );
    };

    // --- Save Component --- 
    const SaveComponent = ( props ) => {
        const { attributes } = props;
        const { url, alt, id, align, borderRadius, width, height } = attributes;

        if ( ! url ) {
            return null; // Don't save anything if no image URL
        }

        const wrapperClasses = align && align !== 'none' ? `wp-block-fca-pages-fca-image align${align}` : 'wp-block-fca-pages-fca-image';
        const figureClasses = align && align !== 'none' ? `align${align}` : ''; // Core WP sometimes uses class on figure too
        const imgStyles = {
            borderRadius: borderRadius + 'px'
        };

        // Image attributes (only add width/height if they have a value)
        const imgAttributes = { 
            src: url, 
            alt: alt, 
            className: id ? `wp-image-${id}` : null, 
            style: imgStyles
        };
        if (width) {
            imgAttributes.width = width;
        }
        if (height) {
            imgAttributes.height = height;
        }

        return createElement( 'figure', { className: wrapperClasses + ' ' + figureClasses }, 
            createElement( 'img', imgAttributes )
        );
    };

    // --- Register Block Type --- 
    registerBlockType( metadata.name, {
        edit: EditComponent,
        save: SaveComponent, 
        icon: metadata.icon // Provide icon here too if not fetching full metadata
    } );

} )( window.wp ); 