<?php
/**
 * Server-side rendering for the FCA Courses List block.
 */

defined('ABSPATH') || exit;

// --- Helper function to get Course URL --- 
if (!function_exists('fca_get_course_url')) {
    function fca_get_course_url($course) { // Changed to accept full course object
        if (empty($course)) {
            return '#';
        }
        
        // Get course slug - check multiple possible properties
        $course_slug = '';
        if (isset($course->slug) && !empty($course->slug)) {
            $course_slug = $course->slug;
        } elseif (isset($course->permalink) && !empty($course->permalink)) {
            $course_slug = $course->permalink;
        } elseif (isset($course->post_name) && !empty($course->post_name)) {
            $course_slug = $course->post_name;
        } else {
            // Fallback to ID if no slug is available
            $course_slug = $course->id ?? 0;
        }
        
        // For debugging - log which property was used
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $source = 'unknown';
            if (isset($course->slug) && !empty($course->slug)) {
                $source = 'slug';
            } elseif (isset($course->permalink) && !empty($course->permalink)) {
                $source = 'permalink';
            } elseif (isset($course->post_name) && !empty($course->post_name)) {
                $source = 'post_name';
            } else {
                $source = 'id (fallback)';
            }
            error_log('FCA Course URL generated using ' . $source . ' property: ' . $course_slug);
        }
        
        // Use the correct URL structure: course/COURSE-SLUG/lessons
        $relative_path = "/course/{$course_slug}/lessons";

        $fc_helper_exists = class_exists('\FluentCommunity\App\Services\Helper');

        if ($fc_helper_exists) {
            return \FluentCommunity\App\Services\Helper::baseUrl($relative_path);
        } else {
            $portal_slug = apply_filters('fluent_community/portal_slug', 'portal'); 
            return home_url("/{$portal_slug}{$relative_path}");
        }
    }
}

// --- Get Attributes --- 
$display_mode = $attributes['displayMode'] ?? 'all';
$selected_courses_str = $attributes['selectedCourses'] ?? '';
$number_of_courses = $attributes['numberOfCourses'] ?? 6; 
$show_image = $attributes['showImage'] ?? true;
$show_description = $attributes['showDescription'] ?? true;
$alignment = $attributes['align'] ?? '';
$card_bg_color = $attributes['cardBackgroundColor'] ?? '';
$card_border_color = $attributes['cardBorderColor'] ?? '';
$card_border_radius = $attributes['cardBorderRadius'] ?? 8;
$number_of_columns = $attributes['numberOfColumns'] ?? 0;

// --- Fetch Courses (Using Correct Method) --- 
$all_courses = [];
$fetch_error = '';

// Use the dedicated getCourses function
if (class_exists('\FluentCommunity\App\Functions\Utility') && method_exists('\FluentCommunity\App\Functions\Utility', 'getCourses')) {
    try {
        // Fetch courses directly, not grouped
        $courses_collection = \FluentCommunity\App\Functions\Utility::getCourses(false); 
        
        if ($courses_collection && method_exists($courses_collection, 'all')) {
            $courses_array = $courses_collection->all(); 
            
             // --- REMOVED DEBUG LOGGING --- 
             // error_log('[FCA Courses List Debug] Raw data from Utility::getCourses(): ' . print_r($courses_array, true));

            // Filter only by published status if necessary (assuming getCourses doesn't already)
            foreach ($courses_array as $course_item) {
                $course_obj = (object) $course_item;
                // Check if status is published (getCourses might already filter this)
                if (isset($course_obj->status) && $course_obj->status === 'published') { 
                    $all_courses[] = $course_obj;
                } else if (!isset($course_obj->status)) {
                    // Assume published if status property doesn't exist (for safety)
                     $all_courses[] = $course_obj;
                }
            }
            
            // Apply sorting if needed (might already be sorted)
            usort($all_courses, function($a, $b) {
                 $time_a = isset($a->created_at) ? strtotime($a->created_at) : 0;
                 $time_b = isset($b->created_at) ? strtotime($b->created_at) : 0;
                 return $time_b <=> $time_a; // Newest first
            });

        } elseif ($courses_collection) {
             $fetch_error = 'Utility::getCourses() returned unexpected format.';
        } else {
             $fetch_error = 'Utility::getCourses() returned no results.';
        }
    } catch (\Exception $e) {
        $fetch_error = 'Error calling Utility::getCourses(): ' . $e->getMessage();
        error_log('FCA Courses List Block Error: ' . $fetch_error);
    }
} else {
    $fetch_error = 'Fluent Community Utility class or getCourses method not found.';
}

if (!empty($fetch_error) && empty($all_courses)) { 
     echo '<p>' . esc_html__('Error fetching courses:', 'fca-pages') . ' ' . esc_html($fetch_error) . '</p>';
     return;
}

// --- Filter Courses Based on Attributes --- 
$final_courses = [];
if ($display_mode === 'specific') {
    $course_ids = array_map('intval', array_filter(explode(',', $selected_courses_str)));
    if (!empty($course_ids)) {
        $keyed_courses = [];
        foreach ($all_courses as $course) {
             if (isset($course->id)) { 
                 $keyed_courses[$course->id] = $course;
             }
        }
        foreach ($course_ids as $id) {
            if (isset($keyed_courses[$id])) {
                $final_courses[] = $keyed_courses[$id];
            }
        }
    } 
} else { // 'all' mode
    $limit = max(1, intval($number_of_courses));
    $final_courses = array_slice($all_courses, 0, $limit);
}

// --- Render Block --- 
if (empty($final_courses)) {
    if ($display_mode === 'specific' && empty($course_ids)){
         echo '<p>' . esc_html__('Please enter valid Course IDs in the block settings.', 'fca-pages') . '</p>';
    } else {
        echo '<p>' . esc_html__('No published courses found matching the criteria.', 'fca-pages') . '</p>';
    }
    return;
}

// Placeholder image - Use correct path for courses
$placeholder_img_path = 'fluent-community/assets/images/course-placeholder.jpg';
$placeholder_img_url = plugins_url($placeholder_img_path);
if (!file_exists(WP_PLUGIN_DIR . '/' . $placeholder_img_path)) {
    $placeholder_img_url = ''; // Fallback if not found
}

$wrapper_attributes = get_block_wrapper_attributes(['class' => 'fca-courses-list-wrapper']);

// --- Generate Grid Class --- 
$grid_classes = 'fca-courses-list fca-spaces-list-grid'; // Base classes
if ($number_of_columns > 0 && $number_of_columns <= 6) { // Add class only for 1-6 columns
    $grid_classes .= ' fca-grid-cols-' . intval($number_of_columns);
}

// Reuse the grid class from spaces list
echo '<div ' . $wrapper_attributes . '>'; 
// Apply the generated classes
echo '<div class="' . esc_attr($grid_classes) . '">'; // NO inline style

// DEBUG: Output first course object properties if WP_DEBUG is enabled
if (defined('WP_DEBUG') && WP_DEBUG && !empty($final_courses)) {
    $first_course = $final_courses[0];
    error_log('FCA Course List - First course object structure: ' . print_r($first_course, true));
}

foreach ($final_courses as $course) {
    $course_id = $course->id ?? 0;
    $course_title = esc_html($course->name ?? $course->title ?? 'Untitled Course'); // Use name or title
    $course_url = fca_get_course_url($course);
    
    // Get description (assuming short_description or description field)
    $course_description_raw = $course->description ?? ($course->short_description ?? '');
    $course_description = $show_description ? wp_trim_words(wp_kses_post($course_description_raw), 15, '...') : '';

    // --- Get Course Image --- 
    // Check cover_photo field first, then use placeholder
    $display_image = $placeholder_img_url; // Default to placeholder
    $course_image_url_raw = $course->cover_photo ?? ''; 

    if ($show_image) {
        // Use cover_photo if it exists and is a valid URL
        if (!empty($course_image_url_raw) && filter_var($course_image_url_raw, FILTER_VALIDATE_URL)) {
             $display_image = $course_image_url_raw;
        } 
        // Removed fallback to settings meta as cover_photo is the primary source
    }

    // --- Apply Card Styles --- 
    $card_styles = '';
    if (!empty($card_bg_color)) {
        $card_styles .= 'background-color:' . esc_attr($card_bg_color) . ';';
    }
    if (!empty($card_border_color)) {
        $card_styles .= 'border-color:' . esc_attr($card_border_color) . ';';
    }
    if (isset($card_border_radius) && is_numeric($card_border_radius)) { 
        $card_styles .= 'border-radius:' . intval($card_border_radius) . 'px;';
    }
    $card_style_attr = !empty($card_styles) ? ' style="' . $card_styles . '"' : '';

    // --- Render Card HTML (Similar to Space Card) --- 
    echo '<div role="region" aria-label="Course Card" class="fca-course-card fca-space-card"' . $card_style_attr . '>';
    
    // Avatar/Image Section (Conditional)
    if ($show_image) {
        echo '<div class="fca-space-card-avatar">'; // Reuse class
        echo '<a href="' . esc_url($course_url) . '" aria-label="Go to Course">';
        echo '<img src="' . esc_url($display_image) . '" alt="' . esc_attr($course_title) . ' image" loading="lazy">'; 
        echo '</a></div>';
    }
    
    // Info Section
    echo '<div class="fca-space-card-info">'; // Reuse class
    // -- Name
    echo '<div class="fca-space-card-name">'; // Reuse class
    echo '<a href="' . esc_url($course_url) . '" class="">' . $course_title . '</a>'; 
    echo '</div>';
    // -- Details (Removed for courses) --
    
    // -- Description (Conditional)
    if ($show_description && !empty($course_description)) {
        echo '<p class="fca-space-card-description">' . $course_description . '</p>'; // Reuse class
    }
    echo '</div>'; // End info

    // Action Section
    echo '<div class="fca-space-card-action">'; // Reuse class
    echo '<a href="' . esc_url($course_url) . '" class="fca-space-card-button">' . __('View Course', 'fca-pages') . '</a>'; // Reuse class
    echo '</div>';

    echo '</div>'; // End fca-course-card
}

echo '</div>'; // End fca-courses-list grid
echo '</div>'; // End wrapper 