/**
 * Editor Script for FCA Courses List Block (No Build Step Version)
 */
( function( wp ) {
    const { __ } = wp.i18n;
    const { registerBlockType } = wp.blocks;
    const { InspectorControls, useBlockProps, BlockControls, AlignmentToolbar, ColorPalette } = wp.blockEditor;
    const { PanelBody, SelectControl, TextControl, TextareaControl, ToggleControl, RangeControl } = wp.components;
    const ServerSideRender = wp.serverSideRender;
    const { createElement, Fragment } = wp.element; 
    const metadata = { 
        "name": "fca-pages/fca-courses-list",
        "attributes": {
             "displayMode": { "type": "string", "default": "all" },
             "selectedCourses": { "type": "string", "default": "" },
             "numberOfCourses": { "type": "number", "default": 6 },
             "showImage": { "type": "boolean", "default": true },
             "showDescription": { "type": "boolean", "default": true },
             "align": { "type": "string", "default": "" },
             "cardBackgroundColor": { "type": "string" },
             "cardBorderColor": { "type": "string" },
             "cardBorderRadius": { "type": "number", "default": 8 },
             "numberOfColumns": { "type": "number", "default": 0 }
        },
    }; 

    // --- Edit Component --- 
    const EditComponent = ( props ) => {
        const { attributes, setAttributes } = props;
        const {
            displayMode, selectedCourses, numberOfCourses, align, 
            showImage, showDescription, 
            cardBackgroundColor, cardBorderColor, cardBorderRadius,
            numberOfColumns
        } = attributes;
        const blockProps = useBlockProps();
        const colorPalette = wp.data.select( "core/block-editor" ).getSettings().colors;

        // Inspector Controls 
        const inspectorControls = createElement( InspectorControls, null, 
            createElement( PanelBody, { title: __( 'Display Settings', 'fca-pages' ), initialOpen: true },
                createElement( SelectControl, {
                    label: __( 'Display Mode', 'fca-pages' ),
                    value: displayMode,
                    options: [
                        { label: __( 'Show All Published Courses', 'fca-pages' ), value: 'all' },
                        { label: __( 'Show Specific Courses', 'fca-pages' ), value: 'specific' },
                    ],
                    onChange: ( value ) => setAttributes( { displayMode: value } ),
                } ),
                displayMode === 'all' && createElement( TextControl, {
                    label: __( 'Number of Courses', 'fca-pages' ),
                    type: "number",
                    value: numberOfCourses,
                    onChange: ( value ) => setAttributes( { numberOfCourses: parseInt( value, 10 ) || 1 } ),
                    min: "1",
                } ),
                displayMode === 'specific' && createElement( TextareaControl, {
                    label: __( 'Course IDs', 'fca-pages' ),
                    help: __( 'Enter comma-separated IDs of the courses to display.', 'fca-pages' ),
                    value: selectedCourses,
                    onChange: ( value ) => setAttributes( { selectedCourses: value } ),
                 } ),
                 createElement( RangeControl, {
                    label: __( 'Number of Columns', 'fca-pages' ),
                    value: numberOfColumns,
                    onChange: ( value ) => setAttributes( { numberOfColumns: value } ),
                    min: 0,
                    max: 6, 
                    step: 1,
                    help: __( 'Set to 0 for automatic column wrapping.', 'fca-pages' ),
                 })
            ),
            createElement( PanelBody, { title: __( 'Visibility Settings', 'fca-pages' ), initialOpen: false },
                 createElement( ToggleControl, {
                    label: __( 'Show Featured Image', 'fca-pages' ),
                    checked: showImage,
                    onChange: ( value ) => setAttributes( { showImage: value } ),
                } ),
                createElement( ToggleControl, {
                    label: __( 'Show Description', 'fca-pages' ),
                    checked: showDescription,
                    onChange: ( value ) => setAttributes( { showDescription: value } ),
                } )
            ),
            createElement( PanelBody, { title: __( 'Card Style Settings', 'fca-pages' ), initialOpen: false },
                 createElement( 'p', null, __( 'Background Color', 'fca-pages' ) ),
                 createElement( ColorPalette, { 
                     colors: colorPalette,
                     value: cardBackgroundColor,
                     onChange: ( color ) => setAttributes( { cardBackgroundColor: color } ),
                     clearable: true,
                     disableCustomColors: false,
                 } ),
                 createElement( 'p', { style: { marginTop: '10px' } }, __( 'Border Color', 'fca-pages' ) ),
                 createElement( ColorPalette, { 
                     colors: colorPalette,
                     value: cardBorderColor,
                     onChange: ( color ) => setAttributes( { cardBorderColor: color } ),
                     clearable: true,
                     disableCustomColors: false,
                 } ),
                 createElement( RangeControl, {
                    label: __( 'Border Radius (px)', 'fca-pages' ),
                    value: cardBorderRadius,
                    onChange: ( value ) => setAttributes( { cardBorderRadius: value } ),
                    min: 0,
                    max: 50, 
                    step: 1,
                 })
            )
        );

        // Block Controls (Alignment)
        const blockControls = createElement( BlockControls, null,
            createElement( AlignmentToolbar, {
                value: align,
                onChange: ( newAlign ) => setAttributes( { align: newAlign } ),
            } )
        );

        return createElement( Fragment, null, 
            blockControls, 
            inspectorControls, 
            createElement( 'div', blockProps, 
                createElement( ServerSideRender, {
                    block: metadata.name,
                    attributes: attributes,
                } )
            )
        );
    };

    // --- Register Block Type --- 
    registerBlockType( metadata.name, {
        edit: EditComponent,
    } );

} )( window.wp ); 