(function(wp) {
    const { __ } = wp.i18n;
    const { registerBlockType } = wp.blocks;
    const { InspectorControls, useBlockProps, RichText, URLInputButton } = wp.blockEditor;
    const { PanelBody, SelectControl, ToggleControl } = wp.components;
    const el = wp.element.createElement;

    registerBlockType('fca-pages/button', {
        apiVersion: 3,
        title: __('FC Button', 'fca-pages'),
        icon: 'button',
        category: 'fca-blocks',
        attributes: {
            text: { type: 'string', default: 'Click Here' },
            url: { type: 'string', default: '#' },
            linkTarget: { type: 'string', default: '_blank' },
            buttonType: { type: 'string', default: 'primary' },
            align: { type: 'string' }
        },
        supports: {
            html: false,
            align: ["left", "center", "right"]
        },

        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { text, url, linkTarget, buttonType, align } = attributes;
            const blockProps = useBlockProps({ className: align ? `has-text-align-${align}` : '' });

            let buttonClass = 'el-button';
            if (buttonType === 'primary') buttonClass += ' fcom_primary_button';
            else if (buttonType === 'secondary') buttonClass += ' fcom_secondary_button';
            else if (buttonType === 'text') buttonClass += ' el-button--text';
            else if (buttonType === 'link') buttonClass = 'fcom_link';

            return el(
                wp.element.Fragment, 
                {}, 
                el(
                    InspectorControls,
                    null,
                    el(
                        PanelBody, 
                        { title: __('Button Settings', 'fca-pages') },
                        el(SelectControl, {
                            label: __('Button Type', 'fca-pages'),
                            value: buttonType,
                            options: [
                                { label: __('Primary', 'fca-pages'), value: 'primary' },
                                { label: __('Secondary', 'fca-pages'), value: 'secondary' },
                                { label: __('Text', 'fca-pages'), value: 'text' },
                                { label: __('Link', 'fca-pages'), value: 'link' }
                            ],
                            onChange: function(newType) { setAttributes({ buttonType: newType }); }
                        }),
                        el(ToggleControl, {
                            label: __('Open in new tab', 'fca-pages'),
                            checked: linkTarget === '_blank',
                            onChange: function(isChecked) { setAttributes({ linkTarget: isChecked ? '_blank' : '_self' }); }
                        })
                    )
                ),
                el(
                    'div', 
                    blockProps,
                    el(RichText, {
                        tagName: buttonType === 'link' ? 'span' : 'a',
                        className: buttonClass,
                        value: text,
                        onChange: function(newText) { setAttributes({ text: newText }); },
                        placeholder: __('Button Text...', 'fca-pages'),
                        withoutInteractiveFormatting: true,
                        allowedFormats: [], // Prevent formatting
                        style: { display: 'inline-block' }, // Editor styling
                        // Prevent link click in editor
                        onClick: buttonType !== 'link' ? function(e) { e.preventDefault(); } : undefined 
                    }),
                    el(URLInputButton, {
                        url: url,
                        onChange: function(newUrl) { setAttributes({ url: newUrl }); }
                    })
                )
            );
        },

        save: function(props) {
            const { attributes } = props;
            const { text, url, linkTarget, buttonType, align } = attributes;
            const blockProps = useBlockProps.save({
                className: align ? `has-text-align-${align}` : ''
            });

            let buttonClass = 'el-button';
            if (buttonType === 'primary') buttonClass += ' fcom_primary_button';
            else if (buttonType === 'secondary') buttonClass += ' fcom_secondary_button';
            else if (buttonType === 'text') buttonClass += ' el-button--text';
            else if (buttonType === 'link') buttonClass = 'fcom_link';

            return el(
                'div', 
                blockProps,
                el(RichText.Content, {
                    tagName: 'a',
                    href: url,
                    target: linkTarget,
                    rel: linkTarget === '_blank' ? 'noopener noreferrer' : null,
                    className: buttonClass,
                    value: text
                })
            );
        }
    });
})(window.wp); 