<?php
/**
 * Fluent Community Theme functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package FluentCommunity_Theme
 */

if ( ! defined( '_S_VERSION' ) ) {
	// Replace with your theme version (matches style.css). Change this for cache busting.
	define( '_S_VERSION', '1.0.0' );
}

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function fluentcommunity_theme_setup() {
	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 */
	load_theme_textdomain( 'fluentcommunity-theme', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
	 */
	add_theme_support( 'post-thumbnails' );

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus(
		array(
			'menu-1' => esc_html__( 'Primary', 'fluentcommunity-theme' ), // For fallback if needed
            'fcom-header-menu' => esc_html__( 'FluentCommunity Header Menu', 'fluentcommunity-theme' ), // New location for FC header
		)
	);

	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support(
		'html5',
		array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
			'style',
			'script',
		)
	);

	// Set up the WordPress core custom background feature.
	add_theme_support(
		'custom-background',
		apply_filters(
			'fluentcommunity_theme_custom_background_args',
			array(
				'default-color' => 'ffffff',
				'default-image' => '',
			)
		)
	);

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );

	/**
	 * Add support for core custom logo.
	 *
	 * @link https://codex.wordpress.org/Theme_Logo
	 */
	add_theme_support(
		'custom-logo',
		array(
			'height'      => 250,
			'width'       => 250,
			'flex-width'  => true,
			'flex-height' => true,
		)
	);
}
add_action( 'after_setup_theme', 'fluentcommunity_theme_setup' );

/**
 * Enqueue scripts and styles.
 */
function fluentcommunity_theme_scripts() {

    // Define plugin path (adjust if FluentCommunity folder name is different)
    $fc_plugin_folder = 'fluent-community'; // ** CHECK THIS FOLDER NAME **
    $fc_plugin_url = plugins_url() . '/' . $fc_plugin_folder . '/assets/';

    // Enqueue FluentCommunity core styles
    // Re-enabling these manual enqueues.
    wp_enqueue_style( 'fluentcommunity-theme-default', $fc_plugin_url . 'theme-default.css', array(), null ); // Use null for version to let plugin manage it, or add a version number
    wp_enqueue_style( 'fluentcommunity-theme-global', $fc_plugin_url . 'global.css', array('fluentcommunity-theme-default'), null );
    wp_enqueue_style( 'fluentcommunity-theme-app', $fc_plugin_url . 'app.css', array('fluentcommunity-theme-global'), null );

	// Enqueue the theme's main stylesheet for overrides.
    // Make it dependent on fluentcommunity-app again.
	wp_enqueue_style( 'fluentcommunity-theme-style',
        get_stylesheet_uri(), 
        array('fluentcommunity-theme-app'), // Restore dependency on 'fluentcommunity-app'
        _S_VERSION 
    );

	// If using threaded comments.
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

    // Add other scripts here if needed for the theme itself (not FC integration)

    // Manually enqueue portal_general.js for basic interactivity
    // Ensure loadClassicPortalAssets is commented out in the template
    $portal_general_script_url = $fc_plugin_url . 'portal_general.js';
    wp_enqueue_script('fluentcommunity-theme-portal-general', $portal_general_script_url, array(), null, true); // Load in footer

    // Simplified inline script to ensure mutually exclusive light/dark classes
    $fix_dark_mode_js = "
        document.addEventListener('DOMContentLoaded', function() {
            const modeSwitchButtons = document.querySelectorAll('.fcom_mode_switch');
            if (!modeSwitchButtons.length) return;

            function enforceModeClass() {
                const htmlEl = document.documentElement;
                // Let portal_general.js add/remove 'dark' first.
                // Then, we clean up.
                setTimeout(() => { 
                    if (htmlEl.classList.contains('dark')) {
                        htmlEl.classList.remove('light');
                    } else {
                        // If dark is not present, ensure light is.
                        if (!htmlEl.classList.contains('light')) {
                             htmlEl.classList.add('light');
                        }
                    }
                }, 0); // Run just after the current execution stack
            }

            // Run on initial load (after portal_general might set initial state)
            enforceModeClass();

            // Run on each click
            modeSwitchButtons.forEach(button => {
                button.addEventListener('click', enforceModeClass);
            });
        });
    ";
    wp_add_inline_script('fluentcommunity-theme-portal-general', $fix_dark_mode_js);

    // Inline script to add active class to sidebar menu items
    $active_menu_js = "
        document.addEventListener('DOMContentLoaded', function() {
            const currentUrl = window.location.href;
            const sidebarMenu = document.getElementById('fluent_community_sidebar_menu');
            if (!sidebarMenu) return;

            const menuLinks = sidebarMenu.querySelectorAll('.fcom_menu_link');
            
            menuLinks.forEach(link => {
                // Remove existing active class from all links first
                link.classList.remove('router-link-exact-active');
                
                // Compare href, handling potential trailing slashes
                const linkHref = link.href.replace(/\/$/, ''); // Remove trailing slash from link
                const currentHref = currentUrl.replace(/\/$/, ''); // Remove trailing slash from current URL

                if (linkHref === currentHref) {
                    link.classList.add('router-link-exact-active');
                }
            });
        });
    ";
    // Add this script after portal_general.js
    wp_add_inline_script('fluentcommunity-theme-portal-general', $active_menu_js, 'after');
}
add_action( 'wp_enqueue_scripts', 'fluentcommunity_theme_scripts' );

/**
 * Add page templates.
 * WordPress 4.7+ automatically detects templates in a `page-templates` subdirectory.
 * If using an older version or for explicit control, you can filter `theme_page_templates`.
 */
// function fluentcommunity_theme_add_page_template( $templates ) {
//     $templates['page-templates/fluentcommunity-template.php'] = __( 'FluentCommunity Page', 'fluentcommunity-theme' );
//     return $templates;
// }
// add_filter( 'theme_page_templates', 'fluentcommunity_theme_add_page_template' );

// --- Potentially Useful Helper Functions ---

/**
 * Check if FluentCommunity plugin is active.
 *
 * @return boolean
 */
function is_fluentcommunity_active() {
    // Check based on a known function or class provided by FluentCommunity.
    // Adjust this check if FluentCommunity uses a different mechanism.
    return function_exists('FluentCommunity'); 
}

/**
 * Maybe load FluentCommunity assets conditionally.
 * 
 * You might hook this into `fluent_community/enqueue_global_assets` if you need
 * your theme's styles/scripts to load *within* the FC SPA context specifically.
 * However, for just using the template, `wp_enqueue_scripts` should suffice for the theme's own CSS.
 */
// function fluentcommunity_theme_enqueue_fc_specific_assets() {
//    if ( is_page_template('page-templates/fluentcommunity-template.php') ) {
//        // Enqueue assets needed ONLY when the FC template is used
//    }
// }
// add_action('wp_enqueue_scripts', 'fluentcommunity_theme_enqueue_fc_specific_assets'); 

/*
 * Attempt to prevent FluentCommunity router from taking over the URL
 * by modifying the perceived portal slug.
 *
 * @param string $slug The original portal slug (e.g., 'social').
 * @return string Modified slug.
 */
/* // REMOVING THIS FILTER - It breaks core FC pages
function fluentcommunity_theme_override_portal_slug($slug) {
    // Try setting it to an empty string or just a slash.
    // This might prevent the SPA router from prepending its base path.
    // NOTE: This is experimental and might break internal FC links.
    if (is_page_template('page-templates/fluentcommunity-template.php')) {
        return '/'; // Or try '' (empty string)
    }
    return $slug; // Return original slug for other contexts
}
add_filter('fluent_community/portal_slug', 'fluentcommunity_theme_override_portal_slug', 20); // High priority 
*/ 

/**
 * Attempt to prevent URL rewriting by adding the WP page slug to FC routes.
 *
 * @param array $routes Existing FluentCommunity routes.
 * @return array Modified routes.
 */
function fluentcommunity_theme_add_wp_page_route($routes) {
    // Check if we are on a singular page using our specific template
    if (is_singular() && is_page_template('page-templates/fluentcommunity-template.php')) {
        global $post;
        if ($post && !empty($post->post_name)) {
            $page_slug = $post->post_name; // Get the slug of the current page

            // Construct the path relative to the portal base
            // Example: If portal is /social/ and page slug is /test-page/,
            // the path the router might see is 'test-page'
            // We need to ensure this path doesn't trigger a redirect/rewrite.

            // Define a basic route structure (guessing based on Vue Router)
            // We want to define the route but not necessarily load a component
            // The key might be just having the path recognized.
            $new_route = [
                'path' => $page_slug, // Use the dynamic page slug
                'name' => 'wp_page_' . $page_slug, // Unique name
                // We might need 'component' or other keys, but let's start minimal
                // Adding 'meta' might signal something? This is speculative.
                'meta' => [
                    'is_wp_page_integration' => true,
                    // Potentially add flags to disable history manipulation if known
                ]
            ];

            // Add the new route, ensuring it doesn't overwrite existing ones if the slug clashes
            if (!isset($routes['wp_page_' . $page_slug])) { // Check based on assumed name key
                 // Assuming $routes is associative array keyed by route name:
                 $routes['wp_page_' . $page_slug] = $new_route;
            }
        }
    }
    return $routes;
}
add_filter('fluent_community/app_route_paths', 'fluentcommunity_theme_add_wp_page_route', 20); // Use priority 20 

/**
 * Add Customizer settings for theme integration.
 */
function fluentcommunity_theme_customize_register( $wp_customize ) {

    // Add Panel for Theme Settings
    $wp_customize->add_panel( 'fc_theme_options_panel', array(
        'title'       => __( 'FluentCommunity Theme Options', 'fluentcommunity-theme' ),
        'priority'    => 10, // High priority to appear near the top
    ) );

    // === General Section ===
    $wp_customize->add_section( 'fc_theme_general_section', array(
        'title'    => __( 'General Layout', 'fluentcommunity-theme' ),
        'panel'    => 'fc_theme_options_panel',
        'priority' => 10,
    ) );

    // Sidebar display on standard pages
    $wp_customize->add_setting( 'fc_theme_display_sidebar_standard', array(
        'default'           => false,
        'sanitize_callback' => 'wp_validate_boolean',
    ) );
    $wp_customize->add_control( 'fc_theme_display_sidebar_standard', array(
        'label'    => __( 'Display FC Sidebar on Standard Pages?', 'fluentcommunity-theme' ),
        'section'  => 'fc_theme_general_section',
        'type'     => 'checkbox',
    ) );

    // Standard Page Content Width
    $wp_customize->add_setting( 'fc_theme_standard_content_max_width', array(
        'default'           => '1100', // Default width
        'sanitize_callback' => 'absint', // Ensure it's a positive integer
        'transport'         => 'postMessage',
    ) );
    $wp_customize->add_control( 'fc_theme_standard_content_max_width', array(
        'label'       => __( 'Standard Page Content Max Width (px)', 'fluentcommunity-theme' ),
        'description' => __( 'Set the maximum width for the main content area on standard pages. Leave empty for full width.', 'fluentcommunity-theme' ),
        'section'     => 'fc_theme_general_section',
        'type'        => 'number',
        'input_attrs' => array(
            'min' => 0,
            'step' => 10,
        ),
    ) );

    // === Standard Header Section ===
    $wp_customize->add_section( 'fc_theme_standard_header_section', array(
        'title'    => __( 'Standard Header (Fluent Style)', 'fluentcommunity-theme' ),
        'panel'    => 'fc_theme_options_panel',
        'priority' => 20,
    ) );

    // Header Menu Selection (using the new location)
    $wp_customize->add_setting( 'nav_menu_locations[fcom-header-menu]', array(
        'theme_supports' => 'menus',
        'sanitize_callback' => 'absint', // Menu IDs are integers
    ) );
    $wp_customize->add_control( new WP_Customize_Nav_Menu_Location_Control( $wp_customize, 'fc_theme_standard_header_menu_control', array(
        'label'       => __( 'Header Menu', 'fluentcommunity-theme' ),
        'description' => __( 'Select the menu to display in the header. Note: To add icons like in FluentCommunity, you may need to use a plugin or custom code (like a custom Walker class) to add HTML (e.g., <i class="el-icon">...</i>) to your menu items.', 'fluentcommunity-theme' ),
        'section'     => 'fc_theme_standard_header_section',
        'location_id' => 'fcom-header-menu', // Target the new location
    ) ) );

    // Header Logo/Title Display
     $wp_customize->add_setting( 'fc_theme_header_logo_display', array(
        'default'           => 'logo_and_title',
        'sanitize_callback' => 'fluentcommunity_theme_sanitize_logo_display_choice',
        'transport'         => 'refresh', // Refresh needed as PHP controls output
    ) );
    $wp_customize->add_control( 'fc_theme_header_logo_display', array(
        'label'    => __( 'Header Branding Display', 'fluentcommunity-theme' ),
        'section'  => 'fc_theme_standard_header_section',
        'type'     => 'radio',
        'choices'  => array(
            'logo_only' => __( 'Logo Only', 'fluentcommunity-theme' ),
            'title_only' => __( 'Site Title Only', 'fluentcommunity-theme' ),
            'logo_and_title' => __( 'Logo and Site Title', 'fluentcommunity-theme' ),
        ),
    ) );

    // Header Background Color
    $wp_customize->add_setting( 'fc_theme_header_bg_color', array(
        'default'           => '#FFFFFF',
        'sanitize_callback' => 'sanitize_hex_color',
        'transport'         => 'postMessage',
    ) );
    $wp_customize->add_control( new WP_Customize_Color_Control( $wp_customize, 'fc_theme_header_bg_color', array(
        'label'    => __( 'Header Background Color', 'fluentcommunity-theme' ),
        'section'  => 'fc_theme_standard_header_section',
    ) ) );

     // Header Border Color
    $wp_customize->add_setting( 'fc_theme_header_border_color', array(
        'default'           => '#DADDE1',
        'sanitize_callback' => 'sanitize_hex_color',
        'transport'         => 'postMessage',
    ) );
    $wp_customize->add_control( new WP_Customize_Color_Control( $wp_customize, 'fc_theme_header_border_color', array(
        'label'    => __( 'Header Border Color', 'fluentcommunity-theme' ),
        'section'  => 'fc_theme_standard_header_section',
    ) ) );

    // Header Menu Text Color
    $wp_customize->add_setting( 'fc_theme_header_menu_text_color', array(
        'default'           => '#65676B',
        'sanitize_callback' => 'sanitize_hex_color',
        'transport'         => 'postMessage',
    ) );
    $wp_customize->add_control( new WP_Customize_Color_Control( $wp_customize, 'fc_theme_header_menu_text_color', array(
        'label'    => __( 'Menu Text Color', 'fluentcommunity-theme' ),
        'section'  => 'fc_theme_standard_header_section',
    ) ) );

    // Header Menu Text Color (Hover)
    $wp_customize->add_setting( 'fc_theme_header_menu_text_hover_color', array(
        'default'           => '#1877F2',
        'sanitize_callback' => 'sanitize_hex_color',
        'transport'         => 'postMessage',
    ) );
    $wp_customize->add_control( new WP_Customize_Color_Control( $wp_customize, 'fc_theme_header_menu_text_hover_color', array(
        'label'    => __( 'Menu Text Hover/Active Color', 'fluentcommunity-theme' ),
        'section'  => 'fc_theme_standard_header_section',
    ) ) );

    // Header Menu Background Color (Hover)
    $wp_customize->add_setting( 'fc_theme_header_menu_bg_hover_color', array(
        'default'           => '#E7F3FF',
        'sanitize_callback' => 'sanitize_hex_color',
        'transport'         => 'postMessage',
    ) );
    $wp_customize->add_control( new WP_Customize_Color_Control( $wp_customize, 'fc_theme_header_menu_bg_hover_color', array(
        'label'    => __( 'Menu Background Hover/Active Color', 'fluentcommunity-theme' ),
        'section'  => 'fc_theme_standard_header_section',
    ) ) );

    // Remove the old menu control if it exists (optional cleanup)
    // $wp_customize->remove_control('fc_theme_standard_header_menu'); 

    // === FluentCommunity Template Section (Keep existing settings) ===
    $wp_customize->add_section( 'fc_template_layout_section' , array(
        'title'      => __( 'FluentCommunity Template Layout', 'fluentcommunity-theme' ),
        'description' => __( 'Settings for pages using the "FluentCommunity Page" template.', 'fluentcommunity-theme' ),
        'panel'      => 'fc_theme_options_panel',
        'priority'   => 30,
    ) );

    // Content Max Width (FC Template)
    $wp_customize->add_setting( 'fc_template_content_max_width', array(
        'default'           => '1200',
        'sanitize_callback' => 'absint',
        'transport'         => 'postMessage',
    ) );
    $wp_customize->add_control( 'fc_template_content_max_width', array(
        'label'       => __( 'Content Max Width (px)', 'fluentcommunity-theme' ),
        'description' => __( 'Set the maximum width for the main content area within the FC template. Leave empty for default.', 'fluentcommunity-theme' ),
        'section'     => 'fc_template_layout_section',
        'type'        => 'number',
        'input_attrs' => array(
            'min' => 0,
            'step' => 10,
        ),
    ) );

    // Content Padding (FC Template)
    $wp_customize->add_setting( 'fc_template_content_padding', array(
        'default'           => '25',
        'sanitize_callback' => 'absint',
        'transport'         => 'postMessage',
    ) );
    $wp_customize->add_control( 'fc_template_content_padding', array(
        'label'       => __( 'Content Padding (px)', 'fluentcommunity-theme' ),
        'description' => __( 'Padding around the main content area within the FC template.', 'fluentcommunity-theme' ),
        'section'     => 'fc_template_layout_section',
        'type'        => 'number',
        'input_attrs' => array(
            'min' => 0,
            'step' => 1,
        ),
    ) );

    // Content Background Color (FC Template)
    $wp_customize->add_setting( 'fc_template_content_bg_color', array(
        'default'           => '#FFFFFF', // Default to white like standard WP
        'sanitize_callback' => 'sanitize_hex_color',
        'transport'         => 'postMessage',
    ) );
    $wp_customize->add_control( new WP_Customize_Color_Control( $wp_customize, 'fc_template_content_bg_color', array(
        'label'    => __( 'Content Background Color', 'fluentcommunity-theme' ),
        'section'  => 'fc_template_layout_section',
    ) ) );

}
add_action( 'customize_register', 'fluentcommunity_theme_customize_register' );

/**
 * Sanitize logo display choice.
 *
 * @param string $choice The selected choice.
 * @return string The sanitized choice.
 */
function fluentcommunity_theme_sanitize_logo_display_choice( $choice ) {
    $valid_choices = array( 'logo_only', 'title_only', 'logo_and_title' );
    if ( in_array( $choice, $valid_choices, true ) ) {
        return $choice;
    }
    return 'logo_and_title'; // Default fallback
}

/**
 * Output Customizer CSS to head.
 */
function fluentcommunity_theme_customizer_css() {
    ?>
    <style type="text/css">
        /* Standard Page Content Width */
        <?php 
        $standard_max_width = get_theme_mod( 'fc_theme_standard_content_max_width', '1100' );
        if ( !empty($standard_max_width) && is_numeric($standard_max_width) ) : ?>
            body.wp-standard-view .standard-content-wrapper > #primary,
            body.wp-standard-view .standard-content-wrapper > .standard-sidebar { /* Apply width to sidebar container too if present */
                /* Adjust max-width application if layout changes */
            }
            body.wp-standard-view .standard-content-wrapper.layout-with-sidebar > #primary {
                 /* Example: Calculate width if sidebar is present */
                 /* max-width: calc(<?php echo esc_html( $standard_max_width ); ?>px - 250px - 40px); /* Max width - sidebar width - gap */ */
            }
            body.wp-standard-view .standard-content-wrapper.layout-no-sidebar > #primary {
                 max-width: <?php echo esc_html( $standard_max_width ); ?>px;
                 margin-left: auto;
                 margin-right: auto;
            }
        <?php else : ?>
            /* Full Width for standard content */
            body.wp-standard-view .standard-content-wrapper.layout-no-sidebar > #primary {
                 max-width: none;
            }
        <?php endif; ?>

        /* Standard Header Colors */
        body.wp-standard-view .fcom_top_menu {
            --fcom-primary-bg: <?php echo sanitize_hex_color( get_theme_mod( 'fc_theme_header_bg_color', '#FFFFFF' ) ); ?>;
            --fcom-primary-border: <?php echo sanitize_hex_color( get_theme_mod( 'fc_theme_header_border_color', '#DADDE1' ) ); ?>;
            --fcom-menu-text: <?php echo sanitize_hex_color( get_theme_mod( 'fc_theme_header_menu_text_color', '#65676B' ) ); ?>;
            --fcom-menu-text-hover: <?php echo sanitize_hex_color( get_theme_mod( 'fc_theme_header_menu_text_hover_color', '#1877F2' ) ); ?>;
            --fcom-menu-text-active: <?php echo sanitize_hex_color( get_theme_mod( 'fc_theme_header_menu_text_hover_color', '#1877F2' ) ); ?>; /* Use hover color for active too */
            --fcom-menu-bg-hover: <?php echo sanitize_hex_color( get_theme_mod( 'fc_theme_header_menu_bg_hover_color', '#E7F3FF' ) ); ?>;
            --fcom-active-bg: <?php echo sanitize_hex_color( get_theme_mod( 'fc_theme_header_menu_bg_hover_color', '#E7F3FF' ) ); ?>; /* Use hover color for active too */
        }

        /* FC Template Content Styles */
        <?php 
        $fc_max_width = get_theme_mod( 'fc_template_content_max_width', '1200' );
        $fc_padding = get_theme_mod( 'fc_template_content_padding', '25' );
        $fc_bg_color = get_theme_mod( 'fc_template_content_bg_color', '#FFFFFF' );
        ?>
        .fcom_wp_page .fhr_content_layout_body {
             <?php if (!empty($fc_bg_color)) : ?>
                background-color: <?php echo sanitize_hex_color( $fc_bg_color ); ?>;
             <?php endif; ?>
             <?php if ( is_numeric($fc_padding) ) : ?>
                padding: <?php echo esc_html( $fc_padding ); ?>px;
             <?php endif; ?>
        }
        /* Apply max-width to the inner scrollable container */
        .fcom_wp_page .fhr_content_layout_body .el-main.fcom_main {
             <?php if ( !empty($fc_max_width) && is_numeric($fc_max_width) ) : ?>
                max-width: <?php echo esc_html( $fc_max_width ); ?>px;
                margin-left: auto;
                margin-right: auto;
            <?php else : ?>
                max-width: none;
            <?php endif; ?>
        }

    </style>
    <?php
}
add_action( 'wp_head', 'fluentcommunity_theme_customizer_css' );

/**
 * Add JS for Customizer controls that use postMessage.
 */
function fluentcommunity_theme_customize_preview_js() {
    wp_enqueue_script( 'fluentcommunity-theme-customizer', get_template_directory_uri() . '/js/customizer.js', array( 'customize-preview' ), _S_VERSION, true );
}
add_action( 'customize_preview_init', 'fluentcommunity_theme_customize_preview_js' );

// --- Filter to potentially add body class based on page template ---
function fluentcommunity_theme_add_body_classes($classes) {
    if (is_page_template('page-templates/fluentcommunity-template.php')) {
        // Remove the standard view class if the FC template is used
        $classes = array_diff($classes, ['wp-standard-view']); 
        // Add a specific class for the FC template if needed
        $classes[] = 'fc-template-view'; 
    } 
    // Add other conditional classes here if necessary
    return $classes;
}
add_filter('body_class', 'fluentcommunity_theme_add_body_classes');

// --- Include Template Tags (Optional) ---
// require get_template_directory() . '/inc/template-tags.php';

// --- Include Template Functions (Optional) ---
// require get_template_directory() . '/inc/template-functions.php';

// --- Start: Added by AI to synchronize dark mode with FluentCommunity portal pages ---
add_action('fluent_community/portal_footer', 'my_theme_sync_fc_dark_mode_script');

function my_theme_sync_fc_dark_mode_script() {
    ?>
    <script id="my-theme-fc-dark-mode-sync">
    (function() {
        const FCOM_GLOBAL_STORAGE_KEY = 'fcom_global_storage';
        const FCOM_COLOR_MODE_PROPERTY = 'fcom_color_mode'; // Assuming this is the property name

        const applyFcThemeFromGlobalStorage = () => {
            let theme = 'light'; // Default theme
            try {
                const globalStorageStr = localStorage.getItem(FCOM_GLOBAL_STORAGE_KEY);
                if (globalStorageStr) {
                    const globalStorage = JSON.parse(globalStorageStr);
                    theme = globalStorage[FCOM_COLOR_MODE_PROPERTY] || 'light';
                }
            } catch (e) {
                console.warn('FluentCommunity Native Sync: Error reading global storage for theme:', e);
            }
            
            const htmlElement = document.documentElement;
            htmlElement.classList.remove('light', 'dark');
            htmlElement.classList.add(theme);
            htmlElement.dataset.colorMode = theme;
            // Note: FluentCommunity's own scripts should handle its UI icons
        };

        // Apply theme on initial load of any FC portal page
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', applyFcThemeFromGlobalStorage);
        } else {
            applyFcThemeFromGlobalStorage();
        }

        // Listen for changes to fcom_global_storage (e.g., from your template pages)
        window.addEventListener('storage', function(event) {
            if (event.key === FCOM_GLOBAL_STORAGE_KEY) {
                const htmlElement = document.documentElement;
                let themeFromStorageEvent = 'light';
                 try {
                    const globalStorageStr = event.newValue; // Use event.newValue for current state
                    if (globalStorageStr) {
                        const globalStorage = JSON.parse(globalStorageStr);
                        themeFromStorageEvent = globalStorage[FCOM_COLOR_MODE_PROPERTY] || 'light';
                    }
                } catch (e) {
                    console.warn('FluentCommunity Native Sync: Error parsing global storage from event:', e);
                }
                const currentDOMTheme = htmlElement.classList.contains('dark') ? 'dark' : 'light';
                if (themeFromStorageEvent !== currentDOMTheme) {
                    applyFcThemeFromGlobalStorage(); // This will re-read from localStorage and apply
                }
            }
        });
    })();
    </script>
    <?php
}
// --- End: Added by AI to synchronize dark mode --- 