jQuery(document).ready(function($) {
    console.log('[FCA Widgets] widget-manager.js loaded.'); // Log script load

    // Get widget counter from data attribute set by PHP
    let widgetCounter = parseInt($('#fc-widgets-container').data('widget-count') || 0);

    // Initialize Select2 for existing widgets
    initializeSelect2();
    initializeExistingEditors();

    // Initialize sortable for widgets
    $('#fc-widgets-container').sortable({
        handle: '.widget-drag-handle',
        placeholder: 'widget-placeholder',
        tolerance: 'pointer'
    });

    // Initialize sortable for the widgets table
    $('#sortable-widgets-table tbody').sortable({
        handle: '.widget-drag-handle',
        placeholder: 'ui-sortable-placeholder',
        helper: function(e, tr) {
            var $originals = tr.children();
            var $helper = tr.clone();
            $helper.children().each(function(index) {
                // Set helper cell sizes to match the original sizes
                $(this).width($originals.eq(index).width());
            });
            return $helper;
        },
        update: function(event, ui) {
            // Update order inputs
            $('#sortable-widgets-table tbody tr').each(function(index) {
                $(this).find('.widget-order-input').val(index);
            });
            
            // Update the widget order hidden input
            updateWidgetOrder();
        }
    });

    // Function to update widget order in hidden input
    function updateWidgetOrder() {
        const order = [];
        $('#sortable-widgets-table tbody tr').each(function() {
            const widgetId = $(this).data('widget-id');
            if (widgetId) {
                order.push(widgetId);
            }
        });
        $('#widget-order').val(JSON.stringify(order));
    }

    // Update widget order on page load
    updateWidgetOrder();

    // Add new widget
    $('#add-widget').on('click', function() {
        const widgetId = widgetCounter++;
        const template = $('#widget-template').html();
        
        if (!template) {
            return;
        }

        const newWidget = template.replace(/WIDGET_ID/g, widgetId);
        $('#fc-widgets-container').append(newWidget);
        
        // Initialize new widget components
        initializeNewWidget(widgetId);
    });

    // Remove widget
    $(document).on('click', '.remove-widget', function() {
        const widgetItem = $(this).closest('.widget-item');
        removeWidget(widgetItem);
    });

    // Helper functions
    function initializeSelect2() {
        $('.space-select').select2({
            width: '100%',
            placeholder: 'Select spaces where this widget should appear'
        });
    }

    function initializeExistingEditors() {
        $('.widget-item').each(function() {
            const editorId = $(this).find('.wp-editor-area').attr('id');
            if (editorId && typeof wp !== 'undefined' && wp.editor) {
                initializeEditor(editorId);
            }
        });
    }

    function initializeNewWidget(widgetId) {
        // Initialize Select2
        const newSelect = $('#fc-widgets-container .space-select').last();
        if (!newSelect.data('select2')) {
            newSelect.select2({
                width: '100%',
                placeholder: 'Select spaces where this widget should appear'
            });
        }

        // Initialize TinyMCE
        const editorId = 'widget_content_' + widgetId;
        if (typeof wp !== 'undefined' && wp.editor) {
            initializeEditor(editorId);
        }
    }

    function initializeEditor(editorId) {
        if (document.getElementById(editorId)) {
            wp.editor.initialize(editorId, {
                tinymce: {
                    height: 200,
                    menubar: false,
                    plugins: 'lists link image media wordpress',
                    toolbar: 'formatselect | bold italic | bullist numlist | link image',
                },
                quicktags: true,
                mediaButtons: true
            });
        }
    }

    function removeWidget(widgetItem) {
        const editorId = widgetItem.find('.wp-editor-area').attr('id');
        
        // Remove TinyMCE instance if it exists
        if (typeof wp !== 'undefined' && wp.editor && editorId) {
            try {
                wp.editor.remove(editorId);
            } catch (e) {
                // console.warn('Error removing editor:', e);
            }
        }
        
        // Remove Select2 instance
        const select2Instance = widgetItem.find('.space-select');
        if (select2Instance.data('select2')) {
            select2Instance.select2('destroy');
        }
        
        // Remove the widget
        widgetItem.remove();
        updateWidgetOrder();
    }

    // Handle "Add New Widget" button click
    $('#add-widget, #add-first-widget').on('click', function(e) {
        e.preventDefault();
        // Get the current URL and properly append the edit parameter
        let url = new URL(window.location.href);
        url.searchParams.set('edit', 'new');
        window.location.href = url.toString();
    });

    // Handle delete widget
    $('.remove-widget').on('click', function() {
        if (!confirm('Are you sure you want to delete this widget?')) {
            return;
        }

        const widgetId = $(this).data('id');
        const $row = $(this).closest('tr');

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'fca_delete_widget',
                widget_id: widgetId,
                nonce: $('#_wpnonce').val() // Get WordPress nonce from the page
            },
            success: function(response) {
                if (response.success) {
                    $row.fadeOut(400, function() {
                        $(this).remove();
                        if ($('tbody tr').length === 0) {
                            $('tbody').append('<tr><td colspan="4">No widgets found. <a href="#" id="add-first-widget">Create your first widget</a>.</td></tr>');
                        }
                    });
                } else {
                    alert('Error deleting widget: ' + response.data);
                }
            },
            error: function() {
                alert('Error deleting widget. Please try again.');
            }
        });
    });

    // Handle widget search
    $('#widget-search').on('input', function() {
        const searchTerm = $(this).val().toLowerCase();
        filterWidgets();
    });

    // Handle space filter
    $('#space-filter').on('change', function() {
        filterWidgets();
    });

    // Clear search
    $('#search-clear').on('click', function() {
        $('#widget-search').val('');
        filterWidgets();
    });

    // Function to filter widgets based on search and space filter
    function filterWidgets() {
        // Ensure search term is always a string before calling toLowerCase()
        const searchTerm = ($('#widget-search').val() || '').toLowerCase();
        const spaceFilter = $('#space-filter').val();
        let hasVisibleWidgets = false;
        
        // Remove any existing no results message
        $('.no-results-message').remove();
        
        // Filter the table rows
        $('#sortable-widgets-table tbody tr').each(function() {
            const $row = $(this);
            
            // Skip the "no widgets found" row
            if ($row.find('td[colspan]').length) {
                return;
            }
            
            // Ensure title is always a string before calling toLowerCase()
            const title = ($row.find('.column-title a').text() || '').toLowerCase();
            
            // Get spaces from data attribute
            let widgetSpaces = [];
            try {
                widgetSpaces = JSON.parse($row.attr('data-spaces') || '[]');
            } catch (e) {
                // console.warn('Error parsing spaces data:', e);
            }
            
            // Check if row matches search term
            const matchesSearch = searchTerm === '' || title.includes(searchTerm);
            
            // Check if row matches space filter
            const matchesSpace = spaceFilter === '' || 
                               (widgetSpaces.length === 0) || // "All Spaces" case
                               widgetSpaces.includes(spaceFilter);
            
            // Show/hide row based on filters
            if (matchesSearch && matchesSpace) {
                $row.removeClass('widget-hidden');
                hasVisibleWidgets = true;
            } else {
                $row.addClass('widget-hidden');
            }
        });
        
        // Show a message if no widgets match the filters
        if (!hasVisibleWidgets && $('#sortable-widgets-table tbody tr').length > 0) {
            const colspan = $('#sortable-widgets-table thead th').length;
            const $noResults = $('<tr class="no-results-message"><td colspan="' + colspan + '">No widgets match your filters. <a href="#" id="clear-filters">Clear filters</a></td></tr>');
            $('#sortable-widgets-table tbody').append($noResults);
            
            // Add click handler for the "clear filters" link
            $('#clear-filters').on('click', function(e) {
                e.preventDefault();
                $('#widget-search').val('');
                $('#space-filter').val('');
                filterWidgets();
            });
        }
        
        // Update the sortable widget to account for hidden items
        if ($.fn.sortable) {
            try {
                $('#sortable-widgets-table tbody').sortable('refresh');
            } catch (e) {
                // console.warn('Error refreshing sortable:', e);
            }
        }
    }

    // Initialize filters on page load
    filterWidgets();

    // --- ThickBox Button Handler --- //

    // Insert button from ThickBox modal
    $('#tb-insert-button').on('click', function() {
        const buttonText = $('#tb-button-text').val().trim();
        const buttonUrl = $('#tb-button-url').val().trim();
        const buttonStyle = $('#tb-button-style').val();
        const editor = window.fcaCurrentEditor;

        if (!buttonText || !buttonUrl) {
            alert('Please enter button text and URL.');
            return;
        }

        if (!editor) {
            alert('Editor not found! Please try again.');
            return;
        }

        let buttonClasses = 'el-button'; // Base class
        if (buttonStyle === 'fcom_primary_button') {
            buttonClasses += ' fcom_primary_button';
        } else if (buttonStyle === 'el-button--text') {
            buttonClasses += ' el-button--text';
        } else if (buttonStyle === 'fcom_link') {
            buttonClasses = 'fcom_link'; // Special case for link style
        }

        // Construct HTML
        const buttonHtml = `<a href="${buttonUrl}" class="${buttonClasses}"><span>${buttonText}</span></a>`;

        // Insert into editor
        editor.execCommand('mceInsertContent', false, buttonHtml);
        
        // Close ThickBox
        tb_remove();
        
        // Clear form fields for next use
        $('#tb-button-text').val('');
        $('#tb-button-url').val('');
        $('#tb-button-style').val('fcom_primary_button');
    });
}); 