/**
 * FCA Sidebar Menu Admin Script
 */
jQuery(document).ready(function($) {

    // --- Configuration ---
    const config = window.fcaSidebarMenu || {}; // Localized data from enqueue
    const optionsData = window.fcaSidebarMenuData || {}; // Localized HTML options
    const optionName = config.optionName || 'fca_sidebar_menu_items';
    const text = config.text || {};

    // --- Helper Functions ---

    // Generate unique IDs
    function generateUniqueId(prefix = 'id-') {
        return prefix + Date.now() + Math.random().toString(16).slice(2);
    }

    // Initialize Select2
    function initSelect2(element) {
        if (!element || !element.length || !$.fn.select2) return;

        if (element.hasClass('fca-select2')) {
            element.select2({
                width: '100%',
                dropdownParent: element.closest('.fc-item-row, .fc-group') // Ensure dropdown is within container
            });
        }
        if (element.hasClass('fca-user-search-select')) {
            element.select2({
                width: '100%',
                ajax: {
                    url: config.ajaxurl,
                    dataType: 'json',
                    delay: 250,
                    data: function(params) {
                        return {
                            action: 'fca_search_users', // AJAX action hook
                            q: params.term,
                            nonce: config.nonce
                        };
                    },
                    processResults: function(data) {
                        return {
                            results: data
                        };
                    },
                    cache: true
                },
                minimumInputLength: element.data('minimum-input-length') || 2,
                placeholder: element.data('placeholder') || 'Search...',
                dropdownParent: element.closest('.fc-item-row')
            });
        }
    }

    // Initialize all Select2 instances
    function initializeAllSelect2() {
        $('.fca-select2, .fca-user-search-select').each(function() {
            // Destroy existing Select2 if it exists to prevent duplication
            if ($(this).data('select2')) {
                $(this).select2('destroy');
            }
            initSelect2($(this));
        });
    }

    // Make items sortable
    function makeItemsSortable(container) {
        if (!container || !container.length || !$.fn.sortable) return;

        container.sortable({
            items: '> .fc-item-row',
            handle: '.item-drag-handle',
            placeholder: 'item-placeholder',
            axis: 'y',
            opacity: 0.7,
            update: function(event, ui) {
                updateGroupItemOrder($(this).closest('.fc-group'));
            }
        }).disableSelection();
    }
    
    // Make groups sortable
    function makeGroupsSortable() {
        const groupsContainer = $('#fca-groups-container');
        if (!groupsContainer.length || !$.fn.sortable) return;

        groupsContainer.sortable({
            items: '> .fc-group',
        handle: '.group-drag-handle',
        placeholder: 'group-placeholder',
            axis: 'y',
            opacity: 0.7,
        update: function(event, ui) {
            updateGroupOrder();
        }
        }).disableSelection();
    }

    // Update hidden input for group order
    function updateGroupOrder() {
        const groupIds = $('#fca-groups-container .fc-group').map(function() {
            return $(this).data('group-id');
        }).get();
        $('#fca-group-order').val(JSON.stringify(groupIds));
    }

    // Update hidden input for item order within a group
    function updateGroupItemOrder(groupElement) {
        if (!groupElement || !groupElement.length) return;
        const itemIds = groupElement.find('.fc-items-tbody .fc-item-row').map(function() {
            return $(this).data('item-id');
        }).get();
        groupElement.find('.fca-item-order').val(JSON.stringify(itemIds));
    }

    // Handle visibility dropdown changes
    function handleVisibilityChange(selectElement) {
        const selectedValue = selectElement.val();
        const parent = selectElement.closest('.fc-group-header, td.column-visibility');
        
        // Hide all visibility details containers first
        parent.find('.visibility-details').hide();
        
        // Show the relevant container based on selection
        if (selectedValue === 'specific_roles') {
            parent.find('.role-visibility-details').show();
        } else if (selectedValue === 'specific_users') {
            parent.find('.user-visibility-details').show();
        } else if (selectedValue === 'space_members') {
            parent.find('.space-visibility-details').show();
        }
    }

    // Handle Link Type dropdown changes
    function handleLinkTypeChange(selectElement) {
        const selectedValue = selectElement.val();
        const fieldsContainer = selectElement.closest('td').find('.link-type-fields');
        const row = selectElement.closest('.fc-item-row');

        // Store current value of the visible content_value select before hiding
        let currentContentValue = '';
        const visibleContentSelect = fieldsContainer.find('.field-group:visible .content-value-select');
        if (visibleContentSelect.length) {
            currentContentValue = visibleContentSelect.val();
        }

        fieldsContainer.find('.field-group').hide(); // Hide all link type fields
        
        if (selectedValue === 'custom') {
            fieldsContainer.find('.custom-url-field').show();
        } else if (selectedValue === 'space') {
            const spaceField = fieldsContainer.find('.space-select-field').show();
            const spaceSelect = spaceField.find('select');
            
            // Only initialize if needed to preserve selection
            if (spaceSelect.find('option').length <= 1) {
                initializeDropdown(spaceSelect, 'spaces');
            }
            
            // If the icon type is "space_icon", update the space icon selection too
            const iconTypeSelect = row.find('.icon-type-select');
            if (iconTypeSelect.val() === 'space_icon') {
                // When the space selection changes, also update the icon space selection
                spaceSelect.off('change.linkToIcon').on('change.linkToIcon', function() {
                    const selectedSpaceId = $(this).val();
                    if (selectedSpaceId) {
                        const spaceIconSelect = row.find('.space-icon-field select');
                        spaceIconSelect.val(selectedSpaceId).trigger('change');
                        
                        // Also update the final value
                        const finalValueInput = row.find('.icon-value-final');
                        finalValueInput.val(selectedSpaceId);
                    }
                });
            }
        } else if (selectedValue === 'course') {
            const courseField = fieldsContainer.find('.course-select-field').show();
            const courseSelect = courseField.find('select');
            
            // Only initialize if needed to preserve selection
            if (courseSelect.find('option').length <= 1) {
                initializeDropdown(courseSelect, 'courses');
            }
        } else if (selectedValue === 'page') {
            const pageField = fieldsContainer.find('.page-select-field').show();
            const pageSelect = pageField.find('select');
            
            // Only initialize if needed to preserve selection
            if (pageSelect.find('option').length <= 1) {
                initializeDropdown(pageSelect, 'pages');
            }
        }
    }

    // Initialize dropdown with AJAX-loaded data
    function initializeDropdown(select, type) {
        if (!select || !select.length) return;
        
        let loadFunction;
        switch (type) {
            case 'spaces':
                loadFunction = loadSpacesForDropdown;
                break;
            case 'courses':
                loadFunction = loadCoursesForDropdown;
                break;
            case 'pages':
                loadFunction = loadPagesForDropdown;
                break;
            default:
                return;
        }
        
        // Store the current value if any
        const currentValue = select.val();
        if (currentValue) {
            select.data('selected-value', currentValue);
        }
        
        loadFunction(select);
    }

    // Handle Icon Type dropdown changes
    function handleIconTypeChange(selectElement) {
        const selectedValue = selectElement.val();
        const fieldsContainer = selectElement.closest('td').find('.icon-type-fields');
        const finalValueInput = fieldsContainer.find('.icon-value-final');
        const valueInputs = fieldsContainer.find('.icon-value-input');
        const row = selectElement.closest('.fc-item-row');

        fieldsContainer.find('.field-group').hide(); // Hide all icon type fields
        valueInputs.prop('disabled', true); // Disable all value inputs
        finalValueInput.val(''); // Clear final value

        let activeInput;
        if (selectedValue === 'emoji') {
            activeInput = fieldsContainer.find('.emoji-field').show().find('input');
        } else if (selectedValue === 'svg_url') {
            activeInput = fieldsContainer.find('.svg-url-field').show().find('input[type="url"]');
        } else if (selectedValue === 'svg_code') {
            activeInput = fieldsContainer.find('.svg-code-field').show().find('textarea');
        } else if (selectedValue === 'space_icon') {
            const spaceIconField = fieldsContainer.find('.space-icon-field').show();
            const spaceIconSelect = spaceIconField.find('.icon-value-space');
            const linkTypeSelect = row.find('.link-type-select');
            if (linkTypeSelect.val() === 'space') {
                const selectedSpaceId = row.find('.space-select-field select').val();
                if (selectedSpaceId) {
                    spaceIconSelect.val(selectedSpaceId).trigger('change');
                    finalValueInput.val(selectedSpaceId);
                }
            }
            activeInput = spaceIconSelect;
        } else if (selectedValue === 'course_icon') {
            const courseIconField = fieldsContainer.find('.course-icon-field').show();
            const courseIconSelect = courseIconField.find('.icon-value-course');
            const linkTypeSelect = row.find('.link-type-select');
            if (linkTypeSelect.val() === 'course') {
                const selectedCourseId = row.find('.course-select-field select').val();
                if (selectedCourseId) {
                    courseIconSelect.val(selectedCourseId).trigger('change');
                    if(finalValueInput.length) {
                         finalValueInput.val(selectedCourseId);
                    }
                }
            }
            activeInput = courseIconSelect;
        }
        
        if(activeInput && activeInput.length) {
            activeInput.prop('disabled', false);
            if (activeInput.is('input, textarea') || (activeInput.is('select') && activeInput.val())) {
                 if(finalValueInput.length) {
                      finalValueInput.val(activeInput.val());
                 }
            }
        }
    }

    // Update the final hidden icon value input when any specific value changes
    function syncIconValue(changedInput) {
        const fieldsContainer = changedInput.closest('.icon-type-fields');
        const finalValueInput = fieldsContainer.find('.icon-value-final');
        finalValueInput.val(changedInput.val());
    }

    // Load dropdown options from AJAX
    function loadSpacesForDropdown(select) {
        var $select = $(select);
        $select.append('<option value="">Loading spaces...</option>');
        
        $.ajax({
            url: config.ajaxurl,
            type: 'POST',
            data: {
                action: 'fca_widgets_get_spaces',
                security: config.nonce
            },
            success: function(response) {
                $select.empty();
                $select.append('<option value="">Select a space</option>');
                
                if (response.success && response.data && response.data.length > 0) {
                    $.each(response.data, function(index, item) {
                        $select.append('<option value="' + item.value + '">' + item.label + '</option>');
                    });
                } else {
                    $select.append('<option value="" disabled>No spaces found</option>');
                }
                
                // If there was a previously selected value, restore it
                if ($select.data('selected-value')) {
                    $select.val($select.data('selected-value'));
                }
                
                $select.trigger('change');
            },
            error: function() {
                $select.empty();
                $select.append('<option value="">Error loading spaces</option>');
            }
        });
    }

    function loadCoursesForDropdown(select) {
        var $select = $(select);
        $select.append('<option value="">Loading courses...</option>');
        
        $.ajax({
            url: config.ajaxurl,
            type: 'POST',
            data: {
                action: 'fca_widgets_get_courses',
                security: config.nonce
            },
            success: function(response) {
                $select.empty();
                $select.append('<option value="">Select a course</option>');
                
                if (response.success && response.data && response.data.length > 0) {
                    $.each(response.data, function(index, item) {
                        $select.append('<option value="' + item.value + '">' + item.label + '</option>');
                    });
                } else {
                    $select.append('<option value="" disabled>No courses found</option>');
                }
                
                // If there was a previously selected value, restore it
                if ($select.data('selected-value')) {
                    $select.val($select.data('selected-value'));
                }
                
                $select.trigger('change');
            },
            error: function() {
                $select.empty();
                $select.append('<option value="">Error loading courses</option>');
            }
        });
    }

    function loadPagesForDropdown(select) {
        var $select = $(select);
        $select.append('<option value="">Loading pages...</option>');
        
        $.ajax({
            url: config.ajaxurl,
            type: 'POST',
            data: {
                action: 'fca_widgets_get_pages',
                security: config.nonce
            },
            success: function(response) {
                $select.empty();
                $select.append('<option value="">Select a page</option>');
                
                if (response.success && response.data && response.data.length > 0) {
                    $.each(response.data, function(index, item) {
                        $select.append('<option value="' + item.value + '">' + item.label + '</option>');
                    });
                } else {
                    $select.append('<option value="" disabled>No pages found</option>');
                }
                
                // If there was a previously selected value, restore it
                if ($select.data('selected-value')) {
                    $select.val($select.data('selected-value'));
                }
                
                $select.trigger('change');
            },
            error: function() {
                $select.empty();
                $select.append('<option value="">Error loading pages</option>');
            }
        });
    }

    // Ensure content_value select is always properly set when showing fields
    function ensureContentValueSelect() {
        $('.link-type-select').each(function() {
            const linkType = $(this).val();
            const row = $(this).closest('.fc-item-row');
            
            // Find the expected select based on link type
            let selectedSelect;
            switch (linkType) {
                case 'space':
                    selectedSelect = row.find('.space-select-field select');
                    break;
                case 'course':
                    selectedSelect = row.find('.course-select-field select');
                    break;
                case 'page':
                    selectedSelect = row.find('.page-select-field select');
                    break;
            }
            
            // Initialize the select if it's visible but has no options
            if (selectedSelect && selectedSelect.is(':visible') && selectedSelect.find('option').length <= 1) {
                // Ensure correct type is passed for initialization
                let initType = linkType;
                if (linkType === 'page') initType = 'pages';
                else if (linkType === 'space') initType = 'spaces';
                else if (linkType === 'course') initType = 'courses';
                
                initializeDropdown(selectedSelect, initType);
            }
        });
    }

    // Save menu data via AJAX - completely rebuilt from scratch
    function saveMenuDataAjax() {
        // Create the menuData object with the expected structure
        const menuData = {
            groups: {},
            items: {},
            group_order: [],
            disable_default_items: 0
        };
        
        // Get the disable default items checkbox value
        const disableCheckbox = $('#fca-disable-default-items');
        if (disableCheckbox.length) {
            menuData.disable_default_items = disableCheckbox.is(':checked') ? 1 : 0;
        }
        
        // Process each group
        $('.fc-group').each(function() {
            const $group = $(this);
            const groupId = $group.data('group-id');
            
            // Get group data
            const title = $group.find('.group-title-input').val();
            const isGroupEnabled = $group.find('.enabled-setting input[type="checkbox"]').is(':checked'); // Get group enabled state
            const visibility = $group.find('.visibility-select').val();
            const topPosition = $group.find('input[name*="[top_position]"]').is(':checked') ? 1 : 0;
            
            // Get spaces for group if applicable
            const spaces = [];
            if (visibility === 'space_members') {
                $group.find('.space-visibility-details select option:selected').each(function() {
                    spaces.push($(this).val());
                });
            }
            
            // Get item order
            let itemOrder = [];
            try {
                const itemOrderJSON = $group.find('.fca-item-order').val();
                if (itemOrderJSON) {
                    itemOrder = JSON.parse(itemOrderJSON);
                }
            } catch (e) {
                // console.error('Error parsing item order:', e);
            }
            
            // Add group to data
            menuData.groups[groupId] = {
                title: title,
                enabled: isGroupEnabled ? 1 : 0, // Add enabled state
                visibility: visibility,
                spaces: spaces,
                top_position: topPosition,
                item_order: itemOrder
            };
            
            // Add to group order
            menuData.group_order.push(groupId);
            
            // Initialize items array for this group
            menuData.items[groupId] = {};
            
            // Process each item in this group
            $group.find('.fc-item-row').each(function() {
                const $item = $(this);
                const itemId = $item.data('item-id');
                const isItemEnabled = $item.find('.column-enabled input[type="checkbox"]').is(':checked');
                const openInNewTab = $item.find('input[name*="[open_in_new_tab]"]').is(':checked'); // Get new value
                
                // Get item data
                const itemData = {
                    title: $item.find('td.column-title input').val(),
                    enabled: isItemEnabled ? 1 : 0,
                    type: $item.find('td.column-link .link-type-select').val(),
                    custom_url: $item.find('td.column-link .custom-url-field input').val(),
                    content_value: '', // Will be set based on type
                    icon_type: $item.find('td.column-icon .icon-type-select').val(),
                    icon_value: $item.find('td.column-icon .icon-value-final').val(),
                    visibility: $item.find('td.column-visibility .visibility-select').val(),
                    roles: [],
                    users: [],
                    spaces: [],
                    open_in_new_tab: openInNewTab ? 1 : 0 // Add new value
                };
                
                // Get content_value based on type
                const type = itemData.type;
                if (type === 'space') {
                    itemData.content_value = $item.find('td.column-link .space-select-field select').val();
                } else if (type === 'course') {
                    itemData.content_value = $item.find('td.column-link .course-select-field select').val();
                } else if (type === 'page') {
                    itemData.content_value = $item.find('td.column-link .page-select-field select').val();
                }
                
                // Get roles, users, spaces based on visibility
                const visibility = itemData.visibility;
                if (visibility === 'specific_roles') {
                    $item.find('td.column-visibility .role-visibility-details select option:selected').each(function() {
                        itemData.roles.push($(this).val());
                    });
                } else if (visibility === 'specific_users') {
                    $item.find('td.column-visibility .user-visibility-details select option:selected').each(function() {
                        itemData.users.push($(this).val());
                    });
                } else if (visibility === 'space_members') {
                    $item.find('td.column-visibility .space-visibility-details select option:selected').each(function() {
                        itemData.spaces.push($(this).val());
                    });
                }
                
                // Handle icon value - make sure it's correctly populated from the active input
                if (itemData.icon_type === 'emoji') {
                    itemData.icon_value = $item.find('td.column-icon .emoji-field input').val();
                } else if (itemData.icon_type === 'svg_url') {
                    itemData.icon_value = $item.find('td.column-icon .svg-url-field input').val();
                } else if (itemData.icon_type === 'svg_code') {
                    itemData.icon_value = $item.find('td.column-icon .svg-code-field textarea').val();
                }
                
                // Add item to data
                menuData.items[groupId][itemId] = itemData;
            });
        });
        
        // Log the data structure for debugging
        // console.log('Saving menu data:', menuData);
        
        // Send AJAX request to save
        $.ajax({
            url: config.ajaxurl,
            type: 'POST',
            data: {
                action: 'fca_widgets_save_sidebar_menu',
                security: config.nonce,
                menu_data: JSON.stringify(menuData)
            },
            beforeSend: function() {
                $('#fca-save-status').html('<span class="saving">Saving...</span>');
            },
            success: function(response) {
                // console.log('Save response:', response);
                if (response.success) {
                    $('#fca-save-status').html('<span class="saved">Saved successfully!</span>');
                    setTimeout(function() {
                        $('#fca-save-status').html('');
                    }, 3000);
                } else {
                    $('#fca-save-status').html('<span class="error">Error: ' + (response.data || 'Unknown error') + '</span>');
                    // console.error('Save error:', response); 
                }
            },
            error: function(xhr, status, error) {
                // console.error('AJAX error:', xhr, status, error);
                $('#fca-save-status').html('<span class="error">Error: ' + error + '</span>');
            }
        });
    }

    // --- Function to update space icon preview ---
    function updateSpaceIconPreview(spaceSelect) {
        const spaceId = spaceSelect.val();
        if (!spaceId) {
            spaceSelect.siblings('.space-icon-preview').html('');
            return;
        }
        
        // Show loading indicator
        spaceSelect.siblings('.space-icon-preview').html('<span style="color:#999;font-style:italic;margin-left:5px;">Loading...</span>');
        
        console.log('Updating space icon preview for space ID:', spaceId);
        
        // Make AJAX request to get space icon preview
        $.ajax({
            url: config.ajaxurl,
            type: 'POST',
            data: {
                action: 'fca_widgets_get_space_icon_preview',
                security: config.nonce,
                space_id: spaceId
            },
            success: function(response) {
                console.log('Space icon preview response:', response);
                if (response.success && response.data) {
                    spaceSelect.siblings('.space-icon-preview').html(response.data);
                } else {
                    const errorMsg = response.data || 'No preview available';
                    console.error('Error getting space icon preview:', errorMsg);
                    spaceSelect.siblings('.space-icon-preview').html('<span style="color:#999;font-style:italic;margin-left:5px;">No preview available</span>');
                }
            },
            error: function(xhr, status, error) {
                console.error('AJAX error getting space icon preview:', error, xhr.responseText);
                spaceSelect.siblings('.space-icon-preview').html('<span style="color:#f00;font-style:italic;margin-left:5px;">Error loading preview</span>');
            }
        });
    }

    // --- Event Handlers ---

    // Add New Group
    $('#fca-add-group').on('click', function() {
        const groupsContainer = $('#fca-groups-container');
        const newGroupId = generateUniqueId('group-');
        const template = wp.template('fca-group');
        const newGroupHtml = template({ group_id: newGroupId });

        groupsContainer.append(newGroupHtml);
        const newGroupElement = groupsContainer.find('.fc-group[data-group-id="' + newGroupId + '"]');
        
        initializeAllSelect2();
        
        // Initialize visibility settings for the new group
        newGroupElement.find('.visibility-select').each(function() {
            handleVisibilityChange($(this));
        });
        
        updateGroupOrder();
        
        newGroupElement.find('.group-title-input').focus();
    });

    // Remove Group
    $('#fca-groups-container').on('click', '.remove-group', function() {
        if (confirm(text.confirmRemoveGroup || 'Are you sure you want to remove this group and all its items?')) {
            $(this).closest('.fc-group').fadeOut(300, function() {
                $(this).remove();
            updateGroupOrder();
            });
        }
    });

    // Add Menu Item
    $('#fca-groups-container').on('click', '.add-menu-item', function() {
        const groupElement = $(this).closest('.fc-group');
        const groupId = groupElement.data('group-id');
        const itemsTbody = groupElement.find('.fc-items-tbody');
        const newItemId = generateUniqueId('item-');
        const template = wp.template('fca-menu-item');

        // Render the basic item structure using the WP template
        const newItemHtml = template({ group_id: groupId, item_id: newItemId });
        const $newItemRow = $(newItemHtml); // Create jQuery object

        // Populate Select Options Dynamically
        if (optionsData.spacesOptionsHtml) {
            $newItemRow.find('select[id^="space-id-"]').html(optionsData.spacesOptionsHtml);
            $newItemRow.find('.space-visibility-details select.space-select').html(optionsData.spacesOptionsHtml);
        }
        if (optionsData.coursesOptionsHtml) {
            $newItemRow.find('select[id^="course-id-"]').html(optionsData.coursesOptionsHtml);
        }
        if (optionsData.pagesOptionsHtml) {
            $newItemRow.find('select[id^="page-id-"]').html(optionsData.pagesOptionsHtml);
        }

        itemsTbody.find('.no-items-row').remove();
        itemsTbody.append($newItemRow);
        
        // Initialize select2 and trigger visibility/type handlers for the new item
        initializeAllSelect2();
        
        // Initialize visibility details for the new item
        $newItemRow.find('.visibility-select').each(function() {
            handleVisibilityChange($(this));
        });
        
        // Initialize link type and icon type settings
        $newItemRow.find('.link-type-select, .icon-type-select').trigger('change');
        
        updateGroupItemOrder(groupElement);

        $newItemRow.find('.column-title input').focus();
    });

    // Remove Menu Item
    $('#fca-groups-container').on('click', '.remove-menu-item', function() {
        if (confirm(text.confirmRemoveItem || 'Are you sure you want to remove this menu item?')) {
            const itemRow = $(this).closest('.fc-item-row');
            const groupElement = itemRow.closest('.fc-group');
            itemRow.fadeOut(300, function() {
                $(this).remove();
                updateGroupItemOrder(groupElement);
                if(groupElement.find('.fc-items-tbody .fc-item-row').length === 0) {
                    groupElement.find('.fc-items-tbody').append('<tr class="no-items-row"><td colspan="6">No items in this group yet.</td></tr>');
                }
            });
        }
    });

    // Handle Changes in Dropdowns (Visibility, Link Type, Icon Type)
    $('#fca-groups-container').on('change', '.visibility-select', function() {
        handleVisibilityChange($(this));
        
        // If we're changing to a type that needs a dropdown, make sure it's properly populated
        const selectedValue = $(this).val();
        const parent = $(this).closest('.fc-group-header, td.column-visibility');
        
        if (selectedValue === 'space_members') {
            const spaceSelect = parent.find('.space-visibility-details select.space-select');
            if (spaceSelect.length && spaceSelect.find('option').length <= 1) {
                if (optionsData.spacesOptionsHtml) {
                    spaceSelect.html(optionsData.spacesOptionsHtml);
                    initSelect2(spaceSelect);
                } else {
                    loadSpacesForDropdown(spaceSelect);
                }
            }
        } else if (selectedValue === 'specific_roles') {
            const roleSelect = parent.find('.role-visibility-details select');
            if (roleSelect.length && !roleSelect.data('select2-id')) {
                initSelect2(roleSelect);
            }
        } else if (selectedValue === 'specific_users') {
            const userSelect = parent.find('.user-visibility-details select');
            if (userSelect.length && !userSelect.data('select2-id')) {
                initSelect2(userSelect);
            }
        }
    });

    $('#fca-groups-container').on('change', '.link-type-select', function() {
        handleLinkTypeChange($(this));
    });

    $('#fca-groups-container').on('change', '.icon-type-select', function() {
        handleIconTypeChange($(this));
    });
    
    // Sync hidden icon value when specific icon inputs change
    $('#fca-groups-container').on('input change', '.icon-value-input', function() {
        syncIconValue($(this));
    });

    // SVG Upload Button
    $('#fca-groups-container').on('click', '.fca-svg-upload-btn', function(e) {
        e.preventDefault();
        const button = $(this);
        const urlInput = button.siblings('input[type="url"]');
        const previewSpan = button.siblings('.svg-preview');

        const frame = wp.media({
            title: 'Select or Upload SVG Icon',
            button: {
                text: 'Use this SVG'
            },
            library: {
                type: 'image/svg+xml' // Only allow SVG
            },
            multiple: false
        });

        frame.on('select', function() {
            const attachment = frame.state().get('selection').first().toJSON();
            if (attachment.mime === 'image/svg+xml') {
                urlInput.val(attachment.url).trigger('change'); // Update URL and trigger change to sync final value
                previewSpan.html(`<img src="${attachment.url}" alt="Preview" style="max-width: 24px; max-height: 24px; vertical-align: middle; margin-left: 5px;">`);
            } else {
                alert('Please select a valid SVG file.');
            }
        });

        frame.open();
    });

    // Initialize media uploader for SVG
    $(document).on('click', '.fca-svg-upload-btn', function(e) {
        e.preventDefault();
        
        var button = $(this);
        var field = button.prev('.icon-value-input');
        var preview = button.next('.svg-preview');
        
        var frame = wp.media({
            title: 'Select or Upload SVG',
            library: {
                type: ['image/svg+xml']
            },
            button: {
                text: 'Use this SVG'
            },
            multiple: false
        });
        
        frame.on('select', function() {
            var attachment = frame.state().get('selection').first().toJSON();
            field.val(attachment.url);
            preview.html('<img src="' + attachment.url + '" alt="Preview" style="max-width: 24px; max-height: 24px; vertical-align: middle; margin-left: 5px;">');
            
            // Update the hidden field with the selected value
            updateIconValueField(field.closest('.column-icon'));
        });
        
        frame.open();
    });
    
    // Initialize media uploader for Images
    $(document).on('click', '.fca-image-upload-btn', function(e) {
        e.preventDefault();
        
        var button = $(this);
        var field = button.prev('.icon-value-input');
        var preview = button.next('.image-preview');
        
        var frame = wp.media({
            title: 'Select or Upload Image',
            library: {
                type: ['image']
            },
            button: {
                text: 'Use this Image'
            },
            multiple: false
        });
        
        frame.on('select', function() {
            var attachment = frame.state().get('selection').first().toJSON();
            field.val(attachment.url);
            preview.html('<img src="' + attachment.url + '" alt="Preview" style="max-width: 24px; max-height: 24px; vertical-align: middle; margin-left: 5px;">');
            
            // Update the hidden field with the selected value
            updateIconValueField(field.closest('.column-icon'));
        });
        
        frame.open();
    });
    
    // Handle icon type change
    $(document).on('change', '.icon-type-select', function() {
        var select = $(this);
        var iconType = select.val();
        var container = select.closest('.column-icon');
        
        // Hide all icon type fields
        container.find('.icon-type-fields > div').hide();
        
        // Show the relevant field based on selection
        if (iconType === 'emoji') {
            container.find('.emoji-field').show();
        } else if (iconType === 'svg_url') {
            container.find('.svg-url-field').show();
        } else if (iconType === 'svg_code') {
            container.find('.svg-code-field').show();
        } else if (iconType === 'image') {
            container.find('.image-field').show();
        } else if (iconType === 'space_icon') {
            container.find('.space-icon-field').show();
        } else if (iconType === 'course_icon') {
            container.find('.course-icon-field').show();
        }
        
        // Update the hidden field with the current selected value
        updateIconValueField(container);
    });
    
    // Function to update the icon_value hidden field
    function updateIconValueField(container) {
        var iconType = container.find('.icon-type-select').val();
        var iconValueField = container.find('.icon-value-final');
        
        if (iconType === 'emoji') {
            iconValueField.val(container.find('[name$="[icon_value_emoji]"]').val());
        } else if (iconType === 'svg_url') {
            iconValueField.val(container.find('[name$="[icon_value_svg_url]"]').val());
        } else if (iconType === 'svg_code') {
            iconValueField.val(container.find('[name$="[icon_value_svg_code]"]').val());
        } else if (iconType === 'image') {
            iconValueField.val(container.find('[name$="[icon_value_image]"]').val());
        } else if (iconType === 'space_icon') {
            iconValueField.val(container.find('[name$="[icon_value_space]"]').val());
        } else if (iconType === 'course_icon') {
            iconValueField.val(container.find('[name$="[icon_value_course]"]').val());
        }
    }
    
    // Update icon value when any input changes
    $(document).on('input change', '.icon-value-input', function() {
        updateIconValueField($(this).closest('.column-icon'));
    });
    
    $(document).on('change', '.icon-value-space', function() {
        updateSpaceIconPreview($(this));
        updateIconValueField($(this).closest('.column-icon'));
    });

    // Add listener for course icon select change
    $(document).on('change', '.icon-value-course', function() {
        updateEntityIconPreview($(this), 'course');
        updateIconValueField($(this).closest('.column-icon'));
    });

    // Initialize space icon previews on page load for visible items
    $('.icon-value-space:visible').each(function() {
        if ($(this).val()) {
            updateSpaceIconPreview($(this));
        }
    });
    
    // When icon type changes to space_icon, update preview if a space is already selected
    $(document).on('change', '.icon-type-select', function() {
        const $select = $(this);
        const $columnIcon = $select.closest('.column-icon');
        if ($select.val() === 'space_icon') {
            const spaceSelect = $columnIcon.find('.icon-value-space');
            if (spaceSelect.val()) {
                updateSpaceIconPreview(spaceSelect);
            }
        } else if ($select.val() === 'course_icon') {
            const courseSelect = $columnIcon.find('.icon-value-course');
            if (courseSelect.val()) {
                updateEntityIconPreview(courseSelect, 'course');
            }
        }
    });

    // --- Generic Entity Icon Preview Function ---
    function updateEntityIconPreview(entitySelect, entityType) {
        const entityId = entitySelect.val();
        const previewContainerSelector = '.' + entityType + '-icon-preview';
        const $previewContainer = entitySelect.siblings(previewContainerSelector);

        if (!entityId) {
            $previewContainer.html('');
            return;
        }

        $previewContainer.html('<span style="color:#999;font-style:italic;margin-left:5px;">Loading...</span>');

        $.ajax({
            url: config.ajaxurl,
            type: 'POST',
            dataType: 'json',
            data: {
                action: 'fca_widgets_get_space_icon_preview',
                security: config.nonce,
                space_id: entityId,
                entity_type_for_preview: entityType
            },
            success: function(response) {
                if (response && typeof response.success !== 'undefined') {
                    if (response.success && typeof response.data !== 'undefined') {
                        $previewContainer.html(response.data);
                    } else {
                        const errorMessage = (response.data && typeof response.data === 'string') ? response.data : 'Preview unavailable.';
                        $previewContainer.html('<span style="color:#e74c3c;font-style:italic;margin-left:5px;">' + errorMessage + '</span>');
                    }
                } else {
                     $previewContainer.html('<span style="color:#c0392b;font-style:italic;margin-left:5px;">Error: Invalid Response</span>');
                }
            },
            error: function() {
                $previewContainer.html('<span style="color:#c0392b;font-style:italic;margin-left:5px;">AJAX Error</span>');
            }
        });
    }

    // --- Initialization ---
    $(document).ready(function() {
        // Add styles for the save button and status indicator
        $('<style>')
            .prop('type', 'text/css')
            .html(`
                #fca-save-status {
                    display: inline-block;
                    margin-left: 10px;
                    font-weight: bold;
                }
                #fca-save-status .saving {
                    color: #f39c12;
                }
                #fca-save-status .saved {
                    color: #27ae60;
                }
                #fca-save-status .error {
                    color: #e74c3c;
                }
                @keyframes spinner {
                    to {transform: rotate(360deg);}
                }
                .saving:before {
                    content: '';
                    display: inline-block;
                    width: 12px;
                    height: 12px;
                    border-radius: 50%;
                    margin-right: 5px;
                    border: 2px solid #f39c12;
                    border-top-color: transparent;
                    animation: spinner .6s linear infinite;
                    vertical-align: middle;
                }
            `)
            .appendTo('head');
            
        // Add a save status indicator next to our buttons
        $('.submit-buttons-row').append('<div id="fca-save-status"></div>');
        
        // Attach click handler to the AJAX save button (now added via PHP)
        $('#save-via-ajax').on('click', function(e) {
            e.preventDefault();
            saveMenuDataAjax();
        });
        
        // Initialize Select2 on initial page load
        initializeAllSelect2();
        
        // Initialize sorting
        makeGroupsSortable();
        $('.fc-items-tbody').each(function() {
            makeItemsSortable($(this));
        });
        
        // Initialize visibility settings on page load
        $('.visibility-select').each(function() {
            handleVisibilityChange($(this));
        });
        
        // Initialize link type and icon type fields on load
        $('.link-type-select, .icon-type-select').each(function() {
            $(this).trigger('change');
        });
        
        // Add new handler for space icon synchronization on existing items
        $('.fc-item-row').each(function() {
            const $row = $(this);
            const linkTypeSelect = $row.find('.link-type-select');
            const iconTypeSelect = $row.find('.icon-type-select');
            
            // If this row has space link type and space icon type already set
            if (linkTypeSelect.val() === 'space' && iconTypeSelect.val() === 'space_icon') {
                // Set up the change event
                const spaceSelect = $row.find('.space-select-field select');
                spaceSelect.off('change.linkToIcon').on('change.linkToIcon', function() {
                    const selectedSpaceId = $(this).val();
                    if (selectedSpaceId) {
                        const spaceIconSelect = $row.find('.space-icon-field select');
                        spaceIconSelect.val(selectedSpaceId).trigger('change');
                        
                        // Also update the final value
                        const finalValueInput = $row.find('.icon-value-final');
                        finalValueInput.val(selectedSpaceId);
                    }
                });
            }
            // Add similar logic for course link type and course icon type
            if (linkTypeSelect.val() === 'course' && iconTypeSelect.val() === 'course_icon') {
                const courseSelect = $row.find('.course-select-field select');
                courseSelect.off('change.linkToIcon').on('change.linkToIcon', function() {
                    const selectedCourseId = $(this).val();
                    if (selectedCourseId) {
                        const courseIconSelect = $row.find('.course-icon-field select');
                        courseIconSelect.val(selectedCourseId).trigger('change');
                        const finalValueInput = $row.find('.icon-value-final');
                         if(finalValueInput.length) {
                              finalValueInput.val(selectedCourseId);
                         }
                    }
                });
            }
        });
        
        // Prevent accidental form submission on Enter key in text inputs
        $('#fca-sidebar-menu-form').on('keypress', 'input[type="text"], input[type="url"]', function(e) {
            if (e.keyCode === 13) {
                e.preventDefault();
                return false;
            }
        });
    });
});

