<?php

class FCA_WidgetManager {
    private $option_name = 'fca_widgets';
    private $nonce_name = 'fca_widget_manager';

    public function __construct() {
        // Check if FluentCommunity is active
        if (!defined('FLUENT_COMMUNITY_PLUGIN_VERSION')) {
            add_action('admin_notices', function() {
                echo '<div class="error"><p>FCA Widgets requires FluentCommunity plugin to be installed and activated.</p></div>';
            });
            return;
        }

        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Frontend actions
        add_action('fluent_community/portal_footer', array($this, 'add_widgets'), 20);
        add_action('fluent_community/portal_footer', array($this, 'add_profile_info'), 20);

        // Add AJAX handler for widget deletion
        add_action('wp_ajax_fca_delete_widget', array($this, 'handle_delete_widget'));
        
        // Add TinyMCE button for FC styled buttons
        add_filter('mce_buttons', array($this, 'register_mce_button'));
        add_filter('mce_external_plugins', array($this, 'add_mce_plugin'));

        // Add redirect after saving a new widget
        add_action('admin_init', array($this, 'maybe_redirect_after_save'));

        // Update the wp_localize_script call
        wp_localize_script('fca-widget-manager', 'fcaSettings', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'optionName' => $this->option_name,
            'nonce' => wp_create_nonce($this->nonce_name)
        ));
    }

    public function register_settings() {
        register_setting('fca_widgets', $this->option_name, array(
            'sanitize_callback' => array($this, 'sanitize_widgets')
        ));
        
        // Add editor styles specifically for the TinyMCE iframe
        add_editor_style(plugin_dir_url(dirname(__FILE__)) . 'css/editor-style.css');
    }

    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'fca-widgets_page_fca-widgets-manager') {
            return;
        }

        // Enqueue WordPress built-in thickbox for reliable modals
        wp_enqueue_script('thickbox');
        wp_enqueue_style('thickbox');

        wp_enqueue_style('select2', 'https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/css/select2.min.css');
        wp_enqueue_script('select2', 'https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/js/select2.min.js', array('jquery'));
        wp_enqueue_script('jquery-ui-sortable');
        wp_enqueue_script('fca-widget-manager', plugin_dir_url(dirname(__FILE__)) . 'js/widget-manager.js', array('jquery', 'select2', 'jquery-ui-sortable', 'thickbox'), '1.0', true);

        wp_localize_script('fca-widget-manager', 'fcaSettings', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'optionName' => $this->option_name,
            'nonce' => wp_create_nonce($this->nonce_name)
        ));
    }

    public function render_admin_page() {
        $widgets = get_option($this->option_name, array());
        // Ensure $widgets is always an array, even if get_option fails unexpectedly
        if (!is_array($widgets)) {
            $widgets = array(); 
        }
        
        // Sort widgets by order if available
        if (!empty($widgets)) {
            uasort($widgets, function($a, $b) {
                $a_order = isset($a['order']) ? $a['order'] : 999;
                $b_order = isset($b['order']) ? $b['order'] : 999;
                return $a_order - $b_order;
            });
        }
        
        // Check if we're editing a specific widget
        $edit_widget_id = isset($_GET['edit']) ? sanitize_text_field($_GET['edit']) : null;
        
        if ($edit_widget_id) {
            if ($edit_widget_id === 'new') {
                // Generate a new unique ID for the widget
                $new_widget_id = 'widget_' . time() . '_' . mt_rand(1000, 9999);
                
                // Render empty widget form for new widget
                $this->render_edit_page($new_widget_id, array(
                    'title' => '',
                    'content' => '',
                    'spaces' => array(),
                    'enabled' => true
                ));
                return;
            } elseif (isset($widgets[$edit_widget_id])) {
                // Render existing widget form
                $this->render_edit_page($edit_widget_id, $widgets[$edit_widget_id]);
                return;
            }
        }
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline">Widget Manager</h1>
            <a href="#" class="page-title-action" id="add-widget">Add New Widget</a>
            
            <?php wp_nonce_field('fca_widget_manager'); ?>
            
            <div class="fca-widgets-list">
                <!-- Add search and filter controls -->
                <div class="fca-widgets-filters">
                    <div class="fca-search-box">
                        <input type="text" id="widget-search" placeholder="Search widgets..." class="regular-text">
                        <button type="button" class="button" id="search-clear">Clear</button>
                    </div>
                    
                    <div class="fca-space-filter">
                        <select id="space-filter">
                            <option value="">All Spaces</option>
                            <?php 
                            $spaces = $this->get_available_spaces();
                            foreach ($spaces as $space) {
                                echo '<option value="' . esc_attr($space['value']) . '">' . esc_html($space['label']) . '</option>';
                            }
                            ?>
                        </select>
                    </div>
                </div>
                
                <form method="post" action="options.php" id="widgets-form">
                    <?php settings_fields('fca_widgets'); ?>
                    
                    <table class="wp-list-table widefat fixed striped" id="sortable-widgets-table">
                    <thead>
                        <tr>
                                <th width="30"></th>
                            <th scope="col" class="column-title column-primary">Title</th>
                            <th scope="col">Status</th>
                            <th scope="col">Spaces</th>
                            <th scope="col">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (!empty($widgets)) : ?>
                            <?php foreach ($widgets as $widget_id => $widget) : ?>
                                    <tr data-widget-id="<?php echo esc_attr($widget_id); ?>" 
                                        data-spaces="<?php echo esc_attr(json_encode(!empty($widget['spaces']) ? $widget['spaces'] : [])); ?>">
                                        <td>
                                            <span class="dashicons dashicons-move widget-drag-handle"></span>
                                        </td>
                                    <td class="column-title column-primary">
                                        <strong>
                                            <a href="<?php echo add_query_arg('edit', $widget_id); ?>">
                                                <?php echo esc_html($widget['title']); ?>
                                            </a>
                                        </strong>
                                            <input type="hidden" name="<?php echo $this->option_name; ?>[<?php echo $widget_id; ?>][order]" 
                                                   value="<?php echo isset($widget['order']) ? intval($widget['order']) : 999; ?>" 
                                                   class="widget-order-input">
                                    </td>
                                    <td>
                                        <?php echo !empty($widget['enabled']) ? 
                                            '<span class="status-active">Active</span>' : 
                                            '<span class="status-inactive">Inactive</span>'; ?>
                                    </td>
                                    <td>
                                        <?php 
                                        if (!empty($widget['spaces'])) {
                                            $space_names = array_map(function($slug) {
                                                $space = $this->get_space_by_slug($slug);
                                                return $space ? $space->title : $slug;
                                            }, $widget['spaces']);
                                            echo esc_html(implode(', ', $space_names));
                                        } else {
                                            echo 'All Spaces';
                                        }
                                        ?>
                                    </td>
                                    <td class="actions">
                                        <a href="<?php echo add_query_arg('edit', $widget_id); ?>" 
                                           class="button button-small">Edit</a>
                                        <button type="button" 
                                                class="button button-small button-link-delete remove-widget" 
                                                data-id="<?php echo esc_attr($widget_id); ?>">
                                            Delete
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                    <td colspan="5">No widgets found. <a href="#" id="add-first-widget">Create your first widget</a>.</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
                    
                    <input type="hidden" name="<?php echo $this->option_name; ?>[widget_order]" id="widget-order" value="">
                    
                    <?php if (!empty($widgets)) : ?>
                        <?php submit_button('Save Widget Order'); ?>
                    <?php endif; ?>
                </form>
            </div>
        </div>

        <style>
            .widget-drag-handle {
                cursor: move;
                color: #999;
            }
            .widget-drag-handle:hover {
                color: #444;
            }
            .ui-sortable-placeholder {
                background-color: #f9f9f9 !important;
                border: 2px dashed #ccc !important;
                visibility: visible !important;
                height: 50px !important;
            }
            .status-active {
                color: #46b450;
                font-weight: 500;
            }
            .status-inactive {
                color: #dc3232;
            }
            .actions {
                display: flex;
                gap: 8px;
            }
            .fca-widgets-list {
                margin-top: 20px;
            }
            
            /* New styles for filters */
            .fca-widgets-filters {
                display: flex;
                justify-content: space-between;
                margin-bottom: 15px;
                align-items: center;
            }
            .fca-search-box {
                display: flex;
                gap: 5px;
                align-items: center;
            }
            .fca-space-filter {
                min-width: 200px;
            }
            .widget-hidden {
                display: none;
            }
            .no-results-message {
                display: none;
                padding: 15px;
                background: #f8f9fa;
                border: 1px solid #ddd;
                text-align: center;
                margin-top: 10px;
            }
        </style>
        <?php
    }

    private function render_edit_page($widget_id, $widget) {
        // Check if this is a new widget that was just saved
        $widget_saved = isset($_GET['settings-updated']) && $_GET['settings-updated'] === 'true';
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline">
                <?php echo $widget_id === 'new' ? 'Add New Widget' : 'Edit Widget'; ?>
            </h1>
            <a href="<?php echo admin_url('admin.php?page=fca-widgets-manager'); ?>" class="page-title-action">← Back to List</a>
            
            <?php if ($widget_saved): ?>
            <div id="widget-saved-message" class="notice notice-success is-dismissible">
                <p><strong>Widget saved successfully!</strong> Redirecting to widget list...</p>
            </div>
            <script>
                // Auto-redirect after 1 second
                setTimeout(function() {
                    window.location.href = '<?php echo admin_url('admin.php?page=fca-widgets-manager&widget-saved=1'); ?>';
                }, 1000);
            </script>
            <?php endif; ?>
            
            <form method="post" action="options.php">
                <?php settings_fields('fca_widgets'); ?>
                
                <div class="fca-widget-edit">
                    <div class="fca-widget-main">
                        <div class="postbox">
                            <div class="postbox-header">
                                <h2>Widget Content</h2>
                            </div>
                            <div class="inside">
                                <p>
                                    <label for="widget-title">Title</label>
                                    <input type="text" 
                                           id="widget-title"
                                           name="<?php echo $this->option_name; ?>[<?php echo $widget_id; ?>][title]"
                                           value="<?php echo esc_attr($widget['title'] ?? ''); ?>"
                                           class="large-text">
                                </p>
                                
                                <?php 
                                wp_editor(
                                    $widget['content'] ?? '',
                                    'widget_content_' . $widget_id,
                                    array(
                                        'textarea_name' => $this->option_name . '[' . $widget_id . '][content]',
                                        'media_buttons' => true,
                                        'textarea_rows' => 15,
                                        'editor_height' => 300
                                    )
                                );
                                ?>
                                
                                <div class="fca-button-preview">
                                    <h3>Button Preview</h3>
                                    <p><em>Note: Actual appearance will match FluentCommunity's styling when displayed on the frontend.</em></p>
                                    <div class="preview-buttons">
                                        <a href="#" class="el-button fcom_primary_button"><span>Primary Button</span></a>
                                        <a href="#" class="el-button"><span>Secondary Button</span></a>
                                        <a href="#" class="el-button el-button--text"><span>Text Button</span></a>
                                        <a href="#" class="fcom_link">Link</a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="fca-widget-sidebar">
                        <div class="postbox">
                            <div class="postbox-header">
                                <h2>Widget Settings</h2>
                            </div>
                            <div class="inside">
                                <p>
                                    <label>
                                        <input type="checkbox" 
                                               name="<?php echo $this->option_name; ?>[<?php echo $widget_id; ?>][enabled]"
                                               value="1"
                                               <?php checked(!empty($widget['enabled'])); ?>>
                                        Enable Widget
                                    </label>
                                </p>
                                
                                <p>
                                    <label>
                                        <input type="checkbox" 
                                               name="<?php echo $this->option_name; ?>[<?php echo $widget_id; ?>][top_position]"
                                               value="1"
                                               <?php checked(!empty($widget['top_position'])); ?>>
                                        Show above default widgets
                                    </label>
                                </p>
                                
                                <p>
                                    <label>Display in Spaces:</label>
                                    <select class="space-select" 
                                            name="<?php echo $this->option_name; ?>[<?php echo $widget_id; ?>][spaces][]"
                                            multiple>
                                        <?php 
                                        $spaces = $this->get_available_spaces();
                                        $selected_spaces = $widget['spaces'] ?? array();
                                        foreach ($spaces as $space) {
                                            printf(
                                                '<option value="%s" %s>%s</option>',
                                                esc_attr($space['value']),
                                                selected(in_array($space['value'], $selected_spaces), true, false),
                                                esc_html($space['label'])
                                            );
                                        }
                                        ?>
                                    </select>
                                    <span class="description">Leave empty to show in all spaces</span>
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <?php submit_button(); ?>
            </form>

            <!-- ThickBox Button Form - Hidden, but will be shown in a ThickBox modal -->
            <div id="fca-button-form" style="display:none;">
                <div style="padding:15px;">
                    <h2 style="margin-top:0;">Insert FluentCommunity Button</h2>
                    <div style="margin:15px 0;">
                        <label for="tb-button-text" style="display:block;margin-bottom:5px;">Button Text</label>
                        <input type="text" id="tb-button-text" style="width:100%;" />
                    </div>
                    <div style="margin:15px 0;">
                        <label for="tb-button-url" style="display:block;margin-bottom:5px;">Button URL</label>
                        <input type="url" id="tb-button-url" style="width:100%;" placeholder="https://..." />
                    </div>
                    <div style="margin:15px 0;">
                        <label for="tb-button-style" style="display:block;margin-bottom:5px;">Button Style</label>
                        <select id="tb-button-style" style="width:100%;">
                            <option value="fcom_primary_button" selected>Primary</option>
                            <option value="">Secondary</option>
                            <option value="el-button--text">Text</option>
                            <option value="fcom_link">Link</option>
                        </select>
                    </div>
                    <div style="margin:15px 0;text-align:right;">
                        <button type="button" class="button button-primary" id="tb-insert-button">Insert Button</button>
                    </div>
                </div>
            </div>
        </div>

        <style>
            .fca-widget-edit {
                display: grid;
                grid-template-columns: 1fr 300px;
                gap: 20px;
                margin-top: 20px;
            }
            .fca-widget-main {
                min-width: 0;
            }
            .fca-widget-sidebar {
                min-width: 0;
            }
            .space-select {
                width: 100%;
                margin-top: 5px;
            }
            .fca-button-preview {
                margin-top: 20px;
                padding: 15px;
                background: #f8f9fa;
                border: 1px solid #ddd;
            }
            .fca-button-preview h3 {
                margin-top: 0;
                margin-bottom: 10px;
            }
            .preview-buttons {
                display: flex;
                gap: 10px;
                flex-wrap: wrap;
            }
            
            /* Very basic styling just for the admin preview */
            .preview-buttons .el-button {
                display: inline-flex;
                padding: 8px 16px;
                border-radius: 4px;
                text-decoration: none;
                border: 1px solid #ddd;
                cursor: pointer;
            }
            .preview-buttons .fcom_primary_button {
                background-color: #2B2E33;
                color: white;
                border-color: #2B2E33;
            }
            .preview-buttons .el-button--text {
                border-color: transparent;
                background: transparent;
                color: #2B2E33;
            }
            .preview-buttons .fcom_link {
                color: #2B2E33;
                text-decoration: none;
            }
        </style>
        <?php
    }

    /**
     * Get available spaces for dropdown selection
     */
    private function get_available_spaces() {
        $spaces = array(
            // Add the All Feeds option at the top
            array(
                'value' => 'all_feeds',
                'label' => 'All Feeds (Homepage)'
            )
        );
        
        // Add regular spaces if FluentCommunity is active
        if (class_exists('\FluentCommunity\App\Functions\Utility')) {
            try {
                $fc_spaces = \FluentCommunity\App\Functions\Utility::getSpaces(false);
                foreach ($fc_spaces as $space) {
                    if (is_object($space) && isset($space->slug) && isset($space->title)) {
                        $spaces[] = array(
                            'value' => $space->slug,
                            'label' => $space->title
                        );
                    }
                }
            } catch (\Exception $e) {
                error_log('Error getting spaces: ' . $e->getMessage());
            }
        }
        
        return $spaces;
    }

    public function add_widgets() {
        $widgets = get_option($this->option_name, array());
        if (empty($widgets)) {
            return;
        }

        ?>
        <script>
            (function(window, document, undefined) {
            document.addEventListener('DOMContentLoaded', function () {
                let isUpdating = false;
                let lastPath = window.location.pathname;
                let updateTimeout = null;

                function addWidgets() {
                    if (isUpdating) {
                        return;
                    }

                    isUpdating = true;

                    // Remove all existing widgets first
                    document.querySelectorAll('[id^="fca-widget-"]').forEach(widget => {
                        widget.remove();
                    });

                    // Try multiple possible selectors for the activity section
                    const activitiesSection = document.querySelector('.app_side_widget.widget_recent_activities') || 
                                             document.querySelector('.sidebar_widgets') ||
                                             document.querySelector('.fcom_sidebar_widgets') ||
                                             document.querySelector('.fc_sidebar_widgets');
                    
                    if (!activitiesSection) {
                        console.log('FCA Widgets: Could not find the sidebar widgets section.');
                        isUpdating = false;
                        return;
                    }

                    // Get current space slug from URL
                    const spaceMatch = window.location.pathname.match(/\/space\/([^\/]+)/);
                    const currentSpaceSlug = spaceMatch ? spaceMatch[1] : null;
                    
                    // Check if we're on the main feed page by looking at the body data-route attribute
                    const isMainFeed = document.body.getAttribute('data-route') === 'all_feeds';

                    // Check if we're on a course page
                    const isCourse = document.body.getAttribute('data-route') === 'course' || 
                                     window.location.pathname.includes('/course/');

                    // Skip adding widgets on course pages
                    if (isCourse) {
                        isUpdating = false;
                        return;
                    }

                    // Define widgets array
                    const widgets = [
                        <?php 
                        // Sort widgets by position (top first, then normal) and by order within each group
                        $top_widgets = array();
                        $normal_widgets = array();
                        
                        foreach ($widgets as $widget_id => $widget) {
                            if (empty($widget['enabled'])) continue;
                            
                            // Process shortcodes and apply filters
                            $content = do_shortcode(wpautop($widget['content']));
                            $content = apply_filters('the_content', $content);
                            
                            $widget_data = "{\n";
                            $widget_data .= "    id: '" . esc_js($widget_id) . "',\n";
                            $widget_data .= "    title: '" . esc_js($widget['title']) . "',\n";
                            $widget_data .= "    content: " . json_encode($content) . ",\n";
                            $widget_data .= "    spaces: " . json_encode(array_map('strval', (array)($widget['spaces'] ?? []))) . ",\n";
                            $widget_data .= "    top_position: " . (!empty($widget['top_position']) ? 'true' : 'false') . ",\n";
                            $widget_data .= "    order: " . (isset($widget['order']) ? intval($widget['order']) : 999) . "\n";
                            $widget_data .= "}";
                            
                            if (!empty($widget['top_position'])) {
                                $top_widgets[] = $widget_data;
                            } else {
                                $normal_widgets[] = $widget_data;
                            }
                        }
                        
                        echo implode(",\n", array_merge($top_widgets, $normal_widgets));
                        ?>
                    ];

                    // Debug log widgets - uncomment if needed
                    // console.log('FCA Widgets: Found', widgets.length, 'widgets to display');

                    // Sort widgets by order
                    widgets.sort((a, b) => {
                        // First sort by position (top first)
                        if (a.top_position !== b.top_position) {
                            return a.top_position ? -1 : 1;
                        }
                        // Then sort by order
                        return a.order - b.order;
                    });

                    // Process each widget
                    widgets.forEach(widget => {
                        // Check if widget should appear in this space
                        if (widget.spaces && widget.spaces.length > 0) {
                            // Check for all_feeds special case
                            if (widget.spaces.includes('all_feeds')) {
                                // Only show on main feed page
                                if (!isMainFeed) {
                                    return;
                                }
                            } else if (!currentSpaceSlug || !widget.spaces.includes(currentSpaceSlug)) {
                                // Not configured for this space
                                return;
                            }
                        }

                        // Create the widget element
                        const customDiv = document.createElement('div');
                        customDiv.id = 'fca-widget-' + widget.id;
                        customDiv.className = 'app_side_widget fca-widget';
                        customDiv.style = `
                            padding: 15px;
                            margin-bottom: 15px;
                            background-color: var(--fcom-primary-bg, #ffffff);
                            color: var(--fcom-primary-text, #000000);
                            border-radius: 8px;
                        `;

                        customDiv.innerHTML = `
                            <h3 style="margin: 0 0 10px 0; font-size: 16px;">${widget.title}</h3>
                            <div class="widget-content entry-content wp-block-post-content">
                                ${widget.content}
                            </div>
                        `;

                        // Insert at the appropriate position
                        if (widget.top_position) {
                            // Insert before the activities section
                            activitiesSection.parentElement.insertBefore(customDiv, activitiesSection);
                        } else {
                            // Insert after the activities section
                            if (activitiesSection.nextSibling) {
                                activitiesSection.parentElement.insertBefore(customDiv, activitiesSection.nextSibling);
                            } else {
                                activitiesSection.parentElement.appendChild(customDiv);
                            }
                        }

                        // Fix links in the widget content
                        customDiv.querySelectorAll('a').forEach(link => {
                            if (link.getAttribute('target') !== '_blank') {
                                link.setAttribute('target', '_self');
                            }
                        });

                        // Re-run any scripts that might have been added via shortcodes
                        const scripts = customDiv.querySelectorAll('script');
                        if (scripts.length > 0) {
                            scripts.forEach(script => {
                                const newScript = document.createElement('script');
                                Array.from(script.attributes).forEach(attr => {
                                    newScript.setAttribute(attr.name, attr.value);
                                });
                                newScript.textContent = script.textContent;
                                script.parentNode.replaceChild(newScript, script);
                            });
                        }
                    });

                    // Debug log
                    // console.log('FCA Widgets: Widget injection complete');

                    setTimeout(() => {
                        isUpdating = false;
                    }, 500);
                }

                // Debounced update function
                function debouncedUpdate() {
                    if (updateTimeout) {
                        clearTimeout(updateTimeout);
                    }
                    updateTimeout = setTimeout(addWidgets, 250);
                }

                // Watch for path changes
                setInterval(() => {
                    if (window.location.pathname !== lastPath) {
                        lastPath = window.location.pathname;
                        debouncedUpdate();
                    }
                }, 100);

                // Initial check
                debouncedUpdate();

                // Set up observer for dynamic content changes
                const observer = new MutationObserver((mutations) => {
                    if (!isUpdating && window.location.pathname === lastPath) {
                        const shouldUpdate = mutations.some(mutation => {
                            return mutation.target.classList && 
                                (mutation.target.classList.contains('app_side_widget') ||
                                 mutation.target.classList.contains('sidebar_widgets') ||
                                 mutation.target.classList.contains('fcom_sidebar_widgets') ||
                                 mutation.target.classList.contains('fc_sidebar_widgets'));
                        });

                        if (shouldUpdate) {
                            debouncedUpdate();
                        }
                    }
                });

                observer.observe(document.body, {
                    childList: true,
                    subtree: true,
                    attributes: false,
                    characterData: false
                });

                // Call addWidgets directly after a short delay to ensure DOM is ready
                setTimeout(addWidgets, 500);
            });
            })(window, document);
        </script>
        <?php
    }

    public function sanitize_widgets($input) {
        if (!is_array($input)) {
            return array();
        }

        // Get existing widgets from the database
        $existing_widgets = get_option($this->option_name, array());
        if (!is_array($existing_widgets)) $existing_widgets = [];

        // Start with the existing data as the base for sanitization
        $sanitized = $existing_widgets;

        // Check if this submission is for saving the widget order from the list page
        if (isset($input['widget_order'])) {
            $widget_order_json = $input['widget_order'];
            $widget_order = json_decode(stripslashes($widget_order_json), true);
            if (!is_array($widget_order)) $widget_order = [];

            // Update the order based on the submitted order array
            if (!empty($widget_order)) {
                $new_order_map = array_flip($widget_order);
                foreach ($sanitized as $widget_id => &$widget_data) {
                     if (is_array($widget_data)) { // Ensure widget_data is an array
                        $widget_data['order'] = isset($new_order_map[$widget_id]) ? $new_order_map[$widget_id] : ($widget_data['order'] ?? 999);
                     }
                }
                unset($widget_data); // Unset reference
            }
            // Since this was an order save, return the sanitized array with updated orders
            return $sanitized;
        } 
        
        // --- Otherwise, assume this is a save from the EDIT page for a SINGLE widget --- 
        // The $input array will contain keys matching a single widget ID
        foreach ($input as $widget_id => $widget_data) {
            if (strpos($widget_id, 'widget_') === 0 && is_array($widget_data)) { // Identify the widget being edited
                 // Ensure spaces is always an array
                $spaces = isset($widget_data['spaces']) ? (array)$widget_data['spaces'] : array();
                
                // Sanitize the data for the specific widget being edited/added
                $sanitized[$widget_id] = array(
                    'title'        => sanitize_text_field($widget_data['title'] ?? ''),
                    'content'      => wp_kses_post($widget_data['content'] ?? ''),
                    'spaces'       => array_map('sanitize_text_field', $spaces),
                    'enabled'      => !empty($widget_data['enabled']),
                    'top_position' => !empty($widget_data['top_position']),
                    // Preserve existing order OR set a default for new widgets
                    'order'        => $existing_widgets[$widget_id]['order'] ?? 999 
                );
                // Since we found and processed the single widget edit, we can stop looping
                // (assuming only one widget is saved at a time from the edit page)
                break; 
            }
        }

        return $sanitized;
    }

    private function get_space_by_slug($slug) {
        if (!class_exists('\FluentCommunity\App\Functions\Utility')) {
            return null;
        }

        try {
            $spaces = \FluentCommunity\App\Functions\Utility::getSpaces(false);
            foreach ($spaces as $space) {
                if (is_object($space) && isset($space->slug) && $space->slug === $slug) {
                    return $space;
                }
            }
        } catch (\Exception $e) {
            error_log('Error getting space by slug: ' . $e->getMessage());
        }
        return null;
    }

    public function add_profile_info() {
        // Only add this script on profile pages
        if (!preg_match('/\/social\/u\//', $_SERVER['REQUEST_URI'])) {
            return;
        }
        ?>
        <script>
            document.addEventListener('DOMContentLoaded', function () {
                // Wait for FluentCommunityApp to be available
                function waitForFluentCommunity(callback, maxAttempts = 20) {
                    let attempts = 0;
                    const checkInterval = setInterval(() => {
                        attempts++;
                        if (window.FluentCommunityApp && window.FluentCommunityApp.appVars) {
                            clearInterval(checkInterval);
                            callback();
                        } else if (attempts >= maxAttempts) {
                            clearInterval(checkInterval);
                        }
                    }, 250);
                }

                waitForFluentCommunity(() => {
                    let isUpdating = false;
                    let lastPath = window.location.pathname;
                    let updateTimeout = null;

                    function addProfileInfo() {
                        if (isUpdating) return;
                        isUpdating = true;

                        // Remove existing profile info
                        document.querySelectorAll('[id^="fca-profile-info-"]').forEach(el => el.remove());

                        // Check if we're on a profile page
                        const profileMatch = window.location.pathname.match(/\/social\/u\/([^\/]+)/);
                        if (!profileMatch) {
                            isUpdating = false;
                            return;
                        }

                        const username = profileMatch[1];
                        const aboutSection = document.querySelector('.about_section');
                        if (!aboutSection) {
                            isUpdating = false;
                            return;
                        }

                        // Create profile info box
                        const profileBox = document.createElement('div');
                        profileBox.id = 'fca-profile-info-box';
                        profileBox.className = 'app_side_widget fca-profile-info';
                        profileBox.style = `
                            padding: 15px;
                            background-color: var(--fcom-primary-bg, #ffffff);
                            color: var(--fcom-primary-text, #000000);
                            border-radius: 8px;
                        `;

                        // Loading state
                        profileBox.innerHTML = '<div class="fca-loading">Loading profile information...</div>';
                        aboutSection.parentNode.insertBefore(profileBox, aboutSection.nextSibling);

                        // Fetch user data
                        const apiBase = window.FluentCommunityApp.appVars.apiEndpoint;
                        const nonce = window.FluentCommunityApp.appVars.nonce;

                        Promise.all([
                            // Fetch user profile
                            fetch(`${apiBase}/users/${username}`, {
                                credentials: 'same-origin',
                                headers: { 'X-WP-Nonce': nonce }
                            }).then(r => r.json()),
                            // Fetch user courses
                            fetch(`${apiBase}/users/${username}/courses`, {
                                credentials: 'same-origin',
                                headers: { 'X-WP-Nonce': nonce }
                            }).then(r => r.json())
                        ])
                        .then(([userData, coursesData]) => {
                            if (!userData || !userData.user) {
                                throw new Error('User data not found');
                            }

                            let html = '<h3>Additional Profile Information</h3><div class="profile-fields">';

                            // Add profile fields
                            Object.entries(userData.user).forEach(([key, value]) => {
                                if (value && typeof value === 'string' && !['ID', 'id', 'user_login', 'display_name'].includes(key)) {
                                    html += `
                                        <div class="profile-field">
                                            <strong>${key.replace(/_/g, ' ')}:</strong>
                                            <span>${value}</span>
                                        </div>
                                    `;
                                }
                            });

                            html += '</div>';

                            // Add courses section
                            html += '<h3 style="margin: 20px 0 15px;">Enrolled Courses</h3>';
                            html += '<div class="enrolled-courses">';

                            if (coursesData && coursesData.length) {
                                coursesData.forEach(course => {
                                    const progress = course.progress || 0;
                                    html += `
                                        <div class="course-item">
                                            <div class="course-title">${course.title}</div>
                                            <div class="course-progress">
                                                <div class="progress-bar">
                                                    <div style="width: ${progress}%"></div>
                                                </div>
                                                <div class="progress-text">${progress}% Complete</div>
                                            </div>
                                        </div>
                                    `;
                                });
                            } else {
                                html += '<p>No courses enrolled</p>';
                            }

                            html += '</div>';
                            profileBox.innerHTML = html;
                        })
                        .catch(error => {
                            profileBox.innerHTML = '<div class="fca-error">Error loading profile information</div>';
                        })
                        .finally(() => {
                            isUpdating = false;
                        });
                    }

                    // Set up observers and event handlers
                    const observer = new MutationObserver((mutations) => {
                        if (!isUpdating && window.location.pathname === lastPath) {
                            if (mutations.some(m => m.target.classList?.contains('about_section'))) {
                                addProfileInfo();
                            }
                        }
                    });

                    observer.observe(document.body, {
                        childList: true,
                        subtree: true,
                        attributes: false
                    });

                    // Check for path changes
                    setInterval(() => {
                        if (window.location.pathname !== lastPath) {
                            lastPath = window.location.pathname;
                            if (lastPath.includes('/social/u/')) {
                                addProfileInfo();
                            }
                        }
                    }, 100);

                    // Initial load
                    if (window.location.pathname.includes('/social/u/')) {
                        addProfileInfo();
                    }
                });
            });
        </script>

        <style>
            .fca-profile-info {
                padding: 15px;
                background-color: var(--fcom-primary-bg, #ffffff);
                color: var(--fcom-primary-text, #000000);
                border-radius: 8px;
            }
            .fca-profile-info h3 {
                margin: 0 0 15px;
                font-size: 16px;
            }
            .fca-profile-info .profile-fields {
                display: grid;
                gap: 10px;
            }
            .fca-profile-info .profile-field {
                display: grid;
                grid-template-columns: 120px 1fr;
                gap: 10px;
                align-items: baseline;
            }
            .fca-profile-info .course-item {
                margin-bottom: 15px;
                padding-bottom: 15px;
                border-bottom: 1px solid rgba(0,0,0,0.1);
            }
            .fca-profile-info .course-item:last-child {
                border-bottom: none;
                margin-bottom: 0;
                padding-bottom: 0;
            }
            .fca-profile-info .progress-bar {
                width: 100%;
                height: 6px;
                background: #eee;
                border-radius: 3px;
                margin-top: 5px;
                overflow: hidden;
            }
            .fca-profile-info .progress-bar > div {
                height: 100%;
                background: #2271b1;
                border-radius: 3px;
                transition: width 0.3s ease;
            }
            .fca-profile-info .progress-text {
                font-size: 12px;
                color: #666;
                margin-top: 3px;
            }
            .fca-loading {
                text-align: center;
                padding: 20px;
                color: #666;
            }
            .fca-error {
                color: #dc3232;
                text-align: center;
                padding: 10px;
            }
        </style>
        <?php
    }

    public function handle_delete_widget() {
        // Add nonce verification
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], $this->nonce_name)) {
            wp_send_json_error('Invalid nonce');
            return;
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        $widget_id = sanitize_text_field($_POST['widget_id']);
        $widgets = get_option($this->option_name, array());
        
        if (isset($widgets[$widget_id])) {
            unset($widgets[$widget_id]);
            update_option($this->option_name, $widgets);
            wp_send_json_success();
        } else {
            wp_send_json_error('Widget not found');
        }
    }

    /**
     * Register new button in the editor.
     */
    public function register_mce_button($buttons) {
        // Add the button to TinyMCE
        if (is_admin() && function_exists('get_current_screen')) {
            $screen = get_current_screen();
            if ($screen && $screen->base === 'fca-widgets_page_fca-widgets-manager') {
                array_push($buttons, 'fcabutton');
            }
        }
        return $buttons;
    }

    /**
     * Add new JavaScript to the plugin.
     */
    public function add_mce_plugin($plugin_array) {
        // Add the TinyMCE plugin
        if (is_admin() && function_exists('get_current_screen')) {
            $screen = get_current_screen();
            if ($screen && $screen->base === 'fca-widgets_page_fca-widgets-manager') {
                // Inline the plugin directly for simplicity
                $plugin_array['fcabutton'] = add_query_arg('ver', FCA_WIDGETS_VERSION, plugin_dir_url(dirname(__FILE__)) . 'js/tinymce-fc-button.js');
            }
        }
        return $plugin_array;
    }

    public function maybe_redirect_after_save() {
        // Check if we're coming from our options page with a redirect flag
        if (isset($_REQUEST['settings-updated']) && 
            isset($_REQUEST['page']) && 
            $_REQUEST['page'] === 'fca-widgets-manager' && 
            isset($_REQUEST['fca_widget_redirect'])) {
            
            // Redirect to the widget list page
            wp_redirect(admin_url('admin.php?page=fca-widgets-manager&widget-saved=1'));
            exit;
        }
    }
} 