<?php

class FCA_SidebarMenuManager {
    private $option_name = 'fca_sidebar_menu_items';
    private $data_file_path = '';
    private $visibility_options = array(
        'all' => 'Everyone',
        'logged_in' => 'Logged In Users',
        'logged_out' => 'Logged Out Users',
        'admin' => 'Admins Only',
        'specific_roles' => 'Specific Roles',
        'specific_users' => 'Specific Users',
        'space_members' => 'Space Members'
    );
    private $link_types = array(
        'custom' => 'Custom URL',
        'space' => 'Space',
        'course' => 'Course',
        'page' => 'FCA Custom Page'
    );
    private $icon_types = array(
        'emoji' => 'Emoji',
        'svg_url' => 'SVG URL',
        'svg_code' => 'SVG Code'
    );

    private $spaces_options_html = '';
    private $courses_options_html = '';
    private $pages_options_html = '';

    public function __construct() {
        // Set the path to the data file
        $this->data_file_path = FCA_WIDGETS_PLUGIN_DIR . 'data/sidebar-menu-data.json';
        
        // Make sure the data directory exists
        $this->ensure_data_directory();
        
        add_action('admin_init', array($this, 'register_settings'));
        add_filter('fluent_community/menu_groups_for_user', array($this, 'add_custom_sidebar_items'), 10, 1);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_fca_search_users', array($this, 'ajax_search_users'));
        add_action('wp_ajax_fca_widgets_add_new_sidebar_menu_item', array($this, 'ajax_add_new_sidebar_menu_item'));
        add_action('wp_ajax_fca_widgets_save_sidebar_menu', array($this, 'ajax_save_sidebar_menu'));
        add_action('wp_ajax_fca_widgets_delete_sidebar_menu_item', array($this, 'ajax_delete_sidebar_menu_item'));
        add_action('wp_ajax_fca_widgets_get_spaces', array($this, 'ajax_get_spaces'));
        add_action('wp_ajax_fca_widgets_get_courses', array($this, 'ajax_get_courses'));
        add_action('wp_ajax_fca_widgets_get_pages', array($this, 'ajax_get_pages'));
        
        // Check for plugin dependencies
        add_action('admin_notices', array($this, 'check_plugin_dependencies'));
    }

    /**
     * Ensure the data directory exists
     */
    private function ensure_data_directory() {
        $data_dir = FCA_WIDGETS_PLUGIN_DIR . 'data';
        
        if (!file_exists($data_dir)) {
            wp_mkdir_p($data_dir);
            
            // Add an index.php file to prevent directory listing
            $index_file = $data_dir . '/index.php';
            if (!file_exists($index_file)) {
                file_put_contents($index_file, '<?php // Silence is golden');
            }
            
            // Add .htaccess to prevent direct access
            $htaccess_file = $data_dir . '/.htaccess';
            if (!file_exists($htaccess_file)) {
                file_put_contents($htaccess_file, "Order deny,allow\nDeny from all");
            }
        }
    }

    /**
     * Load menu data from file
     */
    private function get_menu_data() {
        $defaults = array(
            'groups' => array(),
            'items' => array(),
            'group_order' => array(),
            'disable_default_items' => 0
            // Note: 'enabled' defaults will be handled within sanitize/render logic
        );

        if (!file_exists($this->data_file_path)) {
            return $defaults;
        }
        
        $json_data = file_get_contents($this->data_file_path);
        if (empty($json_data)) {
            return $defaults;
        }
        
        $menu_data = json_decode($json_data, true);
        if (!is_array($menu_data)) {
            return $defaults;
        }
        
        // Merge with defaults to ensure all keys exist
        return wp_parse_args($menu_data, $defaults);
    }

    /**
     * Save menu data to file
     * @param array $menu_data The data to save.
     * @param bool  $is_sanitized Whether the data is already sanitized.
     */
    private function save_menu_data($menu_data, $is_sanitized = false) {
        $this->ensure_data_directory();
        
        // Ensure we have the basic structure
        if (!isset($menu_data['groups'])) {
            $menu_data['groups'] = array();
        }
        if (!isset($menu_data['items'])) {
            $menu_data['items'] = array();
        }
        if (!isset($menu_data['group_order'])) {
            $menu_data['group_order'] = array_keys($menu_data['groups']);
        }
        
        // Sanitize the data if it hasn't been already
        $data_to_save = $is_sanitized ? $menu_data : $this->sanitize_menu_data($menu_data);
        
        // Encode as JSON with pretty formatting
        $json_data = json_encode($data_to_save, JSON_PRETTY_PRINT);
        
        // Save to file
        $result = file_put_contents($this->data_file_path, $json_data);
        
        // Log any errors
        if ($result === false) {
            // error_log('FCA Widgets: Error saving sidebar menu data to file: ' . $this->data_file_path);
            return false;
        }
        
        // error_log('FCA Widgets: Successfully saved sidebar menu data to file: ' . $this->data_file_path . ' (' . $result . ' bytes)');
        return true;
    }

    public function register_settings() {
        // We're no longer using WordPress settings API for saving
        // as we've switched to AJAX-only saving
        
        // However, we'll still register this hook to handle the AJAX save endpoint
        add_action('wp_ajax_fca_widgets_save_sidebar_menu', array($this, 'ajax_save_sidebar_menu'));
    }

    /**
     * This function is now deprecated as we're using AJAX-only saving
     * Keeping it for backwards compatibility
     */
    public function save_form_data($input) {
        return array(); // We no longer use this function
    }

    private function should_show_item($item) {
        if (!isset($item['visibility'])) {
            return true;
        }

        $visibility = $item['visibility'];
        $user_id = get_current_user_id();
        $is_logged_in = is_user_logged_in();

        switch ($visibility) {
            case 'logged_in':
                return $is_logged_in;
            
            case 'logged_out':
                return !$is_logged_in;
                
            case 'admin':
                return $is_logged_in && current_user_can('manage_options');
                
            case 'specific_roles':
                if (!$is_logged_in || empty($item['roles'])) {
                    return false;
                }
                $user = wp_get_current_user();
                foreach ($item['roles'] as $role) {
                    if (in_array($role, (array) $user->roles)) {
                        return true;
                    }
                }
                return false;
                
            case 'specific_users':
                return $is_logged_in && !empty($item['users']) && in_array($user_id, $item['users']);
                
            case 'space_members':
                if (!$is_logged_in || empty($item['spaces'])) {
                    return false;
                }
                try {
                    if (!class_exists('\FluentCommunity\App\Services\Helper')) {
                        // error_log('FCA Widgets Error: \FluentCommunity\App\Services\Helper class not found for space member check.');
                        return false;
                    }
                    $user_space_ids = \FluentCommunity\App\Services\Helper::getUserSpaceIds($user_id);
                    foreach ($item['spaces'] as $space_id) {
                        if (in_array($space_id, $user_space_ids)) {
                            return true;
                        }
                    }
                    return false;
                } catch (\Exception $e) {
                    // error_log('FCA Widgets Error checking space membership: ' . $e->getMessage());
                    return false;
                }
                
            case 'all':
            default:
                return true;
        }
    }

    public function add_custom_sidebar_items($menu_groups) {
        // Load menu data from file instead of options
        $menu_data = $this->get_menu_data();
        
        // Check if default items should be disabled
        $disable_defaults = !empty($menu_data['disable_default_items']);
        
        // If disabling defaults, start with an empty array, otherwise use the provided groups
        if ($disable_defaults) {
            // error_log('FCA Widgets: Default sidebar groups are disabled by setting. Starting with empty groups.');
            $menu_groups = []; // Start fresh
        }
        
        if (empty($menu_data['groups'])) {
             // If no custom groups defined, return either empty array or original groups
            // error_log('FCA Widgets: No custom groups defined.');
            return $menu_groups;
        }

        // Class alias and helper setup
        if (!class_exists('FcaFcUtil') && class_exists('\\FluentCommunity\\App\\Services\\Helper')) {
            class_alias('\\FluentCommunity\\App\\Services\\Helper', 'FcaFcUtil');
        }
        $fc_helper_exists = class_exists('FcaFcUtil');
        $portal_slug = apply_filters('fluent_community/portal_slug', 'portal');

        // Prepare for merging/overwriting
        $existing_groups = array();
        if (!$disable_defaults) {
            // Only build existing_groups index if we are merging
        foreach ($menu_groups as $index => $group) {
                if (isset($group['title'])) { // Ensure title exists
            $existing_groups[$group['title']] = $index;
                }
            }
        }

        $top_menu_groups = array();
        $normal_menu_groups = array();
        
        $group_order = $menu_data['group_order'] ?? array_keys($menu_data['groups']);

        // Loop through OUR custom groups from the file
        foreach ($group_order as $group_id) {
            if (!isset($menu_data['groups'][$group_id])) continue;

            $group = $menu_data['groups'][$group_id];
            
            // ---> Check if group is enabled (default to true if key missing)
            if (!($group['enabled'] ?? true)) {
                 // error_log('FCA Widgets: Skipping disabled group: ' . ($group['title'] ?? $group_id));
                continue; 
            }
            
            if (!$this->should_show_item($group)) {
                continue;
            }

            // Build the items for this custom group
            $group_items = array();
            if (!empty($menu_data['items'][$group_id])) {
                $item_order = $group['item_order'] ?? array_keys($menu_data['items'][$group_id]);
                foreach ($item_order as $item_id) {
                     if (!isset($menu_data['items'][$group_id][$item_id])) continue;

                    $item = $menu_data['items'][$group_id][$item_id];
                    
                    // ---> Check if item is enabled (default to true if key missing)
                    if (!($item['enabled'] ?? true)) {
                        // error_log('FCA Widgets: Skipping disabled item: ' . ($item['title'] ?? $item_id));
                        continue;
                    }
                    
                    if ($this->should_show_item($item)) {
                        $permalink = '#';
                        $item_type = $item['type'] ?? 'custom';
                        $content_value = $item['content_value'] ?? '';
                        $custom_url = $item['custom_url'] ?? '#';

                        // URL Generation Logic (remains the same)
                        $space_specific_classes = '';
                        $course_specific_classes = '';
                        $page_specific_classes = ''; // Always initialize all
                        
                        switch ($item_type) {
                            case 'space':
                                if (!empty($content_value)) {
                                    $space_slug = '';
                                    $space = null;
                                    
                                    if (class_exists('\FluentCommunity\App\Models\Space')) {
                                        $space = \FluentCommunity\App\Models\Space::find($content_value);
                                        if ($space && isset($space->slug)) {
                                            $space_slug = $space->slug;
                                        }
                                    }
                                    
                                    if (!empty($space_slug)) {
                                        $relative_path = "/space/{$space_slug}/home";
                                        $permalink = $fc_helper_exists ? FcaFcUtil::baseUrl($relative_path) : home_url("/{$portal_slug}{$relative_path}");
                                        $space_specific_classes = " fcom_space_id_{$content_value} fcom_space_{$space_slug}";
                                    } else {
                                        // error_log('FCA Widgets: Could not find slug for space ID: ' . $content_value);
                                    }
                                }
                                break;
                            case 'course':
                                if (!empty($content_value)) {
                                    $course_slug = '';
                                    $course_id = $content_value;
                                    $course = null;
                                    
                                    if (class_exists('\FluentCommunity\Modules\Course\Model\Course')) {
                                        $course = \FluentCommunity\Modules\Course\Model\Course::find($course_id);
                                        if ($course && isset($course->slug)) {
                                            $course_slug = $course->slug;
                                        }
                                    }
                                    
                                    if (!empty($course_slug)) {
                                        $relative_path = "/course/{$course_slug}/lessons";
                                        $permalink = $fc_helper_exists ? FcaFcUtil::baseUrl($relative_path) : home_url("/{$portal_slug}{$relative_path}");
                                        $course_specific_classes = " fcom_space_id_{$course_id} fcom_space_{$course_slug}"; // Use space format
                                    } else {
                                        // error_log('FCA Widgets: Could not find slug for course ID: ' . $course_id . '. Falling back to ID-based URL.');
                                        $relative_path = "/courses/{$course_id}";
                                        $permalink = $fc_helper_exists ? FcaFcUtil::baseUrl($relative_path) : home_url("/{$portal_slug}{$relative_path}");
                                    }
                                }
                                break;
                            case 'page':
                                // Add page-specific classes similar to course and space classes
                                if (!empty($content_value)) {
                                    $page_slug = get_post_field('post_name', $content_value);
                                    if ($page_slug) {
                                        $relative_path = "/page/{$page_slug}";
                                        $permalink = $fc_helper_exists ? FcaFcUtil::baseUrl($relative_path) : home_url("/{$portal_slug}{$relative_path}");
                                        // Add page-specific classes
                                        $page_specific_classes = " fcom_page_id_{$content_value} fcom_page_{$page_slug}";
                                    } else {
                                         // error_log('FCA Widgets: Could not find slug for fca_page ID: ' . $content_value);
                                    }
                                }
                                break;
                            case 'custom':
                            default:
                                $permalink = $custom_url;
                                break;
                        }

                        // Icon generation logic (remains the same)
                        $icon_type = $item['icon_type'] ?? 'emoji';
                        $icon_value = $item['icon_value'] ?? '';
                        $emoji = '';
                        $shape_svg = '';
                        if ($icon_type === 'emoji') {
                            $emoji = $icon_value;
                        } elseif ($icon_type === 'svg_code') {
                            $shape_svg = $this->sanitize_svg($icon_value); 
                        } elseif ($icon_type === 'svg_url') {
                            if (filter_var($icon_value, FILTER_VALIDATE_URL)) {
                                $shape_svg = '<svg><use xlink:href="' . esc_url($icon_value) . '"></use></svg>'; 
                            }
                        }

                        // Construct link classes
                        $base_link_classes = 'fcom_menu_link space_menu_item route_url fca-widget-menu-item fca-item-' . $item_id; // Added fca-widget-menu-item

                        // For page type items, add the special fca-page-link class that the FCA Pages plugin looks for
                        if ($item_type === 'page') {
                            $base_link_classes .= ' fca-page-link'; // Add the class that FCA Pages JavaScript recognizes
                        }

                        $link_classes = $base_link_classes;
                        
                        if (!empty($space_specific_classes)) {
                            $link_classes .= $space_specific_classes;
                        }
                        if (!empty($course_specific_classes)) {
                            $link_classes .= $course_specific_classes;
                        }
                        if (!empty($page_specific_classes)) {
                            $link_classes .= $page_specific_classes;
                        }
                        
                        // Add the constructed item to the group
                        $menu_item_data = [
                            'id' => 'fca_item_' . $item_id,
                            'key' => 'fca_item_' . $item_id,
                            'title' => $item['title'],
                            'icon_image' => '',
                            'shape_svg' => $shape_svg,
                            'emoji' => $emoji,
                            'permalink' => $permalink,
                            'link_classes' => trim($link_classes), // Trim any extra spaces
                            'data_attrs' => [
                                'data-fca-permalink' => esc_attr($permalink)
                            ]
                        ];
                        
                        // Add target="_blank" if enabled
                        if (!empty($item['open_in_new_tab'])) {
                            $menu_item_data['target'] = '_blank';
                            // Optional: Add rel attribute for security
                            // $menu_item_data['rel'] = 'noopener noreferrer';
                        }
                        
                        $group_items[] = $menu_item_data;
                    }
                }
            }

            // If the current custom group has no visible items, skip it
            if (empty($group_items)) {
                continue;
            }

            // Determine position and priority
            $is_top_position = !empty($group['top_position']);
            $priority = $is_top_position ? 5 : 15;

            // Add or merge the group
            if (!$disable_defaults && isset($existing_groups[$group['title']])) {
                 // Merge with existing default group
                $index = $existing_groups[$group['title']];
                 // Prepend custom items to maintain order preference
                $menu_groups[$index]['children'] = array_merge(
                    $group_items,
                    $menu_groups[$index]['children'] ?? array()
                );
                 $menu_groups[$index]['priority'] = $priority; // Update priority based on custom setting
                
                if ($is_top_position) {
                    $top_menu_groups[] = $menu_groups[$index];
                     unset($menu_groups[$index]); // Remove from original position
                }
                 // If not top, it stays in its merged position within $menu_groups
                 
            } else {
                 // Add as a new group (either because defaults are disabled or title doesn't match)
                $new_group = [
                    'title' => $group['title'],
                    'children' => $group_items,
                    'priority' => $priority,
                     'id' => 'fca_custom_' . $group_id // Use a unique ID
                ];
                
                if ($is_top_position) {
                    $top_menu_groups[] = $new_group;
                } else {
                    $normal_menu_groups[] = $new_group;
                }
            }
        }
        
        // Combine the groups: Top groups first, then remaining default groups (if not disabled), then normal custom groups
        $final_menu_groups = array_merge($top_menu_groups, array_values($menu_groups), $normal_menu_groups);
        
         // error_log('FCA Widgets: Final menu groups structure: ' . print_r($final_menu_groups, true));
        
        return $final_menu_groups;
    }

    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'fca-widgets_page_fca-sidebar-menu') {
            return;
        }

        wp_enqueue_media(); 

        wp_enqueue_style('select2', 'https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/css/select2.min.css', [], '4.0.13');
        wp_enqueue_script('select2', 'https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/js/select2.min.js', array('jquery'), '4.0.13', true);
        wp_enqueue_script('jquery-ui-sortable');
        
        $js_path = 'js/sidebar-menu.js';
        $js_file_path = FCA_WIDGETS_PLUGIN_DIR . $js_path;
        wp_enqueue_script('fca-sidebar-menu', FCA_WIDGETS_PLUGIN_URL . $js_path, array('jquery', 'select2', 'jquery-ui-sortable', 'wp-util'), filemtime($js_file_path), true);

        // Fetch options HTML *during page render* using render_admin_page, 
        // Here we just pass placeholders or potentially empty strings if needed by JS immediately.
        // The actual population happens later.
        wp_localize_script('fca-sidebar-menu', 'fcaSidebarMenu', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'optionName' => $this->option_name,
            'nonce' => wp_create_nonce('fca_sidebar_menu_nonce'),
            'text' => [
                'confirmRemoveGroup' => __('Are you sure you want to remove this group and all its items?', 'fca-widgets'),
                'confirmRemoveItem' => __('Are you sure you want to remove this menu item?', 'fca-widgets'),
            ],
            'roles' => $this->get_available_roles_for_js(), // Roles are usually safe to get early
            'users' => [], // Users loaded via AJAX
            // Pass empty strings, will be populated via render_admin_page and JS template
            'spacesOptionsHtml' => '', 
            'coursesOptionsHtml' => '',
            'pagesOptionsHtml' => '',
            'visibilityOptions' => $this->visibility_options,
            'linkTypes' => $this->link_types,
            'iconTypes' => $this->icon_types,
            'menuItemTemplate' => $this->get_menu_item_template() // Template needs modification in JS
        ));

        wp_add_inline_style('select2', '
            .fc-group, .fc-item-row { background: #fff; border: 1px solid #ddd; padding: 15px; margin-bottom: 15px; border-radius: 4px; }
            .fc-group-header { display: flex; align-items: center; gap: 10px; margin-bottom: 15px; padding-bottom: 10px; border-bottom: 1px solid #eee; }
            .fc-group-header input[type="text"] { flex-grow: 1; }
            .fc-group-header .visibility-wrapper, .fc-group-header .position-setting { flex-shrink: 0; }
            .fc-items-table { width: 100%; border-collapse: collapse; margin-top: 15px; }
            .fc-items-table th, .fc-items-table td { text-align: left; padding: 8px 10px; border-bottom: 1px solid #eee; vertical-align: top; }
            .fc-items-table th { background: #f9f9f9; font-weight: 600; }
            .fc-items-table td input[type="text"], .fc-items-table td select, .fc-items-table td textarea { width: 100%; box-sizing: border-box; margin-bottom: 5px; }
            .fc-items-table td textarea { min-height: 60px; font-family: monospace; font-size: 12px; }
            .fc-items-table td .small-text { width: 80px; }
            .fc-items-table .item-drag-handle, .group-drag-handle { cursor: move; color: #999; margin-right: 5px; }
            .fc-items-table .item-drag-handle:hover, .group-drag-handle:hover { color: #333; }
            .fc-items-table .remove-menu-item, .remove-group { color: #a00; border-color: #a00; background: #f9e4e4; }
            .fc-items-table .remove-menu-item:hover, .remove-group:hover { color: #fff; background: #a00; border-color: #a00; }
            .fc-header-bar { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; padding-bottom: 15px; border-bottom: 1px solid #ddd; }
            .fc-header-bar h2 { margin: 0; }
            .fc-item-row .field-group { margin-bottom: 10px; padding: 10px; border: 1px dashed #eee; background: #fafafa; }
            .fc-item-row .field-group label { display: block; font-weight: bold; margin-bottom: 5px; }
            .fc-item-row .link-type-fields, .fc-item-row .icon-type-fields, .fc-item-row .visibility-details { margin-top: 8px; }
            .select2-container { width: 100% !important; max-width: 100%; }
            .visibility-details { margin-top: 10px; padding: 10px; background: #f8f9fa; border: 1px solid #eee; display: none; }
            .visibility-details select { min-height: 80px; }
            .user-search-select .select2-selection__choice { background-color: #e4e4e4; }
            .fca-svg-upload-btn { margin-left: 5px; vertical-align: middle; }
        ');
    }

    public function ajax_search_users() {
        check_ajax_referer('fca_sidebar_menu_nonce', 'nonce');

        $search = isset($_GET['q']) ? sanitize_text_field($_GET['q']) : '';
        $results = array();

        if (strlen($search) >= 2) {
            $users = get_users(array(
                'search' => '*' . esc_attr($search) . '*',
                'search_columns' => array('user_login', 'user_email', 'display_name'),
                'number' => 20
            ));

            foreach ($users as $user) {
                $results[] = array(
                    'id' => $user->ID,
                    'text' => sprintf('%s (%s)', $user->display_name, $user->user_email)
                );
            }
        }
        wp_send_json($results);
    }

    public function render_admin_page() {
        // Load menu data from file instead of options
        $menu_data = $this->get_menu_data();

        // --- Fetch data HERE, just before rendering ---
        $spaces = $this->get_available_spaces();
        $courses = $this->get_available_courses();
        $pages = $this->get_available_pages();

        // --- Generate HTML <option> strings ---
        $this->spaces_options_html = $this->generate_options_html($spaces);
        $this->courses_options_html = $this->generate_options_html($courses);
        $this->pages_options_html = $this->generate_options_html($pages);
        
        // Pass the generated HTML options to JS
        wp_localize_script('fca-sidebar-menu', 'fcaSidebarMenuData', array(
            'spacesOptionsHtml' => $this->spaces_options_html,
            'coursesOptionsHtml' => $this->courses_options_html,
            'pagesOptionsHtml' => $this->pages_options_html
        ));

        $group_order = $menu_data['group_order'] ?? array();
        $group_keys = array_keys($menu_data['groups'] ?? array());
        if (count($group_order) !== count($group_keys) || array_diff($group_order, $group_keys) || array_diff($group_keys, $group_order)) {
             $group_order = $group_keys;
        }

        ?>
        <div class="wrap">
            <h1>Sidebar Menu Manager</h1>
            <p><?php _e('Configure custom menu groups and items for the FluentCommunity sidebar.', 'fca-widgets'); ?></p>
            
            <!-- Change to a div instead of a form since we're using AJAX saving only -->
            <div id="fca-sidebar-menu-form">
                <?php
                // Keep the nonce for AJAX security
                wp_nonce_field('fc_sidebar_menu-options');
                ?>
                
                <div class="fc-menu-config">
                    <div class="fc-header-bar">
                        <h2>Menu Groups</h2>
                        <button type="button" class="button button-primary" id="fca-add-group">
                            <span class="dashicons dashicons-plus-alt2"></span> Add New Group
                        </button>
                    </div>

                    <div id="fca-groups-container">
                        <?php 
                        if (!empty($menu_data['groups'])) {
                            $group_order = $menu_data['group_order'] ?? array_keys($menu_data['groups']);
                            // Ensure group order consistency check (as before)
                            $group_keys = array_keys($menu_data['groups']);
                             if (count($group_order) !== count($group_keys) || array_diff($group_order, $group_keys) || array_diff($group_keys, $group_order)) {
                                 $group_order = $group_keys;
                            }
                            
                            foreach ($group_order as $group_id) {
                                if (isset($menu_data['groups'][$group_id])) {
                                    $group = $menu_data['groups'][$group_id];
                                    $group_items = $menu_data['items'][$group_id] ?? array();
                                    // Pass fetched options HTML to render_group if needed, though render_item uses it directly now
                                    $this->render_group($group_id, $group, $group_items); 
                                }
                            }
                        }
                        ?>
                    </div>
                    <!-- Templates are now mostly handled by JS -->
                     <script type="text/html" id="tmpl-fca-group">
                         <?php $this->render_group('{{{ data.group_id }}}', array('title' => ''), array()); ?>
                     </script>
                     <script type="text/html" id="tmpl-fca-menu-item">
                         <?php // Pass empty item array, JS will populate options ?>
                         <?php $this->render_menu_item('{{{ data.group_id }}}', '{{{ data.item_id }}}', array()); ?>
                     </script>

                    <input type="hidden" id="fca-group-order" name="<?php echo esc_attr($this->option_name); ?>[group_order]"
                           value="<?php echo esc_attr(json_encode($group_order ?? array())); // Ensure group_order is defined ?>">

                    <hr style="margin: 20px 0;">
                    
                    <h3><?php _e('General Settings', 'fca-widgets'); ?></h3>
                    <table class="form-table">
                        <tbody>
                            <tr>
                                <th scope="row"><?php _e('Disable Default Items', 'fca-widgets'); ?></th>
                                <td>
                                    <label>
                                        <input type="checkbox" 
                                               id="fca-disable-default-items"
                                               name="<?php echo esc_attr($this->option_name); ?>[disable_default_items]" 
                                               value="1" 
                                               <?php checked($menu_data['disable_default_items'] ?? 0, 1); ?>>
                                        <?php _e('Hide the default Fluent Community sidebar items (Feed, Spaces, Courses, Members, etc.)', 'fca-widgets'); ?>
                                    </label>
                                    <p class="description">
                                        <?php _e('Enable this if you want ONLY your custom menu items (defined above) to appear in the sidebar.', 'fca-widgets'); ?>
                                    </p>
                                </td>
                            </tr>
                        </tbody>
                    </table>

                    <?php 
                    // Display a note about the save option
                    echo '<p class="description" style="margin-top: 20px; margin-bottom: 10px;"><strong>Note:</strong> Changes are saved automatically without page reload.</p>';
                    
                    // Show only the AJAX save button
                    echo '<div class="submit-buttons-row" style="display: flex; gap: 10px; align-items: center;">';
                    
                    // The AJAX save button
                    echo '<button type="button" class="button button-primary" id="save-via-ajax">Save Changes</button>';
                    
                    // Add the save status indicator
                    echo '<div id="fca-save-status"></div>';
                    
                    echo '</div>';
                    ?>
                </div>
            </div>
        </div>
        <?php
    }
    
    // Also fix the Course model path in get_available_courses
    private function get_available_courses() {
        try {
            // First try the same path we use in add_custom_sidebar_items for consistency
            if (class_exists('\FluentCommunity\Modules\Course\Model\Course')) {
                $courses = \FluentCommunity\Modules\Course\Model\Course::select(['id', 'title'])
                    ->where('status', 'published')
                    ->get();
                
                $formatted_courses = array();
                if (is_array($courses) || is_object($courses)) {
                    foreach ($courses as $course) {
                        if (is_object($course) && isset($course->id) && isset($course->title)) {
                            $formatted_courses[] = array(
                                'value' => $course->id,
                                'label' => $course->title
                            );
                        }
                    }
                }
                return $formatted_courses;
            }
            
            // Fallback to alternative path 
            if (class_exists('\FluentCommunity\Modules\Course\Framework\Model\Course')) {
                $courses = \FluentCommunity\Modules\Course\Framework\Model\Course::select(['id', 'title'])
                    ->where('status', 'published')
                    ->get();
                
                $formatted_courses = array();
                if (is_array($courses) || is_object($courses)) {
                    foreach ($courses as $course) {
                        if (is_object($course) && isset($course->id) && isset($course->title)) {
                            $formatted_courses[] = array(
                                'value' => $course->id,
                                'label' => $course->title
                            );
                        }
                    }
                }
                return $formatted_courses;
            }
            
            // Fallback to the utility method if it exists
            if (class_exists('\FluentCommunity\App\Functions\Utility')) {
                $courses = \FluentCommunity\App\Functions\Utility::getCourses(false);
                $formatted_courses = array();
                
                if (is_array($courses)) {
                    foreach ($courses as $course) {
                        if (is_object($course) && isset($course->id) && isset($course->title)) {
                            $formatted_courses[] = array(
                                'value' => $course->id,
                                'label' => $course->title
                            );
                        }
                    }
                }
                return $formatted_courses;
            }
        } catch (\Exception $e) {
            // error_log('Error getting courses: ' . $e->getMessage());
        }
        return array();
    }

    private function render_group($group_id_tpl, $group, $items) {
        // This function remains largely the same, but we ensure render_menu_item inside it
        // has access to the pre-generated options HTML stored in $this->..._options_html
        $group_id = esc_attr($group_id_tpl);
        $group_title = esc_attr($group['title'] ?? '');
        $visibility = $group['visibility'] ?? 'all';
        $selected_spaces_group = $group['spaces'] ?? array(); // Use different var name to avoid conflict
        $is_top_position = !empty($group['top_position']);
        $item_order = $group['item_order'] ?? (!empty($items) ? array_keys($items) : []);
        $is_enabled = $group['enabled'] ?? true; // Default to enabled

        ?>
        <div class="fc-group" data-group-id="<?php echo $group_id; ?>">
            <div class="fc-group-header">
                <span class="group-drag-handle dashicons dashicons-move" title="<?php esc_attr_e('Drag to reorder group', 'fca-widgets'); ?>"></span>
                <input type="text" 
                       name="<?php echo esc_attr($this->option_name); ?>[groups][<?php echo $group_id; ?>][title]"
                       value="<?php echo $group_title; ?>"
                       placeholder="Group Title"
                       class="regular-text group-title-input">
                
                <div class="visibility-wrapper">
                    <label class="screen-reader-text" for="group-vis-<?php echo $group_id; ?>">Group Visibility</label>
                    <select id="group-vis-<?php echo $group_id; ?>"
                            name="<?php echo esc_attr($this->option_name); ?>[groups][<?php echo $group_id; ?>][visibility]"
                            class="visibility-select">
                        <?php foreach ($this->visibility_options as $value => $label): ?>
                             <option value="<?php echo esc_attr($value); ?>" <?php selected($visibility, $value); ?>>
                                <?php echo esc_html($label); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>

                    <div class="visibility-details space-visibility-details"
                         style="<?php echo $visibility === 'space_members' ? 'display:block;' : 'display:none;'; ?>">
                        <label>Show only to members of:</label>
                        <select name="<?php echo esc_attr($this->option_name); ?>[groups][<?php echo $group_id; ?>][spaces][]"
                                class="space-select fca-select2"
                                multiple="multiple"
                                data-placeholder="Select Spaces...">
                            <?php 
                             // Use pre-generated HTML but re-select based on current group's data
                             echo $this->generate_options_html($this->get_available_spaces(), $selected_spaces_group); 
                            ?>
                        </select>
                    </div>
                </div>

                <div class="position-setting">
                    <label>
                        <input type="checkbox" 
                               name="<?php echo esc_attr($this->option_name); ?>[groups][<?php echo $group_id; ?>][top_position]"
                               value="1" 
                               <?php checked($is_top_position); ?>>
                        <?php _e('Show at top', 'fca-widgets'); ?>
                    </label>
                </div>

                <div class="enabled-setting">
                    <label>
                        <input type="checkbox" 
                               name="<?php echo esc_attr($this->option_name); ?>[groups][<?php echo $group_id; ?>][enabled]"
                               value="1" 
                               <?php checked($is_enabled); ?>>
                        <?php _e('Enabled', 'fca-widgets'); ?>
                    </label>
            </div>

                <button type="button" class="button remove-group" title="<?php esc_attr_e('Remove this group and all its items', 'fca-widgets'); ?>">
                    <span class="dashicons dashicons-trash"></span> Remove Group
                 </button>
            </div>

            <div class="fc-items-container">
                <table class="wp-list-table widefat fixed striped fc-items-table">
                <thead>
                    <tr>
                             <th class="column-drag" style="width: 30px;"></th>
                             <th class="column-enabled" style="width: 60px;"><?php _e('Enabled', 'fca-widgets'); ?></th>
                             <th class="column-title" style="width: 20%;"><?php _e('Title', 'fca-widgets'); ?></th>
                             <th class="column-link" style="width: 30%;"><?php _e('Link Type / Destination', 'fca-widgets'); ?></th>
                             <th class="column-icon" style="width: 20%;"><?php _e('Icon Type / Value', 'fca-widgets'); ?></th>
                             <th class="column-visibility" style="width: 20%;"><?php _e('Visibility', 'fca-widgets'); ?></th>
                             <th class="column-actions" style="width: 80px;"><?php _e('Actions', 'fca-widgets'); ?></th>
                    </tr>
                </thead>
                     <tbody class="fc-items-tbody">
                    <?php 
                    if (!empty($items)) {
                        foreach ($item_order as $item_id_tpl) {
                            if (isset($items[$item_id_tpl])) {
                                $this->render_menu_item($group_id_tpl, $item_id_tpl, $items[$item_id_tpl]);
                        }
                        }
                    } else {
                        echo '<tr class="no-items-row"><td colspan="6">' . __('No items in this group yet.', 'fca-widgets') . '</td></tr>';
                    }
                    ?>
                </tbody>
            </table>
                <input type="hidden" class="fca-item-order"
                       name="<?php echo esc_attr($this->option_name); ?>[groups][<?php echo $group_id; ?>][item_order]"
                       value="<?php echo esc_attr(json_encode($item_order)); ?>">
             </div>

            <button type="button" class="button add-menu-item" data-group-id="<?php echo $group_id; ?>">
                <span class="dashicons dashicons-plus-alt"></span> Add Menu Item
            </button>
        </div>
        <?php
    }

    /**
     * Check if FluentCommunity plugin is active
     */
    private function is_fluent_community_active() {
        if (!function_exists('is_plugin_active')) {
            include_once(ABSPATH . 'wp-admin/includes/plugin.php');
        }
        
        $plugin_path = 'fluent-community/fluent-community.php';
        $active = is_plugin_active($plugin_path);
        
        if (!$active) {
            // Log for debugging
            // error_log('FCA Widgets - FluentCommunity plugin is not active');
        }
        
        return $active;
    }

    /**
     * Debug FluentCommunity classes for troubleshooting
     */
    private function debug_fluent_community_classes() {
        // First check if the plugin is active
        $is_active = $this->is_fluent_community_active();
        
        $debug_info = array(
            'plugin_active' => $is_active,
            'spaces_classes' => array(
                '\FluentCommunity\App\Models\Space' => class_exists('\FluentCommunity\App\Models\Space'),
                '\FluentCommunity\App\Functions\Utility' => class_exists('\FluentCommunity\App\Functions\Utility')
            ),
            'courses_classes' => array(
                '\FluentCommunity\Modules\Course\Framework\Model\Course' => class_exists('\FluentCommunity\Modules\Course\Framework\Model\Course'),
                '\FluentCommunity\Modules\Course\Model\Course' => class_exists('\FluentCommunity\Modules\Course\Model\Course')
            )
        );
        
        // error_log('FCA Widgets Debug - FluentCommunity Classes: ' . print_r($debug_info, true));
        return $debug_info;
    }

    /**
     * Get available spaces for dropdown
     */
    private function get_available_spaces() {
        // Debug which classes are available
        $this->debug_fluent_community_classes();
        
        try {
            // Try the most likely path for spaces
            if (class_exists('\FluentCommunity\App\Models\Space')) {
                $spaces = \FluentCommunity\App\Models\Space::select(['id', 'title'])
                    ->where('status', 'published')
                    ->get();
                
                $formatted_spaces = array();
                if (is_array($spaces) || is_object($spaces)) {
                    foreach ($spaces as $space) {
                        if (is_object($space) && isset($space->id) && isset($space->title)) {
                            $formatted_spaces[] = array(
                                'value' => $space->id,
                                'label' => $space->title
                            );
                        }
                    }
                }
                return $formatted_spaces;
            }
            
            // Fallback to the utility method if it exists
            if (class_exists('\FluentCommunity\App\Functions\Utility')) {
                $spaces = \FluentCommunity\App\Functions\Utility::getSpaces(false);
                $formatted_spaces = array();
                
                if (is_array($spaces)) {
                    foreach ($spaces as $space) {
                        if (is_object($space) && isset($space->id) && isset($space->title)) {
                            $formatted_spaces[] = array(
                                'value' => $space->id,
                                'label' => $space->title
                            );
                        } elseif (is_object($space) && isset($space->slug) && isset($space->title)) {
                            $formatted_spaces[] = array(
                                'value' => $space->slug,
                                'label' => $space->title
                            );
                        }
                    }
                }
                return $formatted_spaces;
            }
        } catch (\Exception $e) {
            // error_log('Error getting spaces: ' . $e->getMessage());
        }
            return array();
        }

    /**
     * Get available FCA Custom Pages for dropdown
     */
    private function get_available_pages() {
        $formatted_pages = array();
        
        // Check if the CPT from fca-pages exists
        if (!post_type_exists('fca_page')) {
            // error_log('FCA Widgets: fca_page post type not found. Cannot load FCA Custom Pages.');
            return $formatted_pages; 
        }

        $pages = get_posts(array(
            'post_type' => 'fca_page', // Query the correct CPT
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
        ));

        if ($pages) {
            foreach ($pages as $page) {
                $formatted_pages[] = array(
                    'value' => $page->ID,         // Use ID as value
                    'label' => $page->post_title // Use title as label
                );
            }
        }
        return $formatted_pages;
    }

    private function get_available_roles_for_js() {
        $wp_roles = wp_roles();
        $roles = array();
        foreach ($wp_roles->get_names() as $role_id => $role_name) {
            $roles[] = array('id' => $role_id, 'text' => $role_name);
        }
        return $roles;
    }

    private function get_menu_item_template() {
        // This template generated in PHP is now less useful for dynamic options, 
        // but we still need the basic structure. JS will handle populating selects.
        ob_start();
        $this->render_menu_item('{{{ data.group_id }}}', '{{{ data.item_id }}}', array());
        return ob_get_clean();
    }

    private function render_menu_item($group_id_tpl, $item_id_tpl, $item) {
        // This function now uses the pre-generated HTML options for existing items
        $group_id = esc_attr($group_id_tpl);
        $item_id = esc_attr($item_id_tpl);
        $base_name = esc_attr($this->option_name . "[items][{$group_id}][{$item_id}]");

        $title = esc_attr($item['title'] ?? '');
        $link_type = $item['type'] ?? 'custom';
        // Use content_value directly
        $content_value = $item['content_value'] ?? ''; 
        $custom_url = esc_attr($item['custom_url'] ?? '');
        $icon_type = $item['icon_type'] ?? 'emoji';
        $icon_value = $item['icon_value'] ?? '';
        $visibility = $item['visibility'] ?? 'all';
        $selected_roles = $item['roles'] ?? array();
        $selected_users = $item['users'] ?? array();
        $selected_spaces = $item['spaces'] ?? array();
        $is_enabled = $item['enabled'] ?? true; // Default to enabled
        $open_in_new_tab = !empty($item['open_in_new_tab']); // Get new value

        // Generate options HTML with the selected content_value
        $current_spaces_options = $this->generate_options_html($this->get_available_spaces(), $content_value);
        $current_courses_options = $this->generate_options_html($this->get_available_courses(), $content_value);
        $current_pages_options = $this->generate_options_html($this->get_available_pages(), $content_value);
        $current_roles_options = $this->generate_roles_options_html($selected_roles);
        $current_users_options = $this->generate_selected_users_options_html($selected_users);
        // For visibility, we still compare against the saved array of space slugs
        $current_spaces_vis_options = $this->generate_options_html($this->get_available_spaces(), $selected_spaces);

        ?>
        <tr class="fc-item-row" data-item-id="<?php echo $item_id; ?>">
            <td class="column-drag">
                 <span class="item-drag-handle dashicons dashicons-move" title="<?php esc_attr_e('Drag to reorder item', 'fca-widgets'); ?>"></span>
            </td>
            <td class="column-enabled">
                <label class="screen-reader-text" for="enabled-<?php echo $item_id; ?>">Item Enabled</label>
                <input type="checkbox" id="enabled-<?php echo $item_id; ?>"
                       name="<?php echo $base_name; ?>[enabled]" 
                       value="1" 
                       <?php checked($is_enabled); ?>>
            </td>
             <td class="column-title">
                 <label class="screen-reader-text" for="title-<?php echo $item_id; ?>">Item Title</label>
                 <input type="text" id="title-<?php echo $item_id; ?>"
                        name="<?php echo $base_name; ?>[title]"
                        value="<?php echo $title; ?>"
                        placeholder="<?php esc_attr_e('Menu Item Title', 'fca-widgets'); ?>">
            </td>
             <td class="column-link">
                 <div class="field-group link-type-group">
                      <label for="link-type-<?php echo $item_id; ?>">Link Type</label>
                     <select id="link-type-<?php echo $item_id; ?>" name="<?php echo $base_name; ?>[type]" class="link-type-select">
                          <?php foreach ($this->link_types as $type_key => $type_label): ?>
                              <option value="<?php echo esc_attr($type_key); ?>" <?php selected($link_type, $type_key); ?>>
                                  <?php echo esc_html($type_label); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                  </div>

                <div class="link-type-fields">
                    <div class="field-group custom-url-field" style="<?php echo $link_type === 'custom' ? '' : 'display:none;'; ?>">
                        <label for="custom-url-<?php echo $item_id; ?>">Custom URL</label>
                        <input type="url" id="custom-url-<?php echo $item_id; ?>" name="<?php echo $base_name; ?>[custom_url]"
                               value="<?php echo $custom_url; ?>" placeholder="https://example.com">
                    </div>

                    <div class="field-group space-select-field" style="<?php echo $link_type === 'space' ? '' : 'display:none;'; ?>">
                         <label for="space-id-<?php echo $item_id; ?>">Select Space</label>
                        <select id="space-id-<?php echo $item_id; ?>" name="<?php echo $base_name; ?>[content_value]" 
                               class="content-value-select fca-select2" data-placeholder="Choose a Space...">
                             <?php echo $current_spaces_options; ?>
                        </select>
                    </div>

                    <div class="field-group course-select-field" style="<?php echo $link_type === 'course' ? '' : 'display:none;'; ?>">
                        <label for="course-id-<?php echo $item_id; ?>">Select Course</label>
                         <select id="course-id-<?php echo $item_id; ?>" name="<?php echo $base_name; ?>[content_value]" 
                               class="content-value-select fca-select2" data-placeholder="Choose a Course...">
                              <?php echo $current_courses_options; ?>
                          </select>
                      </div>

                    <div class="field-group page-select-field" style="<?php echo $link_type === 'page' ? '' : 'display:none;'; ?>">
                         <label for="page-id-<?php echo $item_id; ?>">Select FCA Custom Page</label>
                         <select id="page-id-<?php echo $item_id; ?>" name="<?php echo $base_name; ?>[content_value]" 
                               class="content-value-select fca-select2" data-placeholder="Choose an FCA Page...">
                             <?php echo $current_pages_options; // Use options generated from get_available_pages ?>
                        </select>
                    </div>
                </div>
                
                <!-- New Checkbox Field -->
                <div class="field-group open-new-tab-field">
                    <label>
                        <input type="checkbox" 
                               name="<?php echo $base_name; ?>[open_in_new_tab]" 
                               value="1" 
                               <?php checked($open_in_new_tab); ?>>
                        <?php _e('Open in New Tab', 'fca-widgets'); ?>
                    </label>
                </div>
            </td>
            <td class="column-icon">
                 <div class="field-group icon-type-group">
                     <label for="icon-type-<?php echo $item_id; ?>">Icon Type</label>
                    <select id="icon-type-<?php echo $item_id; ?>" name="<?php echo $base_name; ?>[icon_type]" class="icon-type-select">
                         <?php foreach ($this->icon_types as $type_key => $type_label): ?>
                             <option value="<?php echo esc_attr($type_key); ?>" <?php selected($icon_type, $type_key); ?>>
                                 <?php echo esc_html($type_label); ?>
                             </option>
                         <?php endforeach; ?>
                     </select>
                 </div>

                 <div class="icon-type-fields">
                     <div class="field-group emoji-field" style="<?php echo $icon_type === 'emoji' ? '' : 'display:none;'; ?>">
                         <label for="emoji-<?php echo $item_id; ?>">Emoji</label>
                         <input type="text" id="emoji-<?php echo $item_id; ?>" name="<?php echo $base_name; ?>[icon_value_emoji]"
                                value="<?php echo ($icon_type === 'emoji') ? esc_attr($icon_value) : ''; ?>" class="small-text icon-value-input">
                     </div>

                     <div class="field-group svg-url-field" style="<?php echo $icon_type === 'svg_url' ? '' : 'display:none;'; ?>">
                         <label for="svg-url-<?php echo $item_id; ?>">SVG URL</label>
                         <input type="url" id="svg-url-<?php echo $item_id; ?>" name="<?php echo $base_name; ?>[icon_value_svg_url]"
                                value="<?php echo ($icon_type === 'svg_url') ? esc_attr($icon_value) : ''; ?>" class="icon-value-input">
                         <button type="button" class="button button-secondary fca-svg-upload-btn">Select/Upload SVG</button>
                         <span class="svg-preview">
                              <?php if ($icon_type === 'svg_url' && !empty($icon_value)): ?>
                                  <img src="<?php echo esc_url($icon_value); ?>" alt="Preview" style="max-width: 24px; max-height: 24px; vertical-align: middle; margin-left: 5px;">
                              <?php endif; ?>
                          </span>
                      </div>

                     <div class="field-group svg-code-field" style="<?php echo $icon_type === 'svg_code' ? '' : 'display:none;'; ?>">
                         <label for="svg-code-<?php echo $item_id; ?>">SVG Code</label>
                         <textarea id="svg-code-<?php echo $item_id; ?>" name="<?php echo $base_name; ?>[icon_value_svg_code]"
                                   class="icon-value-input"
                                   rows="3"><?php echo ($icon_type === 'svg_code') ? esc_textarea($icon_value) : ''; ?></textarea>
                         <span class="svg-preview">
                              <?php if ($icon_type === 'svg_code' && !empty($icon_value)): ?>
                                  <span style="display: inline-block; width: 24px; height: 24px; vertical-align: middle; margin-left: 5px;"><?php echo $this->sanitize_svg($icon_value); ?></span>
                              <?php endif; ?>
                          </span>
                      </div>
                      <input type="hidden" name="<?php echo $base_name; ?>[icon_value]" class="icon-value-final" value="<?php echo esc_attr($icon_value); ?>">
                  </div>
            </td>
             <td class="column-visibility">
                 <div class="field-group visibility-group">
                      <label for="visibility-<?php echo $item_id; ?>">Visibility</label>
                     <select id="visibility-<?php echo $item_id; ?>" name="<?php echo $base_name; ?>[visibility]" class="visibility-select">
                        <?php foreach ($this->visibility_options as $value => $label): ?>
                              <option value="<?php echo esc_attr($value); ?>" <?php selected($visibility, $value); ?>>
                                  <?php echo esc_html($label); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                  </div>

                 <div class="visibility-details role-visibility-details" style="<?php echo $visibility === 'specific_roles' ? 'display:block;' : ''; ?>">
                        <label>Select Roles:</label>
                     <select name="<?php echo $base_name; ?>[roles][]" class="role-select fca-select2" multiple="multiple" data-placeholder="Select Roles...">
                          <?php echo $current_roles_options; ?>
                        </select>
                    </div>

                <div class="visibility-details user-visibility-details" style="<?php echo $visibility === 'specific_users' ? 'display:block;' : ''; ?>">
                        <label>Select Users:</label>
                    <select name="<?php echo $base_name; ?>[users][]" class="user-select fca-user-search-select" multiple="multiple" data-placeholder="Search for users..." data-minimum-input-length="2">
                         <?php echo $current_users_options; ?>
                        </select>
                    </div>

                 <div class="visibility-details space-visibility-details" style="<?php echo $visibility === 'space_members' ? 'display:block;' : ''; ?>">
                        <label>Select Spaces:</label>
                     <?php // Visibility uses slugs, so use the space slug options ?>
                     <select name="<?php echo $base_name; ?>[spaces][]" class="space-select fca-select2" multiple="multiple" data-placeholder="Select Spaces...">
                          <?php echo $current_spaces_vis_options; ?>
                        </select>
                </div>
            </td>
             <td class="column-actions">
                 <button type="button" class="button button-link-delete remove-menu-item" title="<?php esc_attr_e('Remove this item', 'fca-widgets'); ?>">
                      <span class="dashicons dashicons-trash"></span> Remove
                  </button>
            </td>
        </tr>
        <?php
    }

    public function sanitize_menu_data($input) {
        // Log the raw input to the sanitize function
        // error_log('FCA Widgets - Sanitize Start - Input: ' . print_r($input, true));
        
        if (!is_array($input)) {
            // error_log('FCA Widgets - Sanitize Error: Input is not an array.');
            return array();
        }

        $sanitized = array(
            'groups' => array(),
            'items' => array(),
            'group_order' => array(),
            'disable_default_items' => 0
        );

        // Sanitize Group Order
        if (isset($input['group_order']) && is_array($input['group_order'])) {
            $sanitized['group_order'] = array_map('sanitize_text_field', $input['group_order']);
        } else {
             // error_log('FCA Widgets - Sanitize Warning: group_order is missing or not an array.');
        }

        // Sanitize Disable Default Items setting
        if (isset($input['disable_default_items'])) {
            $sanitized['disable_default_items'] = absint($input['disable_default_items']);
        }

        // Sanitize Groups
        if (!empty($input['groups']) && is_array($input['groups'])) {
            foreach ($input['groups'] as $group_id => $group_data) {
                if (!is_array($group_data)) {
                     // error_log('FCA Widgets - Sanitize Warning: Group data is not an array for group_id: ' . $group_id);
                    continue;
                }
                $sanitized_group_id = sanitize_text_field($group_id);
                
                $sanitized['groups'][$sanitized_group_id] = array(
                    'title' => sanitize_text_field($group_data['title'] ?? ''),
                    'enabled' => !empty($group_data['enabled']) ? 1 : 0, // Sanitize enabled flag
                    'visibility' => isset($group_data['visibility']) && array_key_exists($group_data['visibility'], $this->visibility_options) ? $group_data['visibility'] : 'all',
                    'spaces' => isset($group_data['spaces']) && is_array($group_data['spaces']) ? array_map('sanitize_text_field', $group_data['spaces']) : array(), 
                    'top_position' => !empty($group_data['top_position']) ? 1 : 0,
                    'item_order' => array()
                );

                // Sanitize Item Order
                if (isset($group_data['item_order']) && is_array($group_data['item_order'])) {
                     $sanitized['groups'][$sanitized_group_id]['item_order'] = array_map('sanitize_text_field', $group_data['item_order']);
                } else {
                     // error_log('FCA Widgets - Sanitize Warning: item_order missing or not an array for group_id: ' . $group_id);
                }
            }
        } else {
             // error_log('FCA Widgets - Sanitize Warning: groups array is empty or not an array.');
        }

        // Sanitize Items - ADD DETAILED LOGGING HERE
        if (!empty($input['items']) && is_array($input['items'])) {
            // error_log('FCA Widgets - Sanitize: Processing items array...');
            foreach ($input['items'] as $group_id => $items_in_group) {
                $sanitized_group_id = sanitize_text_field($group_id);
                // error_log('FCA Widgets - Sanitize: Processing items for group: ' . $sanitized_group_id);
                
                // Ensure the group exists in the sanitized groups array
                if (!isset($sanitized['groups'][$sanitized_group_id])) {
                     // error_log('FCA Widgets - Sanitize Warning: Group ID ' . $sanitized_group_id . ' from items array does not exist in sanitized groups. Skipping items.');
                    continue;
                }

                if (is_array($items_in_group)) {
                    $sanitized['items'][$sanitized_group_id] = array();
                    // error_log('FCA Widgets - Sanitize: Found ' . count($items_in_group) . ' item(s) for group ' . $sanitized_group_id);
                
                    foreach ($items_in_group as $item_id => $item_data) {
                         // error_log('FCA Widgets - Sanitize: Processing item ID: ' . $item_id . ' for group ' . $sanitized_group_id);
                        if (!is_array($item_data)) {
                             // error_log('FCA Widgets - Sanitize Warning: Item data is not an array for item_id: ' . $item_id . ' in group ' . $sanitized_group_id);
                            continue;
                        }
                        $sanitized_item_id = sanitize_text_field($item_id);

                        // Get the link type
                        $item_type = isset($item_data['type']) && array_key_exists($item_data['type'], $this->link_types) ? $item_data['type'] : 'custom';
                        
                        // Get the content_value based on the link type
                        $content_value = '';
                        if ($item_type === 'space' || $item_type === 'course') {
                            $content_value = sanitize_text_field($item_data['content_value'] ?? '');
                        } elseif ($item_type === 'page') { // Handle page ID specifically
                             $content_value = absint($item_data['content_value'] ?? 0); // Ensure it is an integer ID
                        }

                        $sanitized_item = array(
                            'title' => sanitize_text_field($item_data['title'] ?? ''),
                            'enabled' => !empty($item_data['enabled']) ? 1 : 0, // Sanitize enabled flag
                            'type' => $item_type,
                            'custom_url' => isset($item_data['custom_url']) ? esc_url_raw($item_data['custom_url']) : '',
                            'content_value' => $content_value,
                            'icon_type' => isset($item_data['icon_type']) && array_key_exists($item_data['icon_type'], $this->icon_types) ? $item_data['icon_type'] : 'emoji',
                            'icon_value' => '', // Will be populated below
                            'visibility' => isset($item_data['visibility']) && array_key_exists($item_data['visibility'], $this->visibility_options) ? $item_data['visibility'] : 'all',
                            'roles' => isset($item_data['roles']) && is_array($item_data['roles']) ? array_map('sanitize_text_field', $item_data['roles']) : array(),
                            'users' => isset($item_data['users']) && is_array($item_data['users']) ? array_map('absint', $item_data['users']) : array(),
                            'spaces' => isset($item_data['spaces']) && is_array($item_data['spaces']) ? array_map('sanitize_text_field', $item_data['spaces']) : array(), 
                            'open_in_new_tab' => !empty($item_data['open_in_new_tab']) ? 1 : 0 // Sanitize new checkbox
                        );
                        
                        // Sanitize icon value
                        if (isset($item_data['icon_value'])) {
                            $icon_type_local = $sanitized_item['icon_type']; // Use local var for clarity
                            $raw_icon_value = $item_data['icon_value'];
                            if ($icon_type_local === 'emoji') {
                                $sanitized_item['icon_value'] = wp_kses($raw_icon_value, array());
                            } elseif ($icon_type_local === 'svg_url') {
                                $sanitized_item['icon_value'] = esc_url_raw($raw_icon_value);
                            } elseif ($icon_type_local === 'svg_code') {
                                $sanitized_item['icon_value'] = $this->sanitize_svg($raw_icon_value);
                            }
                        }

                        // Ensure custom_url is cleared if type is not custom
                        if ($item_type !== 'custom') {
                            $sanitized_item['custom_url'] = '';
                        }

                        $sanitized['items'][$sanitized_group_id][$sanitized_item_id] = $sanitized_item;
                        // error_log('FCA Widgets - Sanitize: Successfully processed item ID: ' . $sanitized_item_id);
                    }
                } else {
                     // error_log('FCA Widgets - Sanitize Warning: items_in_group is not an array for group_id: ' . $sanitized_group_id);
                }
            }
        } else {
             // error_log('FCA Widgets - Sanitize Warning: items array is empty or not an array in the input.');
        }

        // Final consistency checks for orders (remain the same)
        $valid_group_ids = array_keys($sanitized['groups']);
        $sanitized['group_order'] = array_intersect($sanitized['group_order'] ?? [], $valid_group_ids);

        foreach ($sanitized['groups'] as $group_id => &$group_data) {
            if (isset($sanitized['items'][$group_id])) {
                $valid_item_ids = array_keys($sanitized['items'][$group_id]);
                 $group_data['item_order'] = array_intersect($group_data['item_order'] ?? [], $valid_item_ids);
            } else {
                $group_data['item_order'] = [];
                 // error_log('FCA Widgets - Sanitize Consistency: Reset item_order for group ' . $group_id . ' because items array is missing or empty.');
            }
        }
        unset($group_data);
        
        // error_log('FCA Widgets - Sanitize End - Output: ' . print_r($sanitized, true));
        return $sanitized;
    }

    private function sanitize_svg($svg_code) {
        if (empty($svg_code) || !is_string($svg_code)) {
            return '';
        }

        $svg_code = preg_replace('/<!--.*?-->/s', '', $svg_code);
        $svg_code = preg_replace('/<\?.*?\?>/s', '', $svg_code);
        $svg_code = preg_replace('/<!DOCTYPE.*?>/is', '', $svg_code);
        $svg_code = preg_replace('/<script.*?>.*?<\/script>/is', '', $svg_code);

        $allowed_tags = array(
            'svg' => array('width' => true, 'height' => true, 'viewbox' => true, 'xmlns' => true, 'fill' => true, 'stroke' => true, 'class' => true, 'style' => true, 'aria-hidden' => true, 'role' => true),
            'path' => array('d' => true, 'fill' => true, 'stroke' => true, 'stroke-width' => true, 'class' => true, 'style' => true),
            'circle' => array('cx' => true, 'cy' => true, 'r' => true, 'fill' => true, 'stroke' => true, 'stroke-width' => true, 'class' => true, 'style' => true),
            'rect' => array('x' => true, 'y' => true, 'width' => true, 'height' => true, 'fill' => true, 'stroke' => true, 'stroke-width' => true, 'class' => true, 'style' => true, 'rx' => true, 'ry' => true),
            'ellipse' => array('cx' => true, 'cy' => true, 'rx' => true, 'ry' => true, 'fill' => true, 'stroke' => true, 'stroke-width' => true, 'class' => true, 'style' => true),
            'line' => array('x1' => true, 'y1' => true, 'x2' => true, 'y2' => true, 'stroke' => true, 'stroke-width' => true, 'class' => true, 'style' => true),
            'polyline' => array('points' => true, 'fill' => true, 'stroke' => true, 'stroke-width' => true, 'class' => true, 'style' => true),
            'polygon' => array('points' => true, 'fill' => true, 'stroke' => true, 'stroke-width' => true, 'class' => true, 'style' => true),
            'g' => array('fill' => true, 'stroke' => true, 'class' => true, 'style' => true, 'transform' => true),
            'title' => array(),
            'desc' => array(),
            'defs' => array(),
            'symbol' => array('id' => true, 'viewbox' => true),
            'use' => array('xlink:href' => true, 'href' => true, 'transform' => true, 'width' => true, 'height' => true, 'x' => true, 'y' => true) 
        );

        return wp_kses($svg_code, $allowed_tags);
    }

    // Helper function to generate <option> HTML
    // Now handles both objects (id, title) and arrays (value, label)
    private function generate_options_html($items, $selected_value = null) {
        $html = '<option value=""></option>'; // Start with empty option
        if (empty($items)) return $html;

        foreach ($items as $item) {
            $value = '';
            $label = '';

            if (is_object($item) && isset($item->id) && isset($item->title)) {
                $value = $item->id;
                $label = $item->title;
            } elseif (is_array($item) && isset($item['value']) && isset($item['label'])) {
                $value = $item['value'];
                $label = $item['label'];
            }

            if (!empty($value) && !empty($label)) {
                 // Handle comparison for selected, allowing for arrays of selected values
                 $is_selected = is_array($selected_value) ? in_array($value, $selected_value) : ($value == $selected_value);
                 $html .= sprintf('<option value="%s" %s>%s</option>',
                     esc_attr($value),
                     selected($is_selected, true, false),
                     esc_html($label)
                 );
             }
        }
        return $html;
    }
    
    // Helper function to generate <option> HTML for roles
    private function generate_roles_options_html($selected_roles = []) {
         $html = '';
         $roles = get_editable_roles();
         foreach ($roles as $role_id => $role) {
             $html .= sprintf('<option value="%s" %s>%s</option>',
                 esc_attr($role_id),
                 in_array($role_id, (array)$selected_roles) ? 'selected' : '',
                 esc_html($role['name'])
             );
         }
         return $html;
    }
    
    // Helper function to generate <option> HTML for selected users (used by render_menu_item)
    private function generate_selected_users_options_html($selected_users = []) {
        $html = '';
         if (!empty($selected_users)) {
             foreach ($selected_users as $user_id) {
                                    $user = get_user_by('id', $user_id);
                                    if ($user) {
                     $html .= sprintf('<option value="%d" selected>%s (%s)</option>',
                                            $user_id,
                                            esc_html($user->display_name),
                                            esc_html($user->user_email)
                                        );
                                    }
                                }
                            }
        return $html;
    }

    /**
     * Display admin notices for missing dependencies
     */
    public function check_plugin_dependencies() {
        // Only show on our plugin's admin page
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'fca-widgets') === false) {
            return;
        }
        
        if (!$this->is_fluent_community_active()) {
            ?>
            <div class="notice notice-error">
                <p>
                    <strong>FCA Widgets:</strong> 
                    The FluentCommunity plugin is not active. Some features like Spaces and Courses dropdown selectors will not work.
                </p>
                    </div>
                            <?php 
        } else {
            // Check for specific modules/classes
            $debug_info = $this->debug_fluent_community_classes();
            $missing_classes = array();
            
            if (!$debug_info['spaces_classes']['\FluentCommunity\App\Models\Space'] && 
                !$debug_info['spaces_classes']['\FluentCommunity\App\Functions\Utility']) {
                $missing_classes[] = 'Spaces';
            }
            
            if (!$debug_info['courses_classes']['\FluentCommunity\Modules\Course\Framework\Model\Course'] && 
                !$debug_info['courses_classes']['\FluentCommunity\Modules\Course\Model\Course']) {
                $missing_classes[] = 'Courses';
            }
            
            if (!empty($missing_classes)) {
                ?>
                <div class="notice notice-warning">
                    <p>
                        <strong>FCA Widgets:</strong> 
                        Some FluentCommunity modules might not be available: <?php echo implode(', ', $missing_classes); ?>.
                        This may affect dropdown selectors for these items.
                    </p>
                    </div>
        <?php
            }
        }
    }

    /**
     * AJAX handler for getting spaces
     */
    public function ajax_get_spaces() {
        check_ajax_referer('fca_widgets_nonce', 'security');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $spaces = $this->get_available_spaces();
        wp_send_json_success($spaces);
    }

    /**
     * AJAX handler for getting courses
     */
    public function ajax_get_courses() {
        check_ajax_referer('fca_widgets_nonce', 'security');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $courses = $this->get_available_courses();
        wp_send_json_success($courses);
    }

    /**
     * AJAX handler for getting FCA Custom Pages
     */
    public function ajax_get_pages() {
        check_ajax_referer('fca_widgets_nonce', 'security');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        // Use the updated method to get FCA Pages
        $pages = $this->get_available_pages();
        wp_send_json_success($pages);
    }

    /**
     * AJAX handler for saving the sidebar menu
     */
    public function ajax_save_sidebar_menu() {
        check_ajax_referer('fca_sidebar_menu_nonce', 'security');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        // Get the form data
        $menu_data = isset($_POST['menu_data']) ? $_POST['menu_data'] : '';
        
        if (empty($menu_data)) {
            wp_send_json_error('No data provided');
        }
        
        // Log the raw data
        // error_log('FCA Widgets - AJAX Save - Raw data: ' . $menu_data);
        
        // Decode the JSON data
        $menu_data = json_decode(stripslashes($menu_data), true);
        
        if (!is_array($menu_data)) {
            wp_send_json_error('Invalid data format');
        }
        
        // Log the decoded data structure
        // error_log('FCA Widgets - AJAX Save - Decoded data: ' . print_r($menu_data, true));
        
        // Save to file
        $result = $this->save_menu_data($menu_data);
        
        if ($result) {
            wp_send_json_success('Menu saved successfully');
        } else {
            wp_send_json_error('Error saving menu');
        }
    }
} 
