<?php
/**
 * FCA Widgets Migration Manager
 * 
 * Handles migration of data from old plugin versions to the current version.
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class FCA_MigrationManager {
    
    private $migration_notices = array();
    
    public function __construct() {
        // Listen for version update action
        add_action('fca_widgets_version_updated', array($this, 'handle_version_update'), 10, 2);
        
        // Register admin notices for migrations
        add_action('admin_notices', array($this, 'display_migration_notices'));
        
        // Add migration information page
        add_action('admin_menu', array($this, 'add_migration_info_page'), 99); // Low priority to add after main menu
    }
    
    /**
     * Add a migration information page to the admin menu
     */
    public function add_migration_info_page() {
        // Only show if we have recent migration data
        if (get_option('fca_widgets_recent_migration', false)) {
            add_submenu_page(
                'fca-widgets',
                'Migration Information',
                'Migration Info',
                'manage_options',
                'fca-widgets-migration',
                array($this, 'render_migration_info_page')
            );
        }
    }
    
    /**
     * Render the migration information page
     */
    public function render_migration_info_page() {
        $migration_data = get_option('fca_widgets_recent_migration', array());
        ?>
        <div class="wrap">
            <h1>FCA Widgets Migration Information</h1>
            
            <div class="notice notice-info">
                <p>This page provides information about the most recent migration of your FCA Widgets data. This information is intended to help you understand what changes were made during the upgrade process.</p>
            </div>
            
            <?php if (!empty($migration_data)) : ?>
                <div class="card">
                    <h2>Migration Details</h2>
                    <table class="widefat striped">
                        <tr>
                            <th>Previous Version</th>
                            <td><?php echo esc_html($migration_data['old_version']); ?></td>
                        </tr>
                        <tr>
                            <th>Current Version</th>
                            <td><?php echo esc_html($migration_data['new_version']); ?></td>
                        </tr>
                        <tr>
                            <th>Update Date</th>
                            <td><?php echo esc_html(date('F j, Y, g:i a', $migration_data['timestamp'])); ?></td>
                        </tr>
                        <tr>
                            <th>Update Type</th>
                            <td>
                                <?php 
                                $is_pre_105_upgrade = version_compare($migration_data['old_version'], '1.0.5', '<');
                                echo $is_pre_105_upgrade ? 
                                    '<span style="color: #e3a619;">Pre-1.0.5 Migration (Format Change)</span>' : 
                                    '<span style="color: #2271b1;">Post-1.0.5 Update (Data Preservation)</span>';
                                ?>
                            </td>
                        </tr>
                        <tr>
                            <th>Data Preservation Type</th>
                            <td>
                                <?php 
                                $preservation_type = isset($migration_data['preservation_type']) ? 
                                    $migration_data['preservation_type'] : 'unknown';
                                    
                                switch ($preservation_type) {
                                    case 'old_format_migration':
                                        echo '<span style="color: green;">✓ Migrated from old format to new format</span>';
                                        break;
                                    case 'modern_data_preserved':
                                        echo '<span style="color: green;">✓ Existing configuration preserved intact</span>';
                                        break;
                                    case 'data_verified':
                                        echo '<span style="color: green;">✓ Configuration verified across data sources</span>';
                                        break;
                                    case 'none_found':
                                        echo '<span style="color: #e05260;">❌ No valid configuration found to preserve</span>';
                                        break;
                                    default:
                                        echo '<span style="color: gray;">Unknown preservation type</span>';
                                }
                                ?>
                            </td>
                        </tr>
                        <tr>
                            <th>Automatic Backup</th>
                            <td>
                                <?php
                                if (isset($migration_data['backup_created']) && $migration_data['backup_created']) {
                                    $backup_option_name = $migration_data['backup_option_name'] ?? 'unknown';
                                    echo '<span style="color: green;">✓ Backup created in database (option: ' . esc_html($backup_option_name) . ')</span>';
                                } else {
                                    echo '<span style="color: gray;">No backup needed (no data found)</span>';
                                }
                                ?>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div class="card" style="margin-top: 20px;">
                    <h2>Available Backups</h2>
                    <?php
                    global $wpdb;
                    $backup_options = $wpdb->get_results("SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE 'fca_sidebar_menu_items_backup_%'");
                    
                    if (!empty($backup_options)) {
                        echo '<p>The following backup versions of your sidebar menu configuration were found in the database:</p>';
                        echo '<ul style="margin-left: 20px;">';
                        foreach ($backup_options as $option) {
                            $backup_data = get_option($option->option_name);
                            $timestamp = isset($backup_data['_backup_timestamp']) ? 
                                ' (created on ' . date('F j, Y, g:i a', $backup_data['_backup_timestamp']) . ')' : '';
                            $source = isset($backup_data['_backup_source']) ? ' - ' . $backup_data['_backup_source'] : '';
                            echo '<li><code>' . esc_html($option->option_name) . '</code>' . esc_html($timestamp) . esc_html($source) . '</li>';
                        }
                        echo '</ul>';
                        echo '<p>If you need to restore a backup, contact your developer to manually retrieve these values from the WordPress database.</p>';
                    } else {
                        echo '<p>No backups of your sidebar menu configuration were found in the database.</p>';
                    }
                    ?>
                </div>
                
                <div class="card" style="margin-top: 20px;">
                    <h2>What Changed?</h2>
                    <p>In version 1.0.5+, we made the following improvements to how sidebar menu data is stored:</p>
                    <ul style="list-style-type: disc; margin-left: 20px;">
                        <li>Data is now stored in the WordPress options table instead of a JSON file, making it more reliable and compatible with managed hosting environments.</li>
                        <li>The data structure was enhanced to support more features like group ordering, visibility controls, and icon options.</li>
                    </ul>
                    <p>Your existing sidebar menu items were automatically preserved during the update process.</p>
                </div>
                
                <div class="card" style="margin-top: 20px;">
                    <h2>Data Protection Features</h2>
                    <p>To ensure your sidebar menu configuration is preserved across updates, we've implemented:</p>
                    <ul style="list-style-type: disc; margin-left: 20px;">
                        <li><strong>Automatic Data Verification:</strong> On every update, we verify your menu configuration data is preserved.</li>
                        <li><strong>Multiple Data Source Checking:</strong> If no data is found in the primary location, we check other possible storage locations.</li>
                        <li><strong>Configuration Backups:</strong> A backup of your configuration is created with each version update.</li>
                    </ul>
                </div>
                
                <div class="card" style="margin-top: 20px;">
                    <h2>Troubleshooting</h2>
                    <p>If you're having issues with your sidebar menu items after the migration:</p>
                    <ol style="list-style-type: decimal; margin-left: 20px;">
                        <li>Go to the <a href="<?php echo admin_url('admin.php?page=fca-sidebar-menu'); ?>">Sidebar Menu</a> page and verify your items are there.</li>
                        <li>If items are missing, try refreshing the page or logging out and back in.</li>
                        <li>If your menu items are still missing, contact support with the information from this page.</li>
                        <li>For manual data recovery, check the database for option named <code>fca_sidebar_menu_items_backup_X.X.X</code> where X.X.X is a version number.</li>
                    </ol>
                </div>
            <?php else : ?>
                <div class="notice notice-warning">
                    <p>No recent migration data found. This page is most useful immediately after a plugin update.</p>
                </div>
            <?php endif; ?>
        </div>
        <style>
            .card {
                background: #fff;
                border: 1px solid #ccd0d4;
                border-radius: 4px;
                padding: 20px;
                margin-top: 10px;
                box-shadow: 0 1px 1px rgba(0,0,0,.04);
            }
            .card h2 {
                margin-top: 0;
                font-size: 1.3em;
                padding-bottom: 10px;
                border-bottom: 1px solid #eee;
            }
            code {
                background: #f0f0f1;
                padding: 2px 5px;
                border-radius: 3px;
            }
        </style>
        <?php
    }
    
    /**
     * Handle version updates and run appropriate migrations
     */
    public function handle_version_update($old_version, $new_version) {
        // Log the migration attempt
        error_log("FCA Widgets: Migrating from version {$old_version} to {$new_version}");
        
        // Store migration info to display later
        $this->migration_notices['version_update'] = array(
            'old_version' => $old_version,
            'new_version' => $new_version,
            'status' => 'started'
        );
        
        // Setup migration data
        $migration_data = array(
            'old_version' => $old_version,
            'new_version' => $new_version,
            'timestamp' => time(),
            'sidebar_menu_migrated' => false,
            'preservation_type' => 'none', // Will track what kind of data preservation was performed
        );
        
        // Determine what kind of update we're doing
        $is_pre_105_upgrade = version_compare($old_version, '1.0.5', '<');
        $is_post_105_upgrade = !$is_pre_105_upgrade;

        // First, always make a backup of the current configuration regardless of version
        $current_config = get_option('fca_sidebar_menu_items', array());
        if (!empty($current_config) && is_array($current_config)) {
            $backup_copy = $current_config;
            $backup_copy['_backup_timestamp'] = time();
            $backup_copy['_backup_source'] = 'automatic_backup';
            $backup_copy['_from_version'] = $old_version;
            
            // Store with version info in the name
            $backup_option_name = 'fca_sidebar_menu_items_backup_from_' . $old_version . '_to_' . $new_version;
            update_option($backup_option_name, $backup_copy);
            error_log("FCA Widgets: Created pre-update backup as {$backup_option_name}");
            
            $migration_data['backup_created'] = true;
            $migration_data['backup_option_name'] = $backup_option_name;
        }
        
        // Now handle different upgrade scenarios
        if ($is_pre_105_upgrade) {
            // This is an older version upgrade - need to migrate from old format
            $migrated = $this->migrate_sidebar_menu_data();
            $migration_data['sidebar_menu_migrated'] = $migrated;
            
            if ($migrated) {
                $migration_data['preservation_type'] = 'old_format_migration';
                $this->migration_notices['sidebar_menu'] = array(
                    'message' => 'Successfully migrated sidebar menu data from old format to new format.',
                    'type' => 'success'
                );
            }
        } else {
            // This is a 1.0.5+ upgrade - need to PRESERVE existing data, not migrate
            // Check if the current configuration is valid
            $has_valid_config = !empty($current_config) && isset($current_config['groups']) && is_array($current_config['groups']);
            
            if ($has_valid_config) {
                // Already have valid data, just mark as preserved
                $migration_data['preservation_type'] = 'modern_data_preserved';
                $this->migration_notices['data_preserved'] = array(
                    'message' => 'Successfully preserved your existing menu configuration.',
                    'type' => 'success'
                );
                error_log("FCA Widgets: Preserved existing valid configuration (version {$old_version})");
            } else {
                // No valid configuration found from 1.0.5+, try data verification
                $verified = $this->verify_data_integrity();
                
                if ($verified) {
                    $migration_data['preservation_type'] = 'data_verified';
                    $this->migration_notices['data_integrity'] = array(
                        'message' => 'Successfully verified and preserved your configuration data.',
                        'type' => 'success'
                    );
                } else {
                    // Could not find/verify any data
                    $migration_data['preservation_type'] = 'none_found';
                    $this->migration_notices['data_missing'] = array(
                        'message' => 'No valid menu configuration was found to preserve.',
                        'type' => 'warning'
                    );
                    error_log("FCA Widgets: No valid menu configuration found to preserve");
                }
            }
        }
        
        // Save detailed migration data for the info page
        update_option('fca_widgets_recent_migration', $migration_data);
        
        // Update migration status
        $this->migration_notices['version_update']['status'] = 'completed';
        
        // Store notices in a transient to display on next page load
        set_transient('fca_widgets_migration_notices', $this->migration_notices, 60 * 60 * 24); // Expire after 1 day
    }
    
    /**
     * Verify data integrity across any version update
     * This ensures data is preserved even when updating from 1.0.5 to newer versions
     * 
     * @return bool True if data was verified/preserved, false otherwise
     */
    private function verify_data_integrity() {
        $verified = false;
        
        // CRITICAL: First check if we have valid data in the current storage format
        $current_menu_data = get_option('fca_sidebar_menu_items', array());
        
        // Create a backup of the current configuration FIRST before any potential changes
        if (!empty($current_menu_data) && is_array($current_menu_data)) {
            // Add timestamp to backup
            $backup_data = $current_menu_data;
            $backup_data['_backup_timestamp'] = time();
            $backup_data['_backup_source'] = 'pre_migration';
            
            // Store backup with version-specific name
            update_option('fca_sidebar_menu_items_backup_' . FCA_WIDGETS_VERSION, $backup_data);
            error_log('FCA Widgets: Created backup of sidebar menu configuration before any migration');
        }
        
        // Check if current_menu_data is already valid - most important when upgrading from 1.0.5+
        if (!empty($current_menu_data) && is_array($current_menu_data) && !empty($current_menu_data['groups'])) {
            // Data already exists in the correct option and format - we should preserve this!
            $verified = true;
            error_log('FCA Widgets: Valid sidebar menu data already exists in correct format. Preserving existing data.');
            return $verified; // IMPORTANT: Return early to prevent overwriting valid data
        }
        
        // Only proceed with checking alternative data sources if we DON'T have valid data already
        
        // Check known alternative data sources
        $sidebar_menu_option = get_option('fca_sidebar_menu', array());
        $sidebar_items_option = get_option('fca_sidebar_items', array());
        
        // Check if data exists in JSON file
        $data_file_path = FCA_WIDGETS_PLUGIN_DIR . 'data/sidebar-menu-data.json';
        $file_data = array();
        if (file_exists($data_file_path)) {
            $json_data = file_get_contents($data_file_path);
            if (!empty($json_data)) {
                $file_data = json_decode($json_data, true);
            }
        }
        
        // Try to find data in any of the potential sources - only if current format is empty
        if (!empty($sidebar_menu_option) && is_array($sidebar_menu_option)) {
            // Old sidebar_menu data exists, convert and save
            $converted_data = $this->convert_old_menu_format($sidebar_menu_option);
            update_option('fca_sidebar_menu_items', $converted_data);
            $verified = true;
            error_log('FCA Widgets: Data restored from fca_sidebar_menu option');
        } 
        else if (!empty($sidebar_items_option) && is_array($sidebar_items_option)) {
            // Old sidebar_items data exists, convert and save
            $converted_data = $this->convert_old_menu_format($sidebar_items_option);
            update_option('fca_sidebar_menu_items', $converted_data);
            $verified = true;
            error_log('FCA Widgets: Data restored from fca_sidebar_items option');
        }
        else if (!empty($file_data) && is_array($file_data)) {
            // JSON file data exists, save it
            $defaults = array(
                'groups' => array(),
                'items' => array(),
                'group_order' => array(),
                'disable_default_items' => 0
            );
            $data_to_save = wp_parse_args($file_data, $defaults);
            update_option('fca_sidebar_menu_items', $data_to_save);
            $verified = true;
            error_log('FCA Widgets: Data restored from JSON file');
        }
        
        return $verified;
    }
    
    /**
     * Display admin notices for migrations
     */
    public function display_migration_notices() {
        // Check if we have any migration notices to display
        $notices = get_transient('fca_widgets_migration_notices');
        
        if ($notices) {
            // Clear the transient so notices show only once
            delete_transient('fca_widgets_migration_notices');
            
            // Display the main update notice
            if (isset($notices['version_update']) && $notices['version_update']['status'] === 'completed') {
                $old_version = $notices['version_update']['old_version'];
                $new_version = $notices['version_update']['new_version'];
                
                echo '<div class="notice notice-success is-dismissible">';
                echo '<p><strong>FCA Widgets:</strong> Successfully updated from version ' . esc_html($old_version) . ' to ' . esc_html($new_version) . '.</p>';
                
                // Display migration-specific information based on version
                $is_pre_105_upgrade = version_compare($old_version, '1.0.5', '<');
                
                if ($is_pre_105_upgrade) {
                    // Show migration message for pre-1.0.5 upgrades
                    if (isset($notices['sidebar_menu'])) {
                        echo '<p>' . esc_html($notices['sidebar_menu']['message']) . ' <a href="' . admin_url('admin.php?page=fca-widgets-migration') . '">View migration details</a></p>';
                    }
                } else {
                    // Show preservation message for 1.0.5+ upgrades
                    if (isset($notices['data_preserved'])) {
                        echo '<p>' . esc_html($notices['data_preserved']['message']) . ' <a href="' . admin_url('admin.php?page=fca-sidebar-menu') . '">View sidebar menu</a></p>';
                    } elseif (isset($notices['data_integrity'])) {
                        echo '<p>' . esc_html($notices['data_integrity']['message']) . ' <a href="' . admin_url('admin.php?page=fca-widgets-migration') . '">View details</a></p>';
                    } elseif (isset($notices['data_missing'])) {
                        echo '<p><span style="color: #e05260;"><strong>Warning:</strong></span> ' . esc_html($notices['data_missing']['message']) . ' If you previously had menu items, please <a href="' . admin_url('admin.php?page=fca-widgets-migration') . '">check the migration details</a>.</p>';
                    }
                }
                
                // Always show when a file migration happened
                if (isset($notices['file_migration'])) {
                    echo '<p>' . esc_html($notices['file_migration']['message']) . '</p>';
                }
                
                echo '</div>';
            }
        }
    }
    
    /**
     * Migrates sidebar menu data from old format to new format
     * 
     * In older versions (<1.0.5), sidebar menu items were stored in a different option name
     * or with a different data structure.
     * 
     * @return bool True if migration was performed, false otherwise
     */
    private function migrate_sidebar_menu_data() {
        $migrated = false;
        
        // Check for old sidebar menu data (pre 1.0.5)
        $old_menu_data = get_option('fca_sidebar_menu', false);
        
        // If no old data found, check for alternative old storage methods
        if ($old_menu_data === false) {
            $old_menu_data = get_option('fca_sidebar_items', false);
        }
        
        // If we found old data in any location, migrate it
        if ($old_menu_data !== false) {
            error_log("FCA Widgets: Found old sidebar menu data to migrate");
            
            // Get the current menu data (if any)
            $current_menu_data = get_option('fca_sidebar_menu_items', array());
            
            // If current data is empty, it's safe to migrate the old data
            if (empty($current_menu_data) || !is_array($current_menu_data)) {
                // Format conversion may be needed here depending on the old structure
                $new_menu_data = $this->convert_old_menu_format($old_menu_data);
                
                // Save the converted data
                $updated = update_option('fca_sidebar_menu_items', $new_menu_data);
                
                if ($updated) {
                    error_log("FCA Widgets: Successfully migrated sidebar menu data");
                    $migrated = true;
                    
                    // Optionally rename the old option to prevent duplicate migration
                    update_option('fca_sidebar_menu_migrated', $old_menu_data);
                    delete_option('fca_sidebar_menu');
                    delete_option('fca_sidebar_items');
                } else {
                    error_log("FCA Widgets: Failed to save migrated sidebar menu data");
                }
            } else {
                error_log("FCA Widgets: Skipping migration as current menu data exists");
            }
        } else {
            error_log("FCA Widgets: No old sidebar menu data found to migrate");
        }
        
        // Check for old data in the file system (similar to what class-sidebar-menu-manager.php does)
        $file_migrated = $this->migrate_from_data_file();
        
        // Return true if either migration was successful
        return $migrated || $file_migrated;
    }
    
    /**
     * Convert old menu format to the new format
     * 
     * The exact conversion logic depends on the structure of the old data
     */
    private function convert_old_menu_format($old_data) {
        // Default structure for the new format
        $new_data = array(
            'groups' => array(),
            'items' => array(),
            'group_order' => array(),
            'disable_default_items' => 0
        );
        
        // If old data is already in the new format, return it directly
        if (isset($old_data['groups']) && isset($old_data['items'])) {
            return $old_data;
        }
        
        // Conversion for older formats
        // This will vary based on how the old data was structured
        if (is_array($old_data)) {
            // Try to detect the structure of the old data
            
            // Check if it's a flat array of items with no groups
            if (isset($old_data[0]) && is_array($old_data[0]) && isset($old_data[0]['title'])) {
                // Create a default group
                $group_id = 'group_' . time();
                $new_data['groups'][$group_id] = array(
                    'title' => 'Migrated Menu Items',
                    'enabled' => 1,
                    'visibility' => 'all',
                    'spaces' => array(),
                    'top_position' => 0,
                    'item_order' => array()
                );
                
                $new_data['items'][$group_id] = array();
                
                // Convert each item
                foreach ($old_data as $index => $item) {
                    $item_id = 'item_' . time() . '_' . $index;
                    
                    $new_data['items'][$group_id][$item_id] = array(
                        'title' => isset($item['title']) ? $item['title'] : 'Menu Item',
                        'enabled' => 1,
                        'type' => isset($item['type']) ? $item['type'] : 'custom',
                        'custom_url' => isset($item['url']) ? $item['url'] : '',
                        'content_value' => isset($item['content_id']) ? $item['content_id'] : '',
                        'icon_type' => isset($item['icon_type']) ? $item['icon_type'] : 'emoji',
                        'icon_value' => isset($item['icon']) ? $item['icon'] : '',
                        'visibility' => isset($item['visibility']) ? $item['visibility'] : 'all',
                        'roles' => isset($item['roles']) ? $item['roles'] : array(),
                        'users' => isset($item['users']) ? $item['users'] : array(),
                        'spaces' => isset($item['spaces']) ? $item['spaces'] : array(),
                        'open_in_new_tab' => isset($item['new_tab']) ? $item['new_tab'] : 0
                    );
                    
                    $new_data['groups'][$group_id]['item_order'][] = $item_id;
                }
                
                $new_data['group_order'][] = $group_id;
            }
            // Add more conversion logic for other potential old formats
        }
        
        return $new_data;
    }
    
    /**
     * Migrate data from the old JSON file to options table
     * This is a copy of logic from class-sidebar-menu-manager.php for completeness
     * 
     * @return bool True if migration was performed, false otherwise
     */
    private function migrate_from_data_file() {
        $migrated = false;
        $old_data_file_path = FCA_WIDGETS_PLUGIN_DIR . 'data/sidebar-menu-data.json';
        
        if (file_exists($old_data_file_path)) {
            error_log('FCA Widgets: Old data file found. Attempting migration from: ' . $old_data_file_path);
            $json_data = file_get_contents($old_data_file_path);
            if (!empty($json_data)) {
                $file_menu_data = json_decode($json_data, true);
                if ($file_menu_data && is_array($file_menu_data)) {
                    // Data is valid, save it to options
                    $defaults = array(
                        'groups' => array(),
                        'items' => array(),
                        'group_order' => array(),
                        'disable_default_items' => 0
                    );
                    
                    $data_to_migrate = wp_parse_args($file_menu_data, $defaults);
                    $migrated = update_option('fca_sidebar_menu_items', $data_to_migrate);
                    
                    if ($migrated) {
                        error_log('FCA Widgets: Successfully migrated menu data from file to options.');
                        
                        // Add to the migration notices
                        $this->migration_notices['file_migration'] = array(
                            'message' => 'Successfully migrated sidebar menu data from file storage to database.',
                            'type' => 'success'
                        );
                        
                        // Try to rename the old file to prevent re-migration
                        $renamed = rename($old_data_file_path, $old_data_file_path . '.migrated');
                        if (!$renamed) {
                            error_log('FCA Widgets: Could not rename old data file after migration: ' . $old_data_file_path);
                            // As a fallback, try deleting (might fail due to permissions)
                            @unlink($old_data_file_path);
                        }
                    } else {
                        error_log('FCA Widgets: ERROR - Failed to migrate menu data to options table.');
                    }
                }
            }
        }
        
        return $migrated;
    }
}

// Initialize the migration manager
new FCA_MigrationManager(); 