<?php
class FCE_Settings_Page {
    private $options;
    private $fixes_manager;

    public function __construct() {
        add_action('admin_menu', [$this, 'add_plugin_page']);
        add_action('admin_init', [$this, 'page_init']);
        add_action('fluent_community/portal_head', [$this, 'output_custom_css']);
        add_action('fluent_community/portal_head', [$this, 'output_header_code']);
        add_action('fluent_community/portal_footer', [$this, 'output_custom_js']);
        
        // Initialize fixes manager using singleton pattern
        $this->fixes_manager = FCE_Fixes_Manager::instance();
    }

    public function add_plugin_page() {
        add_menu_page(
            'FCE Quick Fixes', 
            'FCE Fixes', 
            'manage_options', 
            'fce-quick-fixes', 
            [$this, 'create_admin_page'],
            'dashicons-admin-generic'
        );
    }

    public function create_admin_page() {
        // Get current settings
        $this->options = get_option('fce_quick_fixes_settings', [
            'custom_css' => '',
            'custom_js' => '',
            'header_code' => ''
        ]);
        
        // Get active fixes
        $active_fixes = get_option('fce_active_fixes', []);
        
        // Get available fixes
        $available_fixes = $this->fixes_manager->get_fixes();
        ?>
        <div class="wrap">
            <h1>FCE Quick Fixes Settings</h1>
            <form method="post" action="options.php">
                <?php settings_fields('fce_quick_fixes_group'); ?>
                
                <!-- Quick Fixes Section -->
                <h2 class="title">Available Quick Fixes</h2>
                <table class="form-table">
                    <tbody>
                        <?php foreach ($available_fixes as $fix_id => $fix): ?>
                            <tr>
                                <th scope="row"><?php echo esc_html($fix['title']); ?></th>
                                <td>
                                    <fieldset>
                                        <label for="fix_<?php echo esc_attr($fix_id); ?>">
                                            <input type="checkbox" 
                                                   id="fix_<?php echo esc_attr($fix_id); ?>"
                                                   name="fce_active_fixes[<?php echo esc_attr($fix_id); ?>]" 
                                                   value="1" 
                                                   <?php checked(isset($active_fixes[$fix_id]) ? $active_fixes[$fix_id] : false); ?>>
                                            Enable this fix
                                        </label>
                                        <p class="description"><?php echo esc_html($fix['description']); ?></p>
                                        
                                        <?php
                                        // Render additional settings if they exist
                                        if (!empty($fix['has_settings']) && isset($active_fixes[$fix_id]) && $active_fixes[$fix_id]) {
                                            call_user_func($fix['settings_callback']);
                                        }
                                        ?>
                                    </fieldset>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <!-- Custom Code Section -->
                <h2 class="title">Custom Code</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">Header Code</th>
                        <td>
                            <textarea 
                                name="fce_quick_fixes_settings[header_code]" 
                                class="large-text code" 
                                rows="10"
                            ><?php echo esc_textarea($this->options['header_code']); ?></textarea>
                            <p class="description">Add custom code to the &lt;head&gt; section (perfect for Google Analytics, meta tags, etc.)</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Custom CSS</th>
                        <td>
                            <textarea 
                                name="fce_quick_fixes_settings[custom_css]" 
                                class="large-text code" 
                                rows="10"
                            ><?php echo esc_textarea($this->options['custom_css']); ?></textarea>
                            <p class="description">Add custom CSS to be applied to the community portal.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Custom JavaScript</th>
                        <td>
                            <textarea 
                                name="fce_quick_fixes_settings[custom_js]" 
                                class="large-text code" 
                                rows="10"
                            ><?php echo esc_textarea($this->options['custom_js']); ?></textarea>
                            <p class="description">Add custom JavaScript to be applied to the community portal.</p>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    public function page_init() {
        // Register Quick Fixes settings
        register_setting(
            'fce_quick_fixes_group',
            'fce_active_fixes',
            [
                'type' => 'array',
                'sanitize_callback' => [$this, 'sanitize_fixes']
            ]
        );

        // Register Fixes Options settings
        register_setting(
            'fce_quick_fixes_group',
            'fce_fixes_options',
            [
                'type' => 'array',
                'sanitize_callback' => [$this, 'sanitize_fixes_options']
            ]
        );

        // Register Custom Code settings
        register_setting(
            'fce_quick_fixes_group',
            'fce_quick_fixes_settings',
            [
                'sanitize_callback' => [$this, 'sanitize_settings']
            ]
        );
    }

    public function sanitize_fixes($input) {
        if (!is_array($input)) {
            return [];
        }
        
        $sanitized = [];
        foreach ($input as $fix_id => $value) {
            $sanitized[$fix_id] = (bool) $value;
        }
        return $sanitized;
    }

    public function sanitize_fixes_options($input) {
        $sanitized = [];
        
        // Sanitize reaction color
        if (isset($input['reaction_color'])) {
            $sanitized['reaction_color'] = sanitize_hex_color($input['reaction_color']);
        }
        
        // Sanitize thumb color (keeping for backward compatibility)
        if (isset($input['thumb_color'])) {
            $sanitized['thumb_color'] = sanitize_hex_color($input['thumb_color']);
        }
        
        // Sanitize copyright text - allow limited HTML tags
        if (isset($input['copyright_text'])) {
            $sanitized['copyright_text'] = wp_kses_post($input['copyright_text']);
        }
        
        // Sanitize copyright link
        if (isset($input['copyright_link'])) {
            $sanitized['copyright_link'] = esc_url_raw($input['copyright_link']);
        }
        
        // Sanitize custom topic order (using semicolons as separators)
        if (isset($input['topic_order'])) {
            $sanitized['topic_order'] = sanitize_text_field($input['topic_order']);
        }
        
        // Sanitize custom space order (using semicolons as separators)
        if (isset($input['space_order'])) {
            $sanitized['space_order'] = sanitize_text_field($input['space_order']);
        }
        
        // Sanitize show footer to admins checkbox
        $sanitized['show_footer_to_admins'] = isset($input['show_footer_to_admins']) ? (bool) $input['show_footer_to_admins'] : false;
        
        return $sanitized;
    }

    public function sanitize_settings($input) {
        $sanitized = [];
        
        // Sanitize custom CSS
        $sanitized['custom_css'] = isset($input['custom_css']) 
            ? wp_strip_all_tags($input['custom_css']) 
            : '';
        
        // Sanitize custom JS
        $sanitized['custom_js'] = isset($input['custom_js']) 
            ? $input['custom_js'] 
            : '';
        
        // Sanitize header code - don't strip tags as we need HTML
        $sanitized['header_code'] = isset($input['header_code']) 
            ? $input['header_code'] 
            : '';
        
        return $sanitized;
    }

    public function output_custom_css() {
        $options = get_option('fce_quick_fixes_settings');
        if (!empty($options['custom_css'])) {
            echo '<style>' . wp_strip_all_tags($options['custom_css']) . '</style>';
        }
    }

    public function output_custom_js() {
        $options = get_option('fce_quick_fixes_settings');
        if (!empty($options['custom_js'])) {
            echo '<script>' . $options['custom_js'] . '</script>';
        }
    }

    public function output_header_code() {
        $options = get_option('fce_quick_fixes_settings');
        if (!empty($options['header_code'])) {
            echo $options['header_code'];
        }
    }
} 