<?php
class FCE_Fixes_Manager {
    private $fixes;
    private $options;
    private static $instance = null;

    /**
     * Get the singleton instance
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Private constructor to enforce singleton pattern
     */
    private function __construct() {
        // Skip initialization during cron or AJAX requests
        if (wp_doing_cron() || wp_doing_ajax()) {
            return;
        }

        $this->options = get_option('fce_fixes_options', []);
        add_action('init', [$this, 'register_fixes']);
        add_action('init', [$this, 'apply_active_fixes']);
    }

    public function register_fixes() {
        // Skip during cron operations
        if (wp_doing_cron()) {
            return;
        }

        $this->fixes = [
            'replace_hearts' => [
                'title' => 'Replace Hearts with Thumbs',
                'description' => 'Replace heart reaction icons with thumbs up throughout the community',
                'callback' => [$this, 'apply_heart_replacement']
            ],
            'custom_reaction_color' => [
                'title' => 'Custom Reaction Color',
                'description' => 'Change the color of reaction icons (hearts or thumbs)',
                'callback' => [$this, 'apply_custom_reaction_color'],
                'has_settings' => true,
                'settings_callback' => [$this, 'render_reaction_color_settings']
            ],
            'replace_profile_icon' => [
                'title' => 'Replace Profile Icon',
                'description' => 'Replace the default profile icon with a cleaner design',
                'callback' => [$this, 'apply_profile_icon_replacement']
            ],
            'fix_comment_editor' => [
                'title' => 'Fix Comment Editor Styles',
                'description' => 'Improves the comment editor appearance and mention dropdown styles',
                'callback' => [$this, 'apply_comment_editor_fixes']
            ],
            'align_welcome_buttons' => [
                'title' => 'Align Welcome Banner Buttons',
                'description' => 'Aligns the welcome banner buttons to the left instead of center',
                'callback' => [$this, 'apply_welcome_button_alignment']
            ],
            'image_viewer_click_outside' => [
                'title' => 'Close Image Viewer on Outside Click',
                'description' => 'Allows closing the image viewer by clicking outside the image',
                'callback' => [$this, 'apply_image_viewer_click_outside']
            ],
            'hide_complete_profile' => [
                'title' => 'Hide Complete Profile Button',
                'description' => 'Hides the "Complete your profile" button while keeping the menu button',
                'callback' => [$this, 'apply_hide_complete_profile']
            ],
            'custom_sidebar_footer' => [
                'title' => 'Custom Sidebar Footer',
                'description' => 'Replace the default "Powered by FluentCommunity" text with your own custom copyright text and link',
                'callback' => [$this, 'apply_custom_sidebar_footer'],
                'has_settings' => true,
                'settings_callback' => [$this, 'render_sidebar_footer_settings']
            ],
            'portal_as_homepage' => [
                'title' => 'Use Portal as Homepage',
                'description' => 'Make the Fluent Community portal your website homepage (empty slug)',
                'callback' => [$this, 'apply_portal_as_homepage'],
                'has_settings' => true,
                'settings_callback' => [$this, 'render_portal_as_homepage_settings']
            ],
            'custom_topic_order' => [
                'title' => 'Custom Topic Order',
                'description' => 'Customize the order of topics in the topic filter bar',
                'callback' => [$this, 'apply_custom_topic_order'],
                'has_settings' => true,
                'settings_callback' => [$this, 'render_topic_order_settings']
            ],
            'custom_space_order' => [
                'title' => 'Custom Space Order',
                'description' => 'Customize the order of spaces on the spaces page',
                'callback' => [$this, 'apply_custom_space_order'],
                'has_settings' => true,
                'settings_callback' => [$this, 'render_space_order_settings']
            ]
        ];
    }

    public function apply_active_fixes() {
        // Skip during cron operations
        if (wp_doing_cron()) {
            return;
        }

        $active_fixes = get_option('fce_active_fixes', []);
        
        foreach ($active_fixes as $fix_id => $is_active) {
            if ($is_active && isset($this->fixes[$fix_id])) {
                call_user_func($this->fixes[$fix_id]['callback']);
            }
        }
    }

    public function render_reaction_color_settings() {
        $options = get_option('fce_fixes_options', []);
        $reaction_color = isset($options['reaction_color']) ? $options['reaction_color'] : '#FFBF00';
        ?>
        <div style="margin-top: 10px;">
            <label for="reaction_color">
                Reaction Color:
                <input type="color" 
                       id="reaction_color" 
                       name="fce_fixes_options[reaction_color]" 
                       value="<?php echo esc_attr($reaction_color); ?>">
            </label>
            <p class="description">Choose the color for reaction icons (hearts or thumbs)</p>
        </div>
        <?php
    }

    public function apply_custom_reaction_color() {
        add_action('wp_head', [$this, 'output_custom_reaction_color']);
        add_action('fluent_community/portal_head', [$this, 'output_custom_reaction_color']);
    }

    public function output_custom_reaction_color() {
        $options = get_option('fce_fixes_options', []);
        $reaction_color = isset($options['reaction_color']) ? $options['reaction_color'] : '#FFBF00';
        ?>
        <style>
            /* Custom reaction color styling */
            .el-icon.user_reacted {
                fill: <?php echo esc_attr($reaction_color); ?> !important;
            }
            .react_active > i > svg {
                fill: <?php echo esc_attr($reaction_color); ?> !important;
            }
            .fcom_is_liked.reply_btn > div > i > svg > g > path {
                fill: <?php echo esc_attr($reaction_color); ?> !important;
            }
        </style>
        <?php
    }

    public function apply_heart_replacement() {
        // Add to both wp_footer and fluent_community/portal_footer to ensure it runs
        add_action('wp_footer', [$this, 'output_heart_replacement'], 99);
        add_action('fluent_community/portal_footer', [$this, 'output_heart_replacement'], 99);
    }

    public function output_heart_replacement() {
        $options = get_option('fce_fixes_options', []);
        $reaction_color = isset($options['reaction_color']) ? $options['reaction_color'] : '#FFBF00';
        ?>
        <style>
            /* Reaction thumbs styling - active state */
            .el-icon.user_reacted {
                fill: <?php echo esc_attr($reaction_color); ?> !important;
            }
            .react_active > i > svg {
                fill: <?php echo esc_attr($reaction_color); ?> !important;
            }
            .fcom_is_liked.reply_btn > div > i > svg > g > path {
                fill: <?php echo esc_attr($reaction_color); ?> !important;
            }
            
            /* Reaction thumbs styling - inactive state (outline only) - LIGHT MODE */
            li.fcom_reaction.fcom_reaction_list > i > svg {
                fill: none;
                stroke: #19283a;
                stroke-width: 2;
            }
            div.comment_wrap > div.reply_box > span:nth-child(2) > div > i > svg {
                fill: none;
                stroke: #19283a;
                stroke-width: 2;
            }
            .feed_actions button > span > i > svg {
                fill: none;
                stroke: #19283a;
                stroke-width: 2;
            }
            
            /* DARK MODE - detected via html.dark selector */
            html.dark li.fcom_reaction.fcom_reaction_list > i > svg,
            html.dark div.comment_wrap > div.reply_box > span:nth-child(2) > div > i > svg,
            html.dark .feed_actions button > span > i > svg {
                stroke: #f0f3f5;
            }
        </style>
        <script>
            (function() {
                function replaceHeartWithThumb() {
                    document.querySelectorAll('.fcom_reaction i.el-icon svg, .fcom_reaction_list i.el-icon svg, .fcom_comments_react i.el-icon svg, button.el-button i.el-icon svg, .react_active .user_reacted i.el-icon svg').forEach(svg => {
                        let path = svg.querySelector('path');
                        if (path && (path.getAttribute("d").includes("M16.697 5.5") || path.getAttribute("d").includes("M20.884 13.19"))) { 
                            path.setAttribute("d", "M14 9V5a3 3 0 0 0-3-3l-4 9v11h11.28a2 2 0 0 0 2-1.7l1.38-9a2 2 0 0 0-2-2.3zM7 22H4a2 2 0 0 1-2-2v-7a2 2 0 0 1 2-2h3");
                        }
                    });
                }

                // Run once in case elements are already present
                replaceHeartWithThumb();

                // Observe the DOM for dynamically added reaction icons
                let observer = new MutationObserver(mutations => {
                    mutations.forEach(mutation => {
                        mutation.addedNodes.forEach(node => {
                            if (node.nodeType === 1) {
                                if (node.matches('.fcom_reaction i.el-icon svg, .fcom_reaction_list i.el-icon svg, .fcom_comments_react i.el-icon svg, button.el-button i.el-icon svg, .react_active .user_reacted i.el-icon svg') || 
                                    node.querySelector('.fcom_reaction i.el-icon svg, .fcom_reaction_list i.el-icon svg, .fcom_comments_react i.el-icon svg, button.el-button i.el-icon svg, .react_active .user_reacted i.el-icon svg')) {
                                    replaceHeartWithThumb();
                                }
                            }
                        });
                    });
                });

                observer.observe(document.body, { childList: true, subtree: true });
            })();
        </script>
        <?php
    }

    public function apply_profile_icon_replacement() {
        add_action('wp_footer', [$this, 'output_profile_icon_replacement'], 99);
        add_action('fluent_community/portal_footer', [$this, 'output_profile_icon_replacement'], 99);
    }

    public function output_profile_icon_replacement() {
        ?>
        <script>
            (function() {
                function replaceProfileIcon() {
                    document.querySelectorAll('.focm_menu_item .el-icon svg').forEach(svg => {
                        let path = svg.querySelector('path');
                        if (path && path.getAttribute("d").includes("M628.736 528.896")) {
                            // Create new paths for the profile icon
                            svg.setAttribute('viewBox', '0 0 24 24');
                            svg.setAttribute('fill', 'none');
                            svg.innerHTML = `
                                <path d="M12.1992 12C14.9606 12 17.1992 9.76142 17.1992 7C17.1992 4.23858 14.9606 2 12.1992 2C9.43779 2 7.19922 4.23858 7.19922 7C7.19922 9.76142 9.43779 12 12.1992 12Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                                <path d="M3 22C3.57038 20.0332 4.74796 18.2971 6.3644 17.0399C7.98083 15.7827 9.95335 15.0687 12 15C16.12 15 19.63 17.91 21 22" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                            `;
                        }
                    });
                }

                // Run once for existing icons
                replaceProfileIcon();

                // Observe for dynamically added icons
                let observer = new MutationObserver(mutations => {
                    mutations.forEach(mutation => {
                        mutation.addedNodes.forEach(node => {
                            if (node.nodeType === 1) {
                                if (node.matches('.focm_menu_item .el-icon svg') || 
                                    node.querySelector('.focm_menu_item .el-icon svg')) {
                                    replaceProfileIcon();
                                }
                            }
                        });
                    });
                });

                observer.observe(document.body, { childList: true, subtree: true });
            })();
        </script>
        <?php
    }

    public function apply_comment_editor_fixes() {
        add_action('wp_head', [$this, 'output_comment_editor_styles']);
        add_action('fluent_community/portal_head', [$this, 'output_comment_editor_styles']);
    }

    public function output_comment_editor_styles() {
        ?>
        <style>
            .feed_commenting .feed_comment_input .fcom_md_seditor_wrapper {
                overflow: visible !important;
            }

            .fcom_mention_pop ul li:hover {
                background: #f0f3f5 !important;
            }
        </style>
        <?php
    }

    public function apply_welcome_button_alignment() {
        add_action('wp_head', [$this, 'output_welcome_button_styles']);
        add_action('fluent_community/portal_head', [$this, 'output_welcome_button_styles']);
    }

    public function output_welcome_button_styles() {
        ?>
        <style>
            .fcom_welcome_box .fcom_welcome_cta_buttons {
                justify-content: left !important;
            }
        </style>
        <?php
    }

    public function apply_image_viewer_click_outside() {
        add_action('wp_footer', [$this, 'output_image_viewer_script']);
        add_action('fluent_community/portal_footer', [$this, 'output_image_viewer_script']);
    }

    public function output_image_viewer_script() {
        ?>
        <script>
            (function() {
                // Function to handle click events
                function handleImageViewerClick(event) {
                    const wrapper = event.target.closest('.el-image-viewer__wrapper');
                    if (!wrapper) return;

                    // Check if click was on the mask or wrapper (outside the image)
                    if (
                        event.target.classList.contains('el-image-viewer__wrapper') || 
                        event.target.classList.contains('el-image-viewer__mask')
                    ) {
                        // Find and click the close button
                        const closeBtn = wrapper.querySelector('.el-image-viewer__close');
                        if (closeBtn) {
                            closeBtn.click();
                        }
                    }
                }

                // Function to set up the observer
                function setupImageViewerObserver() {
                    const observer = new MutationObserver(mutations => {
                        mutations.forEach(mutation => {
                            mutation.addedNodes.forEach(node => {
                                if (node.nodeType === 1 && node.classList.contains('el-image-viewer__wrapper')) {
                                    node.addEventListener('click', handleImageViewerClick);
                                }
                            });
                        });
                    });

                    observer.observe(document.body, {
                        childList: true,
                        subtree: true
                    });
                }

                // Set up the observer when the page loads
                if (document.readyState === 'loading') {
                    document.addEventListener('DOMContentLoaded', setupImageViewerObserver);
                } else {
                    setupImageViewerObserver();
                }
            })();
        </script>
        <?php
    }

    public function apply_hide_complete_profile() {
        add_action('wp_head', [$this, 'output_hide_complete_profile_styles']);
        add_action('fluent_community/portal_head', [$this, 'output_hide_complete_profile_styles']);
    }

    public function output_hide_complete_profile_styles() {
        ?>
        <style>
            /* Hide the Complete Profile button but keep the menu */
            .fhr_page_actions > div:first-child {
                display: none !important;
            }
            
            /* Ensure menu button remains visible and properly aligned */
            .fhr_page_actions {
                justify-content: flex-end !important;
            }
            .fhr_page_actions > div:last-child {
                display: flex !important;
            }
        </style>
        <?php
    }

    public function apply_custom_sidebar_footer() {
        // First, remove all actions with the default priority
        global $wp_filter;
        if (isset($wp_filter['fluent_community/after_portal_sidebar'])) {
            $wp_filter['fluent_community/after_portal_sidebar']->remove_all_filters(10);
        }
        
        // Add our custom sidebar footer
        add_action('fluent_community/after_portal_sidebar', [$this, 'output_custom_sidebar_footer'], 10);
    }

    public function output_custom_sidebar_footer() {
        // Get options
        $options = get_option('fce_fixes_options', []);
        $copyright_text = isset($options['copyright_text']) ? $options['copyright_text'] : '© ' . date('Y') . ' ' . get_bloginfo('name');
        $copyright_link = isset($options['copyright_link']) ? $options['copyright_link'] : home_url('/');
        $show_to_admins = isset($options['show_footer_to_admins']) ? $options['show_footer_to_admins'] : false;
        
        // Check if we're in the sidebar and not in the header/right menu area
        if (did_action('fluent_community/after_header_right_menu_items')) {
            echo '<div role="region" aria-label="Portal Settings" class="fcom_side_footer">';
            
            // Properly check if user is a Fluent Community site admin
            $is_admin = false;
            
            // First check if the Helper class exists with the isSiteAdmin method
            if (class_exists('FluentCommunity\App\Helpers\Helper') && method_exists('FluentCommunity\App\Helpers\Helper', 'isSiteAdmin')) {
                $is_admin = \FluentCommunity\App\Helpers\Helper::isSiteAdmin();
            } else {
                // Fallback: Check for WordPress admin capability
                $is_admin = current_user_can('manage_options');
            }
            
            if ($is_admin) {
                ?>
                <div style="display: flex;justify-content: space-between;" class="fcom_admin_menu">
                    <?php if (!defined('FLUENT_COMMUNITY_PRO')): ?>
                        <a title="Upgrade to Pro" target="_blank" rel="noopener" class="fcom_inline_icon_link_item"
                           href="<?php echo esc_url(class_exists('FluentCommunity\App\Helpers\Utility') && method_exists('FluentCommunity\App\Helpers\Utility', 'getProductUrl') ? \FluentCommunity\App\Helpers\Utility::getProductUrl(true) : '#') ?>">
                            <span class="el-icon">
                                <svg width="126" height="125" viewBox="0 0 126 125" fill="none">
                                <rect x="0.22139" width="125" height="125" rx="17.8571" fill="#4A5FE0"/>
                                <path
                                    d="M47.9424 75.1832L61.6888 67.2467L69.6253 80.9931C62.0334 85.3763 52.3256 82.7751 47.9424 75.1832Z"
                                    fill="white"/>
                                <path fill-rule="evenodd" clip-rule="evenodd"
                                      d="M75.4348 59.3101L61.6884 67.2466L69.6249 80.993L83.3713 73.0565L75.4348 59.3101ZM89.1821 51.3734L75.4356 59.3099L83.3721 73.0564L97.1186 65.1199L89.1821 51.3734Z"
                                      fill="white"/>
                                <path
                                    d="M89.182 51.3736C92.978 49.182 97.8319 50.4826 100.023 54.2786L103.992 61.1518L97.1185 65.12L89.182 51.3736Z"
                                    fill="white"/>
                                <path
                                    d="M64.593 56.4052L50.8466 64.3417L42.9101 50.5953L56.6565 42.6588L64.593 56.4052Z"
                                    fill="white"/>
                                <path
                                    d="M78.3397 48.4683L64.5933 56.4048L56.6568 42.6584C64.2487 38.2752 73.9565 40.8764 78.3397 48.4683Z"
                                    fill="white"/>
                                <path
                                    d="M50.847 64.3418L37.1006 72.2783L29.1641 58.5318L42.9105 50.5953L50.847 64.3418Z"
                                    fill="white"/>
                                <path
                                    d="M37.1011 72.2783C33.3051 74.4699 28.4512 73.1693 26.2596 69.3733L22.2913 62.5001L29.1646 58.5319L37.1011 72.2783Z"
                                    fill="white"/>
                                </svg>
                            </span>
                            <span><?php _e('Upgrade', 'fluent-community'); ?></span>
                        </a>
                    <?php else: ?>
                        <a title="Go to /wp-admin" class="fcom_inline_icon_link_item fcom_wp_admin_link"
                           href="<?php echo esc_url(admin_url()); ?>">
                            <span class="el-icon">
                                <svg viewBox="0 0 122.52 122.523"><g fill="currentColor"><path
                                            d="m8.708 61.26c0 20.802 12.089 38.779 29.619 47.298l-25.069-68.686c-2.916 6.536-4.55 13.769-4.55 21.388z"/><path
                                            d="m96.74 58.608c0-6.495-2.333-10.993-4.334-14.494-2.664-4.329-5.161-7.995-5.161-12.324 0-4.831 3.664-9.328 8.825-9.328.233 0 .454.029.681.042-9.35-8.566-21.807-13.796-35.489-13.796-18.36 0-34.513 9.42-43.91 23.688 1.233.037 2.395.063 3.382.063 5.497 0 14.006-.667 14.006-.667 2.833-.167 3.167 3.994.337 4.329 0 0-2.847.335-6.015.501l19.138 56.925 11.501-34.493-8.188-22.434c-2.83-.166-5.511-.501-5.511-.501-2.832-.166-2.5-4.496.332-4.329 0 0 8.679.667 13.843.667 5.496 0 14.006-.667 14.006-.667 2.835-.167 3.168 3.994.337 4.329 0 0-2.853.335-6.015.501l18.992 56.494 5.242-17.517c2.272-7.269 4.001-12.49 4.001-16.989z"/><path
                                            d="m62.184 65.857-15.768 45.819c4.708 1.384 9.687 2.141 14.846 2.141 6.12 0 11.989-1.058 17.452-2.979-.141-.225-.269-.464-.374-.724z"/><path
                                            d="m107.376 36.046c.226 1.674.354 3.471.354 5.404 0 5.333-.996 11.328-3.996 18.824l-16.053 46.413c15.624-9.111 26.133-26.038 26.133-45.426.001-9.137-2.333-17.729-6.438-25.215z"/><path
                                            d="m61.262 0c-33.779 0-61.262 27.481-61.262 61.26 0 33.783 27.483 61.263 61.262 61.263 33.778 0 61.265-27.48 61.265-61.263-.001-33.779-27.487-61.26-61.265-61.26zm0 119.715c-32.23 0-58.453-26.223-58.453-58.455 0-32.23 26.222-58.451 58.453-58.451 32.229 0 58.45 26.221 58.45 58.451 0 32.232-26.221 58.455-58.45 58.455z"/></g></svg>
                            </span>
                        </a>
                    <?php endif; ?>
                    <a title="Portal Settings" class="fcom_inline_icon_link_item"
                       href="<?php 
                       if (class_exists('FluentCommunity\App\Services\Helper') && method_exists('FluentCommunity\App\Services\Helper', 'baseUrl')) {
                           echo esc_url(\FluentCommunity\App\Services\Helper::baseUrl('admin/settings'));
                       } elseif (class_exists('FluentCommunity\App\Helpers\Helper') && method_exists('FluentCommunity\App\Helpers\Helper', 'baseUrl')) {
                           echo esc_url(\FluentCommunity\App\Helpers\Helper::baseUrl('admin/settings'));
                       } else {
                           echo esc_url(admin_url());
                       }
                       ?>">
                        <span class="el-icon">
                            <svg viewBox="0 0 1024 1024" data-v-d2e47025=""><path fill="currentColor"
                                                                                  d="M600.704 64a32 32 0 0 1 30.464 22.208l35.2 109.376c14.784 7.232 28.928 15.36 42.432 24.512l112.384-24.192a32 32 0 0 1 34.432 15.36L944.32 364.8a32 32 0 0 1-4.032 37.504l-77.12 85.12a357.12 357.12 0 0 1 0 49.024l77.12 85.248a32 32 0 0 1 4.032 37.504l-88.704 153.6a32 32 0 0 1-34.432 15.296L708.8 803.904c-13.44 9.088-27.648 17.28-42.368 24.512l-35.264 109.376A32 32 0 0 1 600.704 960H423.296a32 32 0 0 1-30.464-22.208L357.696 828.48a351.616 351.616 0 0 1-42.56-24.64l-112.32 24.256a32 32 0 0 1-34.432-15.36L79.68 659.2a32 32 0 0 1 4.032-37.504l77.12-85.248a357.12 357.12 0 0 1 0-48.896l-77.12-85.248A32 32 0 0 1 79.68 364.8l88.704-153.6a32 32 0 0 1 34.432-15.296l112.32 24.256c13.568-9.152 27.776-17.408 42.56-24.64l35.2-109.312A32 32 0 0 1 423.232 64H600.64zm-23.424 64H446.72l-36.352 113.088-24.512 11.968a294.113 294.113 0 0 0-34.816 20.096l-22.656 15.36-116.224-25.088-65.28 113.152 79.68 88.192-1.92 27.136a293.12 293.12 0 0 0 0 40.192l1.92 27.136-79.808 88.192 65.344 113.152 116.224-25.024 22.656 15.296a294.113 294.113 0 0 0 34.816 20.096l24.512 11.968L446.72 896h130.688l36.48-113.152 24.448-11.904a288.282 288.282 0 0 0 34.752-20.096l22.592-15.296 116.288 25.024 65.28-113.152-79.744-88.192 1.92-27.136a293.12 293.12 0 0 0 0-40.256l-1.92-27.136 79.808-88.128-65.344-113.152-116.288 24.96-22.592-15.232a287.616 287.616 0 0 0-34.752-20.096l-24.448-11.904L577.344 128zM512 320a192 192 0 1 1 0 384 192 192 0 0 1 0-384m0 64a128 128 0 1 0 0 256 128 128 0 0 0 0-256"></path></svg>
                        </span>
                        <span><?php esc_html_e('Settings', 'fluent-community'); ?></span>
                    </a>
                </div>
                <?php
                
                // Only show custom footer to admin users if the option is enabled
                if ($show_to_admins) {
                    ?>
                    <div style="margin-top: 10px; text-align: center;">
                        <a target="_blank" rel="noopener" style="font-size: 80%; cursor: pointer;"
                           class="fcom_inline_icon_link_item" href="<?php echo esc_url($copyright_link); ?>">
                            <?php echo wp_kses_post($copyright_text); ?>
                        </a>
                    </div>
                    <?php
                }
            } else {
                // Only display custom copyright text for non-admin users
                ?>
                <a target="_blank" rel="noopener" style="font-size: 80%; cursor: pointer;"
                   class="fcom_inline_icon_link_item" href="<?php echo esc_url($copyright_link); ?>">
                    <?php echo wp_kses_post($copyright_text); ?>
                </a>
                <?php
            }
            echo '</div>';
            return;
        }
    }

    public function render_sidebar_footer_settings() {
        // Load WordPress editor scripts
        wp_enqueue_editor();
        
        $options = get_option('fce_fixes_options', []);
        $copyright_text = isset($options['copyright_text']) ? $options['copyright_text'] : '© ' . date('Y') . ' ' . get_bloginfo('name');
        $copyright_link = isset($options['copyright_link']) ? $options['copyright_link'] : home_url('/');
        $show_to_admins = isset($options['show_footer_to_admins']) ? $options['show_footer_to_admins'] : false;
        ?>
        <div style="margin-top: 10px;">
            <label for="copyright_text">
                <strong>Copyright Text:</strong>
            </label>
            <p class="description">Enter the copyright text to display in the sidebar footer. HTML is allowed.</p>
            <?php
            wp_editor(
                $copyright_text,
                'copyright_text',
                [
                    'textarea_name' => 'fce_fixes_options[copyright_text]',
                    'media_buttons' => false,
                    'textarea_rows' => 3,
                    'teeny' => true,
                    'quicktags' => true,
                ]
            );
            ?>
        </div>
        <div style="margin-top: 10px;">
            <label for="copyright_link">
                <strong>Copyright Link:</strong>
            </label>
            <input type="url" 
                   class="widefat"
                   id="copyright_link" 
                   name="fce_fixes_options[copyright_link]" 
                   value="<?php echo esc_attr($copyright_link); ?>">
            <p class="description">Enter the URL to link to from the copyright text.</p>
        </div>
        <div style="margin-top: 10px;">
            <label for="show_footer_to_admins">
                <input type="checkbox" 
                       id="show_footer_to_admins" 
                       name="fce_fixes_options[show_footer_to_admins]" 
                       value="1" 
                       <?php checked($show_to_admins); ?>>
                <strong>Show custom footer to administrators</strong>
            </label>
            <p class="description">By default, administrators will see admin menu links instead of the custom footer. Check this to show both.</p>
        </div>
        <?php
    }

    public function get_fixes() {
        return $this->fixes;
    }

    public function apply_portal_as_homepage() {
        // We don't need to do much here since we're defining the constant and filter
        // in the main plugin file. Just schedule a flush to make sure it takes effect.
        
        // Schedule a one-time flush rewrite on the next request
        update_option('fce_flush_rewrite_rules', 'enable_portal_homepage');
        
        // We'll add a notice to the admin to remind them to flush rewrite rules
        // But only on the relevant screens
        add_action('admin_notices', [$this, 'show_flush_rewrite_notice_if_needed']);
        
        // Check if this feature was just disabled and schedule a flush
        add_action('update_option_fce_active_fixes', [$this, 'check_portal_homepage_toggle'], 10, 2);
        
        // Add a direct filter again just to be extra sure (the main one is in the plugin's main file)
        add_filter('fluent_community/portal_slug', function() {
            return '';
        }, 99999);
    }
    
    public function show_flush_rewrite_notice_if_needed() {
        // Only show this notice on relevant screens
        $screen = get_current_screen();
        if (!$screen || !in_array($screen->id, ['settings_page_permalink', 'toplevel_page_fce-quick-fixes'])) {
            return;
        }
        
        // Get flush status
        $flush_status = get_option('fce_flush_rewrite_rules', false);
        
        // Only show this notice if needed
        if ($flush_status) {
            $action = $flush_status === 'enable_portal_homepage' ? 'enabled' : 'disabled';
            ?>
            <div class="notice notice-warning is-dismissible">
                <p><strong>FCE Quick Fixes:</strong> You've <?php echo esc_html($action); ?> the Portal Homepage feature. Please <a href="<?php echo esc_url(admin_url('options-permalink.php')); ?>">visit the Permalinks Settings page</a> and click "Save Changes" to flush the rewrite rules and activate this change.</p>
            </div>
            <?php
            
            // Mark notice as shown if user is on the permalinks page
            if (isset($screen->id) && $screen->id === 'settings_page_permalink') {
                delete_option('fce_flush_rewrite_rules');
            }
        }
    }
    
    public function render_portal_as_homepage_settings() {
        // Check if Fluent Community is active
        $fluent_community_active = class_exists('FluentCommunity\\App\\App');
        
        // Check current permalink structure
        $permalink_structure = get_option('permalink_structure');
        
        // Get debug info if available
        $debug_info = get_option('fce_portal_homepage_debug', []);
        ?>
        <div style="margin-top: 10px; background-color: #fff8e5; border-left: 4px solid #ffb900; padding: 8px 12px; margin-bottom: 10px;">
            <p><strong>⚠️ Important Instructions:</strong></p>
            <ol>
                <li>Enable this option</li>
                <li>Go to <a href="<?php echo esc_url(admin_url('options-permalink.php')); ?>">Settings → Permalinks</a> and click "Save Changes"</li>
                <li>Clear any browser cache or server caches you might be using</li>
                <li>Your portal should now be accessible at your domain root (e.g., <code>https://example.com/</code>)</li>
            </ol>
            <p><strong>Requirements:</strong></p>
            <ul>
                <li>WordPress must be installed in the root directory, not in a subdirectory</li>
                <li>Permalink structure must be something other than "Plain"</li>
                <li>Fluent Community plugin must be active</li>
            </ul>
        </div>

        <?php if (!$fluent_community_active): ?>
        <div style="margin-top: 10px; background-color: #f8d7da; border-left: 4px solid #dc3545; padding: 8px 12px; margin-bottom: 10px;">
            <p><strong>⚠️ Error:</strong> Fluent Community plugin is not active. This feature requires Fluent Community to be active.</p>
        </div>
        <?php endif; ?>

        <?php if (empty($permalink_structure)): ?>
        <div style="margin-top: 10px; background-color: #f8d7da; border-left: 4px solid #dc3545; padding: 8px 12px; margin-bottom: 10px;">
            <p><strong>⚠️ Warning:</strong> You are using the "Plain" permalink structure. This feature requires a custom permalink structure to work.</p>
            <p>Please go to <a href="<?php echo esc_url(admin_url('options-permalink.php')); ?>">Settings → Permalinks</a> and select any option other than "Plain" (Post name is recommended).</p>
        </div>
        <?php endif; ?>
        
        <?php if (!empty($debug_info)): ?>
        <div style="margin-top: 10px; background-color: #d1ecf1; border-left: 4px solid #0c5460; padding: 8px 12px; margin-bottom: 10px;">
            <p><strong>Debugging Information:</strong></p>
            <ul>
                <li>Constant defined: <?php echo isset($debug_info['constant_defined']) && $debug_info['constant_defined'] ? '✅ Yes' : '❌ No'; ?></li>
                <li>Constant value: <?php echo isset($debug_info['constant_value']) ? (empty($debug_info['constant_value']) ? '✅ Empty (correct)' : '❌ "' . esc_html($debug_info['constant_value']) . '" (incorrect)') : 'Unknown'; ?></li>
                <li>Fluent Community active: <?php echo isset($debug_info['fluent_community_active']) && $debug_info['fluent_community_active'] ? '✅ Yes' : '❌ No'; ?></li>
                <li>Filter present: <?php echo isset($debug_info['filter_present']) && $debug_info['filter_present'] ? '✅ Yes' : '❌ No'; ?></li>
            </ul>
            
            <p><strong>Manual Solution:</strong> If the feature still doesn't work, you can try adding this line to your wp-config.php file:</p>
            <pre style="background: #f5f5f5; padding: 10px; overflow: auto;">define('FLUENT_COMMUNITY_PORTAL_SLUG', '');</pre>
            <p>Add this line near the top of your wp-config.php file, before the line that says "That's all, stop editing".</p>
        </div>
        <?php endif; ?>
        <?php
    }

    public function check_portal_homepage_toggle($old_value, $new_value) {
        // Check if portal_as_homepage was enabled before but is now disabled
        if (isset($old_value['portal_as_homepage']) && $old_value['portal_as_homepage'] && 
            (!isset($new_value['portal_as_homepage']) || !$new_value['portal_as_homepage'])) {
            
            // Schedule a flush with disable flag
            update_option('fce_flush_rewrite_rules', 'disable_portal_homepage');
            
            // Add admin notice
            add_action('admin_notices', [$this, 'show_flush_rewrite_notice_if_needed']);
        }
    }

    public function apply_custom_topic_order() {
        // Register and enqueue the topic sorter script
        add_action('wp_enqueue_scripts', [$this, 'enqueue_topic_sorter_script']);
        add_action('fluent_community/portal_footer', [$this, 'output_topic_sorter_script']);
    }
    
    public function enqueue_topic_sorter_script() {
        // Don't load on admin pages
        if (is_admin()) {
            return;
        }
        
        // Get the custom topic order
        $options = get_option('fce_fixes_options', []);
        $custom_topic_order = isset($options['topic_order']) ? $options['topic_order'] : '';
        
        // Load the script
        wp_enqueue_script(
            'fce-topic-sorter', 
            FCE_QUICK_FIXES_URL . 'assets/js/topic-sorter.js', 
            [], 
            FCE_QUICK_FIXES_VERSION, 
            true
        );
        
        // Convert the custom order to an array and localize it for the script
        $topic_order = [];
        if (!empty($custom_topic_order)) {
            $topic_order = array_map('trim', explode(';', $custom_topic_order));
        }
        
        wp_localize_script('fce-topic-sorter', 'FluentTopicSorter', [
            'customOrder' => $topic_order
        ]);
    }
    
    public function output_topic_sorter_script() {
        // Get the custom order from options
        $options = get_option('fce_fixes_options', []);
        $topic_order = isset($options['topic_order']) ? array_map('trim', explode(';', $options['topic_order'])) : [];
        
        // If no topics are defined, don't output the script
        if (empty($topic_order)) {
            return;
        }
        
        // Since we're in the portal footer, we need to output the script directly
        ?>
        <script>
            window.FluentTopicSorter = {
                customOrder: <?php echo json_encode($topic_order); ?>,
                debug: <?php echo WP_DEBUG ? 'true' : 'false'; ?>
            };
        </script>
        <script src="<?php echo esc_url(FCE_QUICK_FIXES_URL . 'assets/js/topic-sorter.js?ver=' . FCE_QUICK_FIXES_VERSION); ?>"></script>
        <?php
    }
    
    public function render_topic_order_settings() {
        // Get the current topic order
        $options = get_option('fce_fixes_options', []);
        $topic_order = isset($options['topic_order']) ? $options['topic_order'] : '';
        
        // Enqueue jQuery UI for sortable functionality
        wp_enqueue_script('jquery-ui-sortable');
        
        // Create a nonce for AJAX security
        $ajax_nonce = wp_create_nonce('fce_admin_nonce');
        
        ?>
        <div style="margin-top: 10px;">
            <p class="description">Enter your custom topic names in the order you want them to appear. Separate each topic with a semicolon. Topics not in this list will be sorted alphabetically after your custom order.</p>
            <p class="description">For example: <code>Announcements; Bug Report; Feature Request; Tips</code></p>
            
            <div style="margin-top: 10px;">
                <label for="topic_order">
                    <strong>Topic Order:</strong>
                </label>
                <input type="text" 
                       class="widefat"
                       id="topic_order" 
                       name="fce_fixes_options[topic_order]" 
                       value="<?php echo esc_attr($topic_order); ?>"
                       placeholder="Announcements; Bug Report; Feature Request; Tips">
            </div>
            
            <div style="margin-top: 15px; background-color: #f8f9fa; padding: 10px; border: 1px solid #ddd; border-radius: 4px;">
                <h4 style="margin-top: 0;">How to use:</h4>
                <ol>
                    <li>Enter the names of your topics in the order you want them to appear</li>
                    <li>Topics must match the exact text displayed in the buttons (including emojis if present)</li>
                    <li>The "All" button will always remain as the first button</li>
                    <li>Topics not in your list will be sorted alphabetically at the end</li>
                </ol>
                
                <h4>Topic Discovery Tool:</h4>
                <p>Click the button below to scan your site for existing topics. This can help you identify the exact names to use.</p>
                <button type="button" id="discover_topics" class="button button-secondary">Discover Topics</button>
                <div id="topic_discovery_results" style="margin-top: 10px; display: none;">
                    <h4>Discovered Topics:</h4>
                    <div id="topic_list" style="margin-top: 5px; margin-bottom: 10px;"></div>
                    <button type="button" id="use_discovered_topics" class="button button-primary" style="margin-right: 10px;">Use This Order</button>
                    <button type="button" id="sort_alphabetically" class="button button-secondary">Sort Alphabetically</button>
                </div>
                
                <script>
                jQuery(document).ready(function($) {
                    // Topic discovery functionality
                    $('#discover_topics').on('click', function() {
                        const button = $(this);
                        button.prop('disabled', true).text('Scanning...');
                        
                        // Make AJAX request to discover topics
                        $.ajax({
                            url: ajaxurl,
                            type: 'POST',
                            data: {
                                action: 'fce_discover_topics',
                                nonce: '<?php echo esc_js($ajax_nonce); ?>'
                            },
                            success: function(response) {
                                if (response.success && response.data.topics) {
                                    // Display the discovered topics
                                    displayDiscoveredTopics(response.data.topics);
                                    
                                    // Show count
                                    const count = response.data.count || 0;
                                    const plural = count === 1 ? 'topic' : 'topics';
                                    $('#topic_discovery_results h4').text('Discovered Topics: ' + count + ' ' + plural);
                                    
                                    // Display debug information if available
                                    if (response.data.debug) {
                                        const debug = response.data.debug;
                                        let sourceInfo = '';
                                        
                                        if (debug.method_used === 'Example Topics') {
                                            sourceInfo = '<div class="notice notice-warning inline" style="margin: 10px 0; padding: 10px;">' +
                                                '<p><strong>Note:</strong> Using example topics because no actual topics were found on your site. ' +
                                                'Possible reasons:</p>' +
                                                '<ul style="list-style-type: disc; margin-left: 20px;">' +
                                                '<li>Your site has no topics defined yet</li>' +
                                                '<li>Fluent Community active: ' + (debug.fluent_community_active ? 'Yes' : 'No') + '</li>' +
                                                '<li>Utility class found: ' + (debug.utility_class_exists ? 'Yes' : 'No') + '</li>' +
                                                '<li>GetTopics method available: ' + (debug.get_topics_method_exists ? 'Yes' : 'No') + '</li>' +
                                                (debug.utility_error ? '<li>Error: ' + debug.utility_error + '</li>' : '') +
                                                (debug.database_error ? '<li>Database error: ' + debug.database_error + '</li>' : '') +
                                                '</ul>' +
                                                '<p>You can still use these example topics for testing or create your own list.</p>' +
                                                '</div>';
                                        } else {
                                            const methodLabels = {
                                                'Utility::getTopics()': 'Fluent Community API',
                                                'Database Query - Terms Table': 'Database Terms Table',
                                                'Database Query - Post Meta': 'Post Meta Data'
                                            };
                                            
                                            const methodName = methodLabels[debug.method_used] || debug.method_used;
                                            
                                            sourceInfo = '<div class="notice notice-success inline" style="margin: 10px 0; padding: 10px;">' +
                                                '<p><strong>Success!</strong> Found ' + count + ' topics from <strong>' + methodName + '</strong></p>' +
                                                '</div>';
                                        }
                                        
                                        $('#topic_list').before(sourceInfo);
                                    }
                                } else {
                                    alert('No topics found or an error occurred. Using example topics instead.');
                                    // Use fallback example topics
                                    displayDiscoveredTopics([
                                        'Announcements', 
                                        'Bug Report', 
                                        'Feature Request', 
                                        'General', 
                                        'Tips', 
                                        '💡 Idea'
                                    ]);
                                }
                                button.prop('disabled', false).text('Discover Topics');
                            },
                            error: function() {
                                alert('Error discovering topics. Using example topics instead.');
                                // Use fallback example topics
                                displayDiscoveredTopics([
                                    'Announcements', 
                                    'Bug Report', 
                                    'Feature Request', 
                                    'General', 
                                    'Tips', 
                                    '💡 Idea'
                                ]);
                                button.prop('disabled', false).text('Discover Topics');
                            }
                        });
                    });
                    
                    function displayDiscoveredTopics(topics) {
                        const topicList = $('#topic_list');
                        topicList.empty();
                        
                        // Remove any previous debug info
                        topicList.parent().find('.notice').remove();
                        
                        // Create a draggable list of topics
                        const ul = $('<ul>').addClass('topic-sortable-list').css({
                            'list-style': 'none',
                            'padding': 0,
                            'margin': 0
                        });
                        
                        topics.forEach(function(topic) {
                            $('<li>')
                                .text(topic)
                                .appendTo(ul)
                                .css({
                                    'padding': '8px 10px',
                                    'margin-bottom': '5px',
                                    'background-color': '#fff',
                                    'border': '1px solid #ddd',
                                    'border-radius': '3px',
                                    'cursor': 'grab'
                                });
                        });
                        
                        topicList.append(ul);
                        $('#topic_discovery_results').show();
                        
                        // Make the list sortable
                        ul.sortable({
                            placeholder: "ui-state-highlight",
                            axis: "y"
                        });
                        
                        // Button to use the current order
                        $('#use_discovered_topics').off('click').on('click', function() {
                            const orderedTopics = [];
                            ul.find('li').each(function() {
                                orderedTopics.push($(this).text());
                            });
                            $('#topic_order').val(orderedTopics.join('; '));
                        });
                        
                        // Button to sort alphabetically
                        $('#sort_alphabetically').off('click').on('click', function() {
                            const sortedTopics = topics.slice().sort();
                            displayDiscoveredTopics(sortedTopics);
                        });
                    }
                    
                    // If there's already a topic order, try to display it
                    const existingOrder = $('#topic_order').val();
                    if (existingOrder) {
                        const topics = existingOrder.split(';').map(function(topic) {
                            return topic.trim();
                        });
                        if (topics.length > 0) {
                            displayDiscoveredTopics(topics);
                        }
                    }
                });
                </script>
                
                <h4 style="margin-top: 15px;">Examples:</h4>
                <p><strong>Example 1:</strong> <code>Announcements; Bug Report; Feature Request; Tips</code></p>
                <p><strong>Example 2:</strong> <code>💡 Idea; General; Bug, Issues & Problems; Questions</code></p>
            </div>
        </div>
        <?php
    }

    public function apply_custom_space_order() {
        // Register and enqueue the space sorter script
        add_action('wp_enqueue_scripts', [$this, 'enqueue_space_sorter_script']);
        add_action('fluent_community/portal_footer', [$this, 'output_space_sorter_script']);
    }
    
    public function enqueue_space_sorter_script() {
        // Don't load on admin pages
        if (is_admin()) {
            return;
        }
        
        // Get the custom space order
        $options = get_option('fce_fixes_options', []);
        $custom_space_order = isset($options['space_order']) ? $options['space_order'] : '';
        
        // Load the script
        wp_enqueue_script(
            'fce-space-sorter', 
            FCE_QUICK_FIXES_URL . 'assets/js/space-sorter.js', 
            [], 
            FCE_QUICK_FIXES_VERSION, 
            true
        );
        
        // Convert the custom order to an array and localize it for the script
        $space_order = [];
        if (!empty($custom_space_order)) {
            $space_order = array_map('trim', explode(';', $custom_space_order));
        }
        
        wp_localize_script('fce-space-sorter', 'FluentSpaceSorter', [
            'customOrder' => $space_order
        ]);
    }
    
    public function output_space_sorter_script() {
        // Get the custom order from options
        $options = get_option('fce_fixes_options', []);
        $space_order = isset($options['space_order']) ? array_map('trim', explode(';', $options['space_order'])) : [];
        
        // If no spaces are defined, don't output the script
        if (empty($space_order)) {
            return;
        }
        
        // Since we're in the portal footer, we need to output the script directly
        ?>
        <script>
            window.FluentSpaceSorter = {
                customOrder: <?php echo json_encode($space_order); ?>,
                debug: <?php echo WP_DEBUG ? 'true' : 'false'; ?>
            };
        </script>
        <script src="<?php echo esc_url(FCE_QUICK_FIXES_URL . 'assets/js/space-sorter.js?ver=' . FCE_QUICK_FIXES_VERSION); ?>"></script>
        <?php
    }
    
    public function render_space_order_settings() {
        // Get the current space order
        $options = get_option('fce_fixes_options', []);
        $space_order = isset($options['space_order']) ? $options['space_order'] : '';
        
        // Enqueue jQuery UI for sortable functionality
        wp_enqueue_script('jquery-ui-sortable');
        
        // Create a nonce for AJAX security
        $ajax_nonce = wp_create_nonce('fce_admin_nonce');
        
        ?>
        <div style="margin-top: 10px;">
            <p class="description">Enter your custom space names in the order you want them to appear. Separate each space with a semicolon. Spaces not in this list will be sorted by their default order after your custom order.</p>
            <p class="description">For example: <code>FluentCRM; FluentBoards; General; Say Hello</code></p>
            
            <div style="margin-top: 10px;">
                <label for="space_order">
                    <strong>Space Order:</strong>
                </label>
                <input type="text" 
                       class="widefat"
                       id="space_order" 
                       name="fce_fixes_options[space_order]" 
                       value="<?php echo esc_attr($space_order); ?>"
                       placeholder="FluentCRM; FluentBoards; General; Say Hello">
            </div>
            
            <div style="margin-top: 15px; background-color: #f8f9fa; padding: 10px; border: 1px solid #ddd; border-radius: 4px;">
                <h4 style="margin-top: 0;">How to use:</h4>
                <ol>
                    <li>Enter the names of your spaces in the order you want them to appear</li>
                    <li>Spaces must match the exact name displayed in the space cards</li>
                    <li>Spaces not in your list will be displayed after your custom ordering</li>
                </ol>
                
                <h4>Space Discovery Tool:</h4>
                <p>Click the button below to scan your site for existing spaces. This can help you identify the exact names to use.</p>
                <button type="button" id="discover_spaces" class="button button-secondary">Discover Spaces</button>
                <div id="space_discovery_results" style="margin-top: 10px; display: none;">
                    <h4>Discovered Spaces:</h4>
                    <div id="space_list" style="margin-top: 5px; margin-bottom: 10px;"></div>
                    <button type="button" id="use_discovered_spaces" class="button button-primary" style="margin-right: 10px;">Use This Order</button>
                    <button type="button" id="sort_alphabetically" class="button button-secondary">Sort Alphabetically</button>
                </div>
                
                <script>
                jQuery(document).ready(function($) {
                    // Space discovery functionality
                    $('#discover_spaces').on('click', function() {
                        const button = $(this);
                        button.prop('disabled', true).text('Scanning...');
                        
                        // Make AJAX request to discover spaces
                        $.ajax({
                            url: ajaxurl,
                            type: 'POST',
                            data: {
                                action: 'fce_discover_spaces',
                                nonce: '<?php echo esc_js($ajax_nonce); ?>'
                            },
                            success: function(response) {
                                if (response.success && response.data.spaces) {
                                    // Display the discovered spaces
                                    displayDiscoveredSpaces(response.data.spaces);
                                    
                                    // Show count
                                    const count = response.data.count || 0;
                                    const plural = count === 1 ? 'space' : 'spaces';
                                    $('#space_discovery_results h4').text('Discovered Spaces: ' + count + ' ' + plural);
                                    
                                    // Display debug information if available
                                    if (response.data.debug) {
                                        const debug = response.data.debug;
                                        let sourceInfo = '';
                                        
                                        if (debug.method_used === 'Example Spaces') {
                                            sourceInfo = '<div class="notice notice-warning inline" style="margin: 10px 0; padding: 10px;">' +
                                                '<p><strong>Note:</strong> Using example spaces because no actual spaces were found on your site. ' +
                                                'Possible reasons:</p>' +
                                                '<ul style="list-style-type: disc; margin-left: 20px;">' +
                                                '<li>Your site has no spaces defined yet</li>' +
                                                '<li>Fluent Community active: ' + (debug.fluent_community_active ? 'Yes' : 'No') + '</li>' +
                                                '<li>Utility class found: ' + (debug.utility_class_exists ? 'Yes' : 'No') + '</li>' +
                                                '<li>GetSpaces method available: ' + (debug.get_spaces_method_exists ? 'Yes' : 'No') + '</li>' +
                                                (debug.utility_error ? '<li>Error: ' + debug.utility_error + '</li>' : '') +
                                                (debug.database_error ? '<li>Database error: ' + debug.database_error + '</li>' : '') +
                                                '</ul>' +
                                                '<p>You can still use these example spaces for testing or create your own list.</p>' +
                                                '</div>';
                                        } else {
                                            const methodLabels = {
                                                'Utility::getSpaces()': 'Fluent Community API',
                                                'Database Query - Spaces Table': 'Database Spaces Table',
                                                'Database Query - Posts': 'WordPress Posts Data'
                                            };
                                            
                                            const methodName = methodLabels[debug.method_used] || debug.method_used;
                                            
                                            sourceInfo = '<div class="notice notice-success inline" style="margin: 10px 0; padding: 10px;">' +
                                                '<p><strong>Success!</strong> Found ' + count + ' spaces from <strong>' + methodName + '</strong></p>' +
                                                '</div>';
                                        }
                                        
                                        $('#space_list').before(sourceInfo);
                                    }
                                } else {
                                    alert('No spaces found or an error occurred. Using example spaces instead.');
                                    // Use fallback example spaces
                                    displayDiscoveredSpaces([
                                        'FluentCRM', 
                                        'FluentSupport', 
                                        'FluentBoards', 
                                        'General', 
                                        'Say Hello'
                                    ]);
                                }
                                button.prop('disabled', false).text('Discover Spaces');
                            },
                            error: function() {
                                alert('Error discovering spaces. Using example spaces instead.');
                                // Use fallback example spaces
                                displayDiscoveredSpaces([
                                    'FluentCRM', 
                                    'FluentSupport', 
                                    'FluentBoards', 
                                    'General', 
                                    'Say Hello'
                                ]);
                                button.prop('disabled', false).text('Discover Spaces');
                            }
                        });
                    });
                    
                    function displayDiscoveredSpaces(spaces) {
                        const spaceList = $('#space_list');
                        spaceList.empty();
                        
                        // Remove any previous debug info
                        spaceList.parent().find('.notice').remove();
                        
                        // Create a draggable list of spaces
                        const ul = $('<ul>').addClass('space-sortable-list').css({
                            'list-style': 'none',
                            'padding': 0,
                            'margin': 0
                        });
                        
                        spaces.forEach(function(space) {
                            $('<li>')
                                .text(space)
                                .appendTo(ul)
                                .css({
                                    'padding': '8px 10px',
                                    'margin-bottom': '5px',
                                    'background-color': '#fff',
                                    'border': '1px solid #ddd',
                                    'border-radius': '3px',
                                    'cursor': 'grab'
                                });
                        });
                        
                        spaceList.append(ul);
                        $('#space_discovery_results').show();
                        
                        // Make the list sortable
                        ul.sortable({
                            placeholder: "ui-state-highlight",
                            axis: "y"
                        });
                        
                        // Button to use the current order
                        $('#use_discovered_spaces').off('click').on('click', function() {
                            const orderedSpaces = [];
                            ul.find('li').each(function() {
                                orderedSpaces.push($(this).text());
                            });
                            $('#space_order').val(orderedSpaces.join('; '));
                        });
                        
                        // Button to sort alphabetically
                        $('#sort_alphabetically').off('click').on('click', function() {
                            const sortedSpaces = spaces.slice().sort();
                            displayDiscoveredSpaces(sortedSpaces);
                        });
                    }
                    
                    // If there's already a space order, try to display it
                    const existingOrder = $('#space_order').val();
                    if (existingOrder) {
                        const spaces = existingOrder.split(';').map(function(space) {
                            return space.trim();
                        });
                        if (spaces.length > 0) {
                            displayDiscoveredSpaces(spaces);
                        }
                    }
                });
                </script>
                
                <h4 style="margin-top: 15px;">Examples:</h4>
                <p><strong>Example 1:</strong> <code>FluentCRM; FluentBoards; General; Say Hello</code></p>
                <p><strong>Example 2:</strong> <code>General; Say Hello; FluentSupport; FluentCRM</code></p>
            </div>
        </div>
        <?php
    }
} 