<?php
/*
Plugin Name: FCE Quick Fixes
Description: Quick fixes and customizations for Fluent Community
Version: 1.0.5
Author: Fluent Community Add-ons
Author URI: https://fluentcommunityaddons.com
*/

if (!defined('ABSPATH')) {
    exit;
}

// Hard define the constant if the feature is enabled
// Do this BEFORE WordPress has fully loaded to ensure it takes effect
$active_fixes = get_option('fce_active_fixes', []);
if (isset($active_fixes['portal_as_homepage']) && $active_fixes['portal_as_homepage'] && !defined('FLUENT_COMMUNITY_PORTAL_SLUG')) {
    define('FLUENT_COMMUNITY_PORTAL_SLUG', '');
    
    // Also add a direct filter
    add_filter('fluent_community/portal_slug', function() {
        return '';
    }, 99999);
}

define('FCE_QUICK_FIXES_VERSION', '1.0.5');
define('FCE_QUICK_FIXES_PATH', plugin_dir_path(__FILE__));
define('FCE_QUICK_FIXES_URL', plugin_dir_url(__FILE__));

class FCE_Quick_Fixes {
    private static $instance = null;

    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action('plugins_loaded', [$this, 'init_plugin']);
        
        // Check if we need to flush rewrite rules
        add_action('admin_init', [$this, 'maybe_flush_rewrite_rules']);
        
        // Add debug info to help track down issues
        add_action('admin_init', [$this, 'check_portal_homepage_setup']);
        
        // Register AJAX endpoint for topic discovery
        add_action('wp_ajax_fce_discover_topics', [$this, 'ajax_discover_topics']);
        add_action('wp_ajax_fce_discover_spaces', [$this, 'ajax_discover_spaces']);
    }

    public function init_plugin() {
        // Load required files
        require_once FCE_QUICK_FIXES_PATH . 'includes/class-settings-page.php';
        require_once FCE_QUICK_FIXES_PATH . 'includes/class-fixes-manager.php';

        // Initialize components
        new FCE_Settings_Page();
    }
    
    public function check_portal_homepage_setup() {
        // Only run on the plugin's admin page or permalink settings page to avoid cluttering other screens
        $screen = get_current_screen();
        if (!$screen || !in_array($screen->id, ['toplevel_page_fce-quick-fixes', 'options-permalink'])) {
            return;
        }
        
        $active_fixes = get_option('fce_active_fixes', []);
        if (isset($active_fixes['portal_as_homepage']) && $active_fixes['portal_as_homepage']) {
            // Check if constant is defined
            $debug_info = [
                'constant_defined' => defined('FLUENT_COMMUNITY_PORTAL_SLUG'),
                'constant_value' => defined('FLUENT_COMMUNITY_PORTAL_SLUG') ? FLUENT_COMMUNITY_PORTAL_SLUG : 'undefined',
                'checked_time' => current_time('mysql'),
                'fluent_community_active' => class_exists('FluentCommunity\\App\\App'),
                'filter_present' => has_filter('fluent_community/portal_slug'),
            ];
            update_option('fce_portal_homepage_debug', $debug_info);
            
            // Show any issues as notices (only on plugin's admin page for less clutter)
            if ($screen->id === 'toplevel_page_fce-quick-fixes') {
                if (!defined('FLUENT_COMMUNITY_PORTAL_SLUG') || FLUENT_COMMUNITY_PORTAL_SLUG !== '') {
                    add_action('admin_notices', [$this, 'show_portal_constant_issue_notice']);
                }
                
                if (!class_exists('FluentCommunity\\App\\App')) {
                    add_action('admin_notices', [$this, 'show_fluent_missing_notice']);
                }
            }
        }
    }
    
    public function show_portal_constant_issue_notice() {
        ?>
        <div class="notice notice-error is-dismissible">
            <p><strong>FCE Quick Fixes:</strong> The Portal Homepage feature is enabled, but the <code>FLUENT_COMMUNITY_PORTAL_SLUG</code> constant is not set correctly. This may be because another plugin or your theme is defining this constant with a different value.</p>
            <p>Try adding this line directly to your wp-config.php file (before the line that says "That's all, stop editing"):
            <code>define('FLUENT_COMMUNITY_PORTAL_SLUG', '');</code></p>
        </div>
        <?php
    }
    
    public function show_fluent_missing_notice() {
        ?>
        <div class="notice notice-error is-dismissible">
            <p><strong>FCE Quick Fixes:</strong> The Fluent Community plugin doesn't appear to be active. The Portal Homepage feature requires Fluent Community to be active.</p>
        </div>
        <?php
    }
    
    public function maybe_flush_rewrite_rules() {
        // Check if we need to flush rewrite rules
        $flush_status = get_option('fce_flush_rewrite_rules', false);
        
        if ($flush_status) {
            // Check if we're on the permalinks page
            $screen = get_current_screen();
            if (isset($screen->id) && $screen->id === 'options-permalink') {
                // This is the right place to flush
                flush_rewrite_rules(true);
                
                // We can delete the option now
                delete_option('fce_flush_rewrite_rules');
                
                // Add success notice
                add_action('admin_notices', [$this, 'show_rewrite_flushed_notice']);
            }
        }
    }
    
    public function show_rewrite_flushed_notice() {
        ?>
        <div class="notice notice-success is-dismissible">
            <p><strong>FCE Quick Fixes:</strong> Permalinks have been flushed successfully.</p>
            <?php if (class_exists('FluentCommunity\\App\\App')): ?>
            <p>The Portal Homepage feature should now be active. Your portal should now be accessible at your root domain.</p>
            <p>If it's still not working, try the following:
                <ol>
                    <li>Clear your browser cache</li>
                    <li>Clear any server or plugin caches you might be using</li>
                    <li>Make sure your site is not in a subdirectory</li>
                    <li>Check that you're using a permalink structure other than "Plain"</li>
                    <li>Try adding <code>define('FLUENT_COMMUNITY_PORTAL_SLUG', '');</code> directly to your wp-config.php file</li>
                </ol>
            </p>
            <?php else: ?>
            <p><strong>Note:</strong> Fluent Community plugin doesn't appear to be active. Please activate it for the portal homepage feature to work.</p>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * AJAX handler for discovering topics from the site data
     */
    public function ajax_discover_topics() {
        // Check nonce for security
        check_ajax_referer('fce_admin_nonce', 'nonce');
        
        // Ensure user has permission
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'You do not have permission to perform this action']);
            return;
        }
        
        $discovered_topics = [];
        $debug_info = [
            'method_used' => 'none',
            'fluent_community_active' => class_exists('FluentCommunity\\App\\App'),
            'utility_class_exists' => class_exists('FluentCommunity\\App\\Functions\\Utility'),
            'get_topics_method_exists' => class_exists('FluentCommunity\\App\\Functions\\Utility') && 
                method_exists('FluentCommunity\\App\\Functions\\Utility', 'getTopics')
        ];
        
        // Check if Fluent Community is active
        if (class_exists('FluentCommunity\\App\\Functions\\Utility')) {
            try {
                // First approach: Use the Utility class's getTopics method (preferred)
                $topics = \FluentCommunity\App\Functions\Utility::getTopics();
                
                // Let's inspect what we got back and handle different possible structures
                if ($topics) {
                    $debug_info['raw_topics_type'] = is_object($topics) ? get_class($topics) : gettype($topics);
                    
                    // Case 1: It's a collection of term objects
                    if (is_array($topics) || is_object($topics)) {
                        // Convert to array if it's an object that can be iterated
                        $topics_array = is_array($topics) ? $topics : (is_object($topics) && method_exists($topics, 'toArray') ? $topics->toArray() : (array)$topics);
                        
                        foreach ($topics_array as $topic) {
                            // Handle different possible structures
                            if (is_string($topic)) {
                                $discovered_topics[] = $topic;
                            } elseif (is_object($topic) || is_array($topic)) {
                                $topic_obj = (object)$topic;
                                if (isset($topic_obj->title)) {
                                    $discovered_topics[] = $topic_obj->title;
                                } elseif (isset($topic_obj->name)) {
                                    $discovered_topics[] = $topic_obj->name;
                                } elseif (isset($topic_obj->term)) {
                                    $discovered_topics[] = is_object($topic_obj->term) ? 
                                        (isset($topic_obj->term->title) ? $topic_obj->term->title : 
                                        (isset($topic_obj->term->name) ? $topic_obj->term->name : '')) : 
                                        $topic_obj->term;
                                }
                            }
                        }
                    }
                    
                    if (!empty($discovered_topics)) {
                        $debug_info['method_used'] = 'Utility::getTopics()';
                    }
                }
            } catch (\Exception $e) {
                $debug_info['utility_error'] = $e->getMessage();
            }
        }
        
        // If no topics found through Utility class, try the database approach
        if (empty($discovered_topics) && class_exists('FluentCommunity\\App\\Models\\ContentModel')) {
            try {
                // Try to get topics from the database
                global $wpdb;
                
                // Get topic taxonomy term IDs from the Fluent Community tables
                $table_prefix = $wpdb->prefix . 'fc_';
                
                // First attempt: try to get topics from the terms table if it exists
                if ($wpdb->get_var("SHOW TABLES LIKE '{$table_prefix}terms'") === "{$table_prefix}terms") {
                    $topics = $wpdb->get_results("
                        SELECT name 
                        FROM {$table_prefix}terms 
                        WHERE type = 'topic' OR taxonomy_name = 'post_topic' 
                        ORDER BY name ASC
                    ");
                    
                    if ($topics) {
                        foreach ($topics as $topic) {
                            $discovered_topics[] = $topic->name;
                        }
                        $debug_info['method_used'] = 'Database Query - Terms Table';
                    }
                }
                
                // Second attempt: try to extract topics from post meta if the first attempt failed
                if (empty($discovered_topics)) {
                    // Look for posts with topic meta
                    $posts_with_topics = $wpdb->get_results("
                        SELECT meta_value 
                        FROM {$wpdb->postmeta} 
                        WHERE meta_key = '_topics' OR meta_key LIKE '%topic%'
                        AND meta_value IS NOT NULL 
                        AND meta_value != ''
                        LIMIT 100
                    ");
                    
                    $all_topics = [];
                    
                    if ($posts_with_topics) {
                        foreach ($posts_with_topics as $post) {
                            $topics = maybe_unserialize($post->meta_value);
                            if (is_array($topics)) {
                                $all_topics = array_merge($all_topics, $topics);
                            } elseif (is_string($topics)) {
                                $topics_array = explode(',', $topics);
                                $topics_array = array_map('trim', $topics_array);
                                $all_topics = array_merge($all_topics, $topics_array);
                            }
                        }
                    }
                    
                    // Process and deduplicate the topics
                    if (!empty($all_topics)) {
                        $all_topics = array_map('trim', $all_topics);
                        $all_topics = array_filter($all_topics, 'strlen'); // Remove empty values
                        $all_topics = array_unique($all_topics);
                        sort($all_topics);
                        $discovered_topics = $all_topics;
                        $debug_info['method_used'] = 'Database Query - Post Meta';
                    }
                }
            } catch (\Exception $e) {
                $debug_info['database_error'] = $e->getMessage();
            }
        }
        
        // If we still don't have topics, provide some example ones
        if (empty($discovered_topics)) {
            $discovered_topics = [
                'Announcements', 
                'Bug Report', 
                'Feature Request', 
                'General', 
                'Tips', 
                '💡 Idea'
            ];
            $debug_info['method_used'] = 'Example Topics';
        }
        
        // Remove any duplicates and sort the final list
        $discovered_topics = array_unique($discovered_topics);
        sort($discovered_topics);
        
        // Send response
        wp_send_json_success([
            'topics' => $discovered_topics,
            'count' => count($discovered_topics),
            'debug' => $debug_info
        ]);
    }

    /**
     * AJAX handler for discovering spaces from the site data
     */
    public function ajax_discover_spaces() {
        // Check nonce for security
        check_ajax_referer('fce_admin_nonce', 'nonce');
        
        // Ensure user has permission
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'You do not have permission to perform this action']);
            return;
        }
        
        $discovered_spaces = [];
        $debug_info = [
            'method_used' => 'none',
            'fluent_community_active' => class_exists('FluentCommunity\\App\\App'),
            'utility_class_exists' => class_exists('FluentCommunity\\App\\Functions\\Utility'),
            'get_spaces_method_exists' => class_exists('FluentCommunity\\App\\Functions\\Utility') && 
                method_exists('FluentCommunity\\App\\Functions\\Utility', 'getSpaces')
        ];
        
        // Check if Fluent Community is active
        if (class_exists('FluentCommunity\\App\\Functions\\Utility')) {
            try {
                // First approach: Use the Utility class's getSpaces method (preferred)
                $spaces = \FluentCommunity\App\Functions\Utility::getSpaces(false);
                
                // Let's inspect what we got back and handle different possible structures
                if ($spaces) {
                    $debug_info['raw_spaces_type'] = is_object($spaces) ? get_class($spaces) : gettype($spaces);
                    
                    // Convert to array if it's an object that can be iterated
                    $spaces_array = is_array($spaces) ? $spaces : (is_object($spaces) && method_exists($spaces, 'toArray') ? $spaces->toArray() : (array)$spaces);
                    
                    foreach ($spaces_array as $space) {
                        // Handle different possible structures
                        if (is_string($space)) {
                            $discovered_spaces[] = $space;
                        } elseif (is_object($space) || is_array($space)) {
                            $space_obj = (object)$space;
                            
                            if (isset($space_obj->title)) {
                                $discovered_spaces[] = $space_obj->title;
                            } elseif (isset($space_obj->name)) {
                                $discovered_spaces[] = $space_obj->name;
                            } elseif (isset($space_obj->post_title)) {
                                $discovered_spaces[] = $space_obj->post_title;
                            } elseif (isset($space_obj->space_name)) {
                                $discovered_spaces[] = $space_obj->space_name;
                            }
                        }
                    }
                    
                    if (!empty($discovered_spaces)) {
                        $debug_info['method_used'] = 'Utility::getSpaces()';
                    }
                }
            } catch (\Exception $e) {
                $debug_info['utility_error'] = $e->getMessage();
            }
        }
        
        // If no spaces found through Utility class, try the database approach
        if (empty($discovered_spaces)) {
            try {
                // Try to get spaces from the database
                global $wpdb;
                
                // Get space data from WordPress posts
                $spaces_query = $wpdb->prepare(
                    "SELECT post_title 
                    FROM {$wpdb->posts} 
                    WHERE post_type = 'community_space' 
                    AND post_status = 'publish'
                    ORDER BY post_title ASC"
                );
                
                $spaces = $wpdb->get_results($spaces_query);
                
                if ($spaces) {
                    foreach ($spaces as $space) {
                        $discovered_spaces[] = $space->post_title;
                    }
                    $debug_info['method_used'] = 'Database Query - Posts';
                }
                
            } catch (\Exception $e) {
                $debug_info['database_error'] = $e->getMessage();
            }
        }
        
        // If we still don't have spaces, provide some example ones
        if (empty($discovered_spaces)) {
            $discovered_spaces = [
                'FluentCRM',
                'FluentSupport',
                'FluentBoards',
                'FluentCommunity',
                'General',
                'Say Hello'
            ];
            $debug_info['method_used'] = 'Example Spaces';
        }
        
        // Remove any duplicates and sort the final list
        $discovered_spaces = array_unique($discovered_spaces);
        sort($discovered_spaces);
        
        // Send response
        wp_send_json_success([
            'spaces' => $discovered_spaces,
            'count' => count($discovered_spaces),
            'debug' => $debug_info
        ]);
    }
}

// Initialize the plugin
FCE_Quick_Fixes::instance(); 