/**
 * Fluent Community Topic Sorter
 * 
 * This script reorders the topic buttons in Fluent Community
 * based on a predefined custom order from the FCE Quick Fixes plugin.
 */
(function() {
    // Tracking variables to prevent multiple sorts
    let sortedContainers = new WeakMap();
    let isCurrentlySorting = false;
    let lastSortTime = 0;
    let currentUrl = location.href;
    
    // Remove the custom CSS styles that were causing issues
    // and instead rely on the platform's native class names

    // Helper function to get the active topic from URL
    function getActiveTopicFromURL() {
        const urlParams = new URLSearchParams(window.location.search);
        let topic = urlParams.get('topic');
        if (topic) {
            try {
                // Try to decode the topic if it's URL encoded
                return decodeURIComponent(topic);
            } catch (e) {
                return topic;
            }
        }
        return null;
    }

    // Helper function to trigger SPA navigation instead of full page loads
    function navigateWithinSPA(url) {
        // If we have Vue router exposed on window, use it
        if (window.fcApp && window.fcApp.$router) {
            try {
                // Extract the path and query from the URL
                const urlObj = new URL(url, window.location.origin);
                const path = urlObj.pathname;
                const queryParams = {};
                
                // Parse query parameters
                for (const [key, value] of urlObj.searchParams.entries()) {
                    queryParams[key] = value;
                }
                
                // Navigate using Vue router
                window.fcApp.$router.push({ 
                    path: path, 
                    query: queryParams 
                });
                return true;
            } catch (e) {
                return false;
            }
        }

        // Fallback to history API if Vue router is not available
        if (window.history && window.history.pushState) {
            try {
                window.history.pushState(null, '', url);
                // Dispatch a popstate event to trigger SPA router
                const popStateEvent = new PopStateEvent('popstate', { 
                    bubbles: false,
                    cancelable: false,
                    state: null
                });
                window.dispatchEvent(popStateEvent);
                return true;
            } catch (e) {
                return false;
            }
        }

        // Last resort: try to find and trigger a native click handler
        return false;
    }

    // Function to properly apply active state to buttons based on site styling
    function applyButtonActiveState(button, isActive) {
        // Save original class list as a base
        const originalClasses = button.className;
        const usesElementUI = originalClasses.includes('el-button');
        
        // Start from the basic class structure
        if (usesElementUI) {
            // For Element UI buttons we want to maintain the el-button class
            button.className = 'el-button';
        } else {
            // For other buttons, fully reset the class
            button.className = '';
        }
        
        if (isActive) {
            // Apply the active class as used in the platform
            button.classList.add('fcom_active_topic');
            
            // Site-specific active styles
            if (window.location.href.includes('hoolet.uk')) {
                button.classList.add('active');
            }
            
            // Add data attribute for tracking
            button.setAttribute('data-fce-active-topic', 'true');
        } else {
            // Remove our tracking attribute
            button.removeAttribute('data-fce-active-topic');
        }
    }

    // Function to reorder topics
    function reorderTopics() {
        // Prevent simultaneous sorting operations and rate-limit sorting
        if (isCurrentlySorting || (Date.now() - lastSortTime < 500)) {
            return;
        }

        isCurrentlySorting = true;
        
        // Possible class names for the topics container based on different versions/themes
        const possibleContainerClasses = [
            '.fcom_topics_filters',
            '.fc_topics_filters',
            '.fluent_community_topics_filters',
            '.fcom_space_topics .el-scrollbar__view > div', // More generic selector
            '.fcom_feed_topics', // Alternative location
            '.fc_feed_topics'    // Alternative location
        ];
        
        // Find the topics container
        let topicsContainer = null;
        for (const selector of possibleContainerClasses) {
            const container = document.querySelector(selector);
            if (container && container.querySelectorAll('button').length > 0) {
                topicsContainer = container;
                break;
            }
        }
        
        if (!topicsContainer) {
            isCurrentlySorting = false;
            return;
        }

        // Check if we've already sorted this container during this page view
        if (sortedContainers.get(topicsContainer)) {
            isCurrentlySorting = false;
            return;
        }

        // Store the original click handlers if we can find them
        const originalButtons = Array.from(topicsContainer.querySelectorAll('button'));
        const originalHandlers = new Map();
        
        // Try to extract original click handlers or vue component data
        originalButtons.forEach(button => {
            // Store the original element for reference
            originalHandlers.set(button.textContent.trim(), button);
        });

        // Get the active topic from URL if present
        const activeTopicFromURL = getActiveTopicFromURL();

        // Get all topic buttons
        const allButtons = Array.from(topicsContainer.querySelectorAll('button'));
        
        // Find the currently active button (either from URL or existing active class)
        let activeButton = null;
        
        // Check if any button is already active
        for (const button of allButtons) {
            // Check for the active button using classes that might indicate it's active
            if (button.classList.contains('fcom_active_topic')) {
                activeButton = button;
                break;
            }
            
            // If no active class, check if the text matches the URL param
            if (activeTopicFromURL) {
                const buttonText = button.textContent.trim();
                if (buttonText === activeTopicFromURL) {
                    activeButton = button;
                    break;
                }
            }
        }

        // Separate "All" button and topic buttons
        const allButton = topicsContainer.querySelector('button:first-child');
        const topicButtons = Array.from(topicsContainer.querySelectorAll('button:not(:first-child)'));
        
        // If we don't have any buttons or if we don't have the custom order from the plugin, exit
        if (topicButtons.length === 0) {
            isCurrentlySorting = false;
            return;
        }
        
        if (typeof FluentTopicSorter === 'undefined' || !FluentTopicSorter.customOrder || FluentTopicSorter.customOrder.length === 0) {
            isCurrentlySorting = false;
            return;
        }
        
        // Sort buttons based on custom order
        topicButtons.sort((a, b) => {
            const textA = a.textContent.trim();
            const textB = b.textContent.trim();
            
            // Get index in custom order (or -1 if not found)
            const indexA = FluentTopicSorter.customOrder.indexOf(textA);
            const indexB = FluentTopicSorter.customOrder.indexOf(textB);
            
            // If both are in custom order, sort by that order
            if (indexA >= 0 && indexB >= 0) {
                return indexA - indexB;
            }
            
            // If only one is in custom order, prioritize it
            if (indexA >= 0) return -1;
            if (indexB >= 0) return 1;
            
            // If neither is in custom order, sort alphabetically
            return textA.localeCompare(textB);
        });
        
        // Create a clone of the container to avoid Vue rendering issues
        const containerClone = topicsContainer.cloneNode(false);
        
        // Add the "All" button back first
        if (allButton) {
            const newAllButton = allButton.cloneNode(true);
            
            // If "All" button should be active (no topic in URL)
            if (!activeTopicFromURL && (!activeButton || activeButton === allButton)) {
                // Apply active class to the All button (keeping its existing classes)
                if (!newAllButton.classList.contains('fcom_active_topic')) {
                    newAllButton.classList.add('fcom_active_topic');
                }
                newAllButton.setAttribute('data-fce-active-topic', 'true');
            } else {
                // Ensure it's not active
                newAllButton.classList.remove('fcom_active_topic');
                newAllButton.removeAttribute('data-fce-active-topic');
            }
            
            containerClone.appendChild(newAllButton);
        }
        
        // Add the sorted buttons to the clone
        topicButtons.forEach(button => {
            const newButton = button.cloneNode(true);
            const buttonText = newButton.textContent.trim();
            
            // If this button should be active
            if ((activeButton === button) || 
                (activeTopicFromURL && buttonText === activeTopicFromURL)) {
                
                // Add active class - keep other existing classes
                if (!newButton.classList.contains('fcom_active_topic')) {
                    newButton.classList.add('fcom_active_topic');
                }
                newButton.setAttribute('data-fce-active-topic', 'true');
            } else {
                // Ensure it's not active
                newButton.classList.remove('fcom_active_topic');
                newButton.removeAttribute('data-fce-active-topic');
            }
            
            containerClone.appendChild(newButton);
        });
        
        // Replace the original container with the clone
        topicsContainer.parentNode.replaceChild(containerClone, topicsContainer);
        
        // Mark this container as sorted
        sortedContainers.set(containerClone, true);
        
        // Add a slight delay before allowing another sort
        lastSortTime = Date.now();
        
        isCurrentlySorting = false;
        
        // Find if original container has any special/global click handlers
        let containerClickHandler = null;
        const originalContainerEvents = topicsContainer.__events; // Vue event handling
        if (originalContainerEvents && originalContainerEvents.click && originalContainerEvents.click.length > 0) {
            containerClickHandler = originalContainerEvents.click[0];
        }
        
        // Restore clicks on any buttons inside the sorted container
        Array.from(containerClone.querySelectorAll('button')).forEach(button => {
            // Get the button text to identify its purpose
            const buttonText = button.textContent.trim();
            
            // Ensure button clicks still work by re-binding an event listener
            button.addEventListener('click', function(e) {
                // Try to prevent the default behavior that might cause page reload
                e.preventDefault();
                e.stopPropagation();
                
                // Get the current page URL
                const currentUrl = window.location.href;
                const baseUrl = currentUrl.split('?')[0]; // Remove existing query params
                
                // Determine the target URL based on which button was clicked
                let targetUrl;
                
                // Check if this is the All button (first child)
                const isFirstChild = button === containerClone.firstChild;
                
                if (isFirstChild) {
                    // For the "All" button, we just remove the topic parameter
                    targetUrl = baseUrl;
                } else {
                    // For topic buttons, we set the topic parameter
                    const urlObj = new URL(baseUrl);
                    urlObj.searchParams.set('topic', encodeURIComponent(buttonText));
                    targetUrl = urlObj.toString();
                }
                
                // Update active state immediately for better user feedback
                // First, remove active class from all buttons
                Array.from(containerClone.querySelectorAll('button')).forEach(btn => {
                    btn.classList.remove('fcom_active_topic');
                    btn.removeAttribute('data-fce-active-topic');
                });
                
                // Then add active class to the clicked button
                button.classList.add('fcom_active_topic');
                button.setAttribute('data-fce-active-topic', 'true');
                
                // Try to find the original button to trigger its click handler
                const originalButton = originalHandlers.get(buttonText);
                
                // Try SPA navigation first
                if (navigateWithinSPA(targetUrl)) {
                    // Try to trigger the Vue component methods if available
                    if (originalButton && originalButton.__vue__) {
                        try {
                            const vueInstance = originalButton.__vue__;
                            if (typeof vueInstance.handleClick === 'function') {
                                vueInstance.handleClick();
                            }
                        } catch (e) {
                            // Silent fail
                        }
                    }
                    
                    // Also try to trigger container level handlers
                    if (containerClickHandler) {
                        try {
                            containerClickHandler.call(containerClickHandler.context || null, e);
                        } catch (e) {
                            // Silent fail
                        }
                    }
                    
                    return;
                }
                
                // If SPA navigation failed, try to trigger original click
                if (originalButton) {
                    try {
                        originalButton.click();
                        return;
                    } catch (e) {
                        // Silent fail
                    }
                }
                
                // Last resort: direct location change
                window.location.href = targetUrl;
            });
        });
        
        // Immediately ensure the correct button is active based on URL
        ensureActiveButtonState();
    }

    // Standalone function to ensure active button state
    function ensureActiveButtonState() {
        const activeTopicFromURL = getActiveTopicFromURL();
        
        // Get all topic buttons 
        const topicButtons = document.querySelectorAll('.fcom_topics_filters button, .fc_topics_filters button');
        if (!topicButtons || topicButtons.length === 0) return;
        
        // First, remove active class from all buttons
        topicButtons.forEach(button => {
            button.classList.remove('fcom_active_topic');
            button.removeAttribute('data-fce-active-topic');
        });
        
        // Then, find and activate the correct button
        let activeButtonFound = false;
        
        if (activeTopicFromURL) {
            // Find button matching the topic in URL
            topicButtons.forEach(button => {
                const buttonText = button.textContent.trim();
                if (buttonText === activeTopicFromURL) {
                    button.classList.add('fcom_active_topic');
                    button.setAttribute('data-fce-active-topic', 'true');
                    activeButtonFound = true;
                }
            });
        }
        
        // If no active topic in URL or no matching button found, activate the "All" button
        if (!activeButtonFound && topicButtons.length > 0) {
            // The All button is typically the first one
            const allButton = topicButtons[0];
            allButton.classList.add('fcom_active_topic');
            allButton.setAttribute('data-fce-active-topic', 'true');
        }
    }

    // Also add a function to handle active state updates when URL changes
    function updateActiveButtonStates() {
        const activeTopicFromURL = getActiveTopicFromURL();
        
        // Get all topic containers on the page
        const possibleContainerClasses = [
            '.fcom_topics_filters',
            '.fc_topics_filters',
            '.fluent_community_topics_filters',
            '.fcom_space_topics .el-scrollbar__view > div',
            '.fcom_feed_topics',
            '.fc_feed_topics'
        ];
        
        let topicsContainer = null;
        for (const selector of possibleContainerClasses) {
            const container = document.querySelector(selector);
            if (container && container.querySelectorAll('button').length > 0) {
                topicsContainer = container;
                break;
            }
        }
        
        if (!topicsContainer) return;
        
        // Get all buttons
        const allButtons = Array.from(topicsContainer.querySelectorAll('button'));
        if (allButtons.length === 0) return;
        
        // Remove active state from all buttons first
        allButtons.forEach(btn => {
            btn.classList.remove('fcom_active_topic');
            btn.removeAttribute('data-fce-active-topic');
        });
        
        // If we have an active topic from URL, find and activate that button
        if (activeTopicFromURL) {
            let activeButtonFound = false;
            
            // Find the button matching the active topic
            for (const button of allButtons) {
                const buttonText = button.textContent.trim();
                if (buttonText === activeTopicFromURL) {
                    // Add active classes
                    button.classList.add('fcom_active_topic');
                    button.setAttribute('data-fce-active-topic', 'true');
                    activeButtonFound = true;
                    break;
                }
            }
            
            // If no matching button was found, don't do anything else
            if (!activeButtonFound) {
                // No action needed
            }
        } else {
            // No topic in URL, activate the All button (first button)
            const allButton = allButtons[0];
            if (allButton) {
                allButton.classList.add('fcom_active_topic');
                allButton.setAttribute('data-fce-active-topic', 'true');
            }
        }
    }

    // Function to observe DOM changes and reorder when topics first appear
    function observeTopicsContainer() {
        // Get our list of possible container selectors
        const possibleContainerClasses = [
            '.fcom_topics_filters',
            '.fc_topics_filters',
            '.fluent_community_topics_filters',
            '.fcom_space_topics .el-scrollbar__view > div',
            '.fcom_feed_topics',
            '.fc_feed_topics'
        ];
        
        // Helper function to check if any topic container exists
        function checkForTopicContainers() {
            for (const selector of possibleContainerClasses) {
                const container = document.querySelector(selector);
                if (container && container.querySelectorAll('button').length > 0) {
                    return container;
                }
            }
            return null;
        }
        
        // Create a mutation observer that's focused on finding when topic containers are added
        const observer = new MutationObserver((mutations) => {
            let foundTopicContainer = null;
            
            // First check if container already exists
            foundTopicContainer = checkForTopicContainers();
            if (foundTopicContainer && !sortedContainers.get(foundTopicContainer)) {
                setTimeout(() => reorderTopics(), 300);
                return;
            }
            
            // Otherwise look through mutations
            for (const mutation of mutations) {
                if (mutation.addedNodes.length > 0) {
                    // Only check element nodes
                    for (const node of mutation.addedNodes) {
                        if (node.nodeType !== 1) continue;
                        
                        // Check if this node is a topic container
                        for (const selector of possibleContainerClasses) {
                            try {
                                if (node.matches && node.matches(selector)) {
                                    foundTopicContainer = node;
                                    break;
                                }
                            } catch (e) {
                                continue;
                            }
                        }
                        
                        // If not the container itself, check if it contains a topic container
                        if (!foundTopicContainer && node.querySelector) {
                            for (const selector of possibleContainerClasses) {
                                const container = node.querySelector(selector);
                                if (container && container.querySelectorAll('button').length > 0) {
                                    foundTopicContainer = container;
                                    break;
                                }
                            }
                        }
                        
                        if (foundTopicContainer) break;
                    }
                }
                if (foundTopicContainer) break;
            }
            
            if (foundTopicContainer && !sortedContainers.get(foundTopicContainer)) {
                // Wait a moment for Vue to finish rendering all the buttons
                setTimeout(() => reorderTopics(), 300);
            }
        });

        // Start observing the document body
        observer.observe(document.body, { childList: true, subtree: true });
        
        // Also try to reorder immediately in case the container already exists
        const existingContainer = checkForTopicContainers();
        if (existingContainer) {
            setTimeout(() => reorderTopics(), 300);
        }
    }

    // Function to handle Vue SPA navigation
    function handleSpaNavigation() {
        // Watch for URL changes which might indicate navigation in a SPA
        const urlCheckInterval = setInterval(() => {
            if (location.href !== currentUrl) {
                currentUrl = location.href;
                
                // Reset our tracking whenever the URL changes
                sortedContainers = new WeakMap();
                
                // Wait a bit for the page to render after navigation, then reorder and update active states
                setTimeout(() => {
                    reorderTopics();
                    updateActiveButtonStates();
                }, 300);
                
                // Also ensure active state remains after Vue potentially rewrites the DOM
                setTimeout(ensureActiveButtonState, 800);
                setTimeout(ensureActiveButtonState, 1500);
            }
        }, 300);
        
        // Also reorder on popstate events (browser back/forward)
        window.addEventListener('popstate', () => {
            // Reset our tracking
            sortedContainers = new WeakMap();
            setTimeout(() => {
                reorderTopics();
                updateActiveButtonStates();
            }, 300);
            
            // Also ensure active state remains after Vue potentially rewrites the DOM
            setTimeout(ensureActiveButtonState, 800);
            setTimeout(ensureActiveButtonState, 1500);
        });
        
        // Register for Fluent Community's custom navigation events if they exist
        if (window.addEventListener) {
            window.addEventListener('fluent_community_loaded', () => {
                // Reset our tracking
                sortedContainers = new WeakMap();
                setTimeout(() => {
                    reorderTopics();
                    updateActiveButtonStates();
                }, 300);
                
                // Also ensure active state remains after Vue potentially rewrites the DOM
                setTimeout(ensureActiveButtonState, 800);
                setTimeout(ensureActiveButtonState, 1500);
            });
        }
    }

    // Set up a persistent observer that monitors active button styles
    function setupPersistentButtonObserver() {
        const activeTopicFromURL = getActiveTopicFromURL();
        let prevActiveTopicFromURL = activeTopicFromURL;
        
        // Create an observer specifically to monitor when Vue might be updating the button state
        const buttonClassObserver = new MutationObserver((mutations) => {
            for (const mutation of mutations) {
                if (mutation.type === 'attributes' && 
                    mutation.attributeName === 'class' && 
                    mutation.target.tagName === 'BUTTON') {
                    
                    // If Vue adds or removes classes that might affect our active state,
                    // ensure our active state is preserved
                    setTimeout(ensureActiveButtonState, 10);
                    break;
                }
            }
        });
        
        // Start observing
        buttonClassObserver.observe(document.body, {
            attributes: true,
            attributeFilter: ['class'],
            subtree: true
        });
        
        // This interval continuously checks if the active topic has changed
        const continuousStyleInterval = setInterval(() => {
            // Get the current active topic from URL
            const currentActiveTopicFromURL = getActiveTopicFromURL();
            
            // If URL parameter changed, we need to update all button states
            if (currentActiveTopicFromURL !== prevActiveTopicFromURL) {
                prevActiveTopicFromURL = currentActiveTopicFromURL;
                ensureActiveButtonState();
            }
            
            // Periodically check if Vue has changed our button states
            const activeButtons = document.querySelectorAll('[data-fce-active-topic="true"]');
            for (const button of activeButtons) {
                if (!button.classList.contains('fcom_active_topic')) {
                    button.classList.add('fcom_active_topic');
                }
            }
            
        }, 500);
        
        // Return cleanup function in case we need to stop observing
        return () => {
            clearInterval(continuousStyleInterval);
            buttonClassObserver.disconnect();
        };
    }

    // Initialize when the DOM is fully loaded
    function init() {
        observeTopicsContainer();
        handleSpaNavigation();
        
        // Set up the persistent style observer to maintain active state
        setupPersistentButtonObserver();
        
        // Find the Vue app instance if available
        setTimeout(() => {
            if (window.fcApp) {
                // Vue app found
            } else {
                // Try to find Vue app by looking at DOM elements
                const appEl = document.querySelector('#fluent_community_app, #app, #fc_app');
                if (appEl && appEl.__vue__) {
                    window.fcApp = appEl.__vue__;
                }
            }
        }, 1000);
        
        // Set up a timer to regularly check and enforce active button state
        setInterval(ensureActiveButtonState, 500);
    }
    
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})(); 