/**
 * Fluent Community Space Sorter
 * 
 * This script reorders the space cards in Fluent Community
 * based on a predefined custom order from the FCE Quick Fixes plugin.
 */
(function() {
    // Function to check if we're on a spaces page
    function isOnSpacesPage() {
        return location.href.includes('/communities') || 
               location.href.includes('/spaces') || 
               location.href.includes('/all-spaces') ||
               location.href.includes('/discover/spaces');
    }
    
    // Function to check if we're specifically on the discover/spaces page
    function isOnDiscoverSpacesPage() {
        return location.href.includes('/discover/spaces');
    }

    // Immediately hide the filter dropdown with CSS to prevent flickering - only on discover/spaces
    const style = document.createElement('style');
    style.textContent = `
        /* Hide feed filters only on discover/spaces page */
        body.fcom_page_discover_spaces .feed_filters, 
        body.fcom_page_discover_spaces .fcom_feed_filters, 
        body.fcom_page_discover_spaces .fc_feed_filters {
            display: none !important;
        }
        
        /* Grid layout for all space pages */
        .fcom_box_card_list, .fc_box_card_list {
            display: grid !important;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr)) !important;
            gap: 20px !important;
        }
    `;
    document.head.appendChild(style);

    // More targeted feed filter hiding based on URL
    function updateFeedFilterVisibility() {
        const feedFilters = document.querySelectorAll('.feed_filters, .fcom_feed_filters, .fc_feed_filters');
        
        if (isOnDiscoverSpacesPage()) {
            feedFilters.forEach(filter => {
                filter.style.display = 'none';
            });
        } else {
            feedFilters.forEach(filter => {
                filter.style.display = ''; // Reset to default
            });
        }
    }

    // Tracking variables
    let sortedContainers = [];
    let isCurrentlySorting = false;
    let currentUrl = location.href;
    
    // The core sorting function - simplified to focus on the basics
    function sortSpaces() {
        if (isCurrentlySorting || !isOnSpacesPage()) {
            return;
        }
        
        isCurrentlySorting = true;
        
        try {
            // Find the spaces container
            const containerSelectors = [
                '.fcom_communities .fcom_box_card_list',
                '.fc_communities .fc_box_card_list',
                '.fcom_communities',
                '.fc_communities',
                '.fcom_box_card_list',
                '.fc_box_card_list'
            ];
            
            let spacesContainer = null;
            for (const selector of containerSelectors) {
                spacesContainer = document.querySelector(selector);
                if (spacesContainer && spacesContainer.querySelectorAll('.fcom_each_box_card, .fc_each_box_card').length > 0) {
                    break;
                }
            }
            
            if (!spacesContainer) {
                isCurrentlySorting = false;
                return;
            }
            
            // Skip if we've already sorted this container
            if (sortedContainers.includes(spacesContainer)) {
                isCurrentlySorting = false;
                return;
            }
            
            // Get all space cards
            const spaceCards = Array.from(spacesContainer.querySelectorAll('.fcom_each_box_card, .fc_each_box_card'));
            
            // Skip if no cards or no custom order defined
            if (spaceCards.length === 0 || 
                typeof FluentSpaceSorter === 'undefined' || 
                !FluentSpaceSorter.customOrder || 
                FluentSpaceSorter.customOrder.length === 0) {
                isCurrentlySorting = false;
                return;
            }
            
            // Create sorted array of cards
            const sortedCards = [];
            
            // First, add cards that match our custom order
            for (const orderName of FluentSpaceSorter.customOrder) {
                for (const card of spaceCards) {
                    if (sortedCards.includes(card)) continue;
                    
                    const nameEl = card.querySelector('.fcom_box_card_name a, .fc_box_card_name a');
                    if (!nameEl) continue;
                    
                    const spaceName = nameEl.textContent.trim();
                    if (spaceName.toLowerCase() === orderName.toLowerCase() || 
                        spaceName.toLowerCase().includes(orderName.toLowerCase())) {
                        sortedCards.push(card);
                    }
                }
            }
            
            // Then add any remaining cards in their original order
            for (const card of spaceCards) {
                if (!sortedCards.includes(card)) {
                    sortedCards.push(card);
                }
            }
            
            // Apply the sorting by appending cards in order
            // This preserves Vue bindings because we're not cloning
            sortedCards.forEach(card => {
                spacesContainer.appendChild(card);
            });
            
            // Mark container as sorted
            sortedContainers.push(spacesContainer);
            
            // Update feed filter visibility based on current page
            updateFeedFilterVisibility();
        } catch (error) {
            console.error('[Space Sorter] Error:', error);
        }
        
        isCurrentlySorting = false;
    }
    
    // Function to observe DOM changes for spaces container
    function startObserving() {
        // Try to sort immediately if the container exists
        sortSpaces();
        
        // Create a mutation observer to detect when space cards are added
        const observer = new MutationObserver((mutations) => {
            let shouldCheck = false;
            
            // Quick check if any relevant nodes were added
            for (const mutation of mutations) {
                if (mutation.addedNodes.length > 0) {
                    shouldCheck = true;
                    break;
                }
            }
            
            if (shouldCheck && isOnSpacesPage()) {
                // Wait a moment for Vue to finish rendering
                setTimeout(() => {
                    sortSpaces();
                    updateFeedFilterVisibility();
                }, 100);
            }
        });
        
        // Start observing the body for space container additions
        observer.observe(document.body, { childList: true, subtree: true });
        
        // Handle SPA navigation by watching URL changes
        setInterval(() => {
            if (location.href !== currentUrl) {
                currentUrl = location.href;
                
                // Reset tracking for the new page
                sortedContainers = [];
                
                // If we're on a spaces page, try sorting after navigation
                if (isOnSpacesPage()) {
                    setTimeout(() => {
                        sortSpaces();
                        updateFeedFilterVisibility();
                    }, 500);
                } else {
                    // Make sure feed filters are visible on non-space pages
                    setTimeout(updateFeedFilterVisibility, 500);
                }
            }
        }, 500);
        
        // Also handle browser back/forward navigation
        window.addEventListener('popstate', () => {
            sortedContainers = [];
            setTimeout(() => {
                if (isOnSpacesPage()) {
                    sortSpaces();
                }
                updateFeedFilterVisibility();
            }, 500);
        });
    }
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', startObserving);
    } else {
        startObserving();
    }
})(); 