<?php
/**
 * License Checker Helper
 * Provides methods to check license status and feature access
 */

class FCA_CM_LicenseChecker {
    
    private static $instance = null;
    private $license_status = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Check if license is active/valid
     */
    public function is_licensed() {
        if ($this->license_status === null) {
            $this->license_status = $this->check_license_status();
        }
        return $this->license_status;
    }
    
    /**
     * Check actual license status from licensing system
     */
    private function check_license_status() {
        try {
            if (class_exists('\FCA_ContentManager\FluentLicensing')) {
                $licensing = \FCA_ContentManager\FluentLicensing::getInstance();
                $status = $licensing->getStatus();
                
                // Valid statuses that grant access
                return isset($status['status']) && $status['status'] === 'valid';
            }
        } catch (\Exception $e) {
            // If licensing not initialized, treat as unlicensed
            return false;
        }
        
        return false;
    }
    
    /**
     * Check if a specific feature is available
     */
    public function can_access_feature($feature) {
        // Posts are always available (free tier)
        if ($feature === 'posts') {
            return true;
        }
        
        // Overview is always available
        if ($feature === 'overview') {
            return true;
        }
        
        // Courses in read-only mode for free
        if ($feature === 'courses_readonly') {
            return true;
        }
        
        // Comments and reactions - view only for free (limited)
        if ($feature === 'comments_view' || $feature === 'reactions_view') {
            return true;
        }
        
        // Everything else requires license
        $pro_features = [
            'courses_edit',
            'members',
            'comments_edit',
            'reactions_edit',
            'media',
            'metadata',
            'spaces',
            'import',
            'export',
            'bulk_comments',
            'bulk_courses',
            'bulk_spaces',
            'unlimited_view'
        ];
        
        if (in_array($feature, $pro_features)) {
            return $this->is_licensed();
        }
        
        // Default to requiring license for unknown features
        return $this->is_licensed();
    }
    
    /**
     * Get upgrade message for a feature
     */
    public function get_upgrade_message($feature = '') {
        $messages = [
            'courses_edit' => __('Upgrade to Pro to manage courses, reorder sections and lessons, and move items between courses.', 'fca-content-manager'),
            'members' => __('Upgrade to Pro to manage members, handle enrollments, and control user access across spaces and courses.', 'fca-content-manager'),
            'comments_edit' => __('Upgrade to Pro to edit and manage comments with full bulk operations.', 'fca-content-manager'),
            'reactions_edit' => __('Upgrade to Pro to manage reactions without limits.', 'fca-content-manager'),
            'media' => __('Upgrade to Pro to manage media files and see where they\'re used.', 'fca-content-manager'),
            'metadata' => __('Upgrade to Pro to edit and manage metadata entries.', 'fca-content-manager'),
            'spaces' => __('Upgrade to Pro to manage spaces with full editing and bulk operations.', 'fca-content-manager'),
            'import' => __('Upgrade to Pro to import content from CSV and JSON files.', 'fca-content-manager'),
            'export' => __('Upgrade to Pro to export your content for backup or migration.', 'fca-content-manager'),
            'default' => __('Upgrade to Pro to unlock all features and unlimited access.', 'fca-content-manager')
        ];
        
        return isset($messages[$feature]) ? $messages[$feature] : $messages['default'];
    }
    
    /**
     * Get upgrade URL
     */
    public function get_upgrade_url() {
        return 'https://fluentcommunityaddons.com/item/content-manager-for-fluent-community/?utm_source=plugin&utm_medium=upgrade_prompt&utm_campaign=content_manager';
    }
    
    /**
     * Get free tier limits
     */
    public function get_free_limits() {
        return [
            'comments_view_limit' => 50,
            'reactions_view_limit' => 50,
        ];
    }
    
    /**
     * Send JSON error for unlicensed feature access
     */
    public function send_license_required_error($feature = '') {
        wp_send_json_error([
            'message' => $this->get_upgrade_message($feature),
            'license_required' => true,
            'upgrade_url' => $this->get_upgrade_url()
        ], 403);
    }
}

