<?php
/**
 * Import/Export Base Utilities
 * 
 * Provides common functionality for import and export operations
 */

defined('ABSPATH') or die('No direct script access allowed');

class FCA_CM_ImportExport {
    
    /**
     * Supported content types for import/export
     */
    const CONTENT_TYPES = array(
        'posts' => 'Posts',
        'spaces' => 'Spaces',
        'courses' => 'Courses',
        'comments' => 'Comments',
        'media' => 'Media',
        'metadata' => 'Metadata'
    );
    
    /**
     * Supported file formats
     */
    const FORMATS = array(
        'json' => 'JSON',
        'csv' => 'CSV'
    );
    
    /**
     * Get uploads directory for import/export files
     */
    public static function get_uploads_dir() {
        $upload_dir = wp_upload_dir();
        $custom_dir = $upload_dir['basedir'] . '/fca-content-manager';
        
        if (!file_exists($custom_dir)) {
            wp_mkdir_p($custom_dir);
        }
        
        return $custom_dir;
    }
    
    /**
     * Get uploads URL
     */
    public static function get_uploads_url() {
        $upload_dir = wp_upload_dir();
        return $upload_dir['baseurl'] . '/fca-content-manager';
    }
    
    /**
     * Generate unique filename
     */
    public static function generate_filename($type, $format = 'json') {
        $date = date('Y-m-d-His');
        return "fca-export-{$type}-{$date}.{$format}";
    }
    
    /**
     * Validate uploaded file
     */
    public static function validate_upload($file) {
        $errors = array();
        
        // Check for upload errors
        if ($file['error'] !== UPLOAD_ERR_OK) {
            $errors[] = __('File upload error.', 'fca-content-manager');
            return array('valid' => false, 'errors' => $errors);
        }
        
        // Check file size (max 50MB)
        $max_size = 50 * 1024 * 1024;
        if ($file['size'] > $max_size) {
            $errors[] = __('File size exceeds 50MB limit.', 'fca-content-manager');
        }
        
        // Check file extension
        $allowed_extensions = array('json', 'csv', 'zip');
        $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        if (!in_array($file_ext, $allowed_extensions)) {
            $errors[] = __('Invalid file type. Only JSON, CSV, and ZIP files are allowed.', 'fca-content-manager');
        }
        
        return array(
            'valid' => empty($errors),
            'errors' => $errors,
            'extension' => $file_ext
        );
    }
    
    /**
     * Parse CSV file
     */
    public static function parse_csv($file_path, $delimiter = ',') {
        $data = array();
        $headers = array();
        
        if (!file_exists($file_path)) {
            return array('error' => 'File not found');
        }
        
        $handle = fopen($file_path, 'r');
        if ($handle === false) {
            return array('error' => 'Cannot open file');
        }
        
        // Get headers from first row
        $headers = fgetcsv($handle, 0, $delimiter);
        
        if ($headers === false) {
            fclose($handle);
            return array('error' => 'Invalid CSV format');
        }
        
        // Parse data rows
        while (($row = fgetcsv($handle, 0, $delimiter)) !== false) {
            if (count($row) === count($headers)) {
                $data[] = array_combine($headers, $row);
            }
        }
        
        fclose($handle);
        
        return array(
            'headers' => $headers,
            'data' => $data,
            'count' => count($data)
        );
    }
    
    /**
     * Parse JSON file
     */
    public static function parse_json($file_path) {
        if (!file_exists($file_path)) {
            return array('error' => 'File not found');
        }
        
        $content = file_get_contents($file_path);
        if ($content === false) {
            return array('error' => 'Cannot read file');
        }
        
        $data = json_decode($content, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return array('error' => 'Invalid JSON: ' . json_last_error_msg());
        }
        
        return array(
            'data' => $data,
            'count' => is_array($data) ? count($data) : 1
        );
    }
    
    /**
     * Convert array to CSV
     */
    public static function array_to_csv($data, $headers = null) {
        if (empty($data)) {
            return '';
        }
        
        // Get headers from first item if not provided
        if ($headers === null && isset($data[0])) {
            $headers = array_keys($data[0]);
        }
        
        $output = fopen('php://temp', 'r+');
        
        // Write headers
        fputcsv($output, $headers);
        
        // Write data
        foreach ($data as $row) {
            fputcsv($output, $row);
        }
        
        rewind($output);
        $csv = stream_get_contents($output);
        fclose($output);
        
        return $csv;
    }
    
    /**
     * Map user identifier to user ID
     */
    public static function map_user($identifier, $field = 'email') {
        if (empty($identifier)) {
            return get_current_user_id();
        }
        
        if ($field === 'email') {
            $user = get_user_by('email', $identifier);
        } elseif ($field === 'username') {
            $user = get_user_by('login', $identifier);
        } elseif ($field === 'id') {
            $user = get_user_by('id', intval($identifier));
        } else {
            return get_current_user_id();
        }
        
        return $user ? $user->ID : get_current_user_id();
    }
    
    /**
     * Map space slug to space ID
     */
    public static function map_space($slug_or_id) {
        if (empty($slug_or_id)) {
            return null;
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_spaces';
        
        // Try as ID first
        if (is_numeric($slug_or_id)) {
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM {$table} WHERE id = %d",
                intval($slug_or_id)
            ));
            if ($exists) {
                return intval($slug_or_id);
            }
        }
        
        // Try as slug
        $space_id = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE slug = %s",
            $slug_or_id
        ));
        
        return $space_id ? intval($space_id) : null;
    }
    
    /**
     * Log import/export activity
     */
    public static function log($message, $type = 'info') {
        if (defined('WP_DEBUG') && WP_DEBUG === true) {
            error_log("[FCA Content Manager {$type}] " . $message);
        }
    }
    
    /**
     * Clean up old export files (older than 7 days)
     */
    public static function cleanup_old_files() {
        $dir = self::get_uploads_dir();
        $files = glob($dir . '/fca-export-*.{json,csv,zip}', GLOB_BRACE);
        
        $week_ago = time() - (7 * 24 * 60 * 60);
        
        foreach ($files as $file) {
            if (filemtime($file) < $week_ago) {
                @unlink($file);
            }
        }
    }
}
