<?php
/**
 * Import Handler
 * 
 * Handles importing content from JSON and CSV formats
 */

defined('ABSPATH') or die('No direct script access allowed');

class FCA_CM_Import {
    
    /**
     * Import posts from JSON
     */
    public static function import_posts_json($data, $options = array()) {
        $defaults = array(
            'update_existing' => false,
            'skip_duplicates' => true,
            'user_mapping' => 'email',
            'create_spaces' => true,
            'create_terms' => true,
            'dry_run' => false
        );
        
        $options = wp_parse_args($options, $defaults);
        
        $results = array(
            'imported' => 0,
            'updated' => 0,
            'skipped' => 0,
            'errors' => array(),
            'created_spaces' => array(),
            'created_terms' => array()
        );
        
        // Validate data structure
        if (!isset($data['items']) || !is_array($data['items'])) {
            $results['errors'][] = 'Invalid data structure. Expected "items" array.';
            return $results;
        }
        
        global $wpdb;
        $posts_table = $wpdb->prefix . 'fcom_posts';
        
        foreach ($data['items'] as $index => $item) {
            try {
                // Validate required fields
                if (empty($item['title']) || empty($item['message'])) {
                    $results['errors'][] = "Row {$index}: Missing required fields (title or message)";
                    $results['skipped']++;
                    continue;
                }
                
                // Check for existing post by slug
                $existing_id = null;
                if (!empty($item['slug'])) {
                    $existing_id = $wpdb->get_var($wpdb->prepare(
                        "SELECT id FROM {$posts_table} WHERE slug = %s",
                        $item['slug']
                    ));
                }
                
                // Handle duplicate
                if ($existing_id) {
                    if ($options['skip_duplicates']) {
                        $results['skipped']++;
                        continue;
                    } elseif (!$options['update_existing']) {
                        // Create new with modified slug
                        $item['slug'] = $item['slug'] . '-' . time();
                    }
                }
                
                // Map user
                $user_id = FCA_CM_ImportExport::map_user(
                    $item['user_id'] ?? get_current_user_id(),
                    'id'
                );
                
                // Map/create space
                $space_id = null;
                if (!empty($item['space_id'])) {
                    $space_id = FCA_CM_ImportExport::map_space($item['space_id']);
                    
                    if (!$space_id && $options['create_spaces']) {
                        // Space doesn't exist and we should create it
                        // For now, skip or log this
                        $results['errors'][] = "Row {$index}: Space not found: {$item['space_id']}";
                    }
                }
                
                // Prepare post data
                $message = wp_kses_post($item['message']);
                
                // Use provided message_rendered or generate it
                if (!empty($item['message_rendered'])) {
                    $message_rendered = wp_kses_post($item['message_rendered']);
                } else {
                    $message_rendered = $message;
                    if (!empty($message) && strpos($message, '<p>') === false) {
                        // Wrap in paragraph tags if not already HTML
                        $message_rendered = '<p>' . $message . '</p>';
                    }
                }
                
                // Prepare meta field (Fluent Community uses serialized PHP arrays)
                $meta_value = null;
                if (isset($item['meta']) && is_array($item['meta'])) {
                    $meta_value = maybe_serialize($item['meta']);
                } else {
                    // Default meta structure for Fluent Community
                    $meta_value = serialize(array('preview_data' => null));
                }
                
                $post_data = array(
                    'user_id' => $user_id,
                    'title' => sanitize_text_field($item['title']),
                    'slug' => sanitize_title($item['slug'] ?? $item['title']),
                    'message' => $message,
                    'message_rendered' => $message_rendered,
                    'type' => sanitize_text_field($item['type'] ?? 'text'), // Default to 'text' not 'feed'
                    'content_type' => sanitize_text_field($item['content_type'] ?? 'text'),
                    'space_id' => $space_id,
                    'status' => sanitize_text_field($item['status'] ?? 'published'),
                    'privacy' => sanitize_text_field($item['privacy'] ?? 'public'),
                    'meta' => $meta_value,
                    'created_at' => $item['created_at'] ?? current_time('mysql'),
                    'updated_at' => current_time('mysql')
                );
                
                // Dry run - don't actually insert
                if ($options['dry_run']) {
                    $results['imported']++;
                    continue;
                }
                
                // Insert or update
                if ($existing_id && $options['update_existing']) {
                    $result = $wpdb->update($posts_table, $post_data, array('id' => $existing_id));
                    if ($result !== false) {
                        $results['updated']++;
                        $post_id = $existing_id;
                    }
                } else {
                    $result = $wpdb->insert($posts_table, $post_data);
                    if ($result) {
                        $results['imported']++;
                        $post_id = $wpdb->insert_id;
                    }
                }
                
                // Import related data
                if (isset($post_id) && $post_id) {
                    // Import terms if available
                    if (!empty($item['terms']) && is_array($item['terms'])) {
                        self::import_post_terms($post_id, $item['terms'], $options);
                    }
                }
                
            } catch (Exception $e) {
                $results['errors'][] = "Row {$index}: " . $e->getMessage();
                $results['skipped']++;
            }
        }
        
        return $results;
    }
    
    /**
     * Import posts from CSV
     */
    public static function import_posts_csv($csv_data, $field_mapping, $options = array()) {
        // Convert CSV data to JSON-like format based on field mapping
        $json_items = array();
        
        foreach ($csv_data['data'] as $row) {
            $item = array();
            
            foreach ($field_mapping as $csv_column => $db_field) {
                if ($db_field && isset($row[$csv_column])) {
                    $item[$db_field] = $row[$csv_column];
                }
            }
            
            // Parse topics if present (comma-separated)
            if (isset($item['topics'])) {
                $item['terms'] = array_map('trim', explode(',', $item['topics']));
                unset($item['topics']);
            }
            
            $json_items[] = $item;
        }
        
        // Use JSON import with the converted data
        return self::import_posts_json(array('items' => $json_items), $options);
    }
    
    /**
     * Import spaces from JSON
     */
    public static function import_spaces_json($data, $options = array()) {
        $defaults = array(
            'update_existing' => false,
            'skip_duplicates' => true,
            'dry_run' => false
        );
        
        $options = wp_parse_args($options, $defaults);
        
        $results = array(
            'imported' => 0,
            'updated' => 0,
            'skipped' => 0,
            'errors' => array()
        );
        
        if (!isset($data['items']) || !is_array($data['items'])) {
            $results['errors'][] = 'Invalid data structure. Expected "items" array.';
            return $results;
        }
        
        global $wpdb;
        $spaces_table = $wpdb->prefix . 'fcom_spaces';
        
        foreach ($data['items'] as $index => $item) {
            try {
                if (empty($item['title']) || empty($item['slug'])) {
                    $results['errors'][] = "Row {$index}: Missing required fields (title or slug)";
                    $results['skipped']++;
                    continue;
                }
                
                // Check for existing space
                $existing_id = $wpdb->get_var($wpdb->prepare(
                    "SELECT id FROM {$spaces_table} WHERE slug = %s",
                    $item['slug']
                ));
                
                if ($existing_id && $options['skip_duplicates']) {
                    $results['skipped']++;
                    continue;
                }
                
                // Map creator
                $created_by = FCA_CM_ImportExport::map_user(
                    $item['created_by'] ?? get_current_user_id(),
                    'id'
                );
                
                // Prepare space data
                $space_data = array(
                    'created_by' => $created_by,
                    'title' => sanitize_text_field($item['title']),
                    'slug' => sanitize_title($item['slug']),
                    'description' => wp_kses_post($item['description'] ?? ''),
                    'type' => sanitize_text_field($item['type'] ?? ''),
                    'privacy' => sanitize_text_field($item['privacy'] ?? 'public'),
                    'status' => sanitize_text_field($item['status'] ?? 'published'),
                    'logo' => esc_url_raw($item['logo'] ?? ''),
                    'cover_photo' => esc_url_raw($item['cover_photo'] ?? ''),
                    'serial' => intval($item['serial'] ?? 1),
                    'created_at' => $item['created_at'] ?? current_time('mysql'),
                    'updated_at' => current_time('mysql')
                );
                
                if ($options['dry_run']) {
                    $results['imported']++;
                    continue;
                }
                
                // Insert or update
                if ($existing_id && $options['update_existing']) {
                    $result = $wpdb->update($spaces_table, $space_data, array('id' => $existing_id));
                    if ($result !== false) {
                        $results['updated']++;
                    }
                } else {
                    $result = $wpdb->insert($spaces_table, $space_data);
                    if ($result) {
                        $results['imported']++;
                    }
                }
                
            } catch (Exception $e) {
                $results['errors'][] = "Row {$index}: " . $e->getMessage();
                $results['skipped']++;
            }
        }
        
        return $results;
    }
    
    /**
     * Import spaces from CSV
     */
    public static function import_spaces_csv($csv_data, $field_mapping, $options = array()) {
        $json_items = array();
        
        foreach ($csv_data['data'] as $row) {
            $item = array();
            
            foreach ($field_mapping as $csv_column => $db_field) {
                if ($db_field && isset($row[$csv_column])) {
                    $item[$db_field] = $row[$csv_column];
                }
            }
            
            $json_items[] = $item;
        }
        
        return self::import_spaces_json(array('items' => $json_items), $options);
    }
    
    /**
     * Import post terms/topics
     */
    private static function import_post_terms($post_id, $terms, $options) {
        global $wpdb;
        $terms_table = $wpdb->prefix . 'fcom_terms';
        $term_feed_table = $wpdb->prefix . 'fcom_term_feed';
        
        foreach ($terms as $term_data) {
            $term_slug = is_array($term_data) ? $term_data['slug'] : $term_data;
            
            // Find or create term
            $term_id = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM {$terms_table} WHERE slug = %s",
                $term_slug
            ));
            
            if (!$term_id && $options['create_terms']) {
                $term_title = is_array($term_data) ? $term_data['title'] : ucwords(str_replace('-', ' ', $term_slug));
                
                $wpdb->insert($terms_table, array(
                    'title' => $term_title,
                    'slug' => $term_slug,
                    'created_at' => current_time('mysql')
                ));
                
                $term_id = $wpdb->insert_id;
            }
            
            if ($term_id) {
                // Associate term with post
                $wpdb->replace($term_feed_table, array(
                    'term_id' => $term_id,
                    'feed_id' => $post_id,
                    'created_at' => current_time('mysql')
                ));
            }
        }
    }
    
    /**
     * Validate import data
     */
    public static function validate_import($data, $content_type) {
        $errors = array();
        $warnings = array();
        
        // Basic structure validation
        if (!isset($data['items']) || !is_array($data['items'])) {
            $errors[] = 'Invalid data structure. Expected "items" array.';
            return array('valid' => false, 'errors' => $errors, 'warnings' => $warnings);
        }
        
        $items = $data['items'];
        $item_count = count($items);
        
        if ($item_count === 0) {
            $errors[] = 'No items found in import file.';
            return array('valid' => false, 'errors' => $errors, 'warnings' => $warnings);
        }
        
        // Content-type specific validation
        if ($content_type === 'posts') {
            $required_fields = array('title', 'message');
        } elseif ($content_type === 'spaces') {
            $required_fields = array('title', 'slug');
        } else {
            $required_fields = array();
        }
        
        // Sample validation (first 10 items)
        $sample_size = min(10, $item_count);
        for ($i = 0; $i < $sample_size; $i++) {
            $item = $items[$i];
            
            foreach ($required_fields as $field) {
                if (empty($item[$field])) {
                    $warnings[] = "Item {$i}: Missing required field '{$field}'";
                }
            }
        }
        
        return array(
            'valid' => empty($errors),
            'errors' => $errors,
            'warnings' => $warnings,
            'item_count' => $item_count,
            'sample' => array_slice($items, 0, 5)
        );
    }
    
    /**
     * Import comments from JSON
     */
    public static function import_comments_json($data, $options = array()) {
        $defaults = array(
            'update_existing' => false,
            'skip_duplicates' => true,
            'dry_run' => false
        );
        
        $options = wp_parse_args($options, $defaults);
        
        $results = array(
            'imported' => 0,
            'updated' => 0,
            'skipped' => 0,
            'errors' => array()
        );
        
        if (!isset($data['items']) || !is_array($data['items'])) {
            $results['errors'][] = 'Invalid data structure. Expected "items" array.';
            return $results;
        }
        
        global $wpdb;
        $comments_table = $wpdb->prefix . 'fcom_post_comments';
        $posts_table = $wpdb->prefix . 'fcom_posts';
        
        foreach ($data['items'] as $index => $item) {
            try {
                // Validate required fields
                if (empty($item['message']) || empty($item['post_id'])) {
                    $results['errors'][] = "Row {$index}: Missing required fields (message or post_id)";
                    $results['skipped']++;
                    continue;
                }
                
                // Verify post exists
                $post_exists = $wpdb->get_var($wpdb->prepare(
                    "SELECT id FROM {$posts_table} WHERE id = %d",
                    $item['post_id']
                ));
                
                if (!$post_exists) {
                    $results['errors'][] = "Row {$index}: Post ID {$item['post_id']} does not exist";
                    $results['skipped']++;
                    continue;
                }
                
                // Map user
                $user_id = FCA_CM_ImportExport::map_user(
                    $item['user_id'] ?? get_current_user_id(),
                    'id'
                );
                
                // Prepare comment data
                $message = wp_kses_post($item['message']);
                
                // Use provided message_rendered or generate it
                if (!empty($item['message_rendered'])) {
                    $message_rendered = wp_kses_post($item['message_rendered']);
                } else {
                    $message_rendered = $message;
                    if (!empty($message) && strpos($message, '<p>') === false) {
                        $message_rendered = '<p>' . $message . '</p>';
                    }
                }
                
                // Prepare meta field - default to empty array
                $meta_value = serialize(array());
                if (isset($item['meta']) && is_array($item['meta'])) {
                    $meta_value = maybe_serialize($item['meta']);
                } elseif (isset($item['meta']) && is_string($item['meta']) && !empty($item['meta'])) {
                    // Already serialized
                    $meta_value = $item['meta'];
                }
                
                $comment_data = array(
                    'user_id' => $user_id,
                    'post_id' => intval($item['post_id']),
                    'parent_id' => !empty($item['parent_id']) ? intval($item['parent_id']) : null,
                    'message' => $message,
                    'message_rendered' => $message_rendered,
                    'meta' => $meta_value,
                    'type' => sanitize_text_field($item['type'] ?? 'comment'),
                    'content_type' => sanitize_text_field($item['content_type'] ?? 'text'),
                    'status' => sanitize_text_field($item['status'] ?? 'published'),
                    'is_sticky' => intval($item['is_sticky'] ?? 0),
                    'reactions_count' => intval($item['reactions_count'] ?? 0),
                    'created_at' => $item['created_at'] ?? current_time('mysql'),
                    'updated_at' => current_time('mysql')
                );
                
                if ($options['dry_run']) {
                    $results['imported']++;
                    continue;
                }
                
                // Insert comment
                $result = $wpdb->insert($comments_table, $comment_data);
                if ($result) {
                    $results['imported']++;
                } else {
                    $results['errors'][] = "Row {$index}: Failed to insert comment";
                    $results['skipped']++;
                }
                
            } catch (Exception $e) {
                $results['errors'][] = "Row {$index}: " . $e->getMessage();
                $results['skipped']++;
            }
        }
        
        return $results;
    }
    
    /**
     * Import comments from CSV
     */
    public static function import_comments_csv($csv_data, $field_mapping, $options = array()) {
        $json_items = array();
        
        foreach ($csv_data['data'] as $row) {
            $item = array();
            
            foreach ($field_mapping as $csv_column => $db_field) {
                if ($db_field && isset($row[$csv_column])) {
                    $item[$db_field] = $row[$csv_column];
                }
            }
            
            $json_items[] = $item;
        }
        
        return self::import_comments_json(array('items' => $json_items), $options);
    }
    
    /**
     * Import reactions from JSON
     */
    public static function import_reactions_json($data, $options = array()) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_post_reactions';
        
        $results = array(
            'success' => 0,
            'skipped' => 0,
            'errors' => 0,
            'messages' => array()
        );
        
        if (!isset($data['items']) || !is_array($data['items'])) {
            $results['errors']++;
            $results['messages'][] = __('Invalid import data format.', 'fca-content-manager');
            return $results;
        }
        
        foreach ($data['items'] as $index => $item) {
            // Required fields
            if (empty($item['object_id']) || empty($item['object_type']) || empty($item['type'])) {
                $results['errors']++;
                $results['messages'][] = sprintf(__('Row %d: Missing required fields (object_id, object_type, type).', 'fca-content-manager'), $index + 1);
                continue;
            }
            
            // Map user
            $user_id = FCA_CM_ImportExport::map_user($item['user_email'] ?? '', $item['user_id'] ?? null);
            if (!$user_id) {
                $results['errors']++;
                $results['messages'][] = sprintf(__('Row %d: User not found.', 'fca-content-manager'), $index + 1);
                continue;
            }
            
            // Check for duplicates
            $existing = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM $table WHERE user_id = %d AND object_id = %d AND object_type = %s AND type = %s",
                $user_id,
                $item['object_id'],
                $item['object_type'],
                $item['type']
            ));
            
            if ($existing && !empty($options['skip_duplicates'])) {
                $results['skipped']++;
                continue;
            }
            
            $reaction_data = array(
                'user_id' => $user_id,
                'object_id' => intval($item['object_id']),
                'parent_id' => !empty($item['parent_id']) ? intval($item['parent_id']) : null,
                'object_type' => sanitize_text_field($item['object_type']),
                'type' => sanitize_text_field($item['type']),
                'ip_address' => !empty($item['ip_address']) ? sanitize_text_field($item['ip_address']) : null,
                'created_at' => $item['created_at'] ?? current_time('mysql'),
                'updated_at' => current_time('mysql')
            );
            
            if (!empty($options['dry_run'])) {
                $results['success']++;
                continue;
            }
            
            if ($existing && !empty($options['update_existing'])) {
                $wpdb->update($table, $reaction_data, array('id' => $existing));
                $results['success']++;
            } else {
                $wpdb->insert($table, $reaction_data);
                $results['success']++;
            }
        }
        
        return $results;
    }
    
    /**
     * Import term-feed relationships from JSON
     */
    public static function import_term_feed_json($data, $options = array()) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_term_feed';
        
        $results = array(
            'success' => 0,
            'skipped' => 0,
            'errors' => 0,
            'messages' => array()
        );
        
        if (!isset($data['items']) || !is_array($data['items'])) {
            $results['errors']++;
            $results['messages'][] = __('Invalid import data format.', 'fca-content-manager');
            return $results;
        }
        
        foreach ($data['items'] as $index => $item) {
            // Required fields
            if (empty($item['term_id']) || empty($item['post_id'])) {
                $results['errors']++;
                $results['messages'][] = sprintf(__('Row %d: Missing required fields (term_id, post_id).', 'fca-content-manager'), $index + 1);
                continue;
            }
            
            // Check for duplicates
            $existing = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM $table WHERE term_id = %d AND post_id = %d",
                $item['term_id'],
                $item['post_id']
            ));
            
            if ($existing && !empty($options['skip_duplicates'])) {
                $results['skipped']++;
                continue;
            }
            
            $relation_data = array(
                'term_id' => intval($item['term_id']),
                'post_id' => intval($item['post_id']),
                'created_at' => $item['created_at'] ?? current_time('mysql'),
                'updated_at' => current_time('mysql')
            );
            
            if (!empty($options['dry_run'])) {
                $results['success']++;
                continue;
            }
            
            if (!$existing) {
                $wpdb->insert($table, $relation_data);
                $results['success']++;
            } else {
                $results['skipped']++;
            }
        }
        
        return $results;
    }
    
    /**
     * Import space users from JSON
     */
    public static function import_space_users_json($data, $options = array()) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_space_user';
        
        $results = array(
            'success' => 0,
            'skipped' => 0,
            'errors' => 0,
            'messages' => array()
        );
        
        if (!isset($data['items']) || !is_array($data['items'])) {
            $results['errors']++;
            $results['messages'][] = __('Invalid import data format.', 'fca-content-manager');
            return $results;
        }
        
        foreach ($data['items'] as $index => $item) {
            // Required fields
            if (empty($item['space_id'])) {
                $results['errors']++;
                $results['messages'][] = sprintf(__('Row %d: Missing space_id.', 'fca-content-manager'), $index + 1);
                continue;
            }
            
            // Map user
            $user_id = FCA_CM_ImportExport::map_user($item['user_email'] ?? '', $item['user_id'] ?? null);
            if (!$user_id) {
                $results['errors']++;
                $results['messages'][] = sprintf(__('Row %d: User not found.', 'fca-content-manager'), $index + 1);
                continue;
            }
            
            // Map space
            $space_id = FCA_CM_ImportExport::map_space($item['space_slug'] ?? '', $item['space_id'] ?? null);
            if (!$space_id) {
                $results['errors']++;
                $results['messages'][] = sprintf(__('Row %d: Space not found.', 'fca-content-manager'), $index + 1);
                continue;
            }
            
            // Check for duplicates
            $existing = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM $table WHERE space_id = %d AND user_id = %d",
                $space_id,
                $user_id
            ));
            
            if ($existing && !empty($options['skip_duplicates'])) {
                $results['skipped']++;
                continue;
            }
            
            // Serialize meta if needed
            $meta_value = null;
            if (isset($item['meta']) && is_array($item['meta'])) {
                $meta_value = maybe_serialize($item['meta']);
            }
            
            $membership_data = array(
                'space_id' => $space_id,
                'user_id' => $user_id,
                'status' => sanitize_text_field($item['status'] ?? 'active'),
                'role' => sanitize_text_field($item['role'] ?? 'member'),
                'meta' => $meta_value,
                'created_at' => $item['created_at'] ?? current_time('mysql'),
                'updated_at' => current_time('mysql')
            );
            
            if (!empty($options['dry_run'])) {
                $results['success']++;
                continue;
            }
            
            if ($existing && !empty($options['update_existing'])) {
                $wpdb->update($table, $membership_data, array('id' => $existing));
                $results['success']++;
            } else {
                $wpdb->insert($table, $membership_data);
                $results['success']++;
            }
        }
        
        return $results;
    }

    /**
     * Import media from JSON
     */
    public static function import_media_json($data, $options = array()) {
        $defaults = array(
            'update_existing' => false,
            'skip_duplicates' => true,
            'dry_run' => false,
            'map_urls' => true // Convert old URLs to new URLs
        );
        
        $options = wp_parse_args($options, $defaults);
        
        $results = array(
            'imported' => 0,
            'updated' => 0,
            'skipped' => 0,
            'errors' => array(),
            'warnings' => array()
        );
        
        if (!isset($data['items']) || !is_array($data['items'])) {
            $results['errors'][] = 'Invalid data structure. Expected "items" array.';
            return $results;
        }
        
        global $wpdb;
        $media_table = $wpdb->prefix . 'fcom_media_archive';
        
        foreach ($data['items'] as $index => $item) {
            try {
                // Validate required fields
                if (empty($item['media_key']) || empty($item['media_url'])) {
                    $results['errors'][] = "Row {$index}: Missing required fields (media_key or media_url)";
                    $results['skipped']++;
                    continue;
                }
                
                // Check for existing media by key
                $existing_id = $wpdb->get_var($wpdb->prepare(
                    "SELECT id FROM {$media_table} WHERE media_key = %s",
                    $item['media_key']
                ));
                
                if ($existing_id && $options['skip_duplicates']) {
                    $results['skipped']++;
                    continue;
                }
                
                // Map user
                $user_id = FCA_CM_ImportExport::map_user(
                    $item['user_id'] ?? get_current_user_id(),
                    'id'
                );
                
                // Handle settings field - can be serialized or JSON
                $settings_value = 'a:0:{}'; // Default empty array
                if (isset($item['settings'])) {
                    if (is_array($item['settings'])) {
                        $settings_value = maybe_serialize($item['settings']);
                    } elseif (is_string($item['settings']) && !empty($item['settings'])) {
                        // Keep as-is if already formatted
                        $settings_value = $item['settings'];
                    }
                }
                
                // Prepare media data
                $media_data = array(
                    'object_source' => sanitize_text_field($item['object_source'] ?? 'general'),
                    'media_key' => sanitize_text_field($item['media_key']),
                    'user_id' => $user_id,
                    'feed_id' => !empty($item['feed_id']) ? intval($item['feed_id']) : null,
                    'is_active' => intval($item['is_active'] ?? 1),
                    'sub_object_id' => !empty($item['sub_object_id']) ? intval($item['sub_object_id']) : null,
                    'media_type' => sanitize_text_field($item['media_type'] ?? 'image/jpeg'),
                    'driver' => sanitize_text_field($item['driver'] ?? 'local'),
                    'media_path' => sanitize_text_field($item['media_path'] ?? ''),
                    'media_url' => esc_url_raw($item['media_url']),
                    'settings' => $settings_value,
                    'created_at' => $item['created_at'] ?? current_time('mysql'),
                    'updated_at' => current_time('mysql')
                );
                
                // Warn about URL mapping if needed
                if ($options['map_urls'] && !empty($item['media_url'])) {
                    $results['warnings'][] = "Row {$index}: Media URLs may need manual remapping to new site URLs";
                }
                
                if ($options['dry_run']) {
                    $results['imported']++;
                    continue;
                }
                
                // Insert or update
                if ($existing_id && $options['update_existing']) {
                    $result = $wpdb->update($media_table, $media_data, array('id' => $existing_id));
                    if ($result !== false) {
                        $results['updated']++;
                    }
                } else {
                    $result = $wpdb->insert($media_table, $media_data);
                    if ($result) {
                        $results['imported']++;
                    } else {
                        $results['errors'][] = "Row {$index}: Failed to insert media";
                        $results['skipped']++;
                    }
                }
                
            } catch (Exception $e) {
                $results['errors'][] = "Row {$index}: " . $e->getMessage();
                $results['skipped']++;
            }
        }
        
        return $results;
    }
}
