<?php
/**
 * Export Handler
 * 
 * Handles exporting content to JSON and CSV formats
 */

defined('ABSPATH') or die('No direct script access allowed');

class FCA_CM_Export {
    
    /**
     * Export posts to JSON
     */
    public static function export_posts_json($args = array()) {
        $defaults = array(
            'space_id' => '',
            'status' => '',
            'date_from' => '',
            'date_to' => '',
            'include_comments' => true,
            'include_reactions' => true,
            'include_media' => true,
            'include_terms' => true
        );
        
        $args = wp_parse_args($args, $defaults);
        
        // Get posts
        $posts = FCA_CM_Database::get_posts(array(
            'type' => '',
            'space_id' => $args['space_id'],
            'status' => $args['status'],
            'per_page' => -1
        ));
        
        $export_data = array(
            'export_date' => current_time('mysql'),
            'export_version' => '1.0',
            'content_type' => 'posts',
            'total_items' => count($posts['data']),
            'items' => array()
        );
        
        foreach ($posts['data'] as $post) {
            // Decode meta if it's serialized
            $meta_decoded = null;
            if (!empty($post['meta'])) {
                $meta_decoded = maybe_unserialize($post['meta']);
            }
            
            $item = array(
                'id' => $post['id'],
                'title' => $post['title'],
                'slug' => $post['slug'],
                'message' => $post['message'],
                'message_rendered' => $post['message_rendered'] ?? $post['message'],
                'type' => $post['type'],
                'content_type' => $post['content_type'],
                'space_id' => $post['space_id'],
                'user_id' => $post['user_id'],
                'status' => $post['status'],
                'privacy' => $post['privacy'],
                'meta' => $meta_decoded,
                'created_at' => $post['created_at'],
                'updated_at' => $post['updated_at']
            );
            
            // Include related data if requested
            if ($args['include_comments']) {
                $item['comments'] = FCA_CM_Database::get_post_comments($post['id']);
            }
            
            if ($args['include_reactions']) {
                $item['reactions'] = FCA_CM_Database::get_post_reactions($post['id']);
            }
            
            if ($args['include_media']) {
                $item['media'] = FCA_CM_Database::get_post_media($post['id']);
            }
            
            if ($args['include_terms']) {
                $item['terms'] = FCA_CM_Database::get_post_terms($post['id']);
            }
            
            $export_data['items'][] = $item;
        }
        
        return $export_data;
    }
    
    /**
     * Export posts to CSV
     */
    public static function export_posts_csv($args = array()) {
        $posts = FCA_CM_Database::get_posts(array(
            'type' => '',
            'space_id' => $args['space_id'] ?? '',
            'status' => $args['status'] ?? '',
            'per_page' => -1
        ));
        
        $csv_data = array();
        
        foreach ($posts['data'] as $post) {
            // Get space slug for reference
            $space_slug = '';
            if ($post['space_id']) {
                global $wpdb;
                $space_slug = $wpdb->get_var($wpdb->prepare(
                    "SELECT slug FROM {$wpdb->prefix}fcom_spaces WHERE id = %d",
                    $post['space_id']
                ));
            }
            
            // Get user email
            $user = get_userdata($post['user_id']);
            $user_email = $user ? $user->user_email : '';
            
            // Get topics as comma-separated list
            $terms = FCA_CM_Database::get_post_terms($post['id']);
            $topics = array();
            foreach ($terms as $term) {
                $topics[] = $term['slug'];
            }
            
            $csv_data[] = array(
                'id' => $post['id'],
                'title' => $post['title'],
                'slug' => $post['slug'],
                'message' => strip_tags($post['message']),
                'type' => $post['type'],
                'space_slug' => $space_slug,
                'author_email' => $user_email,
                'status' => $post['status'],
                'privacy' => $post['privacy'],
                'topics' => implode(',', $topics),
                'comments_count' => $post['comments_count'],
                'reactions_count' => $post['reactions_count'],
                'created_at' => $post['created_at'],
                'updated_at' => $post['updated_at']
            );
        }
        
        return FCA_CM_ImportExport::array_to_csv($csv_data);
    }
    
    /**
     * Export spaces to JSON
     */
    public static function export_spaces_json($args = array()) {
        $spaces = FCA_CM_Database::get_spaces_list(array(
            'per_page' => -1
        ));
        
        $export_data = array(
            'export_date' => current_time('mysql'),
            'export_version' => '1.0',
            'content_type' => 'spaces',
            'total_items' => count($spaces['data']),
            'items' => $spaces['data']
        );
        
        return $export_data;
    }
    
    /**
     * Export spaces to CSV
     */
    public static function export_spaces_csv($args = array()) {
        $spaces = FCA_CM_Database::get_spaces_list(array(
            'per_page' => -1
        ));
        
        $csv_data = array();
        
        foreach ($spaces['data'] as $space) {
            // Get creator email
            $creator = get_userdata($space['created_by']);
            $creator_email = $creator ? $creator->user_email : '';
            
            // Get parent slug if exists
            $parent_slug = '';
            if ($space['parent_id']) {
                global $wpdb;
                $parent_slug = $wpdb->get_var($wpdb->prepare(
                    "SELECT slug FROM {$wpdb->prefix}fcom_spaces WHERE id = %d",
                    $space['parent_id']
                ));
            }
            
            $csv_data[] = array(
                'id' => $space['id'],
                'title' => $space['title'],
                'slug' => $space['slug'],
                'description' => strip_tags($space['description']),
                'type' => $space['type'],
                'privacy' => $space['privacy'],
                'status' => $space['status'],
                'logo' => $space['logo'],
                'cover_photo' => $space['cover_photo'],
                'parent_slug' => $parent_slug,
                'serial' => $space['serial'],
                'creator_email' => $creator_email,
                'members_count' => $space['members_count'],
                'posts_count' => $space['posts_count'],
                'created_at' => $space['created_at'],
                'updated_at' => $space['updated_at']
            );
        }
        
        return FCA_CM_ImportExport::array_to_csv($csv_data);
    }
    
    /**
     * Export comments to JSON
     */
    public static function export_comments_json($args = array()) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_post_comments';
        
        $where = array('1=1');
        $values = array();
        
        if (!empty($args['space_id'])) {
            $where[] = "post_id IN (SELECT id FROM {$wpdb->prefix}fcom_posts WHERE space_id = %d)";
            $values[] = $args['space_id'];
        }
        
        $where_sql = implode(' AND ', $where);
        $query = "SELECT * FROM {$table} WHERE {$where_sql} ORDER BY created_at ASC";
        
        if (!empty($values)) {
            $query = $wpdb->prepare($query, $values);
        }
        
        $comments = $wpdb->get_results($query, ARRAY_A);
        
        $export_data = array(
            'export_date' => current_time('mysql'),
            'export_version' => '1.0',
            'content_type' => 'comments',
            'total_items' => count($comments),
            'items' => $comments
        );
        
        return $export_data;
    }
    
    /**
     * Export reactions to JSON
     */
    public static function export_reactions_json($args = array()) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_post_reactions';
        
        $where = array('1=1');
        $where_values = array();
        
        if (!empty($args['object_type'])) {
            $where[] = 'object_type = %s';
            $where_values[] = $args['object_type'];
        }
        
        if (!empty($args['type'])) {
            $where[] = 'type = %s';
            $where_values[] = $args['type'];
        }
        
        if (!empty($args['date_from'])) {
            $where[] = 'created_at >= %s';
            $where_values[] = $args['date_from'];
        }
        
        if (!empty($args['date_to'])) {
            $where[] = 'created_at <= %s';
            $where_values[] = $args['date_to'];
        }
        
        $where_sql = implode(' AND ', $where);
        
        if (!empty($where_values)) {
            $query = $wpdb->prepare("SELECT * FROM $table WHERE $where_sql ORDER BY created_at DESC", $where_values);
        } else {
            $query = "SELECT * FROM $table WHERE $where_sql ORDER BY created_at DESC";
        }
        
        $reactions = $wpdb->get_results($query, ARRAY_A);
        
        $export_data = array(
            'export_date' => current_time('mysql'),
            'export_version' => '1.0',
            'content_type' => 'reactions',
            'total_items' => count($reactions),
            'items' => array()
        );
        
        foreach ($reactions as $reaction) {
            // Get user email for mapping
            $user = get_user_by('id', $reaction['user_id']);
            
            $export_data['items'][] = array(
                'id' => $reaction['id'],
                'user_id' => $reaction['user_id'],
                'user_email' => $user ? $user->user_email : '',
                'object_id' => $reaction['object_id'],
                'parent_id' => $reaction['parent_id'],
                'object_type' => $reaction['object_type'],
                'type' => $reaction['type'],
                'ip_address' => $reaction['ip_address'],
                'created_at' => $reaction['created_at'],
                'updated_at' => $reaction['updated_at']
            );
        }
        
        return $export_data;
    }
    
    /**
     * Export term-feed relationships to JSON
     */
    public static function export_term_feed_json($args = array()) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_term_feed';
        
        $where = array('1=1');
        $where_values = array();
        
        if (!empty($args['post_id'])) {
            $where[] = 'post_id = %d';
            $where_values[] = $args['post_id'];
        }
        
        if (!empty($args['term_id'])) {
            $where[] = 'term_id = %d';
            $where_values[] = $args['term_id'];
        }
        
        $where_sql = implode(' AND ', $where);
        
        if (!empty($where_values)) {
            $query = $wpdb->prepare("SELECT * FROM $table WHERE $where_sql ORDER BY id", $where_values);
        } else {
            $query = "SELECT * FROM $table WHERE $where_sql ORDER BY id";
        }
        
        $relationships = $wpdb->get_results($query, ARRAY_A);
        
        $export_data = array(
            'export_date' => current_time('mysql'),
            'export_version' => '1.0',
            'content_type' => 'term_feed',
            'total_items' => count($relationships),
            'items' => array()
        );
        
        foreach ($relationships as $relationship) {
            // Get term and post slugs for reference
            $term = $wpdb->get_row($wpdb->prepare(
                "SELECT slug, title FROM {$wpdb->prefix}fcom_terms WHERE id = %d",
                $relationship['term_id']
            ));
            
            $post = $wpdb->get_row($wpdb->prepare(
                "SELECT slug, title FROM {$wpdb->prefix}fcom_posts WHERE id = %d",
                $relationship['post_id']
            ));
            
            $export_data['items'][] = array(
                'id' => $relationship['id'],
                'term_id' => $relationship['term_id'],
                'term_slug' => $term ? $term->slug : '',
                'term_title' => $term ? $term->title : '',
                'post_id' => $relationship['post_id'],
                'post_slug' => $post ? $post->slug : '',
                'post_title' => $post ? $post->title : '',
                'created_at' => $relationship['created_at'],
                'updated_at' => $relationship['updated_at']
            );
        }
        
        return $export_data;
    }
    
    /**
     * Export space users to JSON
     */
    public static function export_space_users_json($args = array()) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_space_user';
        
        $where = array('1=1');
        $where_values = array();
        
        if (!empty($args['space_id'])) {
            $where[] = 'space_id = %d';
            $where_values[] = $args['space_id'];
        }
        
        if (!empty($args['user_id'])) {
            $where[] = 'user_id = %d';
            $where_values[] = $args['user_id'];
        }
        
        if (!empty($args['role'])) {
            $where[] = 'role = %s';
            $where_values[] = $args['role'];
        }
        
        if (!empty($args['status'])) {
            $where[] = 'status = %s';
            $where_values[] = $args['status'];
        }
        
        $where_sql = implode(' AND ', $where);
        
        if (!empty($where_values)) {
            $query = $wpdb->prepare("SELECT * FROM $table WHERE $where_sql ORDER BY created_at DESC", $where_values);
        } else {
            $query = "SELECT * FROM $table WHERE $where_sql ORDER BY created_at DESC";
        }
        
        $memberships = $wpdb->get_results($query, ARRAY_A);
        
        $export_data = array(
            'export_date' => current_time('mysql'),
            'export_version' => '1.0',
            'content_type' => 'space_users',
            'total_items' => count($memberships),
            'items' => array()
        );
        
        foreach ($memberships as $membership) {
            // Get user email for mapping
            $user = get_user_by('id', $membership['user_id']);
            
            // Get space slug for mapping
            $space = $wpdb->get_row($wpdb->prepare(
                "SELECT slug, title FROM {$wpdb->prefix}fcom_spaces WHERE id = %d",
                $membership['space_id']
            ));
            
            // Decode meta if serialized
            $meta_decoded = null;
            if (!empty($membership['meta'])) {
                $meta_decoded = maybe_unserialize($membership['meta']);
            }
            
            $export_data['items'][] = array(
                'id' => $membership['id'],
                'space_id' => $membership['space_id'],
                'space_slug' => $space ? $space->slug : '',
                'space_title' => $space ? $space->title : '',
                'user_id' => $membership['user_id'],
                'user_email' => $user ? $user->user_email : '',
                'user_name' => $user ? $user->display_name : '',
                'status' => $membership['status'],
                'role' => $membership['role'],
                'meta' => $meta_decoded,
                'created_at' => $membership['created_at'],
                'updated_at' => $membership['updated_at']
            );
        }
        
        return $export_data;
    }
    
    /**
     * Create export file and return download info
     */
    public static function create_export_file($data, $filename, $format = 'json') {
        $dir = FCA_CM_ImportExport::get_uploads_dir();
        $file_path = $dir . '/' . $filename;
        
        if ($format === 'json') {
            $content = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        } else {
            $content = $data; // Already formatted CSV string
        }
        
        $result = file_put_contents($file_path, $content);
        
        if ($result === false) {
            return array('error' => 'Failed to create export file');
        }
        
        return array(
            'success' => true,
            'file_path' => $file_path,
            'file_url' => FCA_CM_ImportExport::get_uploads_url() . '/' . $filename,
            'file_size' => filesize($file_path),
            'filename' => $filename
        );
    }
}
