<?php
/**
 * Database Class
 * Simple, direct database queries for Fluent Community content
 */

// Prevent direct access
defined('ABSPATH') or die('No direct script access allowed');

class FCA_CM_Database {
    
    /**
     * Get posts with filters and pagination
     */
    public static function get_posts($args = array()) {
        global $wpdb;
        
        $defaults = array(
            'page' => 1,
            'per_page' => 20,
            'search' => '',
            'type' => '',  // Empty by default - show all types
            'status' => '',
            'space_id' => '',
            'user_id' => '',
            'orderby' => 'created_at',
            'order' => 'DESC'
        );
        
        $args = wp_parse_args($args, $defaults);
        $table = $wpdb->prefix . 'fcom_posts';
        
        // Build WHERE clause
        $where = array();
        $values = array();
        
        // Only filter by type if specified
        if (!empty($args['type'])) {
            $where[] = "type = %s";
            $values[] = $args['type'];
        } else {
            // Exclude course lessons and sections by default to only show regular posts
            $where[] = "(type IS NULL OR type NOT IN ('course_lesson', 'course_section'))";
        }
        
        if (!empty($args['search'])) {
            $where[] = "(title LIKE %s OR message LIKE %s)";
            $search = '%' . $wpdb->esc_like($args['search']) . '%';
            $values[] = $search;
            $values[] = $search;
        }
        
        if (!empty($args['status'])) {
            $where[] = "status = %s";
            $values[] = $args['status'];
        }
        
        if (!empty($args['space_id'])) {
            $where[] = "space_id = %d";
            $values[] = $args['space_id'];
        }
        
        if (!empty($args['user_id'])) {
            $where[] = "user_id = %d";
            $values[] = $args['user_id'];
        }
        
        $where_sql = !empty($where) ? implode(' AND ', $where) : '1=1';
        
        // Get total count
        $count_query = "SELECT COUNT(*) FROM $table WHERE $where_sql";
        if (!empty($values)) {
            $total = $wpdb->get_var($wpdb->prepare($count_query, $values));
        } else {
            $total = $wpdb->get_var($count_query);
        }
        
        // Get posts
        $offset = ($args['page'] - 1) * $args['per_page'];
        $order_by = sanitize_key($args['orderby']);
        $order = strtoupper($args['order']) === 'ASC' ? 'ASC' : 'DESC';
        
        $query = "SELECT * FROM $table WHERE $where_sql ORDER BY $order_by $order LIMIT %d OFFSET %d";
        $limit_values = array_merge($values, [$args['per_page'], $offset]);
        
        $posts = $wpdb->get_results($wpdb->prepare($query, $limit_values), ARRAY_A);
        
        return array(
            'data' => $posts ? $posts : array(),
            'total' => (int)$total,
            'pages' => ceil($total / $args['per_page']),
            'current_page' => $args['page']
        );
    }
    
    /**
     * Get comments with filters
     */
    public static function get_comments($args = array()) {
        global $wpdb;
        
        $defaults = array(
            'page' => 1,
            'per_page' => 20,
            'search' => '',
            'post_id' => '',
            'user_id' => '',
            'status' => '',
            'type' => 'comment',
            'orderby' => 'created_at',
            'order' => 'DESC'
        );
        
        $args = wp_parse_args($args, $defaults);
        $table = $wpdb->prefix . 'fcom_post_comments';
        
        // Build WHERE clause
        $where = array("type = %s");
        $values = array($args['type']);
        
        if (!empty($args['search'])) {
            $where[] = "message LIKE %s";
            $values[] = '%' . $wpdb->esc_like($args['search']) . '%';
        }
        
        if (!empty($args['post_id'])) {
            $where[] = "post_id = %d";
            $values[] = $args['post_id'];
        }
        
        if (!empty($args['user_id'])) {
            $where[] = "user_id = %d";
            $values[] = $args['user_id'];
        }
        
        if (!empty($args['status'])) {
            $where[] = "status = %s";
            $values[] = $args['status'];
        }
        
        $where_sql = implode(' AND ', $where);
        
        // Get total count
        $count_query = "SELECT COUNT(*) FROM $table WHERE $where_sql";
        $total = $wpdb->get_var($wpdb->prepare($count_query, $values));
        
        // Get comments
        $offset = ($args['page'] - 1) * $args['per_page'];
        $order_by = sanitize_key($args['orderby']);
        $order = strtoupper($args['order']) === 'ASC' ? 'ASC' : 'DESC';
        
        $query = "SELECT * FROM $table WHERE $where_sql ORDER BY $order_by $order LIMIT %d OFFSET %d";
        $values[] = $args['per_page'];
        $values[] = $offset;
        
        $comments = $wpdb->get_results($wpdb->prepare($query, $values), ARRAY_A);
        
        return array(
            'data' => $comments ? $comments : array(),
            'total' => (int)$total,
            'pages' => ceil($total / $args['per_page']),
            'current_page' => $args['page']
        );
    }
    
    /**
     * Get reactions
     */
    public static function get_reactions($args = array()) {
        global $wpdb;
        
        $defaults = array(
            'page' => 1,
            'per_page' => 20,
            'object_id' => '',
            'object_type' => '',
            'user_id' => '',
            'type' => '',
            'orderby' => 'created_at',
            'order' => 'DESC'
        );
        
        $args = wp_parse_args($args, $defaults);
        $table = $wpdb->prefix . 'fcom_post_reactions';
        
        // Build WHERE clause
        $where = array();
        $values = array();
        
        if (!empty($args['object_id'])) {
            $where[] = "object_id = %d";
            $values[] = $args['object_id'];
        }
        
        if (!empty($args['object_type'])) {
            $where[] = "object_type = %s";
            $values[] = $args['object_type'];
        }
        
        if (!empty($args['user_id'])) {
            $where[] = "user_id = %d";
            $values[] = $args['user_id'];
        }
        
        if (!empty($args['type'])) {
            $where[] = "type = %s";
            $values[] = $args['type'];
        }
        
        $where_sql = $where ? implode(' AND ', $where) : '1=1';
        
        // Get total count
        $count_query = "SELECT COUNT(*) FROM $table WHERE $where_sql";
        $total = $values ? $wpdb->get_var($wpdb->prepare($count_query, $values)) : $wpdb->get_var($count_query);
        
        // Get reactions
        $offset = ($args['page'] - 1) * $args['per_page'];
        $order_by = sanitize_key($args['orderby']);
        $order = strtoupper($args['order']) === 'ASC' ? 'ASC' : 'DESC';
        
        $query = "SELECT * FROM $table WHERE $where_sql ORDER BY $order_by $order LIMIT %d OFFSET %d";
        $values[] = $args['per_page'];
        $values[] = $offset;
        
        $reactions = $wpdb->get_results($wpdb->prepare($query, $values), ARRAY_A);
        
        return array(
            'data' => $reactions ? $reactions : array(),
            'total' => (int)$total,
            'pages' => ceil($total / $args['per_page']),
            'current_page' => $args['page']
        );
    }
    
    /**
     * Get media files
     */
    public static function get_media($args = array()) {
        global $wpdb;
        
        $defaults = array(
            'page' => 1,
            'per_page' => 20,
            'user_id' => '',
            'feed_id' => '',
            'space_id' => '',
            'search' => '',
            'media_type' => '',
            'is_active' => '',
            'orderby' => 'created_at',
            'order' => 'DESC'
        );
        
        $args = wp_parse_args($args, $defaults);
        $media_table = $wpdb->prefix . 'fcom_media_archive';
        $posts_table = $wpdb->prefix . 'fcom_posts';
        
        // Build WHERE clause
        $where = array();
        $values = array();
        $join = '';
        
        // Join with posts table if filtering by space
        if (!empty($args['space_id'])) {
            $join = "LEFT JOIN {$posts_table} p ON {$media_table}.feed_id = p.id";
            $where[] = "p.space_id = %d";
            $values[] = $args['space_id'];
        }
        
        if (!empty($args['user_id'])) {
            $where[] = "{$media_table}.user_id = %d";
            $values[] = $args['user_id'];
        }
        
        if (!empty($args['feed_id'])) {
            $where[] = "{$media_table}.feed_id = %d";
            $values[] = $args['feed_id'];
        }
        
        if (!empty($args['media_type'])) {
            $where[] = "{$media_table}.media_type = %s";
            $values[] = $args['media_type'];
        }
        
        if ($args['is_active'] !== '') {
            $where[] = "{$media_table}.is_active = %d";
            $values[] = $args['is_active'];
        }
        
        if (!empty($args['search'])) {
            $where[] = "({$media_table}.media_path LIKE %s OR {$media_table}.media_url LIKE %s)";
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $values[] = $search_term;
            $values[] = $search_term;
        }
        
        $where_sql = $where ? implode(' AND ', $where) : '1=1';
        
        // Get total count
        $count_query = "SELECT COUNT({$media_table}.id) FROM {$media_table} {$join} WHERE {$where_sql}";
        $total = $values ? $wpdb->get_var($wpdb->prepare($count_query, $values)) : $wpdb->get_var($count_query);
        
        // Get media
        $offset = ($args['page'] - 1) * $args['per_page'];
        $order_by = sanitize_key($args['orderby']);
        $order = strtoupper($args['order']) === 'ASC' ? 'ASC' : 'DESC';
        
        $query = "SELECT {$media_table}.* FROM {$media_table} {$join} WHERE {$where_sql} ORDER BY {$media_table}.{$order_by} {$order} LIMIT %d OFFSET %d";
        $values[] = $args['per_page'];
        $values[] = $offset;
        
        $media = $wpdb->get_results($wpdb->prepare($query, $values), ARRAY_A);
        
        return array(
            'data' => $media ? $media : array(),
            'total' => (int)$total,
            'pages' => ceil($total / $args['per_page']),
            'current_page' => $args['page']
        );
    }
    
    /**
     * Get single media file detail
     */
    public static function get_media_detail($media_id) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_media_archive';
        
        $media = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $media_id
        ));
        
        return $media;
    }
    
    /**
     * Delete media file
     */
    public static function delete_media($media_id) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_media_archive';
        
        return $wpdb->delete($table, array('id' => $media_id));
    }
    
    /**
     * Bulk delete media files
     */
    public static function bulk_delete_media($media_ids) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_media_archive';
        
        $placeholders = implode(',', array_fill(0, count($media_ids), '%d'));
        $query = "DELETE FROM $table WHERE id IN ($placeholders)";
        
        return $wpdb->query($wpdb->prepare($query, $media_ids));
    }
    
    /**
     * Get metadata
     */
    public static function get_metadata($args = array()) {
        global $wpdb;
        
        $defaults = array(
            'page' => 1,
            'per_page' => 20,
            'object_type' => '',
            'object_id' => '',
            'meta_key' => '',
            'orderby' => 'created_at',
            'order' => 'DESC'
        );
        
        $args = wp_parse_args($args, $defaults);
        $table = $wpdb->prefix . 'fcom_meta';
        
        // Build WHERE clause
        $where = array();
        $values = array();
        
        if (!empty($args['object_type'])) {
            $where[] = "object_type = %s";
            $values[] = $args['object_type'];
        }
        
        if (!empty($args['object_id'])) {
            $where[] = "object_id = %d";
            $values[] = $args['object_id'];
        }
        
        if (!empty($args['meta_key'])) {
            $where[] = "meta_key LIKE %s";
            $values[] = '%' . $wpdb->esc_like($args['meta_key']) . '%';
        }
        
        $where_sql = $where ? implode(' AND ', $where) : '1=1';
        
        // Get total count
        $count_query = "SELECT COUNT(*) FROM $table WHERE $where_sql";
        $total = $values ? $wpdb->get_var($wpdb->prepare($count_query, $values)) : $wpdb->get_var($count_query);
        
        // Get metadata
        $offset = ($args['page'] - 1) * $args['per_page'];
        $order_by = sanitize_key($args['orderby']);
        $order = strtoupper($args['order']) === 'ASC' ? 'ASC' : 'DESC';
        
        $query = "SELECT * FROM $table WHERE $where_sql ORDER BY $order_by $order LIMIT %d OFFSET %d";
        $values[] = $args['per_page'];
        $values[] = $offset;
        
        $metadata = $wpdb->get_results($wpdb->prepare($query, $values), ARRAY_A);
        
        return array(
            'data' => $metadata ? $metadata : array(),
            'total' => (int)$total,
            'pages' => ceil($total / $args['per_page']),
            'current_page' => $args['page']
        );
    }
    
    /**
     * Get single post
     */
    public static function get_post($id) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_posts';
        return $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $id), ARRAY_A);
    }
    
    /**
     * Update post
     */
    public static function update_post($id, $data) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_posts';
        $data['updated_at'] = current_time('mysql');
        return $wpdb->update($table, $data, array('id' => $id));
    }
    
    /**
     * Delete post
     */
    public static function delete_post($id) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_posts';
        return $wpdb->delete($table, array('id' => $id));
    }
    
    /**
     * Bulk delete posts
     */
    public static function bulk_delete_posts($ids) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_posts';
        $ids_placeholder = implode(',', array_map('absint', $ids));
        return $wpdb->query("DELETE FROM $table WHERE id IN ($ids_placeholder)");
    }
    
    /**
     * Get spaces for filters
     */
    public static function get_spaces() {
        // Use Fluent Community's official helper
        if (!class_exists('\FluentCommunity\App\Functions\Utility')) {
            return array();
        }
        
        $spaces = \FluentCommunity\App\Functions\Utility::getSpaces(false);
        $formatted_spaces = array();
        
        if (is_array($spaces) || is_object($spaces)) {
            foreach ($spaces as $space) {
                // Return space objects directly for compatibility
                if (is_object($space) && isset($space->id) && isset($space->title)) {
                    $formatted_spaces[] = $space;
                }
            }
        }
        
        return $formatted_spaces;
    }
    
    /**
     * Get spaces list with pagination and filters (excluding courses)
     */
    public static function get_spaces_list($args = array()) {
        global $wpdb;
        
        $defaults = array(
            'page' => 1,
            'per_page' => 20,
            'search' => '',
            'status' => '',
            'privacy' => '',
            'orderby' => 'created_at',
            'order' => 'DESC'
        );
        
        $args = wp_parse_args($args, $defaults);
        $table = $wpdb->prefix . 'fcom_spaces';
        
        // Build WHERE clause
        $where = array("type != 'course'"); // Exclude courses
        $values = array();
        
        if (!empty($args['search'])) {
            $where[] = "(title LIKE %s OR description LIKE %s OR slug LIKE %s)";
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $values[] = $search_term;
            $values[] = $search_term;
            $values[] = $search_term;
        }
        
        if (!empty($args['status'])) {
            $where[] = "status = %s";
            $values[] = $args['status'];
        }
        
        if (!empty($args['privacy'])) {
            $where[] = "privacy = %s";
            $values[] = $args['privacy'];
        }
        
        $where_sql = implode(' AND ', $where);
        
        // Get total count
        $count_query = "SELECT COUNT(*) FROM $table WHERE $where_sql";
        $total = $values ? $wpdb->get_var($wpdb->prepare($count_query, $values)) : $wpdb->get_var($count_query);
        
        // Get spaces
        $offset = ($args['page'] - 1) * $args['per_page'];
        $order_by = sanitize_key($args['orderby']);
        $order = strtoupper($args['order']) === 'ASC' ? 'ASC' : 'DESC';
        
        $query = "SELECT * FROM $table WHERE $where_sql ORDER BY $order_by $order LIMIT %d OFFSET %d";
        $values[] = $args['per_page'];
        $values[] = $offset;
        
        $spaces = $wpdb->get_results($wpdb->prepare($query, $values), ARRAY_A);
        
        return array(
            'data' => $spaces ? $spaces : array(),
            'total' => (int)$total,
            'pages' => ceil($total / $args['per_page']),
            'current_page' => $args['page']
        );
    }
    
    /**
     * Get users for filters
     */
    public static function get_users_with_content() {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_posts';
        return $wpdb->get_col("SELECT DISTINCT user_id FROM $table WHERE user_id IS NOT NULL ORDER BY user_id ASC");
    }
    
    /**
     * Get courses (from spaces table where type = 'course')
     */
    public static function get_courses($args = array()) {
        // Use Fluent Community helper
        if (class_exists('\FluentCommunity\App\Functions\Utility')) {
            try {
                $courses = \FluentCommunity\App\Functions\Utility::getCourses(false);
                
                if ($courses) {
                    $coursesArray = [];
                    foreach ($courses as $course) {
                        $coursesArray[] = [
                            'id' => $course->id,
                            'title' => $course->title,
                            'slug' => $course->slug,
                            'description' => $course->description ?? '',
                            'status' => $course->status,
                            'type' => $course->type,
                            'created_at' => $course->created_at,
                            'updated_at' => $course->updated_at
                        ];
                    }
                    return $coursesArray;
                }
            } catch (\Exception $e) {
                // Fall through to direct query
            }
        }
        
        // Fallback: Get from spaces table where type = 'course'
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_spaces';
        
        $where = ["type = 'course'"];
        $values = [];
        
        if (!empty($args['search'])) {
            $where[] = "(title LIKE %s OR slug LIKE %s)";
            $search = '%' . $wpdb->esc_like($args['search']) . '%';
            $values[] = $search;
            $values[] = $search;
        }
        
        if (!empty($args['status'])) {
            $where[] = "status = %s";
            $values[] = $args['status'];
        }
        
        $where_sql = implode(' AND ', $where);
        
        $query = "SELECT * FROM $table WHERE $where_sql ORDER BY title ASC";
        
        if (!empty($values)) {
            return $wpdb->get_results($wpdb->prepare($query, $values), ARRAY_A);
        } else {
            return $wpdb->get_results($query, ARRAY_A);
        }
    }
    
    /**
     * Get course by ID
     */
    public static function get_course($course_id) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_spaces';
        
        $query = "SELECT * FROM $table WHERE id = %d AND type = 'course'";
        return $wpdb->get_row($wpdb->prepare($query, $course_id), ARRAY_A);
    }
    
    /**
     * Get course structure (sections and lessons)
     */
    public static function get_course_structure($course_id) {
        global $wpdb;
        $posts_table = $wpdb->prefix . 'fcom_posts';
        
        // Get all sections and lessons for this course
        $query = "SELECT * FROM $posts_table 
                  WHERE space_id = %d 
                  AND type IN ('course_section', 'course_lesson')
                  ORDER BY priority ASC, created_at ASC";
        
        $items = $wpdb->get_results($wpdb->prepare($query, $course_id), ARRAY_A);
        
        // Organize into structure
        $structure = [
            'sections' => [],
            'lessons' => []
        ];
        
        foreach ($items as $item) {
            if ($item['type'] === 'course_section') {
                $structure['sections'][] = $item;
            } else if ($item['type'] === 'course_lesson') {
                $structure['lessons'][] = $item;
            }
        }
        
        return $structure;
    }
    
    /**
     * Update item priority/order
     */
    public static function update_item_priority($item_id, $priority) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_posts';
        
        return $wpdb->update(
            $table,
            ['priority' => $priority],
            ['id' => $item_id]
        );
    }
    
    /**
     * Get comments for a specific post
     */
    public static function get_post_comments($post_id) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_post_comments';
        
        $query = "SELECT * FROM $table WHERE post_id = %d ORDER BY created_at DESC";
        return $wpdb->get_results($wpdb->prepare($query, $post_id), ARRAY_A);
    }
    
    /**
     * Get reactions for a specific post
     */
    public static function get_post_reactions($post_id) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_post_reactions';
        
        // Reactions use object_id and object_type, not post_id
        $query = "SELECT * FROM $table WHERE object_id = %d AND object_type = 'feed' ORDER BY created_at DESC";
        return $wpdb->get_results($wpdb->prepare($query, $post_id), ARRAY_A);
    }
    
    /**
     * Get media for a specific post
     */
    public static function get_post_media($post_id) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_media_archive';
        
        $query = "SELECT * FROM $table WHERE feed_id = %d ORDER BY created_at DESC";
        return $wpdb->get_results($wpdb->prepare($query, $post_id), ARRAY_A);
    }
    
    /**
     * Get terms/topics for a specific post
     */
    public static function get_post_terms($post_id) {
        global $wpdb;
        $term_feed_table = $wpdb->prefix . 'fcom_term_feed';
        $terms_table = $wpdb->prefix . 'fcom_terms';
        
        $query = "SELECT t.* 
                  FROM $terms_table t
                  INNER JOIN $term_feed_table tf ON t.id = tf.term_id
                  WHERE tf.post_id = %d
                  ORDER BY t.title ASC";
        
        return $wpdb->get_results($wpdb->prepare($query, $post_id), ARRAY_A);
    }
    
    /**
     * Get all available terms/topics
     */
    public static function get_all_terms() {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_terms';
        
        $query = "SELECT * FROM $table ORDER BY title ASC";
        return $wpdb->get_results($query, ARRAY_A);
    }
    
    /**
     * Add term to post
     */
    public static function add_term_to_post($post_id, $term_id) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_term_feed';
        
        // Check if already exists
        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table WHERE post_id = %d AND term_id = %d",
            $post_id, $term_id
        ));
        
        if ($exists) {
            return true;
        }
        
        return $wpdb->insert($table, array(
            'post_id' => $post_id,
            'term_id' => $term_id
        ));
    }
    
    /**
     * Remove term from post
     */
    public static function remove_term_from_post($post_id, $term_id) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_term_feed';
        
        return $wpdb->delete($table, array(
            'post_id' => $post_id,
            'term_id' => $term_id
        ));
    }
    
    /**
     * Bulk update posts
     */
    public static function bulk_update_posts($post_ids, $data) {
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_posts';
        
        if (empty($post_ids) || empty($data)) {
            return false;
        }
        
        // Build SET clause
        $set_parts = array();
        $values = array();
        
        foreach ($data as $key => $value) {
            $set_parts[] = "$key = %s";
            $values[] = $value;
        }
        
        $set_clause = implode(', ', $set_parts);
        $ids_placeholder = implode(',', array_map('absint', $post_ids));
        
        $query = "UPDATE $table SET $set_clause WHERE id IN ($ids_placeholder)";
        
        return $wpdb->query($wpdb->prepare($query, $values));
    }
    
    /**
     * Get members with extended profile data
     */
    public static function get_members($args = array()) {
        global $wpdb;
        
        $defaults = array(
            'page' => 1,
            'per_page' => 20,
            'search' => '',
            'status' => '',
            'role' => '',
            'space_id' => '',
            'orderby' => 'display_name',
            'order' => 'ASC'
        );
        
        $args = wp_parse_args($args, $defaults);
        $offset = ($args['page'] - 1) * $args['per_page'];
        
        $xprofile_table = $wpdb->prefix . 'fcom_xprofile';
        $users_table = $wpdb->users;
        
        // Build query
        $where = array('1=1');
        $values = array();
        
        if (!empty($args['search'])) {
            $where[] = '(xp.display_name LIKE %s OR xp.username LIKE %s OR u.user_email LIKE %s)';
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $values[] = $search_term;
            $values[] = $search_term;
            $values[] = $search_term;
        }
        
        if (!empty($args['status'])) {
            $where[] = 'xp.status = %s';
            $values[] = $args['status'];
        }
        
        // Filter by space membership
        if (!empty($args['space_id'])) {
            $space_user_table = $wpdb->prefix . 'fcom_space_user';
            $where[] = "xp.user_id IN (SELECT user_id FROM $space_user_table WHERE space_id = %d)";
            $values[] = intval($args['space_id']);
        }
        
        // Filter by role in any space
        if (!empty($args['role'])) {
            $space_user_table = $wpdb->prefix . 'fcom_space_user';
            $where[] = "xp.user_id IN (SELECT user_id FROM $space_user_table WHERE role = %s)";
            $values[] = $args['role'];
        }
        
        $where_sql = implode(' AND ', $where);
        
        $allowed_orderby = array('display_name', 'username', 'total_points', 'last_activity', 'created_at');
        $orderby = in_array($args['orderby'], $allowed_orderby) ? $args['orderby'] : 'display_name';
        $order = in_array(strtoupper($args['order']), array('ASC', 'DESC')) ? strtoupper($args['order']) : 'ASC';
        
        $query = "
            SELECT 
                xp.*,
                u.user_email,
                u.user_registered,
                (SELECT COUNT(*) FROM {$wpdb->prefix}fcom_space_user WHERE user_id = xp.user_id) as enrollments_count
            FROM $xprofile_table xp
            LEFT JOIN $users_table u ON xp.user_id = u.ID
            WHERE $where_sql
            ORDER BY xp.$orderby $order
            LIMIT %d OFFSET %d
        ";
        
        $values[] = intval($args['per_page']);
        $values[] = $offset;
        
        if (!empty($values)) {
            $query = $wpdb->prepare($query, $values);
        }
        
        $members = $wpdb->get_results($query, ARRAY_A);
        
        // Get total count
        $count_query = "SELECT COUNT(*) FROM $xprofile_table xp WHERE $where_sql";
        if (count($values) > 2) { // Has filters
            $count_values = array_slice($values, 0, -2); // Remove limit and offset
            $count_query = $wpdb->prepare($count_query, $count_values);
        }
        $total = $wpdb->get_var($count_query);
        
        return array(
            'members' => $members,
            'total' => intval($total),
            'pages' => ceil($total / $args['per_page'])
        );
    }
    
    /**
     * Get single member with all details
     */
    public static function get_member($user_id) {
        global $wpdb;
        
        $xprofile_table = $wpdb->prefix . 'fcom_xprofile';
        $users_table = $wpdb->users;
        
        $member = $wpdb->get_row($wpdb->prepare("
            SELECT 
                xp.*,
                u.user_email,
                u.user_login,
                u.user_registered
            FROM $xprofile_table xp
            LEFT JOIN $users_table u ON xp.user_id = u.ID
            WHERE xp.user_id = %d
        ", $user_id), ARRAY_A);
        
        if (!$member) {
            return null;
        }
        
        // Get space enrollments
        $space_user_table = $wpdb->prefix . 'fcom_space_user';
        $spaces_table = $wpdb->prefix . 'fcom_spaces';
        
        $enrollments = $wpdb->get_results($wpdb->prepare("
            SELECT 
                su.id,
                su.space_id,
                su.role,
                su.status,
                su.created_at,
                s.title as space_title,
                s.slug as space_slug,
                s.type as space_type
            FROM $space_user_table su
            LEFT JOIN $spaces_table s ON su.space_id = s.id
            WHERE su.user_id = %d
            ORDER BY su.created_at DESC
        ", $user_id), ARRAY_A);
        
        $member['enrollments'] = $enrollments;
        
        // Decode meta if it exists
        if (!empty($member['meta'])) {
            $member['meta'] = maybe_unserialize($member['meta']);
        }
        
        return $member;
    }
    
    /**
     * Update member profile
     */
    public static function update_member($user_id, $data) {
        global $wpdb;
        
        $xprofile_table = $wpdb->prefix . 'fcom_xprofile';
        
        $update_data = array();
        
        // Only update allowed fields
        $allowed_fields = array('status', 'display_name', 'username', 'short_description', 'total_points', 'is_verified');
        
        foreach ($allowed_fields as $field) {
            if (isset($data[$field])) {
                $update_data[$field] = $data[$field];
            }
        }
        
        if (isset($data['meta'])) {
            $update_data['meta'] = maybe_serialize($data['meta']);
        }
        
        if (empty($update_data)) {
            return false;
        }
        
        $update_data['updated_at'] = current_time('mysql');
        
        return $wpdb->update(
            $xprofile_table,
            $update_data,
            array('user_id' => $user_id)
        );
    }
    
    /**
     * Add enrollment (space/course membership)
     */
    public static function add_enrollment($user_id, $space_id, $role = 'member') {
        global $wpdb;
        
        $table = $wpdb->prefix . 'fcom_space_user';
        
        // Check if already enrolled
        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table WHERE user_id = %d AND space_id = %d",
            $user_id,
            $space_id
        ));
        
        if ($exists) {
            return false; // Already enrolled
        }
        
        return $wpdb->insert($table, array(
            'user_id' => $user_id,
            'space_id' => $space_id,
            'role' => $role,
            'status' => 'active',
            'created_at' => current_time('mysql'),
            'updated_at' => current_time('mysql')
        ));
    }
    
    /**
     * Remove enrollment
     */
    public static function remove_enrollment($user_id, $space_id) {
        global $wpdb;
        
        $table = $wpdb->prefix . 'fcom_space_user';
        
        return $wpdb->delete($table, array(
            'user_id' => $user_id,
            'space_id' => $space_id
        ));
    }
    
    /**
     * Update enrollment role
     */
    public static function update_enrollment_role($user_id, $space_id, $role) {
        global $wpdb;
        
        $table = $wpdb->prefix . 'fcom_space_user';
        
        return $wpdb->update(
            $table,
            array('role' => $role, 'updated_at' => current_time('mysql')),
            array('user_id' => $user_id, 'space_id' => $space_id)
        );
    }
    
    /**
     * Bulk add enrollments
     */
    public static function bulk_add_enrollments($user_ids, $space_id, $role = 'member') {
        $success = 0;
        foreach ($user_ids as $user_id) {
            if (self::add_enrollment($user_id, $space_id, $role)) {
                $success++;
            }
        }
        return $success;
    }
    
    /**
     * Bulk remove enrollments
     */
    public static function bulk_remove_enrollments($user_ids, $space_id) {
        $success = 0;
        foreach ($user_ids as $user_id) {
            if (self::remove_enrollment($user_id, $space_id)) {
                $success++;
            }
        }
        return $success;
    }
}
