<?php
/**
 * Ajax Class
 * Handles all AJAX requests
 */

// Prevent direct access
defined('ABSPATH') or die('No direct script access allowed');

class FCA_CM_Ajax {
    
    /**
     * Instance of this class
     */
    private static $instance = null;
    
    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Posts
        add_action('wp_ajax_fca_cm_get_posts', array($this, 'get_posts'));
        add_action('wp_ajax_fca_cm_get_post', array($this, 'get_post'));
        add_action('wp_ajax_fca_cm_update_post', array($this, 'update_post'));
        add_action('wp_ajax_fca_cm_delete_post', array($this, 'delete_post'));
        add_action('wp_ajax_fca_cm_bulk_delete_posts', array($this, 'bulk_delete_posts'));
        
        // Comments
        add_action('wp_ajax_fca_cm_get_comments', array($this, 'get_comments'));
        add_action('wp_ajax_fca_cm_delete_comment', array($this, 'delete_comment'));
        add_action('wp_ajax_fca_cm_bulk_delete_comments', array($this, 'bulk_delete_comments'));
        
        // Reactions
        add_action('wp_ajax_fca_cm_get_reactions', array($this, 'get_reactions'));
        add_action('wp_ajax_fca_cm_delete_reaction', array($this, 'delete_reaction'));
        
        // Media
        add_action('wp_ajax_fca_cm_get_media', array($this, 'get_media'));
        add_action('wp_ajax_fca_cm_get_media_detail', array($this, 'get_media_detail'));
        add_action('wp_ajax_fca_cm_delete_media', array($this, 'delete_media'));
        add_action('wp_ajax_fca_cm_bulk_delete_media', array($this, 'bulk_delete_media'));
        
        // Metadata
        add_action('wp_ajax_fca_cm_get_metadata', array($this, 'get_metadata'));
        add_action('wp_ajax_fca_cm_update_metadata', array($this, 'update_metadata'));
        add_action('wp_ajax_fca_cm_delete_metadata', array($this, 'delete_metadata'));
        
        // Terms/Topics
        add_action('wp_ajax_fca_cm_get_all_terms', array($this, 'get_all_terms'));
        add_action('wp_ajax_fca_cm_add_term_to_post', array($this, 'add_term_to_post'));
        add_action('wp_ajax_fca_cm_remove_term_from_post', array($this, 'remove_term_from_post'));
        
        // Bulk Operations
        add_action('wp_ajax_fca_cm_bulk_update_posts', array($this, 'bulk_update_posts'));
        add_action('wp_ajax_fca_cm_bulk_add_term', array($this, 'bulk_add_term'));
        
        // Comment detail
        add_action('wp_ajax_fca_cm_get_comment', array($this, 'get_comment'));
        add_action('wp_ajax_fca_cm_update_comment', array($this, 'update_comment'));
        
        // Courses
        add_action('wp_ajax_fca_cm_get_courses', array($this, 'get_courses'));
        add_action('wp_ajax_fca_cm_get_course', array($this, 'get_course'));
        add_action('wp_ajax_fca_cm_get_course_structure', array($this, 'get_course_structure'));
        add_action('wp_ajax_fca_cm_update_course', array($this, 'update_course'));
        add_action('wp_ajax_fca_cm_bulk_update_courses', array($this, 'bulk_update_courses'));
        add_action('wp_ajax_fca_cm_update_course_order', array($this, 'update_course_order'));
        add_action('wp_ajax_fca_cm_delete_course_item', array($this, 'delete_course_item'));
        add_action('wp_ajax_fca_cm_duplicate_course_item', array($this, 'duplicate_course_item'));
        add_action('wp_ajax_fca_cm_move_course_item', array($this, 'move_course_item'));
        
        // Spaces
        add_action('wp_ajax_fca_cm_get_spaces_list', array($this, 'get_spaces_list'));
        add_action('wp_ajax_fca_cm_get_space', array($this, 'get_space'));
        add_action('wp_ajax_fca_cm_update_space', array($this, 'update_space'));
        add_action('wp_ajax_fca_cm_bulk_update_spaces', array($this, 'bulk_update_spaces'));
        add_action('wp_ajax_fca_cm_bulk_delete_spaces', array($this, 'bulk_delete_spaces'));
        
        // Overview Statistics
        add_action('wp_ajax_fca_cm_get_stats', array($this, 'get_stats'));
        
        // Import/Export
        add_action('wp_ajax_fca_cm_export_posts', array($this, 'export_posts'));
        add_action('wp_ajax_fca_cm_export_spaces', array($this, 'export_spaces'));
        add_action('wp_ajax_fca_cm_export_comments', array($this, 'export_comments'));
        add_action('wp_ajax_fca_cm_export_reactions', array($this, 'export_reactions'));
        add_action('wp_ajax_fca_cm_export_term_feed', array($this, 'export_term_feed'));
        add_action('wp_ajax_fca_cm_export_space_users', array($this, 'export_space_users'));
        add_action('wp_ajax_fca_cm_upload_import_file', array($this, 'upload_import_file'));
        add_action('wp_ajax_fca_cm_process_import', array($this, 'process_import'));
        
        // Members Actions
        add_action('wp_ajax_fca_cm_get_members', array($this, 'get_members'));
        add_action('wp_ajax_fca_cm_get_member', array($this, 'get_member'));
        add_action('wp_ajax_fca_cm_update_member', array($this, 'update_member'));
        add_action('wp_ajax_fca_cm_add_enrollment', array($this, 'add_enrollment'));
        add_action('wp_ajax_fca_cm_remove_enrollment', array($this, 'remove_enrollment'));
        add_action('wp_ajax_fca_cm_bulk_add_enrollments', array($this, 'bulk_add_enrollments'));
        add_action('wp_ajax_fca_cm_bulk_remove_enrollments', array($this, 'bulk_remove_enrollments'));
    }
    
    /**
     * Verify nonce
     */
    private function verify_nonce() {
        if (!check_ajax_referer('fca_cm_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => __('Security check failed.', 'fca-content-manager')));
        }
    }
    
    /**
     * Get posts
     */
    public function get_posts() {
        $this->verify_nonce();
        
        $args = array(
            'page' => isset($_POST['page']) ? intval($_POST['page']) : 1,
            'per_page' => isset($_POST['per_page']) ? intval($_POST['per_page']) : 20,
            'search' => isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '',
            'type' => isset($_POST['type']) ? sanitize_text_field($_POST['type']) : '',  // Empty = show all
            'status' => isset($_POST['status']) ? sanitize_text_field($_POST['status']) : '',
            'space_id' => isset($_POST['space_id']) ? intval($_POST['space_id']) : '',
            'user_id' => isset($_POST['user_id']) ? intval($_POST['user_id']) : '',
            'orderby' => isset($_POST['orderby']) ? sanitize_text_field($_POST['orderby']) : 'created_at',
            'order' => isset($_POST['order']) ? sanitize_text_field($_POST['order']) : 'DESC'
        );
        
        $result = FCA_CM_Database::get_posts($args);
        
        // Enhance data with user info
        if (!empty($result['data'])) {
            foreach ($result['data'] as &$post) {
                $post['author'] = $this->get_user_display_name($post['user_id'] ?? null);
                $post['space_name'] = $this->get_space_name($post['space_id'] ?? null);
                
                // Format dates
                if (isset($post['created_at'])) {
                    $post['created_at_formatted'] = date('M j, Y g:i A', strtotime($post['created_at']));
                }
                
                // Decode meta if it's JSON
                if (isset($post['meta']) && is_string($post['meta'])) {
                    $post['meta'] = json_decode($post['meta'], true);
                }
            }
        }
        
        // Add debug info
        $result['debug'] = array(
            'query_args' => $args,
            'count' => count($result['data']),
            'total' => $result['total']
        );
        
        wp_send_json_success($result);
    }
    
    /**
     * Get single post
     */
    public function get_post() {
        $this->verify_nonce();
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if (!$post_id) {
            wp_send_json_error(array('message' => __('Invalid post ID.', 'fca-content-manager')));
        }
        
        $post = FCA_CM_Database::get_post($post_id);
        
        if (!$post) {
            wp_send_json_error(array('message' => __('Post not found.', 'fca-content-manager')));
        }
        
        // Enhance data
        $post['author'] = $this->get_user_display_name($post['user_id'] ?? null);
        $post['space_name'] = $this->get_space_name($post['space_id'] ?? null);
        
        // Format dates
        if (isset($post['created_at'])) {
            $post['created_at_formatted'] = date('M j, Y g:i A', strtotime($post['created_at']));
        }
        
        // Decode meta if it's JSON
        if (isset($post['meta']) && is_string($post['meta'])) {
            $post['meta'] = json_decode($post['meta'], true);
        }
        
        // Get related data (comments, reactions, media, terms)
        $post['related_comments'] = FCA_CM_Database::get_post_comments($post_id);
        $post['related_reactions'] = FCA_CM_Database::get_post_reactions($post_id);
        $post['related_media'] = FCA_CM_Database::get_post_media($post_id);
        $post['related_terms'] = FCA_CM_Database::get_post_terms($post_id);
        
        // Enhance comments with user names
        if (!empty($post['related_comments'])) {
            foreach ($post['related_comments'] as &$comment) {
                $comment['author'] = $this->get_user_display_name($comment['user_id'] ?? null);
                $comment['created_at_formatted'] = date('M j, Y g:i A', strtotime($comment['created_at']));
            }
        }
        
        // Enhance reactions with user names
        if (!empty($post['related_reactions'])) {
            foreach ($post['related_reactions'] as &$reaction) {
                $reaction['author'] = $this->get_user_display_name($reaction['user_id'] ?? null);
                $reaction['created_at_formatted'] = date('M j, Y g:i A', strtotime($reaction['created_at']));
            }
        }
        
        wp_send_json_success($post);
    }
    
    /**
     * Update post
     */
    public function update_post() {
        $this->verify_nonce();
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if (!$post_id) {
            wp_send_json_error(array('message' => __('Invalid post ID.', 'fca-content-manager')));
        }
        
        $data = array();
        
        if (isset($_POST['title'])) {
            $data['title'] = sanitize_text_field($_POST['title']);
        }
        
        if (isset($_POST['message'])) {
            $data['message'] = wp_kses_post($_POST['message']);
        }
        
        if (isset($_POST['status'])) {
            $data['status'] = sanitize_text_field($_POST['status']);
        }
        
        if (isset($_POST['privacy'])) {
            $data['privacy'] = sanitize_text_field($_POST['privacy']);
        }
        
        if (isset($_POST['is_sticky'])) {
            $data['is_sticky'] = intval($_POST['is_sticky']);
        }
        
        // Metadata fields
        if (isset($_POST['space_id'])) {
            $data['space_id'] = $_POST['space_id'] ? intval($_POST['space_id']) : null;
        }
        
        if (isset($_POST['user_id'])) {
            $data['user_id'] = $_POST['user_id'] ? intval($_POST['user_id']) : null;
        }
        
        if (isset($_POST['created_at']) && !empty($_POST['created_at'])) {
            // Convert from datetime-local format (YYYY-MM-DDTHH:MM) to MySQL datetime
            $data['created_at'] = str_replace('T', ' ', sanitize_text_field($_POST['created_at'])) . ':00';
        }
        
        if (isset($_POST['updated_at']) && !empty($_POST['updated_at'])) {
            // Convert from datetime-local format (YYYY-MM-DDTHH:MM) to MySQL datetime
            $data['updated_at'] = str_replace('T', ' ', sanitize_text_field($_POST['updated_at'])) . ':00';
        }
        
        $result = FCA_CM_Database::update_post($post_id, $data);
        
        if ($result !== false) {
            wp_send_json_success(array('message' => __('Post updated successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to update post.', 'fca-content-manager')));
        }
    }
    
    /**
     * Delete post
     */
    public function delete_post() {
        $this->verify_nonce();
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if (!$post_id) {
            wp_send_json_error(array('message' => __('Invalid post ID.', 'fca-content-manager')));
        }
        
        $result = FCA_CM_Database::delete_post($post_id);
        
        if ($result) {
            wp_send_json_success(array('message' => __('Post deleted successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to delete post.', 'fca-content-manager')));
        }
    }
    
    /**
     * Bulk delete posts
     */
    public function bulk_delete_posts() {
        $this->verify_nonce();
        
        $post_ids = isset($_POST['post_ids']) ? array_map('intval', $_POST['post_ids']) : array();
        
        if (empty($post_ids)) {
            wp_send_json_error(array('message' => __('No posts selected.', 'fca-content-manager')));
        }
        
        $result = FCA_CM_Database::bulk_delete_posts($post_ids);
        
        if ($result) {
            wp_send_json_success(array('message' => sprintf(__('%d posts deleted successfully.', 'fca-content-manager'), count($post_ids))));
        } else {
            wp_send_json_error(array('message' => __('Failed to delete posts.', 'fca-content-manager')));
        }
    }
    
    /**
     * Get comments
     */
    public function get_comments() {
        $this->verify_nonce();
        
        $args = array(
            'page' => isset($_POST['page']) ? intval($_POST['page']) : 1,
            'per_page' => isset($_POST['per_page']) ? intval($_POST['per_page']) : 20,
            'search' => isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '',
            'post_id' => isset($_POST['post_id']) ? intval($_POST['post_id']) : '',
            'user_id' => isset($_POST['user_id']) ? intval($_POST['user_id']) : '',
            'status' => isset($_POST['status']) ? sanitize_text_field($_POST['status']) : '',
            'orderby' => isset($_POST['orderby']) ? sanitize_text_field($_POST['orderby']) : 'created_at',
            'order' => isset($_POST['order']) ? sanitize_text_field($_POST['order']) : 'DESC'
        );
        
        $result = FCA_CM_Database::get_comments($args);
        
        // Enhance data
        foreach ($result['data'] as &$comment) {
            $comment['author'] = $this->get_user_display_name($comment['user_id']);
        }
        
        wp_send_json_success($result);
    }
    
    /**
     * Delete comment
     */
    public function delete_comment() {
        $this->verify_nonce();
        
        // Check license for comment editing
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('comments_edit')) {
            $license_checker->send_license_required_error('comments_edit');
        }
        
        $comment_id = isset($_POST['comment_id']) ? intval($_POST['comment_id']) : 0;
        
        if (!$comment_id) {
            wp_send_json_error(array('message' => __('Invalid comment ID.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_post_comments';
        $result = $wpdb->delete($table, array('id' => $comment_id));
        
        if ($result) {
            wp_send_json_success(array('message' => __('Comment deleted successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to delete comment.', 'fca-content-manager')));
        }
    }
    
    /**
     * Bulk delete comments
     */
    public function bulk_delete_comments() {
        $this->verify_nonce();
        
        // Check license for comment editing
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('comments_edit')) {
            $license_checker->send_license_required_error('comments_edit');
        }
        
        $comment_ids = isset($_POST['comment_ids']) ? array_map('intval', $_POST['comment_ids']) : array();
        
        if (empty($comment_ids)) {
            wp_send_json_error(array('message' => __('No comments selected.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_post_comments';
        $placeholders = implode(',', array_fill(0, count($comment_ids), '%d'));
        $result = $wpdb->query($wpdb->prepare("DELETE FROM $table WHERE id IN ($placeholders)", $comment_ids));
        
        if ($result) {
            wp_send_json_success(array('message' => sprintf(__('%d comments deleted successfully.', 'fca-content-manager'), $result)));
        } else {
            wp_send_json_error(array('message' => __('Failed to delete comments.', 'fca-content-manager')));
        }
    }
    
    /**
     * Get reactions
     */
    public function get_reactions() {
        $this->verify_nonce();
        
        $args = array(
            'page' => isset($_POST['page']) ? intval($_POST['page']) : 1,
            'per_page' => isset($_POST['per_page']) ? intval($_POST['per_page']) : 20,
            'object_id' => isset($_POST['object_id']) ? intval($_POST['object_id']) : '',
            'object_type' => isset($_POST['object_type']) ? sanitize_text_field($_POST['object_type']) : '',
            'user_id' => isset($_POST['user_id']) ? intval($_POST['user_id']) : '',
            'type' => isset($_POST['type']) ? sanitize_text_field($_POST['type']) : '',
            'orderby' => isset($_POST['orderby']) ? sanitize_text_field($_POST['orderby']) : 'created_at',
            'order' => isset($_POST['order']) ? sanitize_text_field($_POST['order']) : 'DESC'
        );
        
        $result = FCA_CM_Database::get_reactions($args);
        
        // Enhance data
        foreach ($result['data'] as &$reaction) {
            $reaction['author'] = $this->get_user_display_name($reaction['user_id']);
        }
        
        wp_send_json_success($result);
    }
    
    /**
     * Delete reaction
     */
    public function delete_reaction() {
        $this->verify_nonce();
        
        $reaction_id = isset($_POST['reaction_id']) ? intval($_POST['reaction_id']) : 0;
        
        if (!$reaction_id) {
            wp_send_json_error(array('message' => __('Invalid reaction ID.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_post_reactions';
        $result = $wpdb->delete($table, array('id' => $reaction_id));
        
        if ($result) {
            wp_send_json_success(array('message' => __('Reaction deleted successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to delete reaction.', 'fca-content-manager')));
        }
    }
    
    /**
     * Get media
     */
    public function get_media() {
        $this->verify_nonce();
        
        // Check license for media access
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('media')) {
            $license_checker->send_license_required_error('media');
        }
        
        $args = array(
            'page' => isset($_POST['page']) ? intval($_POST['page']) : 1,
            'per_page' => isset($_POST['per_page']) ? intval($_POST['per_page']) : 20,
            'search' => isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '',
            'space_id' => isset($_POST['space_id']) ? intval($_POST['space_id']) : '',
            'user_id' => isset($_POST['user_id']) ? intval($_POST['user_id']) : '',
            'feed_id' => isset($_POST['feed_id']) ? intval($_POST['feed_id']) : '',
            'media_type' => isset($_POST['media_type']) ? sanitize_text_field($_POST['media_type']) : '',
            'is_active' => isset($_POST['is_active']) ? intval($_POST['is_active']) : '',
            'orderby' => isset($_POST['orderby']) ? sanitize_text_field($_POST['orderby']) : 'created_at',
            'order' => isset($_POST['order']) ? sanitize_text_field($_POST['order']) : 'DESC'
        );
        
        $result = FCA_CM_Database::get_media($args);
        
        // Enhance data (note: get_media returns ARRAY_A)
        foreach ($result['data'] as &$media) {
            $media['author'] = $this->get_user_display_name($media['user_id'] ?? null);
        }
        
        wp_send_json_success($result);
    }
    
    /**
     * Delete media
     */
    public function delete_media() {
        $this->verify_nonce();
        
        $media_id = isset($_POST['media_id']) ? intval($_POST['media_id']) : 0;
        
        if (!$media_id) {
            wp_send_json_error(array('message' => __('Invalid media ID.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_media_archive';
        $result = $wpdb->delete($table, array('id' => $media_id));
        
        if ($result) {
            wp_send_json_success(array('message' => __('Media deleted successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to delete media.', 'fca-content-manager')));
        }
    }
    
    /**
     * Get media detail
     */
    public function get_media_detail() {
        $this->verify_nonce();
        
        // Check license for media access
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('media')) {
            $license_checker->send_license_required_error('media');
        }
        
        $media_id = isset($_POST['media_id']) ? intval($_POST['media_id']) : 0;
        
        if (!$media_id) {
            wp_send_json_error(array('message' => __('Invalid media ID.', 'fca-content-manager')));
        }
        
        $media = FCA_CM_Database::get_media_detail($media_id);
        
        if (!$media) {
            wp_send_json_error(array('message' => __('Media not found.', 'fca-content-manager')));
        }
        
        // Convert to array for consistency
        $media = (array) $media;
        
        // Enhance data
        $media['author'] = $this->get_user_display_name($media['user_id'] ?? null);
        $media['space_name'] = !empty($media['feed_id']) ? $this->get_space_name_by_feed($media['feed_id']) : '';
        
        // Get usage information
        $media['used_in'] = array();
        
        if (!empty($media['feed_id'])) {
            global $wpdb;
            $posts_table = $wpdb->prefix . 'fcom_posts';
            $spaces_table = $wpdb->prefix . 'fcom_spaces';
            
            $post = $wpdb->get_row($wpdb->prepare(
                "SELECT p.*, s.title as space_title, s.type as space_type 
                 FROM {$posts_table} p
                 LEFT JOIN {$spaces_table} s ON p.space_id = s.id
                 WHERE p.id = %d",
                $media['feed_id']
            ), ARRAY_A);
            
            if ($post) {
                $media['used_in'][] = array(
                    'type' => $post['type'],
                    'id' => $post['id'],
                    'title' => $post['title'] ?: 'Untitled',
                    'space' => $post['space_title'],
                    'space_type' => $post['space_type'],
                    'url' => $this->get_content_url($post)
                );
            }
        }
        
        wp_send_json_success($media);
    }
    
    /**
     * Bulk delete media
     */
    public function bulk_delete_media() {
        $this->verify_nonce();
        
        // Check license for media access
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('media')) {
            $license_checker->send_license_required_error('media');
        }
        
        $media_ids = isset($_POST['media_ids']) ? array_map('intval', $_POST['media_ids']) : array();
        
        if (empty($media_ids)) {
            wp_send_json_error(array('message' => __('No media selected.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_media_archive';
        
        $placeholders = implode(',', array_fill(0, count($media_ids), '%d'));
        $query = "DELETE FROM $table WHERE id IN ($placeholders)";
        $result = $wpdb->query($wpdb->prepare($query, $media_ids));
        
        if ($result) {
            wp_send_json_success(array(
                'message' => sprintf(__('%d media items deleted successfully.', 'fca-content-manager'), $result)
            ));
        } else {
            wp_send_json_error(array('message' => __('Failed to delete media.', 'fca-content-manager')));
        }
    }
    
    /**
     * Get metadata
     */
    public function get_metadata() {
        $this->verify_nonce();
        
        $args = array(
            'page' => isset($_POST['page']) ? intval($_POST['page']) : 1,
            'per_page' => isset($_POST['per_page']) ? intval($_POST['per_page']) : 20,
            'object_type' => isset($_POST['object_type']) ? sanitize_text_field($_POST['object_type']) : '',
            'object_id' => isset($_POST['object_id']) ? intval($_POST['object_id']) : '',
            'meta_key' => isset($_POST['meta_key']) ? sanitize_text_field($_POST['meta_key']) : '',
            'orderby' => isset($_POST['orderby']) ? sanitize_text_field($_POST['orderby']) : 'created_at',
            'order' => isset($_POST['order']) ? sanitize_text_field($_POST['order']) : 'DESC'
        );
        
        $result = FCA_CM_Database::get_metadata($args);
        
        wp_send_json_success($result);
    }
    
    /**
     * Update metadata
     */
    public function update_metadata() {
        $this->verify_nonce();
        
        // Check license for metadata access
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('metadata')) {
            $license_checker->send_license_required_error('metadata');
        }
        
        $meta_id = isset($_POST['meta_id']) ? intval($_POST['meta_id']) : 0;
        $value = isset($_POST['value']) ? $_POST['value'] : '';
        
        if (!$meta_id) {
            wp_send_json_error(array('message' => __('Invalid metadata ID.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_meta';
        
        $result = $wpdb->update(
            $table,
            array(
                'value' => $value,
                'updated_at' => current_time('mysql')
            ),
            array('id' => $meta_id)
        );
        
        if ($result !== false) {
            wp_send_json_success(array('message' => __('Metadata updated successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to update metadata.', 'fca-content-manager')));
        }
    }
    
    /**
     * Delete metadata
     */
    public function delete_metadata() {
        $this->verify_nonce();
        
        // Check license for metadata access
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('metadata')) {
            $license_checker->send_license_required_error('metadata');
        }
        
        $meta_id = isset($_POST['meta_id']) ? intval($_POST['meta_id']) : 0;
        
        if (!$meta_id) {
            wp_send_json_error(array('message' => __('Invalid metadata ID.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_meta';
        $result = $wpdb->delete($table, array('id' => $meta_id));
        
        if ($result) {
            wp_send_json_success(array('message' => __('Metadata deleted successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to delete metadata.', 'fca-content-manager')));
        }
    }
    
    /**
     * Helper: Get user display name
     */
    private function get_user_display_name($user_id) {
        if (!$user_id) {
            return __('Unknown', 'fca-content-manager');
        }
        
        $user = get_userdata($user_id);
        return $user ? $user->display_name : __('Unknown', 'fca-content-manager');
    }
    
    /**
     * Helper: Get space name
     */
    private function get_space_name($space_id) {
        if (!$space_id) {
            return __('No Space', 'fca-content-manager');
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_spaces';
        $space = $wpdb->get_var($wpdb->prepare("SELECT title FROM {$table} WHERE id = %d", $space_id));
        
        return $space ? $space : __('Unknown Space', 'fca-content-manager');
    }
    
    /**
     * Helper: Get space name by feed ID
     */
    private function get_space_name_by_feed($feed_id) {
        if (!$feed_id) {
            return __('No Space', 'fca-content-manager');
        }
        
        global $wpdb;
        $posts_table = $wpdb->prefix . 'fcom_posts';
        $spaces_table = $wpdb->prefix . 'fcom_spaces';
        
        $space_name = $wpdb->get_var($wpdb->prepare(
            "SELECT s.title FROM {$spaces_table} s 
             INNER JOIN {$posts_table} p ON p.space_id = s.id 
             WHERE p.id = %d", 
            $feed_id
        ));
        
        return $space_name ? $space_name : __('Unknown Space', 'fca-content-manager');
    }
    
    /**
     * Helper: Get frontend URL for content
     */
    private function get_content_url($post) {
        // This is a placeholder - adjust based on your actual URL structure
        $base_url = home_url('/community');
        
        switch ($post['type']) {
            case 'course_lesson':
            case 'course_section':
                return $base_url . '/course/' . ($post['space_id'] ?? '');
            default:
                return $base_url . '/feed/' . $post['id'];
        }
    }
    
    /**
     * Get spaces list with pagination and filters
     */
    public function get_spaces_list() {
        $this->verify_nonce();
        
        $args = array(
            'page' => isset($_POST['page']) ? intval($_POST['page']) : 1,
            'per_page' => isset($_POST['per_page']) ? intval($_POST['per_page']) : 20,
            'search' => isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '',
            'status' => isset($_POST['status']) ? sanitize_text_field($_POST['status']) : '',
            'privacy' => isset($_POST['privacy']) ? sanitize_text_field($_POST['privacy']) : '',
            'orderby' => isset($_POST['orderby']) ? sanitize_text_field($_POST['orderby']) : 'created_at',
            'order' => isset($_POST['order']) ? sanitize_text_field($_POST['order']) : 'DESC'
        );
        
        $result = FCA_CM_Database::get_spaces_list($args);
        
        // Enhance data
        foreach ($result['data'] as &$space) {
            $space['creator'] = $this->get_user_display_name($space['created_by'] ?? null);
        }
        
        wp_send_json_success($result);
    }
    
    /**
     * Get single space
     */
    public function get_space() {
        $this->verify_nonce();
        
        $space_id = isset($_POST['space_id']) ? intval($_POST['space_id']) : 0;
        
        if (!$space_id) {
            wp_send_json_error(array('message' => __('Invalid space ID.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_spaces';
        
        $space = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $space_id
        ), ARRAY_A);
        
        if (!$space) {
            wp_send_json_error(array('message' => __('Space not found.', 'fca-content-manager')));
        }
        
        // Enhance data
        $space['creator'] = $this->get_user_display_name($space['created_by'] ?? null);
        
        wp_send_json_success($space);
    }
    
    /**
     * Update space
     */
    public function update_space() {
        $this->verify_nonce();
        
        // Check license for spaces access
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('spaces')) {
            $license_checker->send_license_required_error('spaces');
        }
        
        $space_id = isset($_POST['space_id']) ? intval($_POST['space_id']) : 0;
        
        if (!$space_id) {
            wp_send_json_error(array('message' => __('Invalid space ID.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_spaces';
        
        $data = array(
            'updated_at' => current_time('mysql')
        );
        
        if (isset($_POST['title'])) {
            $data['title'] = sanitize_text_field($_POST['title']);
        }
        
        if (isset($_POST['slug'])) {
            $data['slug'] = sanitize_title($_POST['slug']);
        }
        
        if (isset($_POST['description'])) {
            $data['description'] = wp_kses_post($_POST['description']);
        }
        
        if (isset($_POST['short_description'])) {
            $data['short_description'] = sanitize_text_field($_POST['short_description']);
        }
        
        if (isset($_POST['logo'])) {
            $data['logo'] = esc_url_raw($_POST['logo']);
        }
        
        if (isset($_POST['cover_photo'])) {
            $data['cover_photo'] = esc_url_raw($_POST['cover_photo']);
        }
        
        if (isset($_POST['type'])) {
            $data['type'] = sanitize_text_field($_POST['type']);
        }
        
        if (isset($_POST['parent_id']) && !empty($_POST['parent_id'])) {
            $data['parent_id'] = intval($_POST['parent_id']);
        }
        
        if (isset($_POST['serial']) && !empty($_POST['serial'])) {
            $data['serial'] = intval($_POST['serial']);
        }
        
        if (isset($_POST['settings'])) {
            // Validate JSON
            $settings = $_POST['settings'];
            if (!empty($settings)) {
                $decoded = json_decode($settings);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $data['settings'] = $settings;
                }
            }
        }
        
        if (isset($_POST['status'])) {
            $data['status'] = sanitize_text_field($_POST['status']);
        }
        
        if (isset($_POST['privacy'])) {
            $data['privacy'] = sanitize_text_field($_POST['privacy']);
        }
        
        $result = $wpdb->update($table, $data, array('id' => $space_id));
        
        if ($result !== false) {
            wp_send_json_success(array('message' => __('Space updated successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to update space.', 'fca-content-manager')));
        }
    }
    
    /**
     * Bulk update spaces
     */
    public function bulk_update_spaces() {
        $this->verify_nonce();
        
        $space_ids = isset($_POST['space_ids']) ? array_map('intval', $_POST['space_ids']) : array();
        
        if (empty($space_ids)) {
            wp_send_json_error(array('message' => __('No spaces selected.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_spaces';
        
        $data = array(
            'updated_at' => current_time('mysql')
        );
        
        if (isset($_POST['status']) && !empty($_POST['status'])) {
            $data['status'] = sanitize_text_field($_POST['status']);
        }
        
        if (isset($_POST['privacy']) && !empty($_POST['privacy'])) {
            $data['privacy'] = sanitize_text_field($_POST['privacy']);
        }
        
        if (count($data) == 1) { // Only updated_at
            wp_send_json_error(array('message' => __('No fields to update.', 'fca-content-manager')));
        }
        
        $updated = 0;
        foreach ($space_ids as $space_id) {
            $result = $wpdb->update($table, $data, array('id' => $space_id));
            if ($result !== false) {
                $updated++;
            }
        }
        
        if ($updated > 0) {
            wp_send_json_success(array(
                'message' => sprintf(__('%d spaces updated successfully.', 'fca-content-manager'), $updated)
            ));
        } else {
            wp_send_json_error(array('message' => __('Failed to update spaces.', 'fca-content-manager')));
        }
    }
    
    /**
     * Bulk delete spaces
     */
    public function bulk_delete_spaces() {
        $this->verify_nonce();
        
        // Check license for spaces access
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('spaces')) {
            $license_checker->send_license_required_error('spaces');
        }
        
        $space_ids = isset($_POST['space_ids']) ? array_map('intval', $_POST['space_ids']) : array();
        
        if (empty($space_ids)) {
            wp_send_json_error(array('message' => __('No spaces selected.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_spaces';
        $placeholders = implode(',', array_fill(0, count($space_ids), '%d'));
        $result = $wpdb->query($wpdb->prepare("DELETE FROM $table WHERE id IN ($placeholders) AND type != 'course'", $space_ids));
        
        if ($result) {
            wp_send_json_success(array('message' => sprintf(__('%d spaces deleted successfully.', 'fca-content-manager'), $result)));
        } else {
            wp_send_json_error(array('message' => __('Failed to delete spaces.', 'fca-content-manager')));
        }
    }
    
    /**
     * Get overview statistics
     */
    public function get_stats() {
        $this->verify_nonce();
        
        global $wpdb;
        $posts_table = $wpdb->prefix . 'fcom_posts';
        $comments_table = $wpdb->prefix . 'fcom_post_comments';
        $reactions_table = $wpdb->prefix . 'fcom_post_reactions';
        $media_table = $wpdb->prefix . 'fcom_media_archive';
        $spaces_table = $wpdb->prefix . 'fcom_spaces';
        $xprofile_table = $wpdb->prefix . 'fcom_xprofile';
        
        // Get counts
        $stats = array(
            'posts' => array(
                'total' => $wpdb->get_var("SELECT COUNT(*) FROM $posts_table WHERE type NOT IN ('course', 'course_section', 'course_lesson')"),
                'published' => $wpdb->get_var("SELECT COUNT(*) FROM $posts_table WHERE status = 'published' AND type NOT IN ('course', 'course_section', 'course_lesson')"),
                'draft' => $wpdb->get_var("SELECT COUNT(*) FROM $posts_table WHERE status = 'draft' AND type NOT IN ('course', 'course_section', 'course_lesson')"),
            ),
            'courses' => array(
                'total' => $wpdb->get_var("SELECT COUNT(*) FROM $spaces_table WHERE type = 'course'"),
                'published' => $wpdb->get_var("SELECT COUNT(*) FROM $spaces_table WHERE type = 'course' AND status = 'published'"),
            ),
            'lessons' => array(
                'total' => $wpdb->get_var("SELECT COUNT(*) FROM $posts_table WHERE type = 'course_lesson'"),
            ),
            'members' => array(
                'total' => $wpdb->get_var("SELECT COUNT(DISTINCT user_id) FROM $xprofile_table"),
                'active' => $wpdb->get_var("SELECT COUNT(DISTINCT user_id) FROM $xprofile_table WHERE status = 'active'"),
            ),
            'comments' => array(
                'total' => $wpdb->get_var("SELECT COUNT(*) FROM $comments_table"),
                'published' => $wpdb->get_var("SELECT COUNT(*) FROM $comments_table WHERE status = 'published'"),
                'pending' => $wpdb->get_var("SELECT COUNT(*) FROM $comments_table WHERE status = 'pending'"),
            ),
            'reactions' => array(
                'total' => $wpdb->get_var("SELECT COUNT(*) FROM $reactions_table"),
                'posts' => $wpdb->get_var("SELECT COUNT(*) FROM $reactions_table WHERE object_type = 'feed'"),
                'comments' => $wpdb->get_var("SELECT COUNT(*) FROM $reactions_table WHERE object_type = 'comment'"),
            ),
            'media' => array(
                'total' => $wpdb->get_var("SELECT COUNT(*) FROM $media_table"),
            ),
            'spaces' => array(
                'total' => $wpdb->get_var("SELECT COUNT(*) FROM $spaces_table WHERE type != 'course' OR type IS NULL"),
                'published' => $wpdb->get_var("SELECT COUNT(*) FROM $spaces_table WHERE (type != 'course' OR type IS NULL) AND status = 'published'"),
            ),
        );
        
        wp_send_json_success($stats);
    }
    
    /**
     * Get all available terms/topics
     */
    public function get_all_terms() {
        $this->verify_nonce();
        
        $terms = FCA_CM_Database::get_all_terms();
        
        wp_send_json_success($terms);
    }
    
    /**
     * Add term to post
     */
    public function add_term_to_post() {
        $this->verify_nonce();
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $term_id = isset($_POST['term_id']) ? intval($_POST['term_id']) : 0;
        
        if (!$post_id || !$term_id) {
            wp_send_json_error(array('message' => __('Invalid post or term ID.', 'fca-content-manager')));
        }
        
        $result = FCA_CM_Database::add_term_to_post($post_id, $term_id);
        
        if ($result) {
            wp_send_json_success(array('message' => __('Term added successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to add term.', 'fca-content-manager')));
        }
    }
    
    /**
     * Remove term from post
     */
    public function remove_term_from_post() {
        $this->verify_nonce();
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $term_id = isset($_POST['term_id']) ? intval($_POST['term_id']) : 0;
        
        if (!$post_id || !$term_id) {
            wp_send_json_error(array('message' => __('Invalid post or term ID.', 'fca-content-manager')));
        }
        
        $result = FCA_CM_Database::remove_term_from_post($post_id, $term_id);
        
        if ($result) {
            wp_send_json_success(array('message' => __('Term removed successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to remove term.', 'fca-content-manager')));
        }
    }
    
    /**
     * Bulk update posts
     */
    public function bulk_update_posts() {
        $this->verify_nonce();
        
        $post_ids = isset($_POST['post_ids']) ? array_map('intval', $_POST['post_ids']) : array();
        
        if (empty($post_ids)) {
            wp_send_json_error(array('message' => __('No posts selected.', 'fca-content-manager')));
        }
        
        $data = array();
        
        // Only include fields that are actually being updated
        if (isset($_POST['space_id']) && $_POST['space_id'] !== '') {
            $data['space_id'] = $_POST['space_id'] ? intval($_POST['space_id']) : null;
        }
        
        if (isset($_POST['status']) && $_POST['status'] !== '') {
            $data['status'] = sanitize_text_field($_POST['status']);
        }
        
        if (isset($_POST['privacy']) && $_POST['privacy'] !== '') {
            $data['privacy'] = sanitize_text_field($_POST['privacy']);
        }
        
        if (empty($data)) {
            wp_send_json_error(array('message' => __('No changes specified.', 'fca-content-manager')));
        }
        
        $result = FCA_CM_Database::bulk_update_posts($post_ids, $data);
        
        if ($result !== false) {
            wp_send_json_success(array(
                'message' => sprintf(__('%d posts updated successfully.', 'fca-content-manager'), count($post_ids))
            ));
        } else {
            wp_send_json_error(array('message' => __('Failed to update posts.', 'fca-content-manager')));
        }
    }
    
    /**
     * Bulk add term to multiple posts
     */
    public function bulk_add_term() {
        $this->verify_nonce();
        
        $post_ids = isset($_POST['post_ids']) ? array_map('intval', $_POST['post_ids']) : array();
        $term_id = isset($_POST['term_id']) ? intval($_POST['term_id']) : 0;
        
        if (empty($post_ids) || !$term_id) {
            wp_send_json_error(array('message' => __('Invalid data.', 'fca-content-manager')));
        }
        
        $success_count = 0;
        foreach ($post_ids as $post_id) {
            if (FCA_CM_Database::add_term_to_post($post_id, $term_id)) {
                $success_count++;
            }
        }
        
        wp_send_json_success(array(
            'message' => sprintf(__('Term added to %d posts.', 'fca-content-manager'), $success_count)
        ));
    }
    
    /**
     * Get single comment for modal
     */
    public function get_comment() {
        $this->verify_nonce();
        
        $comment_id = isset($_POST['comment_id']) ? intval($_POST['comment_id']) : 0;
        
        if (!$comment_id) {
            wp_send_json_error(array('message' => __('Invalid comment ID.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $comments_table = $wpdb->prefix . 'fcom_post_comments';
        $reactions_table = $wpdb->prefix . 'fcom_post_reactions';
        $posts_table = $wpdb->prefix . 'fcom_posts';
        
        // Get main comment
        $comment = $wpdb->get_row($wpdb->prepare("SELECT * FROM $comments_table WHERE id = %d", $comment_id), ARRAY_A);
        
        if (!$comment) {
            wp_send_json_error(array('message' => __('Comment not found.', 'fca-content-manager')));
        }
        
        // Enhance comment data
        $comment['author'] = $this->get_user_display_name($comment['user_id'] ?? null);
        $comment['created_at_formatted'] = date('M j, Y g:i A', strtotime($comment['created_at']));
        $comment['updated_at_formatted'] = date('M j, Y g:i A', strtotime($comment['updated_at']));
        
        // Get parent post
        if ($comment['post_id']) {
            $parent_post = $wpdb->get_row($wpdb->prepare("SELECT * FROM $posts_table WHERE id = %d", $comment['post_id']), ARRAY_A);
            if ($parent_post) {
                $parent_post['author'] = $this->get_user_display_name($parent_post['user_id'] ?? null);
                $parent_post['space_name'] = $this->get_space_name($parent_post['space_id'] ?? null);
                $comment['parent_post'] = $parent_post;
            }
        }
        
        // Get reactions (object_type = 'comment', object_id = comment_id)
        $reactions = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $reactions_table WHERE object_type = 'comment' AND object_id = %d ORDER BY created_at DESC",
            $comment_id
        ), ARRAY_A);
        
        if (!empty($reactions)) {
            foreach ($reactions as &$reaction) {
                $reaction['author'] = $this->get_user_display_name($reaction['user_id'] ?? null);
                $reaction['created_at_formatted'] = date('M j, Y g:i A', strtotime($reaction['created_at']));
            }
        }
        $comment['reactions'] = $reactions;
        
        // Get sub-comments/replies (where parent_id = comment_id)
        $replies = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $comments_table WHERE parent_id = %d ORDER BY created_at ASC",
            $comment_id
        ), ARRAY_A);
        
        if (!empty($replies)) {
            foreach ($replies as &$reply) {
                $reply['author'] = $this->get_user_display_name($reply['user_id'] ?? null);
                $reply['created_at_formatted'] = date('M j, Y g:i A', strtotime($reply['created_at']));
            }
        }
        $comment['replies'] = $replies;
        
        wp_send_json_success($comment);
    }
    
    /**
     * Update comment
     */
    public function update_comment() {
        $this->verify_nonce();
        
        // Check license for comment editing
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('comments_edit')) {
            $license_checker->send_license_required_error('comments_edit');
        }
        
        $comment_id = isset($_POST['comment_id']) ? intval($_POST['comment_id']) : 0;
        
        if (!$comment_id) {
            wp_send_json_error(array('message' => __('Invalid comment ID.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_post_comments';
        
        $update_data = array();
        
        if (isset($_POST['message'])) {
            $update_data['message'] = wp_kses_post($_POST['message']);
        }
        
        if (isset($_POST['status'])) {
            $update_data['status'] = sanitize_text_field($_POST['status']);
        }
        
        if (isset($_POST['privacy'])) {
            $update_data['privacy'] = sanitize_text_field($_POST['privacy']);
        }
        
        $update_data['updated_at'] = current_time('mysql');
        
        $result = $wpdb->update($table, $update_data, array('id' => $comment_id));
        
        if ($result !== false) {
            wp_send_json_success(array('message' => __('Comment updated successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to update comment.', 'fca-content-manager')));
        }
    }
    
    /**
     * Get courses
     */
    public function get_courses() {
        $this->verify_nonce();
        
        $args = array(
            'search' => isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '',
            'status' => isset($_POST['status']) ? sanitize_text_field($_POST['status']) : ''
        );
        
        $courses = FCA_CM_Database::get_courses($args);
        
        wp_send_json_success($courses);
    }
    
    /**
     * Get single course with structure
     */
    public function get_course() {
        $this->verify_nonce();
        
        $course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
        
        if (!$course_id) {
            wp_send_json_error(array('message' => __('Invalid course ID.', 'fca-content-manager')));
        }
        
        $course = FCA_CM_Database::get_course($course_id);
        
        if (!$course) {
            wp_send_json_error(array('message' => __('Course not found.', 'fca-content-manager')));
        }
        
        wp_send_json_success($course);
    }
    
    /**
     * Get course structure (sections and lessons)
     */
    public function get_course_structure() {
        $this->verify_nonce();
        
        $course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
        
        if (!$course_id) {
            wp_send_json_error(array('message' => __('Invalid course ID.', 'fca-content-manager')));
        }
        
        $structure = FCA_CM_Database::get_course_structure($course_id);
        
        wp_send_json_success($structure);
    }
    
    /**
     * Update course metadata (courses are spaces)
     */
    public function update_course() {
        $this->verify_nonce();
        
        // Check license for course editing
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('courses_edit')) {
            $license_checker->send_license_required_error('courses_edit');
        }
        
        $course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
        
        if (!$course_id) {
            wp_send_json_error(array('message' => __('Invalid course ID.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_spaces';
        
        // Prepare update data
        $update_data = array();
        
        if (isset($_POST['title'])) {
            $update_data['title'] = sanitize_text_field($_POST['title']);
        }
        
        if (isset($_POST['description'])) {
            $update_data['description'] = wp_kses_post($_POST['description']);
        }
        
        if (isset($_POST['slug'])) {
            $update_data['slug'] = sanitize_title($_POST['slug']);
        }
        
        if (isset($_POST['status'])) {
            $update_data['status'] = sanitize_text_field($_POST['status']);
        }
        
        if (isset($_POST['privacy'])) {
            $update_data['privacy'] = sanitize_text_field($_POST['privacy']);
        }
        
        $update_data['updated_at'] = current_time('mysql');
        
        $result = $wpdb->update(
            $table,
            $update_data,
            array('id' => $course_id)
        );
        
        if ($result !== false) {
            wp_send_json_success(array('message' => __('Course updated successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to update course.', 'fca-content-manager')));
        }
    }
    
    /**
     * Bulk update courses
     */
    public function bulk_update_courses() {
        $this->verify_nonce();
        
        // Check license for course editing
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('courses_edit')) {
            $license_checker->send_license_required_error('courses_edit');
        }
        
        $course_ids = isset($_POST['course_ids']) ? array_map('intval', $_POST['course_ids']) : array();
        
        if (empty($course_ids)) {
            wp_send_json_error(array('message' => __('No courses selected.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_spaces';
        
        // Prepare update data
        $update_data = array();
        
        if (isset($_POST['status']) && !empty($_POST['status'])) {
            $update_data['status'] = sanitize_text_field($_POST['status']);
        }
        
        if (isset($_POST['privacy']) && !empty($_POST['privacy'])) {
            $update_data['privacy'] = sanitize_text_field($_POST['privacy']);
        }
        
        if (empty($update_data)) {
            wp_send_json_error(array('message' => __('No fields to update.', 'fca-content-manager')));
        }
        
        $update_data['updated_at'] = current_time('mysql');
        
        // Update each course
        $updated = 0;
        foreach ($course_ids as $course_id) {
            $result = $wpdb->update(
                $table,
                $update_data,
                array('id' => $course_id)
            );
            if ($result !== false) {
                $updated++;
            }
        }
        
        if ($updated > 0) {
            wp_send_json_success(array(
                'message' => sprintf(__('%d courses updated successfully.', 'fca-content-manager'), $updated)
            ));
        } else {
            wp_send_json_error(array('message' => __('Failed to update courses.', 'fca-content-manager')));
        }
    }
    
    /**
     * Update course item order
     */
    public function update_course_order() {
        $this->verify_nonce();
        
        // Check license for course editing
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('courses_edit')) {
            $license_checker->send_license_required_error('courses_edit');
        }
        
        $items = isset($_POST['items']) ? $_POST['items'] : array();
        
        if (empty($items)) {
            wp_send_json_error(array('message' => __('No items to update.', 'fca-content-manager')));
        }
        
        foreach ($items as $item) {
            if (isset($item['id']) && isset($item['priority'])) {
                FCA_CM_Database::update_item_priority(intval($item['id']), intval($item['priority']));
            }
        }
        
        wp_send_json_success(array('message' => __('Order updated successfully.', 'fca-content-manager')));
    }
    
    /**
     * Delete course item (section or lesson)
     */
    public function delete_course_item() {
        $this->verify_nonce();
        
        // Check license for course editing
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('courses_edit')) {
            $license_checker->send_license_required_error('courses_edit');
        }
        
        $item_id = isset($_POST['item_id']) ? intval($_POST['item_id']) : 0;
        
        if (!$item_id) {
            wp_send_json_error(array('message' => __('Invalid item ID.', 'fca-content-manager')));
        }
        
        $result = FCA_CM_Database::delete_post($item_id);
        
        if ($result) {
            wp_send_json_success(array('message' => __('Item deleted successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to delete item.', 'fca-content-manager')));
        }
    }
    
    /**
     * Duplicate course item (section or lesson)
     */
    public function duplicate_course_item() {
        $this->verify_nonce();
        
        // Check license for course editing
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('courses_edit')) {
            $license_checker->send_license_required_error('courses_edit');
        }
        
        $item_id = isset($_POST['item_id']) ? intval($_POST['item_id']) : 0;
        
        if (!$item_id) {
            wp_send_json_error(array('message' => __('Invalid item ID.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_posts';
        
        // Get the original item
        $original = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $item_id), ARRAY_A);
        
        if (!$original) {
            wp_send_json_error(array('message' => __('Item not found.', 'fca-content-manager')));
        }
        
        // Remove ID and update title
        unset($original['id']);
        $original['title'] = $original['title'] . ' (Copy)';
        $original['created_at'] = current_time('mysql');
        $original['updated_at'] = current_time('mysql');
        
        // Insert duplicate
        $result = $wpdb->insert($table, $original);
        
        if ($result) {
            wp_send_json_success(array(
                'message' => __('Item duplicated successfully.', 'fca-content-manager'),
                'new_id' => $wpdb->insert_id
            ));
        } else {
            wp_send_json_error(array('message' => __('Failed to duplicate item.', 'fca-content-manager')));
        }
    }
    
    /**
     * Move course item (section or lesson) to another course
     */
    public function move_course_item() {
        $this->verify_nonce();
        
        // Check license for course editing
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('courses_edit')) {
            $license_checker->send_license_required_error('courses_edit');
        }
        
        $item_id = isset($_POST['item_id']) ? intval($_POST['item_id']) : 0;
        $item_type = isset($_POST['item_type']) ? sanitize_text_field($_POST['item_type']) : '';
        $target_course_id = isset($_POST['target_course_id']) ? intval($_POST['target_course_id']) : 0;
        $target_section_id = isset($_POST['target_section_id']) ? intval($_POST['target_section_id']) : 0;
        
        if (!$item_id || !$target_course_id) {
            wp_send_json_error(array('message' => __('Invalid parameters.', 'fca-content-manager')));
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'fcom_posts';
        
        // Prepare update data
        $update_data = array();
        
        if ($item_type === 'section') {
            // Moving a section: update space_id and clear parent_id
            $update_data['space_id'] = $target_course_id;
            $update_data['parent_id'] = null;
            
            $result = $wpdb->update($table, $update_data, array('id' => $item_id));
            
            // Also update all lessons in this section
            $wpdb->update(
                $table,
                array('space_id' => $target_course_id),
                array('parent_id' => $item_id, 'type' => 'course_lesson')
            );
            
        } else if ($item_type === 'lesson') {
            // Moving a lesson: update space_id and optionally parent_id
            $update_data['space_id'] = $target_course_id;
            $update_data['parent_id'] = $target_section_id ? $target_section_id : null;
            
            $result = $wpdb->update($table, $update_data, array('id' => $item_id));
        }
        
        if ($result !== false) {
            wp_send_json_success(array('message' => __('Item moved successfully.', 'fca-content-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to move item.', 'fca-content-manager')));
        }
    }
    
    // ========================================================================
    // Import/Export Handlers
    // ========================================================================
    
    /**
     * Export posts
     */
    public function export_posts() {
        $this->verify_nonce();
        
        // Check license for export
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('export')) {
            $license_checker->send_license_required_error('export');
        }
        
        $format = isset($_POST['format']) ? sanitize_text_field($_POST['format']) : 'json';
        $args = array(
            'space_id' => isset($_POST['space_id']) ? intval($_POST['space_id']) : '',
            'status' => isset($_POST['status']) ? sanitize_text_field($_POST['status']) : '',
            'include_comments' => isset($_POST['include_comments']) && $_POST['include_comments'] === 'true',
            'include_reactions' => isset($_POST['include_reactions']) && $_POST['include_reactions'] === 'true',
            'include_media' => isset($_POST['include_media']) && $_POST['include_media'] === 'true',
            'include_terms' => isset($_POST['include_terms']) && $_POST['include_terms'] === 'true',
        );
        
        if ($format === 'json') {
            $data = FCA_CM_Export::export_posts_json($args);
            $filename = FCA_CM_ImportExport::generate_filename('posts', 'json');
        } else {
            $data = FCA_CM_Export::export_posts_csv($args);
            $filename = FCA_CM_ImportExport::generate_filename('posts', 'csv');
        }
        
        $result = FCA_CM_Export::create_export_file($data, $filename, $format);
        
        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        } else {
            wp_send_json_success(array_merge($result, array(
                'total_items' => is_array($data) && isset($data['total_items']) ? $data['total_items'] : 0
            )));
        }
    }
    
    /**
     * Export spaces
     */
    public function export_spaces() {
        $this->verify_nonce();
        
        // Check license for export
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('export')) {
            $license_checker->send_license_required_error('export');
        }
        
        $format = isset($_POST['format']) ? sanitize_text_field($_POST['format']) : 'json';
        $args = array(
            'status' => isset($_POST['status']) ? sanitize_text_field($_POST['status']) : '',
            'privacy' => isset($_POST['privacy']) ? sanitize_text_field($_POST['privacy']) : '',
        );
        
        if ($format === 'json') {
            $data = FCA_CM_Export::export_spaces_json($args);
            $filename = FCA_CM_ImportExport::generate_filename('spaces', 'json');
        } else {
            $data = FCA_CM_Export::export_spaces_csv($args);
            $filename = FCA_CM_ImportExport::generate_filename('spaces', 'csv');
        }
        
        $result = FCA_CM_Export::create_export_file($data, $filename, $format);
        
        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        } else {
            wp_send_json_success(array_merge($result, array(
                'total_items' => is_array($data) && isset($data['total_items']) ? $data['total_items'] : 0
            )));
        }
    }
    
    /**
     * Export comments
     */
    public function export_comments() {
        $this->verify_nonce();
        
        // Check license for export
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('export')) {
            $license_checker->send_license_required_error('export');
        }
        
        $format = isset($_POST['format']) ? sanitize_text_field($_POST['format']) : 'json';
        $args = array(
            'space_id' => isset($_POST['space_id']) ? intval($_POST['space_id']) : '',
        );
        
        $data = FCA_CM_Export::export_comments_json($args);
        $filename = FCA_CM_ImportExport::generate_filename('comments', 'json');
        
        $result = FCA_CM_Export::create_export_file($data, $filename, $format);
        
        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        } else {
            wp_send_json_success(array_merge($result, array(
                'total_items' => is_array($data) && isset($data['total_items']) ? $data['total_items'] : 0
            )));
        }
    }
    
    /**
     * Export reactions
     */
    public function export_reactions() {
        $this->verify_nonce();
        
        // Check license for export
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('export')) {
            $license_checker->send_license_required_error('export');
        }
        
        $args = array(
            'object_type' => isset($_POST['object_type']) ? sanitize_text_field($_POST['object_type']) : '',
            'type' => isset($_POST['reaction_type']) ? sanitize_text_field($_POST['reaction_type']) : '',
            'date_from' => isset($_POST['date_from']) ? sanitize_text_field($_POST['date_from']) : '',
            'date_to' => isset($_POST['date_to']) ? sanitize_text_field($_POST['date_to']) : ''
        );
        
        $format = isset($_POST['format']) ? sanitize_text_field($_POST['format']) : 'json';
        
        $data = FCA_CM_Export::export_reactions_json($args);
        $filename = 'reactions-export-' . date('Y-m-d-His') . '.' . $format;
        
        $result = FCA_CM_Export::create_export_file($data, $filename, $format);
        
        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        } else {
            wp_send_json_success(array_merge($result, array(
                'total_items' => is_array($data) && isset($data['total_items']) ? $data['total_items'] : 0
            )));
        }
    }
    
    /**
     * Export term feed relationships
     */
    public function export_term_feed() {
        $this->verify_nonce();
        
        // Check license for export
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('export')) {
            $license_checker->send_license_required_error('export');
        }
        
        $args = array(
            'post_id' => isset($_POST['post_id']) ? intval($_POST['post_id']) : 0,
            'term_id' => isset($_POST['term_id']) ? intval($_POST['term_id']) : 0
        );
        
        $format = isset($_POST['format']) ? sanitize_text_field($_POST['format']) : 'json';
        
        $data = FCA_CM_Export::export_term_feed_json($args);
        $filename = 'term-feed-export-' . date('Y-m-d-His') . '.' . $format;
        
        $result = FCA_CM_Export::create_export_file($data, $filename, $format);
        
        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        } else {
            wp_send_json_success(array_merge($result, array(
                'total_items' => is_array($data) && isset($data['total_items']) ? $data['total_items'] : 0
            )));
        }
    }
    
    /**
     * Export space users (memberships)
     */
    public function export_space_users() {
        $this->verify_nonce();
        
        // Check license for export
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('export')) {
            $license_checker->send_license_required_error('export');
        }
        
        $args = array(
            'space_id' => isset($_POST['space_id']) ? intval($_POST['space_id']) : 0,
            'user_id' => isset($_POST['user_id']) ? intval($_POST['user_id']) : 0,
            'role' => isset($_POST['role']) ? sanitize_text_field($_POST['role']) : '',
            'status' => isset($_POST['status']) ? sanitize_text_field($_POST['status']) : ''
        );
        
        $format = isset($_POST['format']) ? sanitize_text_field($_POST['format']) : 'json';
        
        $data = FCA_CM_Export::export_space_users_json($args);
        $filename = 'space-users-export-' . date('Y-m-d-His') . '.' . $format;
        
        $result = FCA_CM_Export::create_export_file($data, $filename, $format);
        
        if (isset($result['error'])) {
            wp_send_json_error(array('message' => $result['error']));
        } else {
            wp_send_json_success(array_merge($result, array(
                'total_items' => is_array($data) && isset($data['total_items']) ? $data['total_items'] : 0
            )));
        }
    }
    
    /**
     * Upload and parse import file
     */
    public function upload_import_file() {
        $this->verify_nonce();
        
        // Check license for import
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('import')) {
            $license_checker->send_license_required_error('import');
        }
        
        if (!isset($_FILES['file'])) {
            wp_send_json_error(array('message' => __('No file uploaded.', 'fca-content-manager')));
        }
        
        $file = $_FILES['file'];
        
        // Validate file
        $validation = FCA_CM_ImportExport::validate_upload($file);
        
        if (!$validation['valid']) {
            wp_send_json_error(array('message' => implode(' ', $validation['errors'])));
        }
        
        // Move file to uploads directory
        $upload_dir = FCA_CM_ImportExport::get_uploads_dir();
        $filename = 'import-' . time() . '-' . sanitize_file_name($file['name']);
        $file_path = $upload_dir . '/' . $filename;
        
        if (!move_uploaded_file($file['tmp_name'], $file_path)) {
            wp_send_json_error(array('message' => __('Failed to upload file.', 'fca-content-manager')));
        }
        
        // Parse file
        $extension = $validation['extension'];
        
        if ($extension === 'json') {
            $parsed = FCA_CM_ImportExport::parse_json($file_path);
            
            if (isset($parsed['error'])) {
                @unlink($file_path);
                wp_send_json_error(array('message' => $parsed['error']));
            }
            
            // Validate data structure
            $validation_result = FCA_CM_Import::validate_import($parsed['data'], $parsed['data']['content_type'] ?? 'posts');
            
            if (!$validation_result['valid']) {
                @unlink($file_path);
                wp_send_json_error(array('message' => implode(' ', $validation_result['errors'])));
            }
            
            wp_send_json_success(array(
                'file_path' => $file_path,
                'format' => 'json',
                'content_type' => $parsed['data']['content_type'] ?? 'posts',
                'total_items' => $validation_result['item_count'],
                'sample' => $validation_result['sample']
            ));
            
        } elseif ($extension === 'csv') {
            $parsed = FCA_CM_ImportExport::parse_csv($file_path);
            
            if (isset($parsed['error'])) {
                @unlink($file_path);
                wp_send_json_error(array('message' => $parsed['error']));
            }
            
            wp_send_json_success(array(
                'file_path' => $file_path,
                'format' => 'csv',
                'content_type' => 'posts', // Determine from headers or user selection
                'headers' => $parsed['headers'],
                'total_items' => $parsed['count'],
                'sample' => array_slice($parsed['data'], 0, 5)
            ));
        }
    }
    
    /**
     * Process import
     */
    public function process_import() {
        $this->verify_nonce();
        
        // Check license for import
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('import')) {
            $license_checker->send_license_required_error('import');
        }
        
        $file_path = isset($_POST['file_path']) ? sanitize_text_field($_POST['file_path']) : '';
        $content_type = isset($_POST['content_type']) ? sanitize_text_field($_POST['content_type']) : 'posts';
        $format = isset($_POST['format']) ? sanitize_text_field($_POST['format']) : 'json';
        $field_mapping = isset($_POST['field_mapping']) ? $_POST['field_mapping'] : array();
        $options = isset($_POST['options']) ? $_POST['options'] : array();
        
        if (!file_exists($file_path)) {
            wp_send_json_error(array('message' => __('Import file not found.', 'fca-content-manager')));
        }
        
        // Parse options
        $import_options = array(
            'update_existing' => isset($options['duplicate_handling']) && $options['duplicate_handling'] === 'update',
            'skip_duplicates' => !isset($options['duplicate_handling']) || $options['duplicate_handling'] === 'skip',
            'create_spaces' => isset($options['create_spaces']) && $options['create_spaces'] === 'true',
            'create_terms' => isset($options['create_terms']) && $options['create_terms'] === 'true',
            'dry_run' => isset($options['dry_run']) && $options['dry_run'] === 'true',
        );
        
        // Perform import based on format
        if ($format === 'json') {
            $parsed = FCA_CM_ImportExport::parse_json($file_path);
            
            if (isset($parsed['error'])) {
                wp_send_json_error(array('message' => $parsed['error']));
            }
            
            if ($content_type === 'posts') {
                $results = FCA_CM_Import::import_posts_json($parsed['data'], $import_options);
            } elseif ($content_type === 'spaces') {
                $results = FCA_CM_Import::import_spaces_json($parsed['data'], $import_options);
            } elseif ($content_type === 'comments') {
                $results = FCA_CM_Import::import_comments_json($parsed['data'], $import_options);
            } elseif ($content_type === 'media') {
                $results = FCA_CM_Import::import_media_json($parsed['data'], $import_options);
            } elseif ($content_type === 'reactions') {
                $results = FCA_CM_Import::import_reactions_json($parsed['data'], $import_options);
            } elseif ($content_type === 'term_feed') {
                $results = FCA_CM_Import::import_term_feed_json($parsed['data'], $import_options);
            } elseif ($content_type === 'space_users') {
                $results = FCA_CM_Import::import_space_users_json($parsed['data'], $import_options);
            } else {
                wp_send_json_error(array('message' => __('Unsupported content type.', 'fca-content-manager')));
            }
            
        } elseif ($format === 'csv') {
            $parsed = FCA_CM_ImportExport::parse_csv($file_path);
            
            if (isset($parsed['error'])) {
                wp_send_json_error(array('message' => $parsed['error']));
            }
            
            if ($content_type === 'posts') {
                $results = FCA_CM_Import::import_posts_csv($parsed, $field_mapping, $import_options);
            } elseif ($content_type === 'spaces') {
                $results = FCA_CM_Import::import_spaces_csv($parsed, $field_mapping, $import_options);
            } elseif ($content_type === 'comments') {
                $results = FCA_CM_Import::import_comments_csv($parsed, $field_mapping, $import_options);
            } else {
                wp_send_json_error(array('message' => __('Unsupported content type.', 'fca-content-manager')));
            }
        }
        
        // Clean up file unless dry run
        if (!$import_options['dry_run']) {
            @unlink($file_path);
        }
        
        wp_send_json_success($results);
    }
    
    /**
     * Get Members
     */
    public function get_members() {
        $this->verify_nonce();
        
        // Check license
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('members')) {
            $license_checker->send_license_required_error('members');
        }
        
        $args = array(
            'page' => isset($_POST['page']) ? intval($_POST['page']) : 1,
            'per_page' => isset($_POST['per_page']) ? intval($_POST['per_page']) : 20,
            'search' => isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '',
            'status' => isset($_POST['status']) ? sanitize_text_field($_POST['status']) : '',
            'role' => isset($_POST['role']) ? sanitize_text_field($_POST['role']) : '',
            'space_id' => isset($_POST['space_id']) ? intval($_POST['space_id']) : '',
            'orderby' => isset($_POST['orderby']) ? sanitize_text_field($_POST['orderby']) : 'display_name',
            'order' => isset($_POST['order']) ? sanitize_text_field($_POST['order']) : 'ASC'
        );
        
        $result = FCA_CM_Database::get_members($args);
        
        wp_send_json_success($result);
    }
    
    /**
     * Get single member
     */
    public function get_member() {
        $this->verify_nonce();
        
        // Check license
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('members')) {
            $license_checker->send_license_required_error('members');
        }
        
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        
        if (!$user_id) {
            wp_send_json_error(array('message' => __('User ID is required.', 'fca-content-manager')));
        }
        
        $member = FCA_CM_Database::get_member($user_id);
        
        if (!$member) {
            wp_send_json_error(array('message' => __('Member not found.', 'fca-content-manager')));
        }
        
        wp_send_json_success($member);
    }
    
    /**
     * Update member
     */
    public function update_member() {
        $this->verify_nonce();
        
        // Check license
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('members')) {
            $license_checker->send_license_required_error('members');
        }
        
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        
        if (!$user_id) {
            wp_send_json_error(array('message' => __('User ID is required.', 'fca-content-manager')));
        }
        
        $data = array();
        
        if (isset($_POST['display_name'])) {
            $data['display_name'] = sanitize_text_field($_POST['display_name']);
        }
        
        if (isset($_POST['username'])) {
            $data['username'] = sanitize_user($_POST['username']);
        }
        
        if (isset($_POST['short_description'])) {
            $data['short_description'] = sanitize_textarea_field($_POST['short_description']);
        }
        
        if (isset($_POST['status'])) {
            $data['status'] = sanitize_text_field($_POST['status']);
        }
        
        if (isset($_POST['total_points'])) {
            $data['total_points'] = intval($_POST['total_points']);
        }
        
        if (isset($_POST['is_verified'])) {
            $data['is_verified'] = intval($_POST['is_verified']);
        }
        
        $result = FCA_CM_Database::update_member($user_id, $data);
        
        if ($result === false) {
            wp_send_json_error(array('message' => __('Failed to update member.', 'fca-content-manager')));
        }
        
        wp_send_json_success(array(
            'message' => __('Member updated successfully.', 'fca-content-manager')
        ));
    }
    
    /**
     * Add enrollment
     */
    public function add_enrollment() {
        $this->verify_nonce();
        
        // Check license
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('members')) {
            $license_checker->send_license_required_error('members');
        }
        
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        $space_id = isset($_POST['space_id']) ? intval($_POST['space_id']) : 0;
        $role = isset($_POST['role']) ? sanitize_text_field($_POST['role']) : 'member';
        
        if (!$user_id || !$space_id) {
            wp_send_json_error(array('message' => __('User ID and Space ID are required.', 'fca-content-manager')));
        }
        
        $result = FCA_CM_Database::add_enrollment($user_id, $space_id, $role);
        
        if ($result === false) {
            wp_send_json_error(array('message' => __('Failed to add enrollment or already enrolled.', 'fca-content-manager')));
        }
        
        wp_send_json_success(array(
            'message' => __('Enrollment added successfully.', 'fca-content-manager')
        ));
    }
    
    /**
     * Remove enrollment
     */
    public function remove_enrollment() {
        $this->verify_nonce();
        
        // Check license
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('members')) {
            $license_checker->send_license_required_error('members');
        }
        
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        $space_id = isset($_POST['space_id']) ? intval($_POST['space_id']) : 0;
        
        if (!$user_id || !$space_id) {
            wp_send_json_error(array('message' => __('User ID and Space ID are required.', 'fca-content-manager')));
        }
        
        $result = FCA_CM_Database::remove_enrollment($user_id, $space_id);
        
        if ($result === false) {
            wp_send_json_error(array('message' => __('Failed to remove enrollment.', 'fca-content-manager')));
        }
        
        wp_send_json_success(array(
            'message' => __('Enrollment removed successfully.', 'fca-content-manager')
        ));
    }
    
    /**
     * Bulk add enrollments
     */
    public function bulk_add_enrollments() {
        $this->verify_nonce();
        
        // Check license
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('members')) {
            $license_checker->send_license_required_error('members');
        }
        
        $user_ids = isset($_POST['user_ids']) ? array_map('intval', $_POST['user_ids']) : array();
        $space_id = isset($_POST['space_id']) ? intval($_POST['space_id']) : 0;
        $role = isset($_POST['role']) ? sanitize_text_field($_POST['role']) : 'member';
        
        if (empty($user_ids) || !$space_id) {
            wp_send_json_error(array('message' => __('User IDs and Space ID are required.', 'fca-content-manager')));
        }
        
        $success = FCA_CM_Database::bulk_add_enrollments($user_ids, $space_id, $role);
        
        wp_send_json_success(array(
            'message' => sprintf(__('%d enrollments added successfully.', 'fca-content-manager'), $success),
            'count' => $success
        ));
    }
    
    /**
     * Bulk remove enrollments
     */
    public function bulk_remove_enrollments() {
        $this->verify_nonce();
        
        // Check license
        $license_checker = FCA_CM_LicenseChecker::get_instance();
        if (!$license_checker->can_access_feature('members')) {
            $license_checker->send_license_required_error('members');
        }
        
        $user_ids = isset($_POST['user_ids']) ? array_map('intval', $_POST['user_ids']) : array();
        $space_id = isset($_POST['space_id']) ? intval($_POST['space_id']) : 0;
        
        if (empty($user_ids) || !$space_id) {
            wp_send_json_error(array('message' => __('User IDs and Space ID are required.', 'fca-content-manager')));
        }
        
        $success = FCA_CM_Database::bulk_remove_enrollments($user_ids, $space_id);
        
        wp_send_json_success(array(
            'message' => sprintf(__('%d enrollments removed successfully.', 'fca-content-manager'), $success),
            'count' => $success
        ));
    }
}

