<?php
/**
 * Plugin Name: FCA Content Manager for Fluent Community
 * Plugin URI: https://fluentcommunityaddons.com
 * Description: Modern content management system for Fluent Community - manage posts, courses, comments, reactions, and more
 * Version: 2.0.1
 * Author: Fluent Community Addons
 * Author URI: https://fluentcommunityaddons.com
 * Text Domain: fca-content-manager
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Prevent direct access
defined('ABSPATH') or die('No direct script access allowed');

// Define plugin constants
define('FCA_CM_VERSION', '2.0.1');
define('FCA_CM_FILE', __FILE__);
define('FCA_CM_PATH', plugin_dir_path(__FILE__));
define('FCA_CM_URL', plugin_dir_url(__FILE__));
define('FCA_CM_BASENAME', plugin_basename(__FILE__));

// Check if Fluent Community is active
if (!function_exists('fca_content_manager_check_dependencies')) {
    function fca_content_manager_check_dependencies() {
        // Check for Fluent Community by looking for its main class or constant
        if (!class_exists('\FluentCommunity\App\Services\Helper') && 
            !class_exists('\FluentCommunity\App\Hooks\Handlers\PortalHandler') &&
            !defined('FLUENT_COMMUNITY_PLUGIN_URL')) {
            add_action('admin_notices', function() {
                ?>
                <div class="notice notice-error">
                    <p>
                        <?php 
                        echo sprintf(
                            __('%s requires %s to be installed and activated.', 'fca-content-manager'),
                            '<strong>Fluent Community Content Manager</strong>',
                            '<strong>Fluent Community</strong>'
                        );
                        ?>
                    </p>
                </div>
                <?php
            });
            return false;
        }

        return true;
    }
}

// Initialize plugin
add_action('plugins_loaded', function() {
    if (!fca_content_manager_check_dependencies()) {
        return;
    }
    
    // Load text domain
    load_plugin_textdomain('fca-content-manager', false, dirname(plugin_basename(__FILE__)) . '/languages');
    
    // Load required files
    require_once FCA_CM_PATH . 'includes/LicenseChecker.php';
    require_once FCA_CM_PATH . 'includes/Admin.php';
    require_once FCA_CM_PATH . 'includes/Ajax.php';
    require_once FCA_CM_PATH . 'includes/Database.php';
    require_once FCA_CM_PATH . 'includes/Helpers.php';
    require_once FCA_CM_PATH . 'includes/ImportExport.php';
    require_once FCA_CM_PATH . 'includes/Export.php';
    require_once FCA_CM_PATH . 'includes/Import.php';
    
    // Initialize licensing and updater
    if (!class_exists('\FCA_ContentManager\FluentLicensing')) {
        require_once FCA_CM_PATH . 'updater/FluentLicensing.php';
        require_once FCA_CM_PATH . 'updater/LicenseSettings.php';
    }
    
    try {
        $fluentLicensing = (new \FCA_ContentManager\FluentLicensing())->register([
            'version'  => FCA_CM_VERSION,
            'item_id'  => 2179, // Product ID from FluentCart
            'basename' => FCA_CM_BASENAME,
            'api_url'  => 'https://fluentcommunityaddons.com/'
        ]);
        
        (new \FCA_ContentManager\LicenseSettings())
            ->register($fluentLicensing)
            ->setConfig([
                'menu_title'      => __('Content Manager License', 'fca-content-manager'),
                'page_title'      => __('Content Manager License Settings', 'fca-content-manager'),
                'title'           => __('Fluent Community Content Manager License', 'fca-content-manager'),
                'license_key'     => __('License Key', 'fca-content-manager'),
                'purchase_url'    => 'https://fluentcommunityaddons.com/pricing/?utm_source=plugin&utm_medium=license_page&utm_campaign=content_manager',
                'account_url'     => 'https://fluentcommunityaddons.com/account',
                'plugin_name'     => __('Fluent Community Content Manager', 'fca-content-manager'),
            ])
            ->addPage([
                'type'        => 'submenu',
                'parent_slug' => 'fca-content-manager'
            ]);
    } catch (\Exception $e) {
        // Licensing initialization failed, but plugin can still work
        error_log('FCA Content Manager: Licensing initialization failed - ' . $e->getMessage());
    }
    
    // Initialize admin if in admin area
    if (is_admin()) {
        FCA_CM_Admin::get_instance();
        FCA_CM_Ajax::get_instance();
    }
    
}, 20);

// Activation hook
register_activation_hook(__FILE__, function() {
    // Set default settings if needed
    add_option('fca_content_manager_settings', [
        'enabled' => true,
        'posts_per_page' => 20,
        'courses_per_page' => 20,
    ]);
});

// Deactivation hook
register_deactivation_hook(__FILE__, function() {
    // Clean up if needed
    flush_rewrite_rules();
});

