/**
 * Fluent Community Content Manager - Import/Export
 * Handles wizard functionality for import and export operations
 */

(function($) {
    'use strict';
    
    // Ensure jQuery is available
    if (typeof $ === 'undefined') {
        console.error('FCA Content Manager: jQuery is not loaded!');
        return;
    }
    
    // ========================================================================
    // Export Wizard
    // ========================================================================
    
    const ExportWizard = {
        currentStep: 1,
        totalSteps: 3,
        selectedType: null,
        
        init: function() {
            this.bindEvents();
            this.updateNavigation();
        },
        
        bindEvents: function() {
            console.log('FCA: Binding Export Wizard events');
            
            // Content type selection
            $('.fca-cm-content-type-card:not(.disabled)').on('click', this.selectContentType.bind(this));
            console.log('FCA: Found ' + $('.fca-cm-content-type-card:not(.disabled)').length + ' content type cards');
            
            // Navigation
            $('.fca-cm-wizard-next').on('click', this.nextStep.bind(this));
            $('.fca-cm-wizard-prev').on('click', this.prevStep.bind(this));
            $('.fca-cm-wizard-export').on('click', this.generateExport.bind(this));
            $('.fca-cm-wizard-restart').on('click', this.restart.bind(this));
            
            // Format selection
            $('input[name="export_format"]').on('change', this.updateFormatOptions.bind(this));
        },
        
        selectContentType: function(e) {
            const $card = $(e.currentTarget);
            
            $('.fca-cm-content-type-card').removeClass('selected');
            $card.addClass('selected');
            
            this.selectedType = $card.data('type');
            this.showExportOptions(this.selectedType);
        },
        
        showExportOptions: function(type) {
            $('.fca-cm-export-options').hide();
            $(`.fca-cm-export-options[data-content-type="${type}"]`).show();
        },
        
        nextStep: function() {
            if (!this.validateCurrentStep()) {
                return;
            }
            
            if (this.currentStep < this.totalSteps) {
                this.currentStep++;
                this.updateSteps();
                this.updateNavigation();
                
                if (this.currentStep === 3) {
                    this.showSummary();
                }
            }
        },
        
        prevStep: function() {
            if (this.currentStep > 1) {
                this.currentStep--;
                this.updateSteps();
                this.updateNavigation();
            }
        },
        
        updateSteps: function() {
            $('.fca-cm-wizard-step').removeClass('active');
            $(`.fca-cm-wizard-step-${this.currentStep}`).addClass('active');
        },
        
        updateNavigation: function() {
            // Show/hide buttons based on step
            if (this.currentStep === 1) {
                $('.fca-cm-wizard-prev').hide();
                $('.fca-cm-wizard-next').show();
                $('.fca-cm-wizard-export').hide();
            } else if (this.currentStep === this.totalSteps) {
                $('.fca-cm-wizard-prev').show();
                $('.fca-cm-wizard-next').hide();
                $('.fca-cm-wizard-export').show();
            } else {
                $('.fca-cm-wizard-prev').show();
                $('.fca-cm-wizard-next').show();
                $('.fca-cm-wizard-export').hide();
            }
            
            $('.fca-cm-wizard-restart').hide();
        },
        
        validateCurrentStep: function() {
            if (this.currentStep === 1 && !this.selectedType) {
                alert('Please select a content type to export.');
                return false;
            }
            return true;
        },
        
        showSummary: function() {
            const format = $('input[name="export_format"]:checked').val();
            const formatLabel = format === 'json' ? 'JSON' : 'CSV';
            
            $('#fca-cm-summary-type').text(this.getTypeName(this.selectedType));
            $('#fca-cm-summary-format').text(formatLabel);
            
            // Build filters summary
            let filters = [];
            
            if (this.selectedType === 'posts') {
                const space = $('#fca-cm-export-space option:selected').text();
                const status = $('#fca-cm-export-status option:selected').text();
                
                if ($('#fca-cm-export-space').val()) {
                    filters.push(`Space: ${space}`);
                }
                if ($('#fca-cm-export-status').val()) {
                    filters.push(`Status: ${status}`);
                }
            } else if (this.selectedType === 'spaces') {
                const status = $('#fca-cm-export-spaces-status option:selected').text();
                const privacy = $('#fca-cm-export-spaces-privacy option:selected').text();
                
                if ($('#fca-cm-export-spaces-status').val()) {
                    filters.push(`Status: ${status}`);
                }
                if ($('#fca-cm-export-spaces-privacy').val()) {
                    filters.push(`Privacy: ${privacy}`);
                }
            } else if (this.selectedType === 'comments') {
                const space = $('#fca-cm-export-comments-space option:selected').text();
                if ($('#fca-cm-export-comments-space').val()) {
                    filters.push(`Space: ${space}`);
                }
            }
            
            $('#fca-cm-summary-filters').text(filters.length > 0 ? filters.join(', ') : 'None');
        },
        
        getTypeName: function(type) {
            const names = {
                'posts': 'Posts',
                'spaces': 'Spaces',
                'comments': 'Comments',
                'courses': 'Courses'
            };
            return names[type] || type;
        },
        
        updateFormatOptions: function() {
            const format = $('input[name="export_format"]:checked').val();
            
            // Disable relation checkboxes for CSV
            if (format === 'csv') {
                $('.fca-cm-checkbox-group input[type="checkbox"]').prop('disabled', true).prop('checked', false);
                $('.fca-cm-checkbox-group').css('opacity', '0.5');
            } else {
                $('.fca-cm-checkbox-group input[type="checkbox"]').prop('disabled', false).prop('checked', true);
                $('.fca-cm-checkbox-group').css('opacity', '1');
            }
        },
        
        generateExport: function() {
            $('.fca-cm-export-progress').show();
            $('.fca-cm-export-result').hide();
            $('.fca-cm-export-error').hide();
            $('.fca-cm-wizard-footer .button').hide();
            
            const format = $('input[name="export_format"]:checked').val();
            
            let data = {
                action: 'fca_cm_export_' + this.selectedType,
                nonce: fcaContentManager.nonce,
                format: format
            };
            
            // Add type-specific filters
            if (this.selectedType === 'posts') {
                data.space_id = $('#fca-cm-export-space').val();
                data.status = $('#fca-cm-export-status').val();
                
                if (format === 'json') {
                    data.include_comments = $('input[name="include_comments"]').is(':checked');
                    data.include_reactions = $('input[name="include_reactions"]').is(':checked');
                    data.include_media = $('input[name="include_media"]').is(':checked');
                    data.include_terms = $('input[name="include_terms"]').is(':checked');
                }
            } else if (this.selectedType === 'spaces') {
                data.status = $('#fca-cm-export-spaces-status').val();
                data.privacy = $('#fca-cm-export-spaces-privacy').val();
            } else if (this.selectedType === 'comments') {
                data.space_id = $('#fca-cm-export-comments-space').val();
            }
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: data,
                success: this.handleExportSuccess.bind(this),
                error: this.handleExportError.bind(this)
            });
        },
        
        handleExportSuccess: function(response) {
            $('.fca-cm-export-progress').hide();
            
            if (response.success) {
                $('#fca-cm-export-message').text(`Export file created successfully. ${response.data.total_items || 0} items exported.`);
                $('#fca-cm-download-link').attr('href', response.data.file_url);
                $('.fca-cm-export-result').show();
                $('.fca-cm-wizard-restart').show();
            } else {
                $('#fca-cm-export-error-message').text(response.data.message || 'Export failed.');
                $('.fca-cm-export-error').show();
                $('.fca-cm-wizard-prev').show();
            }
        },
        
        handleExportError: function(xhr, status, error) {
            $('.fca-cm-export-progress').hide();
            $('#fca-cm-export-error-message').text('An error occurred: ' + error);
            $('.fca-cm-export-error').show();
            $('.fca-cm-wizard-prev').show();
        },
        
        restart: function() {
            this.currentStep = 1;
            this.selectedType = null;
            $('.fca-cm-content-type-card').removeClass('selected');
            $('.fca-cm-export-result').hide();
            $('.fca-cm-export-error').hide();
            this.updateSteps();
            this.updateNavigation();
        }
    };
    
    // ========================================================================
    // Import Wizard
    // ========================================================================
    
    const ImportWizard = {
        currentStep: 1,
        totalSteps: 4,
        uploadedFile: null,
        parsedData: null,
        fieldMapping: {},
        
        init: function() {
            this.bindEvents();
            this.updateNavigation();
        },
        
        bindEvents: function() {
            console.log('FCA: Binding Import Wizard events');
            
            // File upload
            $('#fca-cm-browse-btn').on('click', () => $('#fca-cm-import-file').click());
            $('#fca-cm-import-file').on('change', this.handleFileSelect.bind(this));
            $('.fca-cm-remove-file').on('click', this.removeFile.bind(this));
            
            console.log('FCA: Found browse button:', $('#fca-cm-browse-btn').length);
            console.log('FCA: Found template cards:', $('.fca-cm-template-card').length);
            
            // Drag and drop
            const $dropzone = $('#fca-cm-dropzone');
            // Don't add click to dropzone - conflicts with browse button
            
            $dropzone.on('dragover', function(e) {
                e.preventDefault();
                e.stopPropagation();
                $(this).addClass('drag-over');
            });
            
            $dropzone.on('dragleave', function(e) {
                e.preventDefault();
                e.stopPropagation();
                $(this).removeClass('drag-over');
            });
            
            $dropzone.on('drop', this.handleFileDrop.bind(this));
            
            // Template downloads
            $('.fca-cm-template-card').on('click', this.downloadTemplate.bind(this));
            
            // Navigation
            $('.fca-cm-wizard-next').on('click', this.nextStep.bind(this));
            $('.fca-cm-wizard-prev').on('click', this.prevStep.bind(this));
            $('.fca-cm-wizard-import').on('click', this.startImport.bind(this));
            $('.fca-cm-wizard-restart').on('click', this.restart.bind(this));
        },
        
        handleFileSelect: function(e) {
            const file = e.target.files[0];
            if (file) {
                this.processFile(file);
            }
        },
        
        handleFileDrop: function(e) {
            e.preventDefault();
            e.stopPropagation();
            $('#fca-cm-dropzone').removeClass('drag-over');
            
            const file = e.originalEvent.dataTransfer.files[0];
            if (file) {
                this.processFile(file);
            }
        },
        
        processFile: function(file) {
            // Validate file
            const validTypes = ['application/json', 'text/csv', 'application/vnd.ms-excel'];
            const validExtensions = ['json', 'csv'];
            const ext = file.name.split('.').pop().toLowerCase();
            
            if (!validExtensions.includes(ext)) {
                alert('Please upload a JSON or CSV file.');
                return;
            }
            
            if (file.size > 50 * 1024 * 1024) {
                alert('File size exceeds 50MB limit.');
                return;
            }
            
            this.uploadedFile = file;
            this.showFileInfo(file);
            
            // Enable next button
            $('.fca-cm-wizard-next').prop('disabled', false);
            
            // Upload and parse file
            this.uploadFile(file);
        },
        
        showFileInfo: function(file) {
            const sizeKB = (file.size / 1024).toFixed(2);
            const sizeMB = (file.size / (1024 * 1024)).toFixed(2);
            const sizeText = file.size > 1024 * 1024 ? `${sizeMB} MB` : `${sizeKB} KB`;
            
            $('#fca-cm-file-name').text(file.name);
            $('#fca-cm-file-size').text(sizeText);
            
            $('.fca-cm-upload-dropzone').hide();
            $('.fca-cm-upload-result').show();
        },
        
        removeFile: function() {
            this.uploadedFile = null;
            this.parsedData = null;
            $('#fca-cm-import-file').val('');
            $('.fca-cm-upload-result').hide();
            $('.fca-cm-upload-dropzone').show();
            $('.fca-cm-wizard-next').prop('disabled', true);
        },
        
        uploadFile: function(file) {
            const formData = new FormData();
            formData.append('action', 'fca_cm_upload_import_file');
            formData.append('nonce', fcaContentManager.nonce);
            formData.append('file', file);
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: this.handleUploadSuccess.bind(this),
                error: this.handleUploadError.bind(this)
            });
        },
        
        handleUploadSuccess: function(response) {
            if (response.success) {
                this.parsedData = response.data;
                console.log('File parsed:', this.parsedData);
            } else {
                alert('Error parsing file: ' + (response.data.message || 'Unknown error'));
                this.removeFile();
            }
        },
        
        handleUploadError: function(xhr, status, error) {
            alert('Upload error: ' + error);
            this.removeFile();
        },
        
        downloadTemplate: function(e) {
            e.preventDefault();
            const $card = $(e.currentTarget);
            const template = $card.data('template');
            const format = $card.data('format');
            
            window.location.href = `${fcaContentManager.pluginUrl}templates/${template}-sample.${format}`;
        },
        
        nextStep: function() {
            if (!this.validateCurrentStep()) {
                return;
            }
            
            if (this.currentStep < this.totalSteps) {
                this.currentStep++;
                this.updateSteps();
                this.updateNavigation();
                
                if (this.currentStep === 2 && this.parsedData && this.parsedData.format === 'csv') {
                    this.renderFieldMapping();
                } else if (this.currentStep === 2 && this.parsedData && this.parsedData.format === 'json') {
                    // Skip field mapping for JSON, go to step 3
                    this.currentStep = 3;
                    this.updateSteps();
                    this.updateNavigation();
                    this.showPreview();
                } else if (this.currentStep === 3) {
                    this.showPreview();
                }
            }
        },
        
        prevStep: function() {
            if (this.currentStep > 1) {
                this.currentStep--;
                
                // Skip field mapping step when going back from JSON
                if (this.currentStep === 2 && this.parsedData && this.parsedData.format === 'json') {
                    this.currentStep = 1;
                }
                
                this.updateSteps();
                this.updateNavigation();
            }
        },
        
        updateSteps: function() {
            $('.fca-cm-wizard-step').removeClass('active');
            $(`.fca-cm-wizard-step-${this.currentStep}`).addClass('active');
        },
        
        updateNavigation: function() {
            if (this.currentStep === 1) {
                $('.fca-cm-wizard-prev').hide();
                $('.fca-cm-wizard-next').show();
                $('.fca-cm-wizard-import').hide();
                $('.fca-cm-wizard-next').prop('disabled', !this.uploadedFile);
            } else if (this.currentStep === 3) {
                $('.fca-cm-wizard-prev').show();
                $('.fca-cm-wizard-next').hide();
                $('.fca-cm-wizard-import').show();
            } else if (this.currentStep === 4) {
                $('.fca-cm-wizard-prev').hide();
                $('.fca-cm-wizard-next').hide();
                $('.fca-cm-wizard-import').hide();
                $('.fca-cm-wizard-restart').show();
            } else {
                $('.fca-cm-wizard-prev').show();
                $('.fca-cm-wizard-next').show();
                $('.fca-cm-wizard-import').hide();
            }
        },
        
        validateCurrentStep: function() {
            if (this.currentStep === 1 && !this.uploadedFile) {
                alert('Please upload a file to continue.');
                return false;
            }
            return true;
        },
        
        renderFieldMapping: function() {
            if (!this.parsedData || !this.parsedData.headers) {
                return;
            }
            
            const $tbody = $('#fca-cm-field-mapping-body');
            $tbody.empty();
            
            // Field options based on content type
            const fieldOptions = this.getFieldOptions(this.parsedData.content_type);
            
            this.parsedData.headers.forEach((header, index) => {
                const sample = this.parsedData.sample && this.parsedData.sample[0] ? 
                              (this.parsedData.sample[0][header] || '') : '';
                
                const mappedField = this.autoMapField(header);
                
                const $row = $('<tr>').html(`
                    <td><strong>${header}</strong></td>
                    <td>${sample}</td>
                    <td>
                        <select class="fca-cm-field-map" data-column="${header}">
                            <option value="">-- Skip --</option>
                            ${fieldOptions}
                        </select>
                    </td>
                `);
                
                $tbody.append($row);
                
                // Set auto-mapped value
                if (mappedField) {
                    $row.find('select').val(mappedField);
                    this.fieldMapping[header] = mappedField;
                }
            });
            
            // Update mapping on change
            $('.fca-cm-field-map').on('change', (e) => {
                const $select = $(e.target);
                this.fieldMapping[$select.data('column')] = $select.val();
            });
        },
        
        getFieldOptions: function(contentType) {
            const commonFields = {
                'posts': ['title', 'slug', 'message', 'type', 'space_slug', 'author_email', 'status', 'privacy', 'topics', 'created_at'],
                'spaces': ['title', 'slug', 'description', 'type', 'privacy', 'status', 'logo', 'cover_photo', 'parent_slug', 'serial', 'creator_email', 'created_at']
            };
            
            const fields = commonFields[contentType] || commonFields['posts'];
            
            return fields.map(field => `<option value="${field}">${field}</option>`).join('');
        },
        
        autoMapField: function(header) {
            const normalized = header.toLowerCase().replace(/[_\s-]+/g, '_');
            const mapping = {
                'title': 'title',
                'name': 'title',
                'slug': 'slug',
                'message': 'message',
                'content': 'message',
                'body': 'message',
                'type': 'type',
                'space': 'space_slug',
                'space_slug': 'space_slug',
                'author': 'author_email',
                'author_email': 'author_email',
                'user_email': 'author_email',
                'status': 'status',
                'privacy': 'privacy',
                'topics': 'topics',
                'tags': 'topics',
                'created_at': 'created_at',
                'created': 'created_at',
                'date': 'created_at'
            };
            
            return mapping[normalized] || null;
        },
        
        showPreview: function() {
            if (!this.parsedData) {
                return;
            }
            
            $('#fca-cm-preview-total').text(this.parsedData.total_items || 0);
            $('#fca-cm-preview-content-type').text(this.parsedData.content_type || 'Unknown');
            
            // Show sample data
            const sample = this.parsedData.sample || [];
            let sampleHtml = '<pre>' + JSON.stringify(sample, null, 2) + '</pre>';
            $('#fca-cm-preview-sample-data').html(sampleHtml);
        },
        
        startImport: function() {
            this.currentStep = 4;
            this.updateSteps();
            this.updateNavigation();
            
            // Show progress
            $('#fca-cm-import-progress').show();
            $('#fca-cm-import-results').hide();
            
            // Collect options
            const options = {
                duplicate_handling: $('input[name="duplicate_handling"]:checked').val(),
                create_spaces: $('input[name="create_spaces"]').is(':checked'),
                create_terms: $('input[name="create_terms"]').is(':checked'),
                dry_run: $('input[name="dry_run"]').is(':checked')
            };
            
            // Perform import
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'fca_cm_process_import',
                    nonce: fcaContentManager.nonce,
                    file_path: this.parsedData.file_path,
                    content_type: this.parsedData.content_type,
                    format: this.parsedData.format,
                    field_mapping: this.fieldMapping,
                    options: options
                },
                success: this.handleImportSuccess.bind(this),
                error: this.handleImportError.bind(this)
            });
        },
        
        handleImportSuccess: function(response) {
            $('#fca-cm-import-progress').hide();
            
            if (response.success) {
                const results = response.data;
                
                $('.fca-cm-result-imported').text(results.imported || 0);
                $('.fca-cm-result-updated').text(results.updated || 0);
                $('.fca-cm-result-skipped').text(results.skipped || 0);
                $('.fca-cm-result-errors').text(results.errors ? results.errors.length : 0);
                
                if (results.errors && results.errors.length > 0) {
                    const $errorList = $('#fca-cm-error-list');
                    $errorList.empty();
                    results.errors.forEach(error => {
                        $errorList.append(`<li>${error}</li>`);
                    });
                    $('#fca-cm-import-errors').show();
                }
                
                $('#fca-cm-import-results').show();
            } else {
                alert('Import failed: ' + (response.data.message || 'Unknown error'));
            }
        },
        
        handleImportError: function(xhr, status, error) {
            $('#fca-cm-import-progress').hide();
            alert('Import error: ' + error);
        },
        
        restart: function() {
            window.location.reload();
        }
    };
    
    // ========================================================================
    // Initialize
    // ========================================================================
    
    $(document).ready(function() {
        // Initialize export wizard if on export page
        if ($('.fca-cm-export-wizard').length) {
            console.log('FCA Content Manager: Initializing Export Wizard');
            ExportWizard.init();
        }
        
        // Initialize import wizard if on import page
        if ($('.fca-cm-import-wizard').length) {
            console.log('FCA Content Manager: Initializing Import Wizard');
            ImportWizard.init();
        }
    });
    
})(jQuery);
