/**
 * Fluent Community Content Manager - Admin JavaScript
 * Handles all interactions and AJAX operations
 */

(function($) {
    'use strict';

    const FCA_CM = {
        // Current state
        currentPage: 1,
        currentPostId: null,
        currentCourseId: null,
        filters: {},
        selectedItems: new Set(),
        allTerms: [],
        selectedPosts: [],
        isLicensed: typeof fcaCMData !== 'undefined' ? fcaCMData.isLicensed : false,

        /**
         * Initialize
         */
        init: function() {
            this.checkLicenseRestrictions();
            this.initEventListeners();
            this.loadPageContent();
        },
        
        /**
         * Check license and apply restrictions
         */
        checkLicenseRestrictions: function() {
            if (this.isLicensed) {
                return; // No restrictions for Pro users
            }
            
            // Apply restrictions based on current page
            const currentPage = this.getCurrentPageType();
            
                if (['members', 'media', 'metadata', 'spaces', 'export', 'import'].includes(currentPage)) {
                    this.showProOverlay(currentPage);
                } else if (currentPage === 'courses') {
                this.makeCoursesReadOnly();
            } else if (currentPage === 'comments') {
                this.showViewLimitNotice('comments', 50);
            } else if (currentPage === 'reactions') {
                this.showViewLimitNotice('reactions', 50);
            }
        },
        
        /**
         * Get current page type
         */
        getCurrentPageType: function() {
            if ($('#fca-cm-posts-list').length) return 'posts';
            if ($('#fca-cm-courses-list').length) return 'courses';
            if ($('#fca-cm-members-list').length) return 'members';
            if ($('#fca-cm-comments-list').length) return 'comments';
            if ($('#fca-cm-reactions-list').length) return 'reactions';
            if ($('#fca-cm-media-list').length) return 'media';
            if ($('#fca-cm-metadata-list').length) return 'metadata';
            if ($('#fca-cm-spaces-list').length) return 'spaces';
            if ($('.fca-cm-export-wizard').length) return 'export';
            if ($('.fca-cm-import-wizard').length) return 'import';
            return 'overview';
        },
        
        /**
         * Show Pro overlay on locked pages
         */
        showProOverlay: function(pageType) {
            const messages = {
                members: 'Manage community members, handle enrollments, and control user access across spaces and courses.',
                media: 'Manage media files, see where they\'re used, and keep your uploads organized.',
                metadata: 'View and edit all metadata entries across your community.',
                spaces: 'Manage spaces with full editing capabilities and bulk operations.',
                export: 'Export your content for backup, migration, or analysis.',
                import: 'Import content from CSV and JSON files with advanced mapping options.'
            };
            
            const overlay = $(`
                <div class="fca-cm-pro-overlay">
                    <div class="fca-cm-pro-message">
                        <span class="dashicons dashicons-star-filled"></span>
                        <h3>${fcaCMData.strings.proFeature}</h3>
                        <p>${messages[pageType]}</p>
                        <a href="${fcaCMData.upgradeUrl}" class="fca-cm-upgrade-button" target="_blank">
                            ${fcaCMData.strings.upgradeToPro}
                        </a>
                    </div>
                </div>
            `);
            
            $('.fca-cm-content').css('position', 'relative').append(overlay);
        },
        
        /**
         * Make courses read-only for free users
         */
        makeCoursesReadOnly: function() {
            // Add read-only badge to header
            $('.fca-cm-page-title').append(`
                <span class="fca-cm-read-only-badge">${fcaCMData.strings.readOnly}</span>
            `);
            
            // Disable edit buttons after they're rendered
            $(document).on('DOMNodeInserted', '.fca-cm-detail-content', function() {
                $('.fca-cm-action-button').not('.fca-cm-disabled').each(function() {
                    const $btn = $(this);
                    const originalText = $btn.text();
                    
                    $btn.addClass('fca-cm-free-disabled')
                        .attr('title', fcaCMData.strings.readOnly)
                        .on('click', function(e) {
                            e.preventDefault();
                            e.stopPropagation();
                            alert('This feature requires a Pro license. Upgrade to edit courses!');
                            return false;
                        });
                });
            });
            
            // Disable bulk edit and other action buttons
            $(document).on('click', '.fca-cm-bulk-action, [data-action]', function(e) {
                if (!FCA_CM.isLicensed && $(this).closest('#fca-cm-courses-list').length) {
                    e.preventDefault();
                    e.stopPropagation();
                    alert('Course editing requires a Pro license!');
                    return false;
                }
            });
        },
        
        /**
         * Show view limit notice
         */
        showViewLimitNotice: function(type, limit) {
            const notice = $(`
                <div class="fca-cm-view-limit-notice">
                    <p>
                        <strong>Free Version:</strong> Viewing first ${limit} ${type}. 
                        Upgrade to Pro for unlimited access and editing capabilities.
                    </p>
                    <a href="${fcaCMData.upgradeUrl}" class="fca-cm-upgrade-button" target="_blank">
                        ${fcaCMData.strings.upgradeToPro}
                    </a>
                </div>
            `);
            
            $('.fca-cm-header').after(notice);
            
            // Disable edit/delete buttons on free tier
            $(document).on('DOMNodeInserted', '.fca-cm-list', function() {
                $('.fca-cm-list-item').each(function(index) {
                    if (index < limit) {
                        $(this).find('.fca-cm-delete-btn, .fca-cm-edit-btn').addClass('fca-cm-free-disabled')
                            .attr('title', 'Pro feature - Upgrade to edit')
                            .on('click', function(e) {
                                e.preventDefault();
                                e.stopPropagation();
                                alert(`Editing ${type} requires a Pro license!`);
                                return false;
                            });
                    }
                });
            });
            
            // Disable bulk operations
            $('#fca-cm-bulk-delete-' + type).addClass('fca-cm-free-disabled')
                .on('click', function(e) {
                    e.preventDefault();
                    alert(`Bulk operations require a Pro license!`);
                    return false;
                });
        },

        /**
         * Initialize event listeners
         */
        initEventListeners: function() {
            // Common listeners
            $(document).on('click', '#fca-cm-refresh', () => this.refreshList());
            $(document).on('click', '#fca-cm-debug-counts', () => this.showDebugInfo());
            $(document).on('click', '.fca-cm-modal-backdrop, [data-close]', (e) => this.closeModal(e));
            
            // Overview page
            if ($('#fca-cm-stats-grid').length) {
                this.initOverviewListeners();
            }
            
            // Posts page
            if ($('#fca-cm-posts-list').length) {
                this.initPostsListeners();
            }
            
            // Courses page
            if ($('#fca-cm-courses-list').length) {
                this.initCoursesListeners();
            }
            
            // Members page
            if ($('#fca-cm-members-list').length) {
                this.initMembersListeners();
            }
            
            // Comments page
            if ($('#fca-cm-comments-list').length) {
                this.initCommentsListeners();
            }
            
            // Reactions page
            if ($('#fca-cm-reactions-table').length) {
                this.initReactionsListeners();
            }
            
            // Media page
            if ($('#fca-cm-media-list').length) {
                this.initMediaListeners();
            }
            
            // Spaces page
            if ($('#fca-cm-spaces-list').length) {
                this.initSpacesListeners();
            }
            
            // Metadata page
            if ($('#fca-cm-metadata-table').length) {
                this.initMetadataListeners();
            }
        },

        /**
         * Initialize overview page listeners
         */
        initOverviewListeners: function() {
            $(document).on('click', '#fca-cm-refresh-overview', () => {
                this.loadOverview();
            });
            
            this.loadOverview();
        },
        
        /**
         * Load overview statistics
         */
        loadOverview: function() {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_stats',
                    nonce: fcaCMData.nonce
                },
                success: (response) => {
                    if (response.success) {
                        this.renderOverview(response.data);
                    } else {
                        this.showError('Failed to load statistics');
                    }
                },
                error: () => {
                    this.showError('Failed to load statistics');
                }
            });
        },
        
        /**
         * Render overview statistics
         */
        renderOverview: function(stats) {
            const $grid = $('#fca-cm-stats-grid');
            $grid.empty();
            
            const statsCards = [
                {
                    title: 'Posts',
                    icon: 'dashicons-admin-post',
                    total: stats.posts.total,
                    subtitle: `${stats.posts.published} Published, ${stats.posts.draft} Draft`,
                    link: 'admin.php?page=fca-cm-posts',
                    color: '#2271b1'
                },
                {
                    title: 'Comments',
                    icon: 'dashicons-admin-comments',
                    total: stats.comments.total,
                    subtitle: `${stats.comments.published} Published, ${stats.comments.pending} Pending`,
                    link: 'admin.php?page=fca-cm-comments',
                    color: '#00a32a'
                },
                {
                    title: 'Courses',
                    icon: 'dashicons-book-alt',
                    total: stats.courses.total,
                    subtitle: `${stats.courses.published} Published`,
                    link: 'admin.php?page=fca-cm-courses',
                    color: '#8c3bda'
                },
                {
                    title: 'Lessons',
                    icon: 'dashicons-welcome-learn-more',
                    total: stats.lessons.total,
                    subtitle: 'Course lessons',
                    link: 'admin.php?page=fca-cm-courses',
                    color: '#f0b849'
                },
                {
                    title: 'Members',
                    icon: 'dashicons-admin-users',
                    total: stats.members.total,
                    subtitle: `${stats.members.active} Active`,
                    link: 'admin.php?page=fca-cm-members',
                    color: '#d63638'
                },
                {
                    title: 'Reactions',
                    icon: 'dashicons-heart',
                    total: stats.reactions.total,
                    subtitle: `${stats.reactions.posts} on Posts, ${stats.reactions.comments} on Comments`,
                    link: 'admin.php?page=fca-cm-reactions',
                    color: '#e91e63'
                },
                {
                    title: 'Media Files',
                    icon: 'dashicons-format-image',
                    total: stats.media.total,
                    subtitle: 'Images, videos, and files',
                    link: 'admin.php?page=fca-cm-media',
                    color: '#00bcd4'
                },
                {
                    title: 'Spaces',
                    icon: 'dashicons-admin-multisite',
                    total: stats.spaces.total,
                    subtitle: `${stats.spaces.published || 0} Published`,
                    link: 'admin.php?page=fca-cm-spaces',
                    color: '#4caf50'
                }
            ];
            
            statsCards.forEach(card => {
                const $card = $(`
                    <div class="fca-cm-stat-card">
                        <div class="fca-cm-stat-icon" style="background-color: ${card.color}20; color: ${card.color};">
                            <span class="dashicons ${card.icon}"></span>
                        </div>
                        <div class="fca-cm-stat-content">
                            <div class="fca-cm-stat-number">${card.total || 0}</div>
                            <div class="fca-cm-stat-title">${card.title}</div>
                            <div class="fca-cm-stat-subtitle">${card.subtitle}</div>
                        </div>
                        <div class="fca-cm-stat-action">
                            <a href="${card.link}" class="fca-cm-btn fca-cm-btn-sm fca-cm-btn-secondary">
                                View All
                                <span class="dashicons dashicons-arrow-right-alt2"></span>
                            </a>
                        </div>
                    </div>
                `);
                $grid.append($card);
            });
        },
        
        /**
         * Initialize posts page listeners
         */
        initPostsListeners: function() {
            // Load all terms
            this.loadAllTerms();
            
            // Search
            let searchTimer;
            $(document).on('input', '#fca-cm-search', (e) => {
                clearTimeout(searchTimer);
                searchTimer = setTimeout(() => {
                    this.filters.search = $(e.target).val();
                    this.currentPage = 1;
                    this.loadPosts();
                }, 500);
            });

            // Filters
            $(document).on('change', '#fca-cm-filter-status, #fca-cm-filter-space, #fca-cm-per-page', () => {
                this.filters.status = $('#fca-cm-filter-status').val();
                this.filters.space_id = $('#fca-cm-filter-space').val();
                this.filters.per_page = $('#fca-cm-per-page').val();
                this.currentPage = 1;
                this.loadPosts();
            });

            // Select all
            $(document).on('change', '#fca-cm-select-all', (e) => {
                const checked = $(e.target).prop('checked');
                $('.fca-cm-item-checkbox').prop('checked', checked);
                this.updateBulkButtons();
            });

            // Individual checkbox
            $(document).on('change', '.fca-cm-item-checkbox', () => {
                this.updateBulkButtons();
            });

            // Bulk delete
            $(document).on('click', '#fca-cm-bulk-delete', () => {
                this.bulkDeletePosts();
            });

            // Post item click
            $(document).on('click', '.fca-cm-list-item', (e) => {
                if (!$(e.target).is('input[type="checkbox"]')) {
                    const postId = $(e.currentTarget).data('post-id');
                    this.loadPostDetail(postId);
                }
            });

            // Close detail panel
            $(document).on('click', '#fca-cm-close-detail', () => {
                this.closePostDetail();
            });

            // Save post
            $(document).on('click', '#fca-cm-save-post', () => {
                this.savePost();
            });

            // Delete post
            $(document).on('click', '#fca-cm-delete-post', () => {
                this.deletePost();
            });
            
            // Bulk edit
            $(document).on('click', '#fca-cm-bulk-edit', () => {
                this.enterBulkEditMode();
            });
            
            $(document).on('click', '#fca-cm-close-bulk-edit', () => {
                this.exitBulkEditMode();
            });
            
            $(document).on('click', '#fca-cm-save-bulk', () => {
                this.saveBulkEdit();
            });
            
            // Term management
            $(document).on('change', '#fca-cm-add-term-select', (e) => {
                this.addTermToPost(e);
            });
            
            $(document).on('click', '.fca-cm-tag-remove', (e) => {
                this.removeTermFromPost(e);
            });
            
            // Comment modal
            $(document).on('click', '.fca-cm-relationship-item', (e) => {
                this.openCommentModal(e);
            });
            
            $(document).on('click', '.fca-cm-modal-close, .fca-cm-modal-overlay', () => {
                this.closeModal();
            });
            
            // Bulk add term
            $(document).on('change', '#fca-cm-bulk-add-term', (e) => {
                this.bulkAddTerm(e);
            });
            
            // Debug button
            $(document).on('click', '#fca-cm-debug-counts', () => {
                this.showDebugInfo();
            });

            // Load initial data
            this.loadPosts();
        },

        /**
         * Initialize courses page listeners
         */
        initCoursesListeners: function() {
            // Search
            let searchTimer;
            $(document).on('input', '#fca-cm-course-search', (e) => {
                clearTimeout(searchTimer);
                searchTimer = setTimeout(() => {
                    this.filters.search = $(e.target).val();
                    this.currentPage = 1;
                    this.loadCourses();
                }, 500);
            });

            // Filters
            $(document).on('change', '#fca-cm-course-status, #fca-cm-course-space', () => {
                this.filters.status = $('#fca-cm-course-status').val();
                this.filters.space_id = $('#fca-cm-course-space').val();
                this.currentPage = 1;
                this.loadCourses();
            });

            // Course item click
            $(document).on('click', '.fca-cm-course-item', (e) => {
                const courseId = $(e.currentTarget).data('course-id');
                this.loadCourseDetail(courseId);
            });

            // Close course detail
            $(document).on('click', '#fca-cm-close-course-detail', () => {
                this.closeCourseDetail();
            });

            // Edit course
            $(document).on('click', '#fca-cm-edit-course', () => {
                this.openCourseModal();
            });

            // Save course
            $(document).on('click', '#fca-cm-save-course-edit', () => {
                this.saveCourse();
            });

            // Edit lesson
            $(document).on('click', '.fca-cm-edit-lesson', (e) => {
                const lessonId = $(e.currentTarget).data('lesson-id');
                this.openLessonModal(lessonId);
            });

            // Save lesson
            $(document).on('click', '#fca-cm-save-lesson-edit', () => {
                this.saveLesson();
            });
            
            // Edit course item (section or lesson)
            $(document).on('click', '.fca-cm-edit-item', (e) => {
                e.stopPropagation();
                const itemId = $(e.currentTarget).data('item-id');
                const itemType = $(e.currentTarget).data('item-type');
                this.editCourseItem(itemId, itemType);
            });
            
            // Delete course item
            $(document).on('click', '.fca-cm-delete-item', (e) => {
                e.stopPropagation();
                const itemId = $(e.currentTarget).data('item-id');
                if (confirm('Are you sure you want to delete this item?')) {
                    this.deleteCourseItem(itemId);
                }
            });
            
            // Duplicate course item
            $(document).on('click', '.fca-cm-duplicate-item', (e) => {
                e.stopPropagation();
                const itemId = $(e.currentTarget).data('item-id');
                this.duplicateCourseItem(itemId);
            });
            
            // Move course item
            $(document).on('click', '.fca-cm-move-item', (e) => {
                e.stopPropagation();
                const itemId = $(e.currentTarget).data('item-id');
                const itemType = $(e.currentTarget).data('item-type');
                this.openMoveItemModal(itemId, itemType);
            });
            
            // Course checkboxes
            $(document).on('change', '#fca-cm-select-all-courses', (e) => {
                const isChecked = $(e.target).is(':checked');
                $('.fca-cm-course-checkbox').prop('checked', isChecked);
                $('.fca-cm-course-item').toggleClass('selected', isChecked);
                this.updateCourseBulkButtons();
            });
            
            $(document).on('change', '.fca-cm-course-checkbox', (e) => {
                const $item = $(e.target).closest('.fca-cm-course-item');
                $item.toggleClass('selected', $(e.target).is(':checked'));
                
                const totalCheckboxes = $('.fca-cm-course-checkbox').length;
                const checkedCheckboxes = $('.fca-cm-course-checkbox:checked').length;
                $('#fca-cm-select-all-courses').prop('checked', totalCheckboxes === checkedCheckboxes);
                
                this.updateCourseBulkButtons();
            });
            
            // Course bulk edit
            $(document).on('click', '#fca-cm-bulk-edit-courses', () => {
                const selected = $('.fca-cm-course-checkbox:checked').map(function() { return $(this).val(); }).get();
                if (selected.length > 0) {
                    this.openCourseBulkEditModal(selected);
                }
            });
            
            // Course bulk delete
            $(document).on('click', '#fca-cm-bulk-delete-courses', () => {
                const selected = $('.fca-cm-course-checkbox:checked').map(function() { return $(this).val(); }).get();
                if (selected.length > 0 && confirm(`Are you sure you want to delete ${selected.length} courses? This cannot be undone.`)) {
                    this.bulkDeleteCourses(selected);
                }
            });
            
            // Edit course details button - opens modal
            $(document).on('click', '#fca-cm-edit-course-details', () => {
                if (this.currentCourseData) {
                    this.openCourseDetailsModal();
                }
            });
            
            // Save course details
            $(document).on('click', '#fca-cm-save-course-details', () => {
                this.saveCourseDetails();
            });
            
            // Save bulk course edit
            $(document).on('click', '#fca-cm-save-bulk-courses', () => {
                this.saveCourseBulkEdit();
            });
            
            // Close course details modal
            $(document).on('click', '#fca-cm-course-details-modal .fca-cm-modal-close, #fca-cm-course-details-modal .fca-cm-modal-overlay', () => {
                $('#fca-cm-course-details-modal').fadeOut(200);
            });
            
            // Close course bulk edit modal
            $(document).on('click', '#fca-cm-course-bulk-edit-modal .fca-cm-modal-close, #fca-cm-course-bulk-edit-modal .fca-cm-modal-overlay', () => {
                $('#fca-cm-course-bulk-edit-modal').fadeOut(200);
            });
            
            // Move modal - course selection change
            $(document).on('change', '#fca-cm-move-target-course', (e) => {
                const courseId = $(e.target).val();
                if (courseId && this.moveItemType === 'lesson') {
                    this.loadSectionsForMove(courseId);
                }
            });
            
            // Move modal - confirm button
            $(document).on('click', '#fca-cm-confirm-move', () => {
                this.confirmMoveItem();
            });
            
            // Move modal - close
            $(document).on('click', '#fca-cm-move-item-modal .fca-cm-modal-close, #fca-cm-move-item-modal .fca-cm-modal-overlay', () => {
                $('#fca-cm-move-item-modal').fadeOut(200);
            });

            // Delete lesson
            $(document).on('click', '.fca-cm-delete-lesson', (e) => {
                const lessonId = $(e.currentTarget).data('lesson-id');
                this.deleteLesson(lessonId);
            });

            // Refresh courses
            $(document).on('click', '#fca-cm-refresh-courses', () => {
                this.loadCourses();
            });

            // Load initial data
            this.loadCourses();
        },

        /**
         * Initialize comments page listeners
         */
        initCommentsListeners: function() {
            let searchTimer;
            $(document).on('input', '#fca-cm-comments-search', (e) => {
                clearTimeout(searchTimer);
                searchTimer = setTimeout(() => {
                    this.filters.search = $(e.target).val();
                    this.currentPage = 1;
                    this.loadComments();
                }, 500);
            });

            $(document).on('change', '#fca-cm-comments-status, #fca-cm-comments-per-page', () => {
                this.filters.status = $('#fca-cm-comments-status').val();
                this.filters.per_page = $('#fca-cm-comments-per-page').val();
                this.currentPage = 1;
                this.loadComments();
            });

            $(document).on('click', '.fca-cm-comment-item', (e) => {
                const commentId = $(e.currentTarget).data('comment-id');
                this.loadCommentDetail(commentId);
            });

            $(document).on('click', '#fca-cm-refresh-comments', () => {
                this.loadComments();
            });
            
            // Close comment detail
            $(document).on('click', '#fca-cm-close-comment', () => {
                $('#fca-cm-comment-detail .fca-cm-detail-content').hide();
                $('#fca-cm-comment-detail .fca-cm-detail-empty').show();
                $('.fca-cm-comment-item').removeClass('active');
            });
            
            // Save comment
            $(document).on('click', '#fca-cm-save-comment', () => {
                this.saveComment();
            });
            
            // Space filter
            $(document).on('change', '#fca-cm-comments-space', () => {
                this.filters.space_id = $('#fca-cm-comments-space').val();
                this.currentPage = 1;
                this.loadComments();
            });
            
            // Bulk delete comments
            $(document).on('click', '#fca-cm-bulk-delete-comments', () => {
                const selected = this.getSelectedItems('.fca-cm-comment-checkbox:checked');
                if (selected.length > 0) {
                    this.bulkDeleteComments(selected);
                }
            });

            this.loadComments();
        },

        /**
         * Initialize reactions page listeners
         */
        initReactionsListeners: function() {
            $(document).on('change', '#fca-cm-reactions-type, #fca-cm-reactions-per-page', () => {
                this.filters.reaction_type = $('#fca-cm-reactions-type').val();
                this.filters.per_page = $('#fca-cm-reactions-per-page').val();
                this.currentPage = 1;
                this.loadReactions();
            });

            $(document).on('click', '.fca-cm-delete-reaction', (e) => {
                const reactionId = $(e.currentTarget).data('reaction-id');
                this.deleteReaction(reactionId);
            });

            $(document).on('click', '#fca-cm-refresh-reactions', () => {
                this.loadReactions();
            });

            this.loadReactions();
        },

        /**
         * Initialize media page listeners
         */
        initMediaListeners: function() {
            // Search
            let searchTimer;
            $(document).on('input', '#fca-cm-media-search', (e) => {
                clearTimeout(searchTimer);
                searchTimer = setTimeout(() => {
                    this.filters.search = $(e.target).val();
                    this.currentPage = 1;
                    this.loadMedia();
                }, 500);
            });
            
            // Filters
            $(document).on('change', '#fca-cm-media-type, #fca-cm-media-space, #fca-cm-media-per-page', () => {
                this.filters.media_type = $('#fca-cm-media-type').val();
                this.filters.space_id = $('#fca-cm-media-space').val();
                this.filters.per_page = $('#fca-cm-media-per-page').val();
                this.currentPage = 1;
                this.loadMedia();
            });
            
            // Select all
            $(document).on('change', '#fca-cm-select-all-media', (e) => {
                const isChecked = $(e.target).is(':checked');
                $('.fca-cm-media-checkbox').prop('checked', isChecked);
                $('.fca-cm-media-item').toggleClass('selected', isChecked);
                this.updateMediaBulkButtons();
            });
            
            // Individual checkbox
            $(document).on('change', '.fca-cm-media-checkbox', (e) => {
                const $item = $(e.target).closest('.fca-cm-media-item');
                $item.toggleClass('selected', $(e.target).is(':checked'));
                
                const totalCheckboxes = $('.fca-cm-media-checkbox').length;
                const checkedCheckboxes = $('.fca-cm-media-checkbox:checked').length;
                $('#fca-cm-select-all-media').prop('checked', totalCheckboxes === checkedCheckboxes);
                
                this.updateMediaBulkButtons();
            });
            
            // Bulk delete
            $(document).on('click', '#fca-cm-bulk-delete-media', () => {
                const selected = $('.fca-cm-media-checkbox:checked').map(function() { return $(this).val(); }).get();
                if (selected.length > 0 && confirm(`Are you sure you want to delete ${selected.length} media files? This cannot be undone.`)) {
                    this.bulkDeleteMedia(selected);
                }
            });

            // Media item click
            $(document).on('click', '.fca-cm-media-item', (e) => {
                if (!$(e.target).is('input[type="checkbox"]')) {
                    const mediaId = $(e.currentTarget).data('media-id');
                    this.loadMediaDetail(mediaId);
                }
            });
            
            // Close detail
            $(document).on('click', '#fca-cm-close-media', () => {
                $('#fca-cm-media-detail .fca-cm-detail-content').hide();
                $('#fca-cm-media-detail .fca-cm-detail-empty').show();
                $('.fca-cm-media-item').removeClass('active');
            });
            
            // Delete single media
            $(document).on('click', '#fca-cm-delete-media', () => {
                if (this.currentMediaId && confirm('Are you sure you want to delete this media file?')) {
                    this.deleteMedia(this.currentMediaId);
                }
            });

            $(document).on('click', '#fca-cm-refresh-media', () => {
                this.loadMedia();
            });

            this.loadMedia();
        },
        
        /**
         * Update media bulk buttons
         */
        updateMediaBulkButtons: function() {
            const selectedCount = $('.fca-cm-media-checkbox:checked').length;
            const $bulkDelete = $('#fca-cm-bulk-delete-media');
            const $selectedCount = $('.fca-cm-bulk-actions .fca-cm-selected-count');
            
            if (selectedCount > 0) {
                $bulkDelete.prop('disabled', false);
                $selectedCount.text(`${selectedCount} selected`);
            } else {
                $bulkDelete.prop('disabled', true);
                $selectedCount.text('');
            }
        },

        /**
         * Initialize metadata page listeners
         */
        initMetadataListeners: function() {
            let searchTimer;
            $(document).on('input', '#fca-cm-metadata-search', (e) => {
                clearTimeout(searchTimer);
                searchTimer = setTimeout(() => {
                    this.filters.meta_key = $(e.target).val();
                    this.currentPage = 1;
                    this.loadMetadata();
                }, 500);
            });

            $(document).on('change', '#fca-cm-metadata-object-type, #fca-cm-metadata-per-page', () => {
                this.filters.object_type = $('#fca-cm-metadata-object-type').val();
                this.filters.per_page = $('#fca-cm-metadata-per-page').val();
                this.currentPage = 1;
                this.loadMetadata();
            });

            $(document).on('click', '.fca-cm-edit-metadata', (e) => {
                const metaId = $(e.currentTarget).data('meta-id');
                this.editMetadata(metaId);
            });

            $(document).on('click', '.fca-cm-delete-metadata', (e) => {
                const metaId = $(e.currentTarget).data('meta-id');
                this.deleteMetadata(metaId);
            });

            $(document).on('click', '#fca-cm-refresh-metadata', () => {
                this.loadMetadata();
            });
            
            // Save metadata
            $(document).on('click', '#fca-cm-save-metadata', () => {
                this.saveMetadata();
            });
            
            // Close metadata modal
            $(document).on('click', '#fca-cm-metadata-edit-modal .fca-cm-modal-close, #fca-cm-metadata-edit-modal .fca-cm-modal-overlay', () => {
                $('#fca-cm-metadata-edit-modal').fadeOut(200);
            });

            this.loadMetadata();
        },

        /**
         * Load page content based on current page
         */
        loadPageContent: function() {
            // This is called on initial load
            // Specific page loaders are called in their init functions
        },

        /**
         * Load posts list
         */
        loadPosts: function() {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_posts',
                    nonce: fcaCMData.nonce,
                    page: this.currentPage,
                    per_page: this.filters.per_page || 20,
                    search: this.filters.search || '',
                    status: this.filters.status || '',
                    space_id: this.filters.space_id || '',
                    type: ''  // Empty = show all post types
                },
                success: (response) => {
                    if (response.success) {
                        this.renderPostsList(response.data);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Render posts list
         */
        renderPostsList: function(data) {
            const $list = $('#fca-cm-posts-list');
            $list.empty();

            if (data.data.length === 0) {
                $list.html(this.getEmptyState(fcaCMData.strings.noResults));
                return;
            }

            data.data.forEach(post => {
                const $item = $(`
                    <div class="fca-cm-list-item has-checkbox" data-post-id="${post.id}">
                        <input type="checkbox" class="fca-cm-item-checkbox" value="${post.id}">
                        <div class="fca-cm-list-item-content">
                            <h4 class="fca-cm-list-item-title">
                                ${this.escapeHtml(post.title || 'Untitled')}
                            </h4>
                            <p class="fca-cm-list-item-excerpt">
                                ${this.stripHtml(post.message || '').substring(0, 100)}...
                            </p>
                            <div class="fca-cm-list-item-meta">
                                <span>${post.author}</span>
                                <span>•</span>
                                <span>${this.formatDate(post.created_at)}</span>
                                <span>•</span>
                                ${this.getStatusBadge(post.status)}
                            </div>
                        </div>
                    </div>
                `);
                $list.append($item);
            });

            this.renderPagination(data, '#fca-cm-pagination');
        },

        /**
         * Load post detail
         */
        loadPostDetail: function(postId) {
            this.currentPostId = postId;
            
            // Update active state
            $('.fca-cm-list-item').removeClass('active');
            $(`.fca-cm-list-item[data-post-id="${postId}"]`).addClass('active');

            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_post',
                    nonce: fcaCMData.nonce,
                    post_id: postId
                },
                success: (response) => {
                    if (response.success) {
                        this.renderPostDetail(response.data);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Render post detail
         */
        renderPostDetail: function(post) {
            $('.fca-cm-detail-empty').hide();
            $('.fca-cm-detail-content').show();

            // Basic fields
            $('#fca-cm-post-id').val(post.id);
            $('#fca-cm-post-title').val(post.title || '');
            $('#fca-cm-post-message').val(post.message || '');
            $('#fca-cm-post-status').val(post.status);
            $('#fca-cm-post-privacy').val(post.privacy);
            $('#fca-cm-post-sticky').prop('checked', post.is_sticky == 1);

            // Editable metadata fields
            $('#fca-cm-post-space-select').val(post.space_id || '');
            $('#fca-cm-post-author-id').val(post.user_id || '');
            $('#fca-cm-post-author-name').text(post.author ? `(${post.author})` : '');
            
            // Format dates for datetime-local input (YYYY-MM-DDTHH:MM)
            if (post.created_at) {
                $('#fca-cm-post-created-date').val(this.formatDateTimeLocal(post.created_at));
            }
            if (post.updated_at) {
                $('#fca-cm-post-updated-date').val(this.formatDateTimeLocal(post.updated_at));
            }

            // Read-only metadata display
            $('#fca-cm-post-id-display').text(post.id);
            $('#fca-cm-post-type').text(post.type || 'N/A');
            $('#fca-cm-post-content-type').text(post.content_type || 'N/A');
            $('#fca-cm-post-comments-count').text(post.comments_count || 0);
            $('#fca-cm-post-reactions-count').text(post.reactions_count || 0);
            
            // Render related data
            this.renderPostTerms(post.related_terms || []);
            this.renderPostMedia(post.related_media || []);
            this.renderPostComments(post.related_comments || []);
            this.renderPostReactions(post.related_reactions || []);
        },
        
        /**
         * Render post terms/topics
         */
        renderPostTerms: function(terms) {
            const $container = $('#fca-cm-post-terms');
            if (!terms || terms.length === 0) {
                $container.html('<p class="fca-cm-empty-message">No topics assigned to this post</p>');
                return;
            }
            
            let html = '<div class="fca-cm-tags-list">';
            terms.forEach(term => {
                html += `<span class="fca-cm-tag" data-term-id="${term.id}">
                    ${this.escapeHtml(term.title || term.slug)}
                    <button type="button" class="fca-cm-tag-remove" title="Remove topic"></button>
                </span>`;
            });
            html += '</div>';
            $container.html(html);
        },
        
        /**
         * Render post media
         */
        renderPostMedia: function(mediaItems) {
            const $container = $('#fca-cm-post-media');
            if (!mediaItems || mediaItems.length === 0) {
                $container.html('<p class="fca-cm-empty-message">No media attached to this post</p>');
                return;
            }
            
            let html = '<div class="fca-cm-media-grid">';
            mediaItems.forEach(media => {
                const mediaType = media.media_type || 'file';
                const isImage = mediaType.includes('image');
                const isVideo = mediaType.includes('video');
                const icon = isImage ? 'format-image' : isVideo ? 'video-alt2' : 'media-document';
                
                html += `
                    <div class="fca-cm-media-item" data-media-id="${media.id}">
                        <div class="fca-cm-media-icon">
                            ${isImage && media.media_url ? 
                                `<img src="${media.media_url}" alt="" style="width: 100%; height: 100%; object-fit: cover; border-radius: 4px;">` :
                                `<span class="dashicons dashicons-${icon}"></span>`}
                        </div>
                        <div class="fca-cm-media-info">
                            <div class="fca-cm-media-type">${this.escapeHtml(mediaType)}</div>
                            ${media.media_url ? `<a href="${media.media_url}" target="_blank" class="fca-cm-media-link">View</a>` : ''}
                        </div>
                    </div>
                `;
            });
            html += '</div>';
            $container.html(html);
        },
        
        /**
         * Render post comments
         */
        renderPostComments: function(comments) {
            const $container = $('#fca-cm-post-comments-list');
            if (!comments || comments.length === 0) {
                $container.html('<p class="fca-cm-empty-message">No comments on this post</p>');
                return;
            }
            
            let html = '<div class="fca-cm-relationship-list">';
            comments.forEach(comment => {
                const message = comment.message || '';
                const preview = message.length > 100 ? message.substring(0, 100) + '...' : message;
                html += `
                    <div class="fca-cm-relationship-item" data-comment-id="${comment.id}">
                        <div class="fca-cm-relationship-header">
                            <strong>${this.escapeHtml(comment.author || 'Unknown')}</strong>
                            <span class="fca-cm-relationship-date">${comment.created_at_formatted}</span>
                        </div>
                        <div class="fca-cm-relationship-content">${this.escapeHtml(preview)}</div>
                        <div class="fca-cm-relationship-meta">
                            Status: ${comment.status} | Reactions: ${comment.reactions_count || 0}
                        </div>
                    </div>
                `;
            });
            html += '</div>';
            $container.html(html);
        },
        
        /**
         * Render post reactions
         */
        renderPostReactions: function(reactions) {
            const $container = $('#fca-cm-post-reactions-list');
            if (!reactions || reactions.length === 0) {
                $container.html('<p class="fca-cm-empty-message">No reactions on this post</p>');
                return;
            }
            
            // Group reactions by type
            const grouped = {};
            reactions.forEach(reaction => {
                const type = reaction.type || reaction.reaction_type || 'like';
                if (!grouped[type]) {
                    grouped[type] = [];
                }
                grouped[type].push(reaction);
            });
            
            let html = '<div class="fca-cm-reactions-grid">';
            Object.keys(grouped).forEach(type => {
                const count = grouped[type].length;
                html += `
                    <div class="fca-cm-reaction-group">
                        <div class="fca-cm-reaction-type">
                            <span class="fca-cm-reaction-emoji">${this.getReactionEmoji(type)}</span>
                            <span class="fca-cm-reaction-label">${this.escapeHtml(type)}</span>
                            <span class="fca-cm-reaction-count">${count}</span>
                        </div>
                        <div class="fca-cm-reaction-users">
                            ${grouped[type].slice(0, 5).map(r => this.escapeHtml(r.author || 'Unknown')).join(', ')}
                            ${count > 5 ? ` and ${count - 5} more` : ''}
                        </div>
                    </div>
                `;
            });
            html += '</div>';
            $container.html(html);
        },
        
        /**
         * Get emoji for reaction type
         */
        getReactionEmoji: function(type) {
            const emojis = {
                'like': '👍',
                'love': '❤️',
                'laugh': '😂',
                'wow': '😮',
                'sad': '😢',
                'angry': '😠'
            };
            return emojis[type] || '👍';
        },

        /**
         * Save post
         */
        savePost: function() {
            const formData = {
                action: 'fca_cm_update_post',
                nonce: fcaCMData.nonce,
                post_id: $('#fca-cm-post-id').val(),
                title: $('#fca-cm-post-title').val(),
                message: $('#fca-cm-post-message').val(),
                status: $('#fca-cm-post-status').val(),
                privacy: $('#fca-cm-post-privacy').val(),
                is_sticky: $('#fca-cm-post-sticky').is(':checked') ? 1 : 0,
                // Editable metadata
                space_id: $('#fca-cm-post-space-select').val() || null,
                user_id: $('#fca-cm-post-author-id').val() || null,
                created_at: $('#fca-cm-post-created-date').val() || null,
                updated_at: $('#fca-cm-post-updated-date').val() || null
            };

            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: formData,
                success: (response) => {
                    if (response.success) {
                        this.showSuccess(fcaCMData.strings.saved);
                        this.loadPosts();
                        // Reload the detail to show updated values
                        this.loadPostDetail(formData.post_id);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Delete post
         */
        deletePost: function() {
            if (!confirm(fcaCMData.strings.confirmDelete)) {
                return;
            }

            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_delete_post',
                    nonce: fcaCMData.nonce,
                    post_id: this.currentPostId
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess(response.data.message);
                        this.closePostDetail();
                        this.loadPosts();
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Bulk delete posts
         */
        bulkDeletePosts: function() {
            const postIds = Array.from(this.selectedItems);
            
            if (postIds.length === 0) {
                return;
            }

            if (!confirm(fcaCMData.strings.confirmBulkDelete)) {
                return;
            }

            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_bulk_delete_posts',
                    nonce: fcaCMData.nonce,
                    post_ids: postIds
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess(response.data.message);
                        this.selectedItems.clear();
                        this.updateSelectedItems();
                        this.loadPosts();
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },
        
        /**
         * Bulk delete comments
         */
        bulkDeleteComments: function(commentIds) {
            if (!confirm(fcaCMData.strings.confirmBulkDelete || 'Are you sure you want to delete the selected comments?')) {
                return;
            }

            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_bulk_delete_comments',
                    nonce: fcaCMData.nonce,
                    comment_ids: commentIds
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess(response.data.message);
                        this.loadComments();
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error || 'An error occurred');
                }
            });
        },
        
        /**
         * Bulk delete spaces
         */
        bulkDeleteSpaces: function(spaceIds) {
            if (!confirm(fcaCMData.strings.confirmBulkDelete || 'Are you sure you want to delete the selected spaces?')) {
                return;
            }

            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_bulk_delete_spaces',
                    nonce: fcaCMData.nonce,
                    space_ids: spaceIds
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess(response.data.message);
                        this.selectedSpaces.clear();
                        this.loadSpaces();
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error || 'An error occurred');
                }
            });
        },

        /**
         * Close post detail
         */
        closePostDetail: function() {
            $('.fca-cm-detail-content').hide();
            $('.fca-cm-detail-empty').show();
            $('.fca-cm-list-item').removeClass('active');
            this.currentPostId = null;
        },

        /**
         * Update selected items
         */
        updateSelectedItems: function() {
            this.selectedItems.clear();
            $('.fca-cm-list-item-checkbox input:checked').each((i, el) => {
                this.selectedItems.add($(el).val());
            });

            const count = this.selectedItems.size;
            $('#fca-cm-bulk-delete').prop('disabled', count === 0);
            $('.fca-cm-selected-count').text(count > 0 ? `${count} selected` : '');
        },

        /**
         * Load courses list
         */
        loadCourses: function() {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_courses',
                    nonce: fcaCMData.nonce,
                    search: this.filters.search || '',
                    status: this.filters.status || ''
                },
                success: (response) => {
                    if (response.success) {
                        this.renderCoursesList(response.data);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Render courses list
         */
        renderCoursesList: function(courses) {
            const $list = $('#fca-cm-courses-list');
            $list.empty();

            if (!courses || courses.length === 0) {
                $list.html(this.getEmptyState('No courses found'));
                return;
            }

            courses.forEach(course => {
                const $item = $(`
                    <div class="fca-cm-list-item fca-cm-course-item has-checkbox" data-course-id="${course.id}">
                        <input type="checkbox" class="fca-cm-item-checkbox fca-cm-course-checkbox" value="${course.id}">
                        <div class="fca-cm-list-item-content">
                            <h4 class="fca-cm-list-item-title">
                                ${this.escapeHtml(course.title || 'Untitled Course')}
                            </h4>
                            <div class="fca-cm-list-item-meta">
                                <span>${this.getStatusBadge(course.status)}</span>
                            </div>
                        </div>
                    </div>
                `);
                $list.append($item);
            });
        },

        /**
         * Load course detail
         */
        loadCourseDetail: function(courseId) {
            this.currentCourseId = courseId;
            
            $('.fca-cm-course-item').removeClass('active');
            $(`.fca-cm-course-item[data-course-id="${courseId}"]`).addClass('active');

            // Load course basic info
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_course',
                    nonce: fcaCMData.nonce,
                    course_id: courseId
                },
                success: (response) => {
                    if (response.success) {
                        this.renderCourseDetail(response.data);
                        this.loadCourseStructure(courseId);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Render course detail
         */
        renderCourseDetail: function(course) {
            $('#fca-cm-course-detail .fca-cm-detail-empty').hide();
            $('#fca-cm-course-detail .fca-cm-detail-content').show();

            // Update header title
            $('#fca-cm-course-title').text(course.title || 'Untitled Course');
            
            // Populate form fields
            $('#fca-cm-course-id').val(course.id);
            $('#fca-cm-course-title-input').val(course.title || '');
            $('#fca-cm-course-description').val(course.description || '');
            $('#fca-cm-course-slug').val(course.slug || '');
            $('#fca-cm-course-status-select').val(course.status || 'draft');
            $('#fca-cm-course-privacy').val(course.privacy || 'public');
            $('#fca-cm-course-created').val(this.formatDate(course.created_at));
            $('#fca-cm-course-updated').val(this.formatDate(course.updated_at));

            // Store course data for editing
            this.currentCourseData = course;
        },

        /**
         * Load course structure (sections and lessons)
         */
        loadCourseStructure: function(courseId) {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_course_structure',
                    nonce: fcaCMData.nonce,
                    course_id: courseId
                },
                success: (response) => {
                    if (response.success) {
                        this.renderCourseStructure(response.data);
                    } else {
                        $('#fca-cm-structure-tree').html('<div class="fca-cm-empty"><p>Could not load course structure</p></div>');
                    }
                },
                error: () => {
                    this.showError('Failed to load course structure');
                }
            });
        },

        /**
         * Render course structure with sections and lessons
         */
        renderCourseStructure: function(structure) {
            const $tree = $('#fca-cm-structure-tree');
            $tree.empty();

            const sections = structure.sections || [];
            const lessons = structure.lessons || [];

            if (sections.length === 0 && lessons.length === 0) {
                $tree.html('<div class="fca-cm-empty"><p>No content in this course yet</p></div>');
                return;
            }

            // Build section-lesson hierarchy
            const sectionsWithLessons = sections.map(section => {
                return {
                    ...section,
                    lessons: lessons.filter(lesson => lesson.parent_id == section.id)
                };
            });

            // Get orphan lessons (lessons without parent section)
            const orphanLessons = lessons.filter(lesson => !lesson.parent_id || !sections.find(s => s.id == lesson.parent_id));

            // Render sections
            sectionsWithLessons.forEach((section, index) => {
                const $section = $(`
                    <div class="fca-cm-structure-section" data-item-id="${section.id}" data-item-type="section" data-priority="${section.priority || index}">
                        <div class="fca-cm-section-header">
                            <span class="dashicons dashicons-menu fca-cm-drag-handle"></span>
                            <h4 class="fca-cm-section-title">${this.escapeHtml(section.title || 'Untitled Section')}</h4>
                            <div class="fca-cm-section-actions">
                                <button type="button" class="fca-cm-btn fca-cm-btn-icon fca-cm-edit-item" data-item-id="${section.id}" data-item-type="section" title="Edit">
                                    <span class="dashicons dashicons-edit"></span>
                                </button>
                                <button type="button" class="fca-cm-btn fca-cm-btn-icon fca-cm-move-item" data-item-id="${section.id}" data-item-type="section" title="Move to Course">
                                    <span class="dashicons dashicons-randomize"></span>
                                </button>
                                <button type="button" class="fca-cm-btn fca-cm-btn-icon fca-cm-duplicate-item" data-item-id="${section.id}" title="Duplicate">
                                    <span class="dashicons dashicons-admin-page"></span>
                                </button>
                                <button type="button" class="fca-cm-btn fca-cm-btn-icon fca-cm-delete-item" data-item-id="${section.id}" title="Delete">
                                    <span class="dashicons dashicons-trash"></span>
                                </button>
                            </div>
                        </div>
                        <div class="fca-cm-lessons-container"></div>
                    </div>
                `);

                // Add lessons to section
                if (section.lessons && section.lessons.length > 0) {
                    const $lessonsContainer = $section.find('.fca-cm-lessons-container');
                    section.lessons.forEach((lesson, lessonIndex) => {
                        const $lesson = this.renderLesson(lesson, lessonIndex);
                        $lessonsContainer.append($lesson);
                    });
                }

                $tree.append($section);
            });

            // Render orphan lessons at the end
            if (orphanLessons.length > 0) {
                const $orphanSection = $(`
                    <div class="fca-cm-structure-section fca-cm-orphan-section">
                        <div class="fca-cm-section-header">
                            <h4 class="fca-cm-section-title"><em>Unassigned Lessons</em></h4>
                        </div>
                        <div class="fca-cm-lessons-container"></div>
                    </div>
                `);

                orphanLessons.forEach((lesson, index) => {
                    const $lesson = this.renderLesson(lesson, index);
                    $orphanSection.find('.fca-cm-lessons-container').append($lesson);
                });

                $tree.append($orphanSection);
            }

            // Initialize drag and drop
            this.initCourseDragDrop();
        },

        /**
         * Render a single lesson
         */
        renderLesson: function(lesson, index) {
            return $(`
                <div class="fca-cm-lesson-item" data-item-id="${lesson.id}" data-item-type="lesson" data-priority="${lesson.priority || index}">
                    <span class="dashicons dashicons-menu fca-cm-drag-handle"></span>
                    <span class="fca-cm-lesson-title">${this.escapeHtml(lesson.title || 'Untitled Lesson')}</span>
                    <div class="fca-cm-lesson-meta">
                        ${lesson.content_type ? `<span class="fca-cm-badge">${lesson.content_type}</span>` : ''}
                    </div>
                    <div class="fca-cm-lesson-actions">
                        <button type="button" class="fca-cm-btn fca-cm-btn-icon fca-cm-edit-item" data-item-id="${lesson.id}" data-item-type="lesson" title="Edit">
                            <span class="dashicons dashicons-edit"></span>
                        </button>
                        <button type="button" class="fca-cm-btn fca-cm-btn-icon fca-cm-move-item" data-item-id="${lesson.id}" data-item-type="lesson" title="Move to Course">
                            <span class="dashicons dashicons-randomize"></span>
                        </button>
                        <button type="button" class="fca-cm-btn fca-cm-btn-icon fca-cm-duplicate-item" data-item-id="${lesson.id}" title="Duplicate">
                            <span class="dashicons dashicons-admin-page"></span>
                        </button>
                        <button type="button" class="fca-cm-btn fca-cm-btn-icon fca-cm-delete-item" data-item-id="${lesson.id}" title="Delete">
                            <span class="dashicons dashicons-trash"></span>
                        </button>
                    </div>
                </div>
            `);
        },

        /**
         * Initialize drag and drop for course structure
         */
        initCourseDragDrop: function() {
            const self = this;
            
            // Make sections sortable
            $('#fca-cm-structure-tree').sortable({
                handle: '.fca-cm-section-header .fca-cm-drag-handle',
                items: '.fca-cm-structure-section:not(.fca-cm-orphan-section)',
                axis: 'y',
                cursor: 'move',
                opacity: 0.7,
                update: function() {
                    self.saveCourseOrder();
                }
            });

            // Make lessons sortable within sections
            $('.fca-cm-lessons-container').sortable({
                handle: '.fca-cm-drag-handle',
                items: '.fca-cm-lesson-item',
                axis: 'y',
                cursor: 'move',
                opacity: 0.7,
                connectWith: '.fca-cm-lessons-container',
                update: function() {
                    self.saveCourseOrder();
                }
            });
        },

        /**
         * Save course order after drag and drop
         */
        saveCourseOrder: function() {
            const items = [];

            // Get section order
            $('#fca-cm-structure-tree .fca-cm-structure-section').each(function(index) {
                const sectionId = $(this).data('item-id');
                if (sectionId) {
                    items.push({
                        id: sectionId,
                        priority: index
                    });

                    // Get lesson order within this section
                    $(this).find('.fca-cm-lesson-item').each(function(lessonIndex) {
                        const lessonId = $(this).data('item-id');
                        items.push({
                            id: lessonId,
                            priority: lessonIndex
                        });
                    });
                }
            });

            // Save to database
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_update_course_order',
                    nonce: fcaCMData.nonce,
                    items: items
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess('Order updated');
                    }
                }
            });
        },
        
        /**
         * Edit course item (section or lesson) - opens post editor
         */
        editCourseItem: function(itemId, itemType) {
            // Load the item as a post
            this.loadPostDetail(itemId);
        },
        
        /**
         * Delete course item
         */
        deleteCourseItem: function(itemId) {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_delete_course_item',
                    nonce: fcaCMData.nonce,
                    item_id: itemId
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess('Item deleted');
                        // Reload course structure
                        if (this.currentCourseId) {
                            this.loadCourseStructure(this.currentCourseId);
                        }
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError('Failed to delete item');
                }
            });
        },
        
        /**
         * Duplicate course item
         */
        duplicateCourseItem: function(itemId) {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_duplicate_course_item',
                    nonce: fcaCMData.nonce,
                    item_id: itemId
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess('Item duplicated');
                        // Reload course structure
                        if (this.currentCourseId) {
                            this.loadCourseStructure(this.currentCourseId);
                        }
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError('Failed to duplicate item');
                }
            });
        },
        
        /**
         * Update course bulk action buttons
         */
        updateCourseBulkButtons: function() {
            const selectedCount = $('.fca-cm-course-checkbox:checked').length;
            const $bulkEdit = $('#fca-cm-bulk-edit-courses');
            const $bulkDelete = $('#fca-cm-bulk-delete-courses');
            const $selectedCount = $('.fca-cm-bulk-actions .fca-cm-selected-count');
            
            if (selectedCount > 0) {
                $bulkEdit.prop('disabled', false);
                $bulkDelete.prop('disabled', false);
                $selectedCount.text(`${selectedCount} selected`);
            } else {
                $bulkEdit.prop('disabled', true);
                $bulkDelete.prop('disabled', true);
                $selectedCount.text('');
            }
        },
        
        /**
         * Bulk delete courses
         */
        bulkDeleteCourses: function(courseIds) {
            // Note: Courses are spaces, so we'd need to delete from wp_fcom_spaces table
            // For now, show a message
            this.showError('Bulk delete for courses requires special handling (they are spaces). Coming soon!');
        },
        
        /**
         * Open move item modal
         */
        openMoveItemModal: function(itemId, itemType) {
            this.moveItemId = itemId;
            this.moveItemType = itemType;
            
            // Set description
            const description = itemType === 'section' 
                ? 'Move this section and all its lessons to another course.' 
                : 'Move this lesson to another course or section.';
            $('#fca-cm-move-item-description').text(description);
            
            // Show/hide section selector for lessons
            if (itemType === 'lesson') {
                $('#fca-cm-move-section-container').show();
            } else {
                $('#fca-cm-move-section-container').hide();
            }
            
            // Load available courses
            this.loadCoursesForMove();
            
            // Show modal
            $('#fca-cm-move-item-modal').fadeIn(200);
        },
        
        /**
         * Load courses for move modal
         */
        loadCoursesForMove: function() {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_courses',
                    nonce: fcaCMData.nonce
                },
                success: (response) => {
                    if (response.success) {
                        const $select = $('#fca-cm-move-target-course');
                        $select.empty().append('<option value="">Select a course...</option>');
                        
                        response.data.forEach(course => {
                            // Don't show current course
                            if (course.id != this.currentCourseId) {
                                $select.append(`<option value="${course.id}">${this.escapeHtml(course.title)}</option>`);
                            }
                        });
                    }
                }
            });
        },
        
        /**
         * Load sections for selected course in move modal
         */
        loadSectionsForMove: function(courseId) {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_course_structure',
                    nonce: fcaCMData.nonce,
                    course_id: courseId
                },
                success: (response) => {
                    if (response.success) {
                        const $select = $('#fca-cm-move-target-section');
                        $select.empty().append('<option value="">No section (make orphan)</option>');
                        
                        const sections = response.data.sections || [];
                        sections.forEach(section => {
                            $select.append(`<option value="${section.id}">${this.escapeHtml(section.title)}</option>`);
                        });
                    }
                }
            });
        },
        
        /**
         * Confirm move item
         */
        confirmMoveItem: function() {
            const targetCourseId = $('#fca-cm-move-target-course').val();
            const targetSectionId = $('#fca-cm-move-target-section').val();
            
            if (!targetCourseId) {
                this.showError('Please select a target course');
                return;
            }
            
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_move_course_item',
                    nonce: fcaCMData.nonce,
                    item_id: this.moveItemId,
                    item_type: this.moveItemType,
                    target_course_id: targetCourseId,
                    target_section_id: targetSectionId || null
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess('Item moved successfully');
                        $('#fca-cm-move-item-modal').fadeOut(200);
                        // Reload course structure
                        if (this.currentCourseId) {
                            this.loadCourseStructure(this.currentCourseId);
                        }
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError('Failed to move item');
                }
            });
        },
        
        /**
         * Open course details modal
         */
        openCourseDetailsModal: function() {
            const course = this.currentCourseData;
            if (!course) return;
            
            // Populate modal
            $('#fca-cm-course-id').val(course.id);
            $('#fca-cm-course-title-input').val(course.title || '');
            $('#fca-cm-course-description').val(course.description || '');
            $('#fca-cm-course-slug').val(course.slug || '');
            $('#fca-cm-course-status-select').val(course.status || 'draft');
            $('#fca-cm-course-privacy').val(course.privacy || 'public');
            $('#fca-cm-course-created').val(this.formatDate(course.created_at));
            $('#fca-cm-course-updated').val(this.formatDate(course.updated_at));
            
            // Show modal
            $('#fca-cm-course-details-modal').fadeIn(200);
        },
        
        /**
         * Save course details from modal
         */
        saveCourseDetails: function() {
            const courseId = $('#fca-cm-course-id').val();
            
            if (!courseId) {
                this.showError('No course selected');
                return;
            }
            
            const data = {
                action: 'fca_cm_update_course',
                nonce: fcaCMData.nonce,
                course_id: courseId,
                title: $('#fca-cm-course-title-input').val(),
                description: $('#fca-cm-course-description').val(),
                slug: $('#fca-cm-course-slug').val(),
                status: $('#fca-cm-course-status-select').val(),
                privacy: $('#fca-cm-course-privacy').val()
            };
            
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: data,
                success: (response) => {
                    if (response.success) {
                        this.showSuccess('Course updated successfully');
                        // Update the header title
                        $('#fca-cm-course-title').text(data.title);
                        // Close modal
                        $('#fca-cm-course-details-modal').fadeOut(200);
                        // Reload the course list to reflect changes
                        this.loadCourses();
                        // Update current course data
                        if (this.currentCourseData) {
                            this.currentCourseData.title = data.title;
                            this.currentCourseData.description = data.description;
                            this.currentCourseData.slug = data.slug;
                            this.currentCourseData.status = data.status;
                            this.currentCourseData.privacy = data.privacy;
                        }
                    } else {
                        this.showError(response.data.message || 'Failed to update course');
                    }
                },
                error: () => {
                    this.showError('Failed to update course');
                }
            });
        },

        /**
         * Open course bulk edit modal
         */
        openCourseBulkEditModal: function(courseIds) {
            this.bulkEditCourseIds = courseIds;
            
            // Reset form
            $('#fca-cm-bulk-course-status').val('');
            $('#fca-cm-bulk-course-privacy').val('');
            
            // Update description
            $('#fca-cm-bulk-edit-description').text(`Update ${courseIds.length} courses at once. Only fill in the fields you want to change.`);
            
            // Show modal
            $('#fca-cm-course-bulk-edit-modal').fadeIn(200);
        },

        /**
         * Save course bulk edit
         */
        saveCourseBulkEdit: function() {
            if (!this.bulkEditCourseIds || this.bulkEditCourseIds.length === 0) {
                this.showError('No courses selected');
                return;
            }
            
            const status = $('#fca-cm-bulk-course-status').val();
            const privacy = $('#fca-cm-bulk-course-privacy').val();
            
            if (!status && !privacy) {
                this.showError('Please select at least one field to update');
                return;
            }
            
            const data = {
                action: 'fca_cm_bulk_update_courses',
                nonce: fcaCMData.nonce,
                course_ids: this.bulkEditCourseIds
            };
            
            if (status) {
                data.status = status;
            }
            
            if (privacy) {
                data.privacy = privacy;
            }
            
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: data,
                success: (response) => {
                    if (response.success) {
                        this.showSuccess(response.data.message || 'Courses updated successfully');
                        // Close modal
                        $('#fca-cm-course-bulk-edit-modal').fadeOut(200);
                        // Uncheck all checkboxes
                        $('.fca-cm-course-checkbox').prop('checked', false);
                        $('#fca-cm-select-all-courses').prop('checked', false);
                        this.updateCourseBulkButtons();
                        // Reload courses
                        this.loadCourses();
                        this.bulkEditCourseIds = null;
                    } else {
                        this.showError(response.data.message || 'Failed to update courses');
                    }
                },
                error: () => {
                    this.showError('Failed to update courses');
                }
            });
        },

        /**
         * Render section lessons
         */
        renderSectionLessons: function(sectionId, lessons) {
            const $list = $(`#fca-cm-lessons-${sectionId}`);
            $list.empty();

            if (lessons.length === 0) {
                return;
            }

            lessons.forEach(lesson => {
                const $lesson = $(`
                    <li class="fca-cm-lesson-item">
                        <span class="dashicons dashicons-media-text"></span>
                        <h5 class="fca-cm-lesson-title">${this.escapeHtml(lesson.title)}</h5>
                        <div class="fca-cm-lesson-actions">
                            <button type="button" class="fca-cm-btn fca-cm-btn-icon fca-cm-edit-lesson" data-lesson-id="${lesson.id}">
                                <span class="dashicons dashicons-edit"></span>
                            </button>
                            <button type="button" class="fca-cm-btn fca-cm-btn-icon fca-cm-delete-lesson" data-lesson-id="${lesson.id}">
                                <span class="dashicons dashicons-trash"></span>
                            </button>
                        </div>
                    </li>
                `);
                $list.append($lesson);
            });
        },

        /**
         * Open course modal
         */
        openCourseModal: function() {
            if (!this.currentCourseData) return;

            $('#fca-cm-course-edit-id').val(this.currentCourseData.id);
            $('#fca-cm-course-edit-title').val(this.currentCourseData.title || '');
            $('#fca-cm-course-edit-message').val(this.currentCourseData.message || '');
            $('#fca-cm-course-edit-status').val(this.currentCourseData.status);
            $('#fca-cm-course-edit-privacy').val(this.currentCourseData.privacy);

            $('#fca-cm-course-modal').fadeIn(200);
        },

        /**
         * Open lesson modal
         */
        openLessonModal: function(lessonId) {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_post',
                    nonce: fcaCMData.nonce,
                    post_id: lessonId
                },
                success: (response) => {
                    if (response.success) {
                        const lesson = response.data;
                        $('#fca-cm-lesson-edit-id').val(lesson.id);
                        $('#fca-cm-lesson-edit-title').val(lesson.title || '');
                        $('#fca-cm-lesson-edit-message').val(lesson.message || '');
                        $('#fca-cm-lesson-edit-status').val(lesson.status);
                        $('#fca-cm-lesson-modal').fadeIn(200);
                    }
                }
            });
        },

        /**
         * Save course
         */
        saveCourse: function() {
            const formData = {
                action: 'fca_cm_update_post',
                nonce: fcaCMData.nonce,
                post_id: $('#fca-cm-course-edit-id').val(),
                title: $('#fca-cm-course-edit-title').val(),
                message: $('#fca-cm-course-edit-message').val(),
                status: $('#fca-cm-course-edit-status').val(),
                privacy: $('#fca-cm-course-edit-privacy').val()
            };

            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: formData,
                success: (response) => {
                    if (response.success) {
                        this.showSuccess(fcaCMData.strings.saved);
                        this.closeModal();
                        this.loadCourses();
                        if (this.currentCourseId) {
                            this.loadCourseDetail(this.currentCourseId);
                        }
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Save lesson
         */
        saveLesson: function() {
            const formData = {
                action: 'fca_cm_update_post',
                nonce: fcaCMData.nonce,
                post_id: $('#fca-cm-lesson-edit-id').val(),
                title: $('#fca-cm-lesson-edit-title').val(),
                message: $('#fca-cm-lesson-edit-message').val(),
                status: $('#fca-cm-lesson-edit-status').val()
            };

            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: formData,
                success: (response) => {
                    if (response.success) {
                        this.showSuccess(fcaCMData.strings.saved);
                        this.closeModal();
                        if (this.currentCourseId) {
                            this.loadCourseDetail(this.currentCourseId);
                        }
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Delete lesson
         */
        deleteLesson: function(lessonId) {
            if (!confirm('Are you sure you want to delete this lesson?')) {
                return;
            }

            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_delete_post',
                    nonce: fcaCMData.nonce,
                    post_id: lessonId
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess('Lesson deleted successfully');
                        if (this.currentCourseId) {
                            this.loadCourseDetail(this.currentCourseId);
                        }
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Close course detail
         */
        closeCourseDetail: function() {
            $('#fca-cm-course-detail .fca-cm-detail-content').hide();
            $('#fca-cm-course-detail .fca-cm-detail-empty').show();
            $('.fca-cm-course-item').removeClass('active');
            this.currentCourseId = null;
            this.currentCourseData = null;
        },

        /**
         * Load comments
         */
        loadComments: function() {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_comments',
                    nonce: fcaCMData.nonce,
                    page: this.currentPage,
                    per_page: this.filters.per_page || 20,
                    search: this.filters.search || '',
                    status: this.filters.status || ''
                },
                success: (response) => {
                    if (response.success) {
                        this.renderCommentsList(response.data);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Render comments list
         */
        renderCommentsList: function(data) {
            const $list = $('#fca-cm-comments-list');
            $list.empty();

            if (data.data.length === 0) {
                $list.html(this.getEmptyState('No comments found'));
                return;
            }

            data.data.forEach(comment => {
                const $item = $(`
                    <div class="fca-cm-list-item fca-cm-comment-item has-checkbox" data-comment-id="${comment.id}">
                        <input type="checkbox" class="fca-cm-item-checkbox fca-cm-comment-checkbox" value="${comment.id}">
                        <div class="fca-cm-list-item-content">
                            <p class="fca-cm-list-item-excerpt">
                                ${this.stripHtml(comment.message || '').substring(0, 150)}...
                            </p>
                            <div class="fca-cm-list-item-meta">
                                <span>${comment.author}</span>
                                <span>•</span>
                                <span>Post #${comment.post_id}</span>
                                <span>•</span>
                                <span>${this.formatDate(comment.created_at)}</span>
                                <span>•</span>
                                ${this.getStatusBadge(comment.status)}
                            </div>
                        </div>
                    </div>
                `);
                $list.append($item);
            });

            this.renderPagination(data, '#fca-cm-comments-pagination');
        },

        /**
         * Load comment detail
         */
        loadCommentDetail: function(commentId) {
            this.currentCommentId = commentId;
            
            $('.fca-cm-comment-item').removeClass('active');
            $(`.fca-cm-comment-item[data-comment-id="${commentId}"]`).addClass('active');
            
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_comment',
                    nonce: fcaCMData.nonce,
                    comment_id: commentId
                },
                success: (response) => {
                    if (response.success) {
                        this.renderCommentDetail(response.data);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },
        
        renderCommentDetail: function(comment) {
            $('#fca-cm-comment-detail .fca-cm-detail-empty').hide();
            $('#fca-cm-comment-detail .fca-cm-detail-content').show();
            
            // Populate form fields
            $('#fca-cm-comment-id').val(comment.id);
            $('#fca-cm-comment-message').val(comment.message || '');
            $('#fca-cm-comment-status').val(comment.status);
            $('#fca-cm-comment-privacy').val(comment.privacy || 'public');
            
            // Metadata
            $('#fca-cm-comment-id-display').text(comment.id);
            $('#fca-cm-comment-author').text(comment.author || 'Unknown');
            $('#fca-cm-comment-created').text(comment.created_at_formatted);
            $('#fca-cm-comment-updated').text(comment.updated_at_formatted);
            
            // Parent Post
            if (comment.parent_post) {
                const post = comment.parent_post;
                const message = post.message || '';
                const preview = message.length > 150 ? message.substring(0, 150) + '...' : message;
                $('#fca-cm-comment-parent-post').html(`
                    <div class="fca-cm-relationship-header">
                        <strong>${this.escapeHtml(post.title || 'Untitled Post')}</strong>
                        <span class="fca-cm-badge">${post.type}</span>
                    </div>
                    <div class="fca-cm-relationship-content">${this.escapeHtml(preview)}</div>
                    <div class="fca-cm-relationship-meta">
                        By ${this.escapeHtml(post.author)} in ${this.escapeHtml(post.space_name || 'No Space')}
                    </div>
                `);
            } else {
                $('#fca-cm-comment-parent-post').html('<p class="fca-cm-empty-message">No parent post found</p>');
            }
            
            // Reactions
            this.renderCommentReactions(comment.reactions || []);
            
            // Replies
            this.renderCommentReplies(comment.replies || []);
        },
        
        renderCommentReactions: function(reactions) {
            const $container = $('#fca-cm-comment-reactions-list');
            if (!reactions || reactions.length === 0) {
                $container.html('<p class="fca-cm-empty-message">No reactions on this comment</p>');
                return;
            }
            
            // Group by type
            const grouped = {};
            reactions.forEach(reaction => {
                const type = reaction.type || 'like';
                if (!grouped[type]) grouped[type] = [];
                grouped[type].push(reaction);
            });
            
            let html = '<div class="fca-cm-reactions-grid">';
            Object.keys(grouped).forEach(type => {
                html += `
                    <div class="fca-cm-reaction-group">
                        <div class="fca-cm-reaction-type">
                            <span class="fca-cm-reaction-emoji">${this.getReactionEmoji(type)}</span>
                            <span class="fca-cm-reaction-label">${this.escapeHtml(type)}</span>
                            <span class="fca-cm-reaction-count">${grouped[type].length}</span>
                        </div>
                    </div>
                `;
            });
            html += '</div>';
            $container.html(html);
        },
        
        renderCommentReplies: function(replies) {
            const $container = $('#fca-cm-comment-replies-list');
            if (!replies || replies.length === 0) {
                $container.html('<p class="fca-cm-empty-message">No replies to this comment</p>');
                return;
            }
            
            let html = '<div class="fca-cm-relationship-list">';
            replies.forEach(reply => {
                const message = reply.message || '';
                const preview = message.length > 100 ? message.substring(0, 100) + '...' : message;
                html += `
                    <div class="fca-cm-relationship-item">
                        <div class="fca-cm-relationship-header">
                            <strong>${this.escapeHtml(reply.author || 'Unknown')}</strong>
                            <span class="fca-cm-relationship-date">${reply.created_at_formatted}</span>
                        </div>
                        <div class="fca-cm-relationship-content">${this.escapeHtml(preview)}</div>
                        <div class="fca-cm-relationship-meta">
                            Status: ${reply.status} | Reactions: ${reply.reactions_count || 0}
                        </div>
                    </div>
                `;
            });
            html += '</div>';
            $container.html(html);
        },
        
        saveComment: function() {
            const commentId = $('#fca-cm-comment-id').val();
            if (!commentId) return;
            
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_update_comment',
                    nonce: fcaCMData.nonce,
                    comment_id: commentId,
                    message: $('#fca-cm-comment-message').val(),
                    status: $('#fca-cm-comment-status').val(),
                    privacy: $('#fca-cm-comment-privacy').val()
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess('Comment updated successfully');
                        this.loadComments();
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError('Failed to update comment');
                }
            });
        },

        /**
         * Load reactions
         */
        loadReactions: function() {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_reactions',
                    nonce: fcaCMData.nonce,
                    page: this.currentPage,
                    per_page: this.filters.per_page || 20,
                    reaction_type: this.filters.reaction_type || ''
                },
                success: (response) => {
                    if (response.success) {
                        this.renderReactionsTable(response.data);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Render reactions table
         */
        renderReactionsTable: function(data) {
            const $tbody = $('#fca-cm-reactions-table tbody');
            $tbody.empty();

            if (data.data.length === 0) {
                $tbody.html('<tr><td colspan="7" class="fca-cm-text-center">No reactions found</td></tr>');
                return;
            }

            data.data.forEach(reaction => {
                const $row = $(`
                    <tr>
                        <td>${reaction.id}</td>
                        <td>${this.escapeHtml(reaction.author)}</td>
                        <td>${this.getReactionIcon(reaction.reaction_type)} ${reaction.reaction_type}</td>
                        <td>${reaction.post_id || '-'}</td>
                        <td>${reaction.comment_id || '-'}</td>
                        <td>${this.formatDate(reaction.created_at)}</td>
                        <td>
                            <button type="button" class="fca-cm-btn fca-cm-btn-sm fca-cm-btn-danger fca-cm-delete-reaction" data-reaction-id="${reaction.id}">
                                <span class="dashicons dashicons-trash"></span>
                            </button>
                        </td>
                    </tr>
                `);
                $tbody.append($row);
            });

            this.renderPagination(data, '#fca-cm-reactions-pagination');
        },

        /**
         * Delete reaction
         */
        deleteReaction: function(reactionId) {
            if (!confirm('Are you sure you want to delete this reaction?')) {
                return;
            }

            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_delete_reaction',
                    nonce: fcaCMData.nonce,
                    reaction_id: reactionId
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess(response.data.message);
                        this.loadReactions();
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Load media
         */
        loadMedia: function() {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_media',
                    nonce: fcaCMData.nonce,
                    page: this.currentPage,
                    per_page: this.filters.per_page || 20,
                    search: this.filters.search || '',
                    media_type: this.filters.media_type || '',
                    space_id: this.filters.space_id || ''
                },
                success: (response) => {
                    if (response.success) {
                        this.renderMediaList(response.data);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Render media list
         */
        renderMediaList: function(data) {
            const $list = $('#fca-cm-media-list');
            $list.empty();

            if (!data.data || data.data.length === 0) {
                $list.html(this.getEmptyState('No media files found'));
                return;
            }

            data.data.forEach(media => {
                const mediaUrl = media.media_url;
                const mediaType = media.media_type || 'file';
                const isImage = mediaType.includes('image');
                const isVideo = mediaType.includes('video');
                
                const $item = $(`
                    <div class="fca-cm-list-item fca-cm-media-item has-checkbox" data-media-id="${media.id}">
                        <input type="checkbox" class="fca-cm-item-checkbox fca-cm-media-checkbox" value="${media.id}">
                        <div class="fca-cm-list-item-thumb">
                            ${isImage ? `<img src="${mediaUrl}" alt="">` : 
                              isVideo ? '<span class="dashicons dashicons-video-alt2"></span>' :
                              '<span class="dashicons dashicons-media-document"></span>'}
                        </div>
                        <div class="fca-cm-list-item-content">
                            <h4 class="fca-cm-list-item-title">${this.escapeHtml(media.media_key || 'Untitled')}</h4>
                            <div class="fca-cm-list-item-meta">
                                <span>${mediaType}</span>
                                ${media.author ? `<span>•</span><span>${this.escapeHtml(media.author)}</span>` : ''}
                                <span>•</span>
                                <span>${this.formatDate(media.created_at)}</span>
                            </div>
                        </div>
                    </div>
                `);
                $list.append($item);
            });

            this.renderPagination(data, '#fca-cm-media-pagination');
        },
        
        /**
         * Load media detail
         */
        loadMediaDetail: function(mediaId) {
            this.currentMediaId = mediaId;
            
            $('.fca-cm-media-item').removeClass('active');
            $(`.fca-cm-media-item[data-media-id="${mediaId}"]`).addClass('active');
            
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_media_detail',
                    nonce: fcaCMData.nonce,
                    media_id: mediaId
                },
                success: (response) => {
                    if (response.success) {
                        this.renderMediaDetail(response.data);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError('Failed to load media details');
                }
            });
        },
        
        /**
         * Render media detail
         */
        renderMediaDetail: function(media) {
            $('#fca-cm-media-detail .fca-cm-detail-empty').hide();
            $('#fca-cm-media-detail .fca-cm-detail-content').show();
            
            // Preview
            const isImage = media.media_type === 'image';
            const isVideo = media.media_type === 'video';
            let previewHtml = '';
            
            if (isImage) {
                previewHtml = `<img src="${media.media_url}" alt="${this.escapeHtml(media.title || '')}" style="max-width: 100%; height: auto;">`;
            } else if (isVideo) {
                previewHtml = `<video src="${media.media_url}" controls style="max-width: 100%;"></video>`;
            } else {
                previewHtml = `<div class="fca-cm-media-icon"><span class="dashicons dashicons-media-document" style="font-size: 64px;"></span></div>`;
            }
            $('#fca-cm-media-preview').html(previewHtml);
            
            // Info
            $('#fca-cm-media-id').text(media.id);
            $('#fca-cm-media-filename').text(media.title || media.media_key || 'Untitled');
            $('#fca-cm-media-type-display').text(media.media_type || 'file');
            $('#fca-cm-media-size').text(this.formatFileSize(media.file_size || 0));
            $('#fca-cm-media-url').attr('href', media.media_url).text(media.media_url);
            $('#fca-cm-media-user').text(media.author || 'Unknown');
            $('#fca-cm-media-created').text(this.formatDate(media.created_at));
            
            // Used in
            const $usedIn = $('#fca-cm-media-used-in');
            if (media.used_in && media.used_in.length > 0) {
                let html = '<div class="fca-cm-relationship-list">';
                media.used_in.forEach(item => {
                    const typeLabel = item.type === 'course_lesson' ? 'Lesson' : 
                                     item.type === 'course_section' ? 'Section' : 
                                     item.type === 'feed' ? 'Post' : item.type;
                    html += `
                        <div class="fca-cm-relationship-item">
                            <div class="fca-cm-relationship-header">
                                <strong>${typeLabel}: ${this.escapeHtml(item.title)}</strong>
                                ${item.space ? `<span class="fca-cm-relationship-meta">in ${this.escapeHtml(item.space)}</span>` : ''}
                            </div>
                            ${item.url ? `<a href="${item.url}" target="_blank" class="fca-cm-media-link">View Content →</a>` : ''}
                        </div>
                    `;
                });
                html += '</div>';
                $usedIn.html(html);
            } else {
                $usedIn.html('<p class="fca-cm-empty-message">Not used in any content</p>');
            }
        },

        /**
         * Delete media
         */
        deleteMedia: function(mediaId) {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_delete_media',
                    nonce: fcaCMData.nonce,
                    media_id: mediaId
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess('Media deleted successfully');
                        $('#fca-cm-media-detail .fca-cm-detail-content').hide();
                        $('#fca-cm-media-detail .fca-cm-detail-empty').show();
                        this.loadMedia();
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError('Failed to delete media');
                }
            });
        },
        
        /**
         * Bulk delete media
         */
        bulkDeleteMedia: function(mediaIds) {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_bulk_delete_media',
                    nonce: fcaCMData.nonce,
                    media_ids: mediaIds
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess(`${mediaIds.length} media files deleted`);
                        this.loadMedia();
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError('Failed to delete media files');
                }
            });
        },

        /**
         * Load metadata
         */
        loadMetadata: function() {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_metadata',
                    nonce: fcaCMData.nonce,
                    page: this.currentPage,
                    per_page: this.filters.per_page || 20,
                    object_type: this.filters.object_type || '',
                    meta_key: this.filters.meta_key || ''
                },
                success: (response) => {
                    if (response.success) {
                        this.currentMetadata = response.data.data;
                        this.renderMetadataTable(response.data);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Render metadata table
         */
        renderMetadataTable: function(data) {
            const $tbody = $('#fca-cm-metadata-table tbody');
            $tbody.empty();

            if (data.data.length === 0) {
                $tbody.html('<tr><td colspan="7" class="fca-cm-text-center">No metadata found</td></tr>');
                return;
            }

            data.data.forEach(meta => {
                const $row = $(`
                    <tr>
                        <td>${meta.id}</td>
                        <td>${this.escapeHtml(meta.object_type)}</td>
                        <td>${meta.object_id}</td>
                        <td>${this.escapeHtml(meta.meta_key)}</td>
                        <td>${this.truncate(this.escapeHtml(meta.value || ''), 50)}</td>
                        <td>${this.formatDate(meta.created_at)}</td>
                        <td>
                            <button type="button" class="fca-cm-btn fca-cm-btn-sm fca-cm-btn-secondary fca-cm-edit-metadata" data-meta-id="${meta.id}" title="Edit">
                                <span class="dashicons dashicons-edit"></span>
                            </button>
                            <button type="button" class="fca-cm-btn fca-cm-btn-sm fca-cm-btn-danger fca-cm-delete-metadata" data-meta-id="${meta.id}" title="Delete">
                                <span class="dashicons dashicons-trash"></span>
                            </button>
                        </td>
                    </tr>
                `);
                $tbody.append($row);
            });

            this.renderPagination(data, '#fca-cm-metadata-pagination');
        },

        /**
         * Edit metadata
         */
        editMetadata: function(metaId) {
            // Find metadata in current data
            const metaData = this.currentMetadata ? this.currentMetadata.find(m => m.id == metaId) : null;
            
            if (!metaData) {
                this.showError('Metadata not found');
                return;
            }
            
            // Populate modal
            $('#fca-cm-meta-id').val(metaData.id);
            $('#fca-cm-meta-object-type').val(metaData.object_type);
            $('#fca-cm-meta-object-id').val(metaData.object_id);
            $('#fca-cm-meta-key').val(metaData.meta_key);
            $('#fca-cm-meta-value').val(metaData.value || '');
            
            // Show modal
            $('#fca-cm-metadata-edit-modal').fadeIn(200);
        },

        /**
         * Save metadata
         */
        saveMetadata: function() {
            const metaId = $('#fca-cm-meta-id').val();
            const value = $('#fca-cm-meta-value').val();
            
            if (!metaId) {
                this.showError('Invalid metadata ID');
                return;
            }
            
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_update_metadata',
                    nonce: fcaCMData.nonce,
                    meta_id: metaId,
                    value: value
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess(response.data.message || 'Metadata updated successfully');
                        $('#fca-cm-metadata-edit-modal').fadeOut(200);
                        this.loadMetadata();
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError('Failed to update metadata');
                }
            });
        },

        /**
         * Delete metadata
         */
        deleteMetadata: function(metaId) {
            if (!confirm('Are you sure you want to delete this metadata?')) {
                return;
            }

            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_delete_metadata',
                    nonce: fcaCMData.nonce,
                    meta_id: metaId
                },
                success: (response) => {
                    if (response.success) {
                        this.showSuccess(response.data.message);
                        this.loadMetadata();
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError(fcaCMData.strings.error);
                }
            });
        },

        /**
         * Initialize spaces page listeners
         */
        initSpacesListeners: function() {
            // Search
            let searchTimer;
            $(document).on('input', '#fca-cm-space-search', (e) => {
                clearTimeout(searchTimer);
                searchTimer = setTimeout(() => {
                    this.filters.search = $(e.target).val();
                    this.currentPage = 1;
                    this.loadSpaces();
                }, 500);
            });
            
            // Filters
            $(document).on('change', '#fca-cm-space-status, #fca-cm-space-privacy, #fca-cm-space-per-page', () => {
                this.filters.status = $('#fca-cm-space-status').val();
                this.filters.privacy = $('#fca-cm-space-privacy').val();
                this.filters.per_page = $('#fca-cm-space-per-page').val();
                this.currentPage = 1;
                this.loadSpaces();
            });
            
            // Select all
            $(document).on('change', '#fca-cm-select-all-spaces', (e) => {
                const isChecked = $(e.target).is(':checked');
                $('.fca-cm-space-checkbox').prop('checked', isChecked);
                $('.fca-cm-list-item').toggleClass('selected', isChecked);
                this.updateSpaceBulkButtons();
            });
            
            // Individual checkbox
            $(document).on('change', '.fca-cm-space-checkbox', (e) => {
                const $item = $(e.target).closest('.fca-cm-list-item');
                $item.toggleClass('selected', $(e.target).is(':checked'));
                
                const totalCheckboxes = $('.fca-cm-space-checkbox').length;
                const checkedCheckboxes = $('.fca-cm-space-checkbox:checked').length;
                $('#fca-cm-select-all-spaces').prop('checked', totalCheckboxes === checkedCheckboxes);
                
                this.updateSpaceBulkButtons();
            });
            
            // Bulk edit
            $(document).on('click', '#fca-cm-bulk-edit-spaces', () => {
                const selected = $('.fca-cm-space-checkbox:checked').map(function() { return $(this).val(); }).get();
                if (selected.length > 0) {
                    this.enterBulkEditModeSpaces(selected);
                }
            });
            
            // Bulk delete
            $(document).on('click', '#fca-cm-bulk-delete-spaces', () => {
                const selected = $('.fca-cm-space-checkbox:checked').map(function() { return $(this).val(); }).get();
                if (selected.length > 0) {
                    this.bulkDeleteSpaces(selected);
                }
            });
            
            // Cancel bulk edit
            $(document).on('click', '#fca-cm-cancel-bulk-edit-spaces', () => {
                this.exitBulkEditModeSpaces();
            });
            
            // Save bulk edit
            $(document).on('click', '#fca-cm-save-bulk-spaces', () => {
                this.saveBulkSpaces();
            });
            
            // Space item click
            $(document).on('click', '.fca-cm-list-item[data-space-id]', (e) => {
                if (!$(e.target).is('input[type="checkbox"]')) {
                    const spaceId = $(e.currentTarget).data('space-id');
                    this.loadSpaceDetail(spaceId);
                }
            });
            
            // Save space
            $(document).on('click', '#fca-cm-save-space', () => {
                this.saveSpace();
            });
            
            // Refresh
            $(document).on('click', '#fca-cm-refresh-spaces', () => {
                this.loadSpaces();
            });
            
            this.loadSpaces();
        },
        
        /**
         * Load spaces
         */
        loadSpaces: function() {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_spaces_list',
                    nonce: fcaCMData.nonce,
                    page: this.currentPage,
                    per_page: this.filters.per_page || 20,
                    search: this.filters.search || '',
                    status: this.filters.status || '',
                    privacy: this.filters.privacy || ''
                },
                success: (response) => {
                    if (response.success) {
                        this.renderSpacesList(response.data);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError('Failed to load spaces');
                }
            });
        },
        
        /**
         * Render spaces list
         */
        renderSpacesList: function(data) {
            const $list = $('#fca-cm-spaces-list');
            $list.empty();
            
            if (!data.data || data.data.length === 0) {
                $list.html(this.getEmptyState('No spaces found'));
                return;
            }
            
            data.data.forEach(space => {
                const $item = $(`
                    <div class="fca-cm-list-item has-checkbox" data-space-id="${space.id}">
                        <input type="checkbox" class="fca-cm-item-checkbox fca-cm-space-checkbox" value="${space.id}">
                        <div class="fca-cm-list-item-content">
                            <h4 class="fca-cm-list-item-title">${this.escapeHtml(space.title || 'Untitled')}</h4>
                            <div class="fca-cm-list-item-meta">
                                <span class="fca-cm-status-badge fca-cm-status-${space.status}">${space.status}</span>
                                <span>•</span>
                                <span>${space.members_count || 0} members</span>
                                <span>•</span>
                                <span>${space.posts_count || 0} posts</span>
                                <span>•</span>
                                <span>${space.privacy}</span>
                            </div>
                        </div>
                    </div>
                `);
                $list.append($item);
            });
            
            this.renderPagination(data, '#fca-cm-spaces-pagination');
        },
        
        /**
         * Load space detail
         */
        loadSpaceDetail: function(spaceId) {
            this.currentSpaceId = spaceId;
            
            $('.fca-cm-list-item').removeClass('active');
            $(`.fca-cm-list-item[data-space-id="${spaceId}"]`).addClass('active');
            
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_get_space',
                    nonce: fcaCMData.nonce,
                    space_id: spaceId
                },
                success: (response) => {
                    if (response.success) {
                        this.renderSpaceDetail(response.data);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError('Failed to load space details');
                }
            });
        },
        
        /**
         * Render space detail
         */
        renderSpaceDetail: function(space) {
            $('.fca-cm-detail-panel .fca-cm-detail-empty').hide();
            $('.fca-cm-detail-panel .fca-cm-detail-content').show();
            
            // Set form values
            $('#fca-cm-space-id').val(space.id);
            $('#fca-cm-space-title').text(space.title || 'Untitled Space');
            $('#fca-cm-space-title-input').val(space.title || '');
            $('#fca-cm-space-slug').val(space.slug || '');
            $('#fca-cm-space-description').val(space.description || '');
            $('#fca-cm-space-short-description').val(space.short_description || '');
            $('#fca-cm-space-logo').val(space.logo || '');
            $('#fca-cm-space-cover-photo').val(space.cover_photo || '');
            $('#fca-cm-space-type').val(space.type || '');
            $('#fca-cm-space-parent-id').val(space.parent_id || '');
            $('#fca-cm-space-serial').val(space.serial || 1);
            $('#fca-cm-space-settings').val(space.settings || '');
            $('#fca-cm-space-status-select').val(space.status);
            $('#fca-cm-space-privacy-select').val(space.privacy);
            
            // Set metadata
            $('#fca-cm-space-created-by').text(space.creator || 'Unknown');
            $('#fca-cm-space-members-count').text(space.members_count || 0);
            $('#fca-cm-space-posts-count').text(space.posts_count || 0);
            $('#fca-cm-space-created').text(this.formatDate(space.created_at));
            $('#fca-cm-space-updated').text(this.formatDate(space.updated_at));
        },
        
        /**
         * Save space
         */
        saveSpace: function() {
            const spaceId = $('#fca-cm-space-id').val();
            
            if (!spaceId) {
                this.showError('No space selected');
                return;
            }
            
            const data = {
                action: 'fca_cm_update_space',
                nonce: fcaCMData.nonce,
                space_id: spaceId,
                title: $('#fca-cm-space-title-input').val(),
                slug: $('#fca-cm-space-slug').val(),
                description: $('#fca-cm-space-description').val(),
                short_description: $('#fca-cm-space-short-description').val(),
                logo: $('#fca-cm-space-logo').val(),
                cover_photo: $('#fca-cm-space-cover-photo').val(),
                type: $('#fca-cm-space-type').val(),
                parent_id: $('#fca-cm-space-parent-id').val(),
                serial: $('#fca-cm-space-serial').val(),
                settings: $('#fca-cm-space-settings').val(),
                status: $('#fca-cm-space-status-select').val(),
                privacy: $('#fca-cm-space-privacy-select').val()
            };
            
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: data,
                success: (response) => {
                    if (response.success) {
                        this.showSuccess('Space updated successfully');
                        $('#fca-cm-space-title').text(data.title);
                        this.loadSpaces();
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError('Failed to update space');
                }
            });
        },
        
        /**
         * Enter bulk edit mode for spaces
         */
        enterBulkEditModeSpaces: function(spaceIds) {
            this.selectedSpaces = spaceIds;
            $('#fca-cm-space-bulk-edit-panel').slideDown(200);
            $('.fca-cm-detail-panel .fca-cm-detail-content').hide();
            $('.fca-cm-detail-panel .fca-cm-detail-empty').hide();
        },
        
        /**
         * Exit bulk edit mode for spaces
         */
        exitBulkEditModeSpaces: function() {
            this.selectedSpaces = [];
            $('#fca-cm-space-bulk-edit-panel').slideUp(200);
            $('.fca-cm-space-checkbox').prop('checked', false);
            $('.fca-cm-list-item').removeClass('selected');
            $('#fca-cm-select-all-spaces').prop('checked', false);
            this.updateSpaceBulkButtons();
            $('.fca-cm-detail-panel .fca-cm-detail-empty').show();
        },
        
        /**
         * Save bulk edit for spaces
         */
        saveBulkSpaces: function() {
            if (!this.selectedSpaces || this.selectedSpaces.length === 0) {
                this.showError('No spaces selected');
                return;
            }
            
            const status = $('#fca-cm-bulk-space-status').val();
            const privacy = $('#fca-cm-bulk-space-privacy').val();
            
            if (!status && !privacy) {
                this.showError('Please select at least one field to update');
                return;
            }
            
            const data = {
                action: 'fca_cm_bulk_update_spaces',
                nonce: fcaCMData.nonce,
                space_ids: this.selectedSpaces
            };
            
            if (status) {
                data.status = status;
            }
            
            if (privacy) {
                data.privacy = privacy;
            }
            
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: data,
                success: (response) => {
                    if (response.success) {
                        this.showSuccess(response.data.message);
                        this.exitBulkEditModeSpaces();
                        this.loadSpaces();
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                    this.showError('Failed to update spaces');
                }
            });
        },
        
        /**
         * Update space bulk buttons
         */
        updateSpaceBulkButtons: function() {
            const selectedCount = $('.fca-cm-space-checkbox:checked').length;
            const $bulkEdit = $('#fca-cm-bulk-edit-spaces');
            const $selectedCount = $('.fca-cm-bulk-actions .fca-cm-selected-count');
            
            if (selectedCount > 0) {
                $bulkEdit.prop('disabled', false);
                $selectedCount.text(`${selectedCount} selected`);
            } else {
                $bulkEdit.prop('disabled', true);
                $selectedCount.text('');
            }
        },

        /**
         * Render pagination
         */
        renderPagination: function(data, selector) {
            const $pagination = $(selector);
            $pagination.empty();

            if (data.pages <= 1) {
                return;
            }

            const $info = $(`<div class="fca-cm-pagination-info">Page ${data.current_page} of ${data.pages} (${data.total} total)</div>`);
            const $buttons = $('<div class="fca-cm-pagination-buttons"></div>');

            // Previous button
            $buttons.append(`
                <button type="button" class="fca-cm-pagination-btn" data-page="${data.current_page - 1}" ${data.current_page === 1 ? 'disabled' : ''}>
                    <span class="dashicons dashicons-arrow-left-alt2"></span>
                </button>
            `);

            // Page numbers
            for (let i = 1; i <= data.pages; i++) {
                if (i === 1 || i === data.pages || (i >= data.current_page - 2 && i <= data.current_page + 2)) {
                    $buttons.append(`
                        <button type="button" class="fca-cm-pagination-btn ${i === data.current_page ? 'active' : ''}" data-page="${i}">
                            ${i}
                        </button>
                    `);
                } else if (i === data.current_page - 3 || i === data.current_page + 3) {
                    $buttons.append('<span>...</span>');
                }
            }

            // Next button
            $buttons.append(`
                <button type="button" class="fca-cm-pagination-btn" data-page="${data.current_page + 1}" ${data.current_page === data.pages ? 'disabled' : ''}>
                    <span class="dashicons dashicons-arrow-right-alt2"></span>
                </button>
            `);

            // Click handler
            $buttons.on('click', '.fca-cm-pagination-btn:not(:disabled)', (e) => {
                this.currentPage = parseInt($(e.currentTarget).data('page'));
                this.refreshList();
            });

            $pagination.append($info, $buttons);
        },

        /**
         * Refresh current list
         */
        refreshList: function() {
            if ($('#fca-cm-posts-list').length) {
                this.loadPosts();
            } else if ($('#fca-cm-courses-list').length) {
                this.loadCourses();
            } else if ($('#fca-cm-members-list').length) {
                this.loadMembers();
            } else if ($('#fca-cm-comments-list').length) {
                this.loadComments();
            } else if ($('#fca-cm-reactions-table').length) {
                this.loadReactions();
            } else if ($('#fca-cm-media-grid').length) {
                this.loadMedia();
            } else if ($('#fca-cm-metadata-table').length) {
                this.loadMetadata();
            }
        },

        /**
         * Close modal
         */
        closeModal: function(e) {
            if (e && !$(e.target).hasClass('fca-cm-modal-backdrop') && !$(e.target).is('[data-close]') && !$(e.target).parent().is('[data-close]')) {
                return;
            }
            $('.fca-cm-modal').fadeOut(200);
        },

        /**
         * Helper: Get empty state HTML
         */
        getEmptyState: function(message) {
            return `
                <div class="fca-cm-empty">
                    <span class="dashicons dashicons-info"></span>
                    <p>${message}</p>
                </div>
            `;
        },

        /**
         * Helper: Get status badge
         */
        getStatusBadge: function(status) {
            const colors = {
                'published': 'green',
                'draft': 'gray',
                'pending': 'yellow',
                'archived': 'red',
                'private': 'blue'
            };
            const color = colors[status] || 'gray';
            return `<span class="fca-cm-badge fca-cm-badge-${color}">${status}</span>`;
        },

        /**
         * Helper: Get reaction icon
         */
        getReactionIcon: function(type) {
            const icons = {
                'like': '👍',
                'love': '❤️',
                'laugh': '😂',
                'wow': '😮',
                'sad': '😢',
                'angry': '😠'
            };
            return icons[type] || '👍';
        },

        /**
         * Helper: Format date
         */
        formatDate: function(dateString) {
            if (!dateString) return 'Never';
            const date = new Date(dateString);
            return date.toLocaleString();
        },

        /**
         * Helper: Format file size in bytes to human readable
         */
        formatFileSize: function(bytes) {
            if (!bytes || bytes === 0) return '0 Bytes';
            
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            
            return Math.round((bytes / Math.pow(k, i)) * 100) / 100 + ' ' + sizes[i];
        },

        /**
         * Helper: Format date for datetime-local input
         */
        formatDateTimeLocal: function(dateString) {
            if (!dateString) return '';
            const date = new Date(dateString);
            // Format: YYYY-MM-DDTHH:MM
            const year = date.getFullYear();
            const month = String(date.getMonth() + 1).padStart(2, '0');
            const day = String(date.getDate()).padStart(2, '0');
            const hours = String(date.getHours()).padStart(2, '0');
            const minutes = String(date.getMinutes()).padStart(2, '0');
            return `${year}-${month}-${day}T${hours}:${minutes}`;
        },

        /**
         * Helper: Escape HTML
         */
        escapeHtml: function(text) {
            const map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.replace(/[&<>"']/g, m => map[m]);
        },

        /**
         * Helper: Strip HTML
         */
        stripHtml: function(html) {
            const tmp = document.createElement('DIV');
            tmp.innerHTML = html;
            return tmp.textContent || tmp.innerText || '';
        },

        /**
         * Helper: Truncate text
         */
        truncate: function(text, length) {
            if (text.length <= length) return text;
            return text.substring(0, length) + '...';
        },

        /**
         * Show success message
         */
        showSuccess: function(message) {
            // You can implement a toast notification here
            alert(message);
        },

        /**
         * Show error message
         */
        showError: function(message) {
            alert('Error: ' + message);
        },
        
        /**
         * Show debug info
         */
        showDebugInfo: function() {
            $.ajax({
                url: fcaCMData.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fca_cm_debug_counts',
                    nonce: fcaCMData.nonce
                },
                success: (response) => {
                    if (response.success) {
                        const data = response.data;
                        let msg = 'DATABASE DEBUG INFO\n\n';
                        msg += 'Table Counts:\n';
                        msg += `- All Posts: ${data.counts.posts_all}\n`;
                        msg += `- Feed Posts: ${data.counts.posts_feed}\n`;
                        msg += `- Course Posts: ${data.counts.posts_course}\n`;
                        msg += `- Lesson Posts: ${data.counts.posts_lesson}\n`;
                        msg += `- Comments: ${data.counts.comments}\n`;
                        msg += `- Reactions: ${data.counts.reactions}\n`;
                        msg += `- Media: ${data.counts.media}\n`;
                        msg += `- Metadata: ${data.counts.metadata}\n`;
                        msg += `- Spaces: ${data.counts.spaces}\n\n`;
                        
                        msg += `\nSpaces Debug:\n`;
                        msg += `- Has Utility Class: ${data.has_utility_class ? 'Yes' : 'No'}\n`;
                        msg += `- Processed Spaces: ${data.spaces_count}\n`;
                        msg += `- Raw Spaces in DB: ${data.raw_spaces_count}\n\n`;
                        
                        if (data.raw_spaces && data.raw_spaces.length > 0) {
                            msg += 'All Spaces from API:\n';
                            data.raw_spaces.forEach(space => {
                                msg += `- ID ${space.id}: ${space.title || space.slug} (${space.slug}) [${space.status}/${space.type}]\n`;
                            });
                            msg += '\n';
                        } else {
                            msg += 'No spaces found via API.\n\n';
                        }
                        
                        if (data.spaces && data.spaces.length > 0) {
                            msg += 'Processed Spaces (for dropdown):\n';
                            data.spaces.forEach(space => {
                                msg += `- ID ${space.id}: ${space.name}\n`;
                            });
                            msg += '\n';
                        }
                        
                        if (data.recent_posts && data.recent_posts.length > 0) {
                            msg += 'Recent Posts:\n';
                            data.recent_posts.forEach(post => {
                                msg += `- ID ${post.id}: ${post.title || 'Untitled'} (${post.type}/${post.status})\n`;
                            });
                        } else {
                            msg += 'No posts found in database.\n';
                        }
                        
                        msg += `\nTable Prefix: ${data.table_prefix}`;
                        
                        console.log('Debug Data:', data);
                        alert(msg);
                    } else {
                        this.showError(response.data.message);
                    }
                },
                error: () => {
                this.showError('Failed to get debug info');
            }
        });
    },
    
    /**
     * Load all available terms/topics
     */
    loadAllTerms: function() {
        $.ajax({
            url: fcaCMData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'fca_cm_get_all_terms',
                nonce: fcaCMData.nonce
            },
            success: (response) => {
                if (response.success) {
                    this.allTerms = response.data;
                    this.populateTermSelects();
                }
            }
        });
    },
    
    /**
     * Populate term select dropdowns
     */
    populateTermSelects: function() {
        if (!this.allTerms || this.allTerms.length === 0) {
            return;
        }
        
        const options = this.allTerms.map(term =>
            `<option value="${term.id}">${this.escapeHtml(term.title || term.slug)}</option>`
        ).join('');
        
        $('#fca-cm-add-term-select').append(options);
        $('#fca-cm-bulk-add-term').append(options);
    },
    
    /**
     * Add term to post
     */
    addTermToPost: function(e) {
        const termId = $(e.target).val();
        if (!termId || !this.currentPostId) return;
        
        $.ajax({
            url: fcaCMData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'fca_cm_add_term_to_post',
                nonce: fcaCMData.nonce,
                post_id: this.currentPostId,
                term_id: termId
            },
            success: (response) => {
                if (response.success) {
                    this.showSuccess('Topic added');
                    this.loadPostDetail(this.currentPostId);
                    $(e.target).val('');
                } else {
                    this.showError(response.data.message);
                }
            },
            error: () => {
                this.showError('Failed to add topic');
            }
        });
    },
    
    /**
     * Remove term from post
     */
    removeTermFromPost: function(e) {
        e.stopPropagation();
        const $tag = $(e.target).closest('.fca-cm-tag');
        const termId = $tag.data('term-id');
        
        if (!termId || !this.currentPostId) return;
        
        $.ajax({
            url: fcaCMData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'fca_cm_remove_term_from_post',
                nonce: fcaCMData.nonce,
                post_id: this.currentPostId,
                term_id: termId
            },
            success: (response) => {
                if (response.success) {
                    $tag.fadeOut(300, function() { $(this).remove(); });
                    this.showSuccess('Topic removed');
                } else {
                    this.showError(response.data.message);
                }
            },
            error: () => {
                this.showError('Failed to remove topic');
            }
        });
    },
    
    /**
     * Open comment modal
     */
    openCommentModal: function(e) {
        const commentId = $(e.currentTarget).data('comment-id');
        
        if (!commentId) return;
        
        $.ajax({
            url: fcaCMData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'fca_cm_get_comment',
                nonce: fcaCMData.nonce,
                comment_id: commentId
            },
            success: (response) => {
                if (response.success) {
                    const comment = response.data;
                    $('#fca-cm-modal-comment-author').text(comment.author || 'Unknown');
                    $('#fca-cm-modal-comment-date').text(comment.created_at_formatted);
                    $('#fca-cm-modal-comment-content').html(comment.message_rendered || this.escapeHtml(comment.message || ''));
                    $('#fca-cm-modal-comment-status').text(comment.status);
                    $('#fca-cm-modal-comment-reactions').text(comment.reactions_count || 0);
                    $('#fca-cm-comment-modal').fadeIn(200);
                } else {
                    this.showError(response.data.message);
                }
            },
            error: () => {
                this.showError('Failed to load comment');
            }
        });
    },
    
    /**
     * Close modal
     */
    closeModal: function() {
        $('.fca-cm-modal').fadeOut(200);
    },
    
    /**
     * Update bulk edit buttons
     */
    updateBulkButtons: function() {
        const checkedCount = $('.fca-cm-item-checkbox:checked').length;
        $('#fca-cm-bulk-edit, #fca-cm-bulk-delete').prop('disabled', checkedCount === 0);
        $('.fca-cm-selected-count').text(checkedCount > 0 ? `${checkedCount} selected` : '');
        
        // Update list item appearance
        $('.fca-cm-list-item').each(function() {
            $(this).toggleClass('selected', $(this).find('.fca-cm-item-checkbox').is(':checked'));
        });
    },
    
    /**
     * Enter bulk edit mode
     */
    enterBulkEditMode: function() {
        this.selectedPosts = $('.fca-cm-item-checkbox:checked').map(function() {
            return $(this).val();
        }).get();
        
        if (this.selectedPosts.length === 0) {
            return;
        }
        
        $('#fca-cm-bulk-count').text(this.selectedPosts.length);
        $('#fca-cm-detail-panel').hide();
        $('#fca-cm-bulk-edit-panel').show();
        
        // Reset form
        $('#fca-cm-bulk-form')[0].reset();
    },
    
    /**
     * Exit bulk edit mode
     */
    exitBulkEditMode: function() {
        $('#fca-cm-bulk-edit-panel').hide();
        $('#fca-cm-detail-panel').show();
        this.selectedPosts = [];
    },
    
    /**
     * Save bulk edit
     */
    saveBulkEdit: function() {
        if (!this.selectedPosts || this.selectedPosts.length === 0) {
            return;
        }
        
        const data = {
            action: 'fca_cm_bulk_update_posts',
            nonce: fcaCMData.nonce,
            post_ids: this.selectedPosts,
            space_id: $('#fca-cm-bulk-space').val(),
            status: $('#fca-cm-bulk-status').val(),
            privacy: $('#fca-cm-bulk-privacy').val()
        };
        
        $.ajax({
            url: fcaCMData.ajaxUrl,
            type: 'POST',
            data: data,
            success: (response) => {
                if (response.success) {
                    this.showSuccess(response.data.message);
                    this.loadPosts();
                    this.exitBulkEditMode();
                    $('.fca-cm-item-checkbox').prop('checked', false);
                    $('#fca-cm-select-all').prop('checked', false);
                    this.updateBulkButtons();
                } else {
                    this.showError(response.data.message);
                }
            },
            error: () => {
                this.showError('Failed to update posts');
            }
        });
    },
    
    /**
     * Bulk add term to selected posts
     */
    bulkAddTerm: function(e) {
        const termId = $(e.target).val();
        
        if (!termId || !this.selectedPosts || this.selectedPosts.length === 0) {
            return;
        }
        
        $.ajax({
            url: fcaCMData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'fca_cm_bulk_add_term',
                nonce: fcaCMData.nonce,
                post_ids: this.selectedPosts,
                term_id: termId
            },
            success: (response) => {
                if (response.success) {
                    this.showSuccess(response.data.message);
                    $(e.target).val('');
                } else {
                    this.showError(response.data.message);
                }
            },
            error: () => {
                this.showError('Failed to add topic to posts');
            }
        });
    },
    
    /**
     * ===================================
     * MEMBERS MANAGEMENT
     * ===================================
     */
    
    /**
     * Initialize members page listeners
     */
    initMembersListeners: function() {
        // Search
        let searchTimer;
        $(document).on('input', '#fca-cm-members-search', (e) => {
            clearTimeout(searchTimer);
            searchTimer = setTimeout(() => {
                this.filters.search = $(e.target).val();
                this.currentPage = 1;
                this.loadMembers();
            }, 500);
        });

        // Filters
        $(document).on('change', '#fca-cm-member-status-filter, #fca-cm-member-role-filter, #fca-cm-member-space-filter', () => {
            this.filters.status = $('#fca-cm-member-status-filter').val();
            this.filters.role = $('#fca-cm-member-role-filter').val();
            this.filters.space_id = $('#fca-cm-member-space-filter').val();
            this.currentPage = 1;
            this.loadMembers();
        });

        // Select all
        $(document).on('change', '#fca-cm-select-all-members', (e) => {
            const checked = $(e.target).prop('checked');
            $('.fca-cm-member-checkbox').prop('checked', checked);
            this.updateMemberBulkButtons();
        });

        // Individual checkbox
        $(document).on('change', '.fca-cm-member-checkbox', () => {
            this.updateMemberBulkButtons();
        });

        // Bulk enroll
        $(document).on('click', '#fca-cm-bulk-enroll-members', () => {
            this.openBulkEnrollModal();
        });

        // Bulk unenroll
        $(document).on('click', '#fca-cm-bulk-unenroll-members', () => {
            this.openBulkUnenrollModal();
        });

        // Save bulk enroll
        $(document).on('click', '#fca-cm-save-bulk-enroll', () => {
            this.saveBulkEnroll();
        });

        // Save bulk unenroll
        $(document).on('click', '#fca-cm-save-bulk-unenroll', () => {
            this.saveBulkUnenroll();
        });

        // Cancel selection
        $(document).on('click', '#fca-cm-cancel-selection', () => {
            $('.fca-cm-member-checkbox').prop('checked', false);
            $('#fca-cm-select-all-members').prop('checked', false);
            this.updateMemberBulkButtons();
        });

        // Refresh
        $(document).on('click', '#fca-cm-refresh-members', () => {
            this.loadMembers();
        });

        // Select member from list
        $(document).on('click', '#fca-cm-members-list .fca-cm-list-item', function(e) {
            // Don't trigger if clicking checkbox
            if ($(e.target).hasClass('fca-cm-item-checkbox') || $(e.target).hasClass('fca-cm-member-checkbox')) {
                return;
            }
            
            // Highlight selected
            $('#fca-cm-members-list .fca-cm-list-item').removeClass('active');
            $(this).addClass('active');
            
            const userId = $(this).data('user-id');
            FCA_CM.loadMemberDetail(userId);
        });

        // Save member
        $(document).on('click', '#fca-cm-save-member', () => {
            this.saveMember();
        });

        // Add enrollment
        $(document).on('click', '#fca-cm-add-member-enrollment', () => {
            this.openAddEnrollmentModal();
        });

        // Close bulk enroll modal
        $(document).on('click', '#fca-cm-bulk-enroll-modal .fca-cm-modal-close', () => {
            $('#fca-cm-bulk-enroll-modal').fadeOut(200);
        });

        // Close bulk unenroll modal
        $(document).on('click', '#fca-cm-bulk-unenroll-modal .fca-cm-modal-close', () => {
            $('#fca-cm-bulk-unenroll-modal').fadeOut(200);
        });

        // Close add enrollment modal
        $(document).on('click', '#fca-cm-add-enrollment-modal .fca-cm-modal-close', () => {
            $('#fca-cm-add-enrollment-modal').fadeOut(200);
        });

        // Save add enrollment
        $(document).on('click', '#fca-cm-save-add-enrollment', () => {
            this.saveAddEnrollment();
        });

        // Remove enrollment
        $(document).on('click', '.fca-cm-remove-enrollment', function() {
            const userId = $(this).data('user-id');
            const spaceId = $(this).data('space-id');
            FCA_CM.removeEnrollment(userId, spaceId);
        });

        // Initial load
        this.loadMembers();
    },

    /**
     * Load members list
     */
    loadMembers: function() {
        const $list = $('#fca-cm-members-list');
        $list.html('<div class="fca-cm-loader"><div class="fca-cm-spinner"></div><p>Loading members...</p></div>');

        $.ajax({
            url: fcaCMData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'fca_cm_get_members',
                nonce: fcaCMData.nonce,
                page: this.currentPage,
                per_page: this.filters.per_page || 20,
                search: this.filters.search || '',
                status: this.filters.status || '',
                role: this.filters.role || '',
                space_id: this.filters.space_id || '',
                orderby: 'display_name',
                order: 'ASC'
            },
            success: (response) => {
                if (response.success) {
                    this.renderMembersList(response.data);
                } else {
                    this.showError(response.data.message || 'Failed to load members');
                }
            },
            error: () => {
                this.showError('Failed to load members');
            }
        });
    },

    /**
     * Render members list
     */
    renderMembersList: function(data) {
        const $list = $('#fca-cm-members-list');
        
        if (!data.members || data.members.length === 0) {
            $list.html(`
                <div class="fca-cm-empty">
                    <span class="dashicons dashicons-admin-users"></span>
                    <h3>No members found</h3>
                    <p>Try adjusting your filters or search terms.</p>
                </div>
            `);
            return;
        }

        let html = '';

        data.members.forEach(member => {
            const avatar = member.avatar || 'https://www.gravatar.com/avatar/?d=mp&s=40';
            const displayName = this.escapeHtml(member.display_name || member.username || 'No Name');
            const email = this.escapeHtml(member.user_email || '');
            const status = member.status || 'active';
            const points = member.total_points || 0;
            const enrollmentsCount = member.enrollments_count || 0;
            
            const statusClass = status === 'active' ? 'green' : (status === 'blocked' ? 'red' : 'yellow');
            
            html += `
                <div class="fca-cm-list-item has-checkbox" data-user-id="${member.user_id}">
                    <input type="checkbox" class="fca-cm-item-checkbox fca-cm-member-checkbox" value="${member.user_id}" onclick="event.stopPropagation()">
                    <img src="${avatar}" alt="${displayName}" class="fca-cm-member-avatar">
                    <div class="fca-cm-list-item-content">
                        <div class="fca-cm-member-name">${displayName}</div>
                        <div class="fca-cm-list-item-meta">
                            <span>${email}</span>
                            <span class="fca-cm-badge fca-cm-badge-${statusClass}">${status}</span>
                            <span>${points} pts</span>
                            <span>${enrollmentsCount} enrollments</span>
                        </div>
                    </div>
                </div>
            `;
        });

        $list.html(html);

        // Render pagination
        this.renderPagination(data.total, data.pages, '.fca-cm-pagination', () => this.loadMembers());
    },

    /**
     * Load member detail
     */
    loadMemberDetail: function(userId) {
        const $detail = $('#fca-cm-member-detail');
        const $empty = $('.fca-cm-detail-empty');

        $detail.html('<div class="fca-cm-loader"><div class="fca-cm-spinner"></div></div>').show();
        $empty.hide();

        // Highlight selected member
        $('.fca-cm-member-item').removeClass('fca-cm-active');
        $(`.fca-cm-member-item[data-user-id="${userId}"]`).addClass('fca-cm-active');

        this.currentPostId = userId; // Reuse this property

        $.ajax({
            url: fcaCMData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'fca_cm_get_member',
                nonce: fcaCMData.nonce,
                user_id: userId
            },
            success: (response) => {
                if (response.success) {
                    this.renderMemberDetail(response.data);
                } else {
                    this.showError(response.data.message || 'Failed to load member');
                }
            },
            error: () => {
                this.showError('Failed to load member');
            }
        });
    },

    /**
     * Render member detail
     */
    renderMemberDetail: function(member) {
        const $detail = $('#fca-cm-member-detail');
        const avatar = member.avatar || 'https://www.gravatar.com/avatar/?d=mp&s=80';
        const meta = member.meta || {};

        let enrollmentsHtml = '';
        if (member.enrollments && member.enrollments.length > 0) {
            enrollmentsHtml = member.enrollments.map(enrollment => {
                const roleClass = enrollment.role === 'admin' ? 'danger' : (enrollment.role === 'moderator' ? 'warning' : 'info');
                return `
                    <div class="fca-cm-enrollment-item">
                        <div class="fca-cm-enrollment-info">
                            <strong>${enrollment.space_title || 'Unknown Space'}</strong>
                            <span class="fca-cm-badge fca-cm-badge-${roleClass}">${enrollment.role}</span>
                            <span class="fca-cm-meta-text">${enrollment.space_type || 'space'}</span>
                        </div>
                        <button type="button" class="fca-cm-btn-icon fca-cm-remove-enrollment" 
                                data-user-id="${member.user_id}" data-space-id="${enrollment.space_id}" 
                                title="Remove enrollment">
                            <span class="dashicons dashicons-no"></span>
                        </button>
                    </div>
                `;
            }).join('');
        } else {
            enrollmentsHtml = '<p class="fca-cm-empty-text">No enrollments</p>';
        }

        const html = `
            <div class="fca-cm-detail-content">
                <div class="fca-cm-detail-header">
                    <img src="${avatar}" alt="${member.display_name}" class="fca-cm-detail-avatar">
                    <div>
                        <h3>${member.display_name || 'No Name'}</h3>
                        <p class="fca-cm-meta-text">${member.user_email}</p>
                    </div>
                </div>
                
                <div class="fca-cm-detail-body">
                <div class="fca-cm-form-group">
                    <label>Display Name</label>
                    <input type="text" id="fca-cm-member-display-name" class="fca-cm-input" 
                           value="${member.display_name || ''}" placeholder="Display Name">
                </div>
                
                <div class="fca-cm-form-group">
                    <label>Username</label>
                    <input type="text" id="fca-cm-member-username" class="fca-cm-input" 
                           value="${member.username || ''}" placeholder="Username">
                </div>
                
                <div class="fca-cm-form-group">
                    <label>Email</label>
                    <input type="email" class="fca-cm-input" value="${member.user_email || ''}" disabled>
                    <small class="fca-cm-help-text">Email cannot be changed here. Use WordPress user settings.</small>
                </div>
                
                <div class="fca-cm-form-group">
                    <label>Bio / Description</label>
                    <textarea id="fca-cm-member-description" class="fca-cm-input" rows="3" 
                              placeholder="Short description...">${member.short_description || ''}</textarea>
                </div>
                
                <div class="fca-cm-form-row">
                    <div class="fca-cm-form-group">
                        <label>Status</label>
                        <select id="fca-cm-member-status" class="fca-cm-input">
                            <option value="active" ${member.status === 'active' ? 'selected' : ''}>Active</option>
                            <option value="blocked" ${member.status === 'blocked' ? 'selected' : ''}>Blocked</option>
                            <option value="pending" ${member.status === 'pending' ? 'selected' : ''}>Pending</option>
                        </select>
                    </div>
                    
                    <div class="fca-cm-form-group">
                        <label>Total Points</label>
                        <input type="number" id="fca-cm-member-points" class="fca-cm-input" 
                               value="${member.total_points || 0}" min="0">
                    </div>
                </div>
                
                <div class="fca-cm-form-group">
                    <label class="fca-cm-checkbox-label">
                        <input type="checkbox" id="fca-cm-member-verified" 
                               ${member.is_verified ? 'checked' : ''}>
                        Verified Member
                    </label>
                </div>
                
                <div class="fca-cm-form-group">
                    <label class="fca-cm-label-with-action">
                        <span>Enrollments (${member.enrollments ? member.enrollments.length : 0})</span>
                        <button type="button" id="fca-cm-add-member-enrollment" class="fca-cm-btn-text">
                            <span class="dashicons dashicons-plus-alt"></span> Add Enrollment
                        </button>
                    </label>
                    <div class="fca-cm-enrollments-list">
                        ${enrollmentsHtml}
                    </div>
                </div>
                
                <div class="fca-cm-meta-info">
                    <small><strong>User ID:</strong> ${member.user_id}</small>
                    <small><strong>Registered:</strong> ${member.user_registered || 'N/A'}</small>
                    <small><strong>Last Activity:</strong> ${member.last_activity || 'Never'}</small>
                </div>
                
                <div class="fca-cm-form-group" style="margin-top: 24px; padding-top: 24px; border-top: 1px solid #f0f0f1;">
                    <button type="button" id="fca-cm-save-member" class="fca-cm-btn fca-cm-btn-primary" style="width: 100%;">
                        <span class="dashicons dashicons-yes"></span>
                        Save Changes
                    </button>
                </div>
                </div>
            </div>
        `;

        $('.fca-cm-detail-empty').hide();
        $detail.html(html).show();
    },

    /**
     * Save member
     */
    saveMember: function() {
        const userId = this.currentPostId;
        const data = {
            action: 'fca_cm_update_member',
            nonce: fcaCMData.nonce,
            user_id: userId,
            display_name: $('#fca-cm-member-display-name').val(),
            username: $('#fca-cm-member-username').val(),
            short_description: $('#fca-cm-member-description').val(),
            status: $('#fca-cm-member-status').val(),
            total_points: $('#fca-cm-member-points').val(),
            is_verified: $('#fca-cm-member-verified').is(':checked') ? 1 : 0
        };

        $.ajax({
            url: fcaCMData.ajaxUrl,
            type: 'POST',
            data: data,
            success: (response) => {
                if (response.success) {
                    this.showSuccess('Member updated successfully');
                    this.loadMembers();
                    this.loadMemberDetail(userId);
                } else {
                    this.showError(response.data.message || 'Failed to update member');
                }
            },
            error: () => {
                this.showError('Failed to update member');
            }
        });
    },

    /**
     * Update member bulk buttons
     */
    updateMemberBulkButtons: function() {
        const checked = $('.fca-cm-member-checkbox:checked').length;
        const $enrollBtn = $('#fca-cm-bulk-enroll-members');
        const $unenrollBtn = $('#fca-cm-bulk-unenroll-members');
        const $selectedCount = $('.fca-cm-selected-count');
        
        if (checked > 0) {
            $enrollBtn.prop('disabled', false);
            $unenrollBtn.prop('disabled', false);
            $selectedCount.text(`${checked} selected`);
        } else {
            $enrollBtn.prop('disabled', true);
            $unenrollBtn.prop('disabled', true);
            $selectedCount.text('');
        }
    },

    /**
     * Open bulk enroll modal
     */
    openBulkEnrollModal: function() {
        $('#fca-cm-bulk-enroll-modal').fadeIn(200);
    },

    /**
     * Open bulk unenroll modal
     */
    openBulkUnenrollModal: function() {
        $('#fca-cm-bulk-unenroll-modal').fadeIn(200);
    },

    /**
     * Save bulk enroll
     */
    saveBulkEnroll: function() {
        const userIds = $('.fca-cm-member-checkbox:checked').map(function() {
            return $(this).val();
        }).get();

        const spaceId = $('#fca-cm-bulk-enroll-space').val();
        const role = $('#fca-cm-bulk-enroll-role').val();

        if (!spaceId) {
            this.showError('Please select a space or course');
            return;
        }

        $.ajax({
            url: fcaCMData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'fca_cm_bulk_add_enrollments',
                nonce: fcaCMData.nonce,
                user_ids: userIds,
                space_id: spaceId,
                role: role
            },
            success: (response) => {
                if (response.success) {
                    this.showSuccess(response.data.message);
                    this.closeModal();
                    this.loadMembers();
                    if (this.currentPostId) {
                        this.loadMemberDetail(this.currentPostId);
                    }
                } else {
                    this.showError(response.data.message || 'Failed to enroll members');
                }
            },
            error: () => {
                this.showError('Failed to enroll members');
            }
        });
    },

    /**
     * Save bulk unenroll
     */
    saveBulkUnenroll: function() {
        const userIds = $('.fca-cm-member-checkbox:checked').map(function() {
            return $(this).val();
        }).get();

        const spaceId = $('#fca-cm-bulk-unenroll-space').val();

        if (!spaceId) {
            this.showError('Please select a space or course');
            return;
        }

        if (!confirm('Are you sure you want to remove these members from the selected space?')) {
            return;
        }

        $.ajax({
            url: fcaCMData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'fca_cm_bulk_remove_enrollments',
                nonce: fcaCMData.nonce,
                user_ids: userIds,
                space_id: spaceId
            },
            success: (response) => {
                if (response.success) {
                    this.showSuccess(response.data.message);
                    this.closeModal();
                    this.loadMembers();
                    if (this.currentPostId) {
                        this.loadMemberDetail(this.currentPostId);
                    }
                } else {
                    this.showError(response.data.message || 'Failed to remove enrollments');
                }
            },
            error: () => {
                this.showError('Failed to remove enrollments');
            }
        });
    },

    /**
     * Open add enrollment modal
     */
    openAddEnrollmentModal: function() {
        $('#fca-cm-add-enrollment-modal').fadeIn(200);
    },

    /**
     * Save add enrollment
     */
    saveAddEnrollment: function() {
        const userId = this.currentPostId;
        const spaceId = $('#fca-cm-add-enrollment-space').val();
        const role = $('#fca-cm-add-enrollment-role').val();

        if (!spaceId) {
            this.showError('Please select a space or course');
            return;
        }

        $.ajax({
            url: fcaCMData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'fca_cm_add_enrollment',
                nonce: fcaCMData.nonce,
                user_id: userId,
                space_id: spaceId,
                role: role
            },
            success: (response) => {
                if (response.success) {
                    this.showSuccess('Enrollment added successfully');
                    this.closeModal();
                    this.loadMemberDetail(userId);
                } else {
                    this.showError(response.data.message || 'Failed to add enrollment');
                }
            },
            error: () => {
                this.showError('Failed to add enrollment');
            }
        });
    },

    /**
     * Remove enrollment
     */
    removeEnrollment: function(userId, spaceId) {
        if (!confirm('Are you sure you want to remove this enrollment?')) {
            return;
        }

        $.ajax({
            url: fcaCMData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'fca_cm_remove_enrollment',
                nonce: fcaCMData.nonce,
                user_id: userId,
                space_id: spaceId
            },
            success: (response) => {
                if (response.success) {
                    this.showSuccess('Enrollment removed successfully');
                    this.loadMemberDetail(userId);
                } else {
                    this.showError(response.data.message || 'Failed to remove enrollment');
                }
            },
            error: () => {
                this.showError('Failed to remove enrollment');
            }
        });
    }
};

// Initialize on document ready
$(document).ready(() => {
    FCA_CM.init();
});

})(jQuery);

