<?php

/**
 * Demo Content Importer
 * 
 * Provides functionality to import demo courses and lessons
 */

class FCE_Demo_Importer {
    
    /**
     * Constructor
     */
    public function __construct() {
        // Handle demo import
        add_action('admin_post_fce_import_demo_content', [$this, 'handle_demo_import']);
    }
    
    /**
     * Add submenu page
     */
    public function add_admin_menu() {
        add_submenu_page(
            'fce-shortcodes',
            'Import Demo Content',
            'Import Demo Content',
            'manage_options',
            'fce-demo-importer',
            [$this, 'render_importer_page']
        );
    }
    
    /**
     * Render the importer page
     */
    public function render_importer_page() {
        ?>
        <div class="wrap">
            <h1>FCE Shortcodes - Import Demo Content</h1>
            
            <?php
            // Show result message if we've just imported
            if (isset($_GET['imported'])) {
                $success = $_GET['imported'] === '1';
                $courses = isset($_GET['courses']) ? intval($_GET['courses']) : 0;
                $lessons = isset($_GET['lessons']) ? intval($_GET['lessons']) : 0;
                $categories = isset($_GET['categories']) ? intval($_GET['categories']) : 0;
                
                if ($success) {
                    ?>
                    <div class="fce-demo-import-result">
                        <h3>Demo Content Imported Successfully!</h3>
                        <p>The following content was imported:</p>
                        <ul>
                            <li><strong><?php echo $courses; ?></strong> courses</li>
                            <li><strong><?php echo $lessons; ?></strong> lessons</li>
                            <?php if ($categories > 0): ?>
                                <li><strong><?php echo $categories; ?></strong> categories</li>
                            <?php endif; ?>
                        </ul>
                        <p>You can now use the <code>[fce_course_search]</code> shortcode to display these courses.</p>
                    </div>
                    <?php
                } else {
                    ?>
                    <div class="fce-demo-import-result error">
                        <h3>Error Importing Demo Content</h3>
                        <p>There was an error importing the demo content. Please check the error logs for more information.</p>
                    </div>
                    <?php
                }
            }
            ?>
            
            <div class="fce-shortcodes-help">
                <p>This tool will import demo courses and lessons into your Fluent Community database. This is useful for testing the FCE Shortcodes plugin functionality.</p>
                <p class="fce-demo-import-warning">Warning: This is for testing purposes only. The demo content will be added to your actual Fluent Community database.</p>
                
                <form method="post" action="<?php echo admin_url('admin-post.php'); ?>">
                    <input type="hidden" name="action" value="fce_import_demo_content">
                    <?php wp_nonce_field('fce_import_demo_content', 'fce_demo_nonce'); ?>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">Number of Courses</th>
                            <td>
                                <input type="number" name="fce_demo_courses" value="5" min="1" max="20">
                                <p class="description">How many demo courses to create (1-20)</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Lessons per Course</th>
                            <td>
                                <input type="number" name="fce_demo_lessons" value="5" min="1" max="15">
                                <p class="description">How many lessons to create per course (1-15)</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Categories</th>
                            <td>
                                <input type="checkbox" name="fce_demo_categories" value="1" checked>
                                <label for="fce_demo_categories">Create demo categories</label>
                            </td>
                        </tr>
                    </table>
                    
                    <?php submit_button('Import Demo Content', 'primary', 'submit', true, ['id' => 'fce-import-demo-btn']); ?>
                </form>
            </div>
        </div>
        <?php
    }
    
    /**
     * Handle the demo import
     */
    public function handle_demo_import() {
        // Check nonce
        check_admin_referer('fce_import_demo_content', 'fce_demo_nonce');
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('You do not have sufficient permissions to access this page.');
        }
        
        // Get form data
        $num_courses = isset($_POST['fce_demo_courses']) ? intval($_POST['fce_demo_courses']) : 5;
        $num_lessons = isset($_POST['fce_demo_lessons']) ? intval($_POST['fce_demo_lessons']) : 5;
        $create_categories = isset($_POST['fce_demo_categories']) && $_POST['fce_demo_categories'] == '1';
        
        // Limit the numbers
        $num_courses = max(1, min(20, $num_courses));
        $num_lessons = max(1, min(15, $num_lessons));
        
        // Import the demo content
        $result = $this->import_demo_content($num_courses, $num_lessons, $create_categories);
        
        // Redirect back with result
        wp_redirect(add_query_arg(
            [
                'page' => 'fce-demo-importer',
                'imported' => $result['success'] ? '1' : '0',
                'courses' => $result['courses'],
                'lessons' => $result['lessons'],
                'categories' => $result['categories']
            ],
            admin_url('admin.php')
        ));
        exit;
    }
    
    /**
     * Import demo content
     */
    private function import_demo_content($num_courses, $num_lessons, $create_categories) {
        global $wpdb;
        
        $result = [
            'success' => false,
            'courses' => 0,
            'lessons' => 0,
            'categories' => 0
        ];
        
        // Demo categories
        $categories = [
            ['title' => 'Programming', 'slug' => 'programming'],
            ['title' => 'Design', 'slug' => 'design'],
            ['title' => 'Business', 'slug' => 'business'],
            ['title' => 'Marketing', 'slug' => 'marketing'],
            ['title' => 'Personal Development', 'slug' => 'personal-development']
        ];
        
        // Demo course titles and descriptions
        $course_templates = [
            [
                'title' => 'Introduction to Web Development',
                'description' => 'Learn the fundamentals of web development including HTML, CSS, and JavaScript. This course is perfect for beginners who want to start building websites.',
                'category' => 'programming'
            ],
            [
                'title' => 'Advanced JavaScript Programming',
                'description' => 'Take your JavaScript skills to the next level with advanced concepts like closures, prototypes, async/await, and modern ES6+ features.',
                'category' => 'programming'
            ],
            [
                'title' => 'UI/UX Design Fundamentals',
                'description' => 'Learn the principles of user interface and user experience design. Create beautiful, intuitive designs that users will love.',
                'category' => 'design'
            ],
            [
                'title' => 'Digital Marketing Mastery',
                'description' => 'Comprehensive guide to digital marketing including SEO, social media, email marketing, and paid advertising strategies.',
                'category' => 'marketing'
            ],
            [
                'title' => 'Business Strategy and Management',
                'description' => 'Learn essential business management skills including strategic planning, team leadership, and operational excellence.',
                'category' => 'business'
            ],
            [
                'title' => 'Personal Productivity and Time Management',
                'description' => 'Boost your productivity with proven time management techniques, goal setting strategies, and personal organization systems.',
                'category' => 'personal-development'
            ],
            [
                'title' => 'Mobile App Development with React Native',
                'description' => 'Build cross-platform mobile applications using React Native. Create apps that work on both iOS and Android from a single codebase.',
                'category' => 'programming'
            ],
            [
                'title' => 'Graphic Design for Beginners',
                'description' => 'Master the fundamentals of graphic design including typography, color theory, layout design, and industry-standard tools.',
                'category' => 'design'
            ],
            [
                'title' => 'Content Marketing Strategy',
                'description' => 'Learn how to create and distribute valuable content to attract and engage your target audience and drive profitable customer action.',
                'category' => 'marketing'
            ],
            [
                'title' => 'Entrepreneurship: Start Your Own Business',
                'description' => 'A comprehensive guide to starting and growing your own business, from idea validation to launch and beyond.',
                'category' => 'business'
            ]
        ];
        
        // Demo lesson titles
        $lesson_templates = [
            'Getting Started with %s',
            'Understanding the Basics of %s',
            'Advanced Techniques in %s',
            'Practical Applications of %s',
            'Case Studies: %s in Action',
            'Future Trends in %s',
            'Building Your First %s Project',
            'Troubleshooting Common %s Issues',
            'Best Practices for %s',
            'Mastering %s: Final Project',
            'Introduction to %s Tools',
            'Working with %s Teams',
            'Optimizing Your %s Workflow',
            'Measuring Success in %s',
            'Career Opportunities in %s'
        ];
        
        try {
            // Start transaction
            $wpdb->query('START TRANSACTION');
            
            // Get current user ID for author
            $current_user_id = get_current_user_id();
            
            // Create categories if needed
            $category_ids = [];
            if ($create_categories) {
                foreach ($categories as $category) {
                    // Check if category exists
                    $existing = $wpdb->get_var($wpdb->prepare(
                        "SELECT id FROM {$wpdb->prefix}fcom_terms WHERE slug = %s AND taxonomy_name = 'category'",
                        $category['slug']
                    ));
                    
                    if (!$existing) {
                        // Insert category
                        $wpdb->insert(
                            $wpdb->prefix . 'fcom_terms',
                            [
                                'title' => $category['title'],
                                'slug' => $category['slug'],
                                'taxonomy_name' => 'category',
                                'created_at' => current_time('mysql'),
                                'updated_at' => current_time('mysql')
                            ]
                        );
                        
                        $category_ids[$category['slug']] = $wpdb->insert_id;
                        $result['categories']++;
                    } else {
                        $category_ids[$category['slug']] = $existing;
                    }
                }
            }
            
            // Create courses
            for ($i = 0; $i < $num_courses; $i++) {
                // Get a random course template or create a generic one if we've used all templates
                $template_index = $i % count($course_templates);
                $course_template = $course_templates[$template_index];
                
                // Create a unique title if we're creating more courses than templates
                $course_title = $course_template['title'];
                if ($i >= count($course_templates)) {
                    $course_title .= ' ' . ceil(($i + 1) / count($course_templates));
                }
                
                // Generate a unique slug
                $base_slug = sanitize_title($course_title);
                $slug = $base_slug;
                $counter = 1;
                
                // Check if slug exists
                while ($wpdb->get_var($wpdb->prepare(
                    "SELECT id FROM {$wpdb->prefix}fcom_spaces WHERE slug = %s",
                    $slug
                ))) {
                    $slug = $base_slug . '-' . $counter;
                    $counter++;
                }
                
                // Insert course
                $wpdb->insert(
                    $wpdb->prefix . 'fcom_spaces',
                    [
                        'created_by' => $current_user_id,
                        'parent_id' => 0, // Top-level course
                        'title' => $course_title,
                        'slug' => $slug,
                        'description' => $course_template['description'],
                        'type' => 'course',
                        'privacy' => 'private',
                        'status' => 'published',
                        'serial' => $i + 1,
                        'settings' => serialize([
                            'restricted_post_only' => 'no',
                            'emoji' => '',
                            'shape_svg' => '',
                            'custom_lock_screen' => 'no',
                            'can_request_join' => 'no',
                            'layout_style' => 'timeline',
                            'show_sidebar' => 'yes',
                            'og_image' => '',
                            'links' => [],
                            'topic_required' => 'no',
                            'hide_members_count' => 'no',
                            'members_page_status' => 'members_only',
                            'course_type' => 'self_paced',
                            'disable_comments' => 'no'
                        ]),
                        'created_at' => current_time('mysql'),
                        'updated_at' => current_time('mysql')
                    ]
                );
                
                $course_id = $wpdb->insert_id;
                
                // Add course price (random between free and paid)
                $is_free = rand(0, 1) == 1;
                $price = $is_free ? 0 : rand(10, 100) + 0.99;
                
                $wpdb->insert(
                    $wpdb->prefix . 'fcom_meta',
                    [
                        'object_id' => $course_id,
                        'object_type' => 'course',
                        'meta_key' => 'price',
                        'value' => $price,
                        'created_at' => current_time('mysql'),
                        'updated_at' => current_time('mysql')
                    ]
                );
                
                $wpdb->insert(
                    $wpdb->prefix . 'fcom_meta',
                    [
                        'object_id' => $course_id,
                        'object_type' => 'course',
                        'meta_key' => 'is_free',
                        'value' => $is_free ? '1' : '0',
                        'created_at' => current_time('mysql'),
                        'updated_at' => current_time('mysql')
                    ]
                );
                
                // Add author relationship
                $wpdb->insert(
                    $wpdb->prefix . 'fcom_space_user',
                    [
                        'space_id' => $course_id,
                        'user_id' => $current_user_id,
                        'role' => 'admin',
                        'created_at' => current_time('mysql'),
                        'updated_at' => current_time('mysql')
                    ]
                );
                
                // Create a main section for the course
                $section_title = 'Main Section';
                $wpdb->insert(
                    $wpdb->prefix . 'fcom_posts',
                    [
                        'created_by' => $current_user_id,
                        'title' => $section_title,
                        'type' => 'course_section',
                        'space_id' => $course_id,
                        'status' => 'published',
                        'parent_id' => 0,
                        'slug' => sanitize_title($section_title),
                        'created_at' => current_time('mysql'),
                        'updated_at' => current_time('mysql')
                    ]
                );
                
                $section_id = $wpdb->insert_id;
                
                // Create lessons for this course
                $subject = preg_replace('/^(Introduction to|Advanced|Mastering) /', '', $course_title);
                
                for ($j = 0; $j < $num_lessons; $j++) {
                    // Get lesson template
                    $lesson_template = $lesson_templates[$j % count($lesson_templates)];
                    $lesson_title = sprintf($lesson_template, $subject);
                    
                    // Generate lesson content
                    $lesson_content = $this->generate_lesson_content($lesson_title, $subject);
                    
                    // Insert lesson post
                    $wpdb->insert(
                        $wpdb->prefix . 'fcom_posts',
                        [
                            'created_by' => $current_user_id,
                            'title' => $lesson_title,
                            'type' => 'course_lesson',
                            'space_id' => $course_id,
                            'parent_id' => $section_id, // Link to the section
                            'status' => 'published',
                            'slug' => sanitize_title($lesson_title),
                            'message' => $lesson_content, // Use message field for content
                            'created_at' => current_time('mysql'),
                            'updated_at' => current_time('mysql')
                        ]
                    );
                    
                    $lesson_id = $wpdb->insert_id;
                    
                    // Add lesson order meta
                    $wpdb->insert(
                        $wpdb->prefix . 'fcom_meta',
                        [
                            'object_id' => $lesson_id,
                            'object_type' => 'course_lesson',
                            'meta_key' => 'order',
                            'value' => $j + 1,
                            'created_at' => current_time('mysql'),
                            'updated_at' => current_time('mysql')
                        ]
                    );
                    
                    // Add lesson to category if available
                    if ($create_categories && isset($category_ids[$course_template['category']])) {
                        $wpdb->insert(
                            $wpdb->prefix . 'fcom_term_feed',
                            [
                                'term_id' => $category_ids[$course_template['category']],
                                'post_id' => $lesson_id,
                                'created_at' => current_time('mysql'),
                                'updated_at' => current_time('mysql')
                            ]
                        );
                    }
                    
                    $result['lessons']++;
                }
                
                $result['courses']++;
            }
            
            // Commit transaction
            $wpdb->query('COMMIT');
            $result['success'] = true;
            
        } catch (Exception $e) {
            // Rollback on error
            $wpdb->query('ROLLBACK');
            error_log('FCE Demo Import Error: ' . $e->getMessage());
        }
        
        return $result;
    }
    
    /**
     * Generate dummy lesson content
     */
    private function generate_lesson_content($lesson_title, $subject) {
        $paragraphs = [
            "Welcome to this lesson on {$lesson_title}. In this comprehensive guide, we'll explore everything you need to know about {$subject} and how it can benefit your projects.",
            
            "Understanding {$subject} is crucial for success in today's competitive landscape. Whether you're a beginner or an experienced professional, mastering these concepts will give you a significant advantage.",
            
            "<h2>Key Concepts</h2>",
            
            "Let's start by exploring the fundamental principles of {$subject}. These core concepts form the foundation upon which all advanced techniques are built.",
            
            "<ul>
                <li>The importance of proper planning in {$subject}</li>
                <li>How to identify and solve common problems</li>
                <li>Best practices for implementing {$subject} in real-world scenarios</li>
                <li>Tools and resources that can enhance your workflow</li>
                <li>Measuring success and tracking progress</li>
            </ul>",
            
            "<h2>Practical Applications</h2>",
            
            "Theory is important, but practical application is where real learning happens. Let's look at some ways you can apply {$subject} in your daily work:",
            
            "<ol>
                <li>Start with a small project to build confidence</li>
                <li>Gradually incorporate more complex techniques</li>
                <li>Seek feedback from peers and mentors</li>
                <li>Iterate and improve based on real-world results</li>
                <li>Document your process for future reference</li>
            </ol>",
            
            "<h2>Case Study</h2>",
            
            "Consider the case of Company XYZ, which implemented {$subject} techniques to solve their business challenges. Within six months, they saw a 40% improvement in efficiency and a 25% increase in customer satisfaction.",
            
            "Their approach involved careful planning, stakeholder engagement, and iterative implementation. By focusing on small wins and building momentum, they were able to achieve remarkable results.",
            
            "<h2>Common Mistakes to Avoid</h2>",
            
            "Even experienced professionals can make mistakes when working with {$subject}. Here are some pitfalls to watch out for:",
            
            "<ul>
                <li>Rushing the process without proper planning</li>
                <li>Failing to consider all stakeholders</li>
                <li>Overlooking important details in the implementation phase</li>
                <li>Not measuring results effectively</li>
                <li>Resisting adaptation when circumstances change</li>
            </ul>",
            
            "<h2>Conclusion</h2>",
            
            "Mastering {$subject} is a journey that requires dedication and practice. By understanding the core principles, applying them in practical situations, and learning from both successes and failures, you'll develop expertise that sets you apart.",
            
            "In our next lesson, we'll dive deeper into advanced techniques and explore cutting-edge developments in the field of {$subject}. Until then, practice what you've learned and experiment with different approaches to solidify your understanding."
        ];
        
        return implode("\n\n", $paragraphs);
    }
}

// Initialize the class
new FCE_Demo_Importer(); 