<?php

class FCE_Course_Search {
    private $option_name = 'fce_shortcodes_course_search';
    
    public function __construct() {
        // Register AJAX handlers
        add_action('wp_ajax_fce_get_courses', [$this, 'handle_ajax_get_courses']);
        add_action('wp_ajax_nopriv_fce_get_courses', [$this, 'handle_ajax_get_courses']);
        
        add_action('wp_ajax_fce_get_course_categories', [$this, 'handle_ajax_get_course_categories']);
        add_action('wp_ajax_nopriv_fce_get_course_categories', [$this, 'handle_ajax_get_course_categories']);
        
        add_action('wp_ajax_fce_get_course_authors', [$this, 'handle_ajax_get_course_authors']);
        add_action('wp_ajax_nopriv_fce_get_course_authors', [$this, 'handle_ajax_get_course_authors']);
    }
    
    /**
     * Get courses based on search parameters
     */
    public function get_courses($args = []) {
        global $wpdb;
        
        $defaults = [
            'per_page' => 10,
            'page' => 1,
            'search' => '',
            'category' => '',
            'author' => '',
            'status' => 'published',
            'orderby' => 'title',
            'order' => 'ASC'
        ];
        
        $args = wp_parse_args($args, $defaults);
        
        // Calculate offset
        $offset = ($args['page'] - 1) * $args['per_page'];
        
        // Make sure we're selecting the cover_photo field
        $select_fields = "s.id, s.title, s.description, s.privacy, s.status, s.slug, s.logo, s.cover_photo, s.settings, s.created_by, s.created_at";
        
        // The SQL query should include the cover_photo field
        $sql = "SELECT $select_fields FROM {$wpdb->prefix}fcom_spaces s WHERE s.type = 'course'";
        
        // Add status filter
        if ($args['status']) {
            $sql .= $wpdb->prepare(" AND s.status = %s", $args['status']);
        }
        
        // Add search filter
        if ($args['search']) {
            $sql .= $wpdb->prepare(" AND (s.title LIKE %s OR s.description LIKE %s)", 
                '%' . $wpdb->esc_like($args['search']) . '%',
                '%' . $wpdb->esc_like($args['search']) . '%'
            );
        }
        
        // Add category filter
        if ($args['category']) {
            $sql .= $wpdb->prepare(" AND s.id IN (
                SELECT p.space_id FROM {$wpdb->prefix}fcom_posts p
                JOIN {$wpdb->prefix}fcom_term_feed tf ON p.id = tf.post_id
                JOIN {$wpdb->prefix}fcom_terms t ON tf.term_id = t.id
                WHERE t.slug = %s
            )", $args['category']);
        }
        
        // Add author filter
        if ($args['author']) {
            $sql .= $wpdb->prepare(" AND s.id IN (
                SELECT space_id FROM {$wpdb->prefix}fcom_space_user
                WHERE user_id = %d AND role = 'admin'
            )", $args['author']);
        }
        
        // Count total before adding limit
        $count_query = str_replace("SELECT s.*, ", "SELECT COUNT(*) ", $sql);
        $total = $wpdb->get_var($count_query);
        
        // Add ordering
        $sql .= " ORDER BY s.{$args['orderby']} {$args['order']}";
        
        // Add pagination
        $sql .= $wpdb->prepare(" LIMIT %d OFFSET %d", $args['per_page'], $offset);
        
        // Get courses
        $courses = $wpdb->get_results($sql);
        
        // Get course links class
        $course_links = new FCE_Course_Links();
        
        // Get additional data for each course
        foreach ($courses as $course) {
            // Get course image
            $course->image_url = $this->get_course_image($course->id);
            
            // Get course price
            $course->price = $this->get_course_price($course->id);
            
            // Get course categories
            $course->categories = $this->get_course_categories($course->id);
            
            // Get lesson count
            $course->lesson_count = $this->get_course_lesson_count($course->id);
            
            // Get course URL
            $course->course_url = $course_links->get_course_permalink($course->id);
        }
        
        // Store pagination info in a global variable for the template to use
        global $fce_course_pagination;
        $fce_course_pagination = [
            'total' => $total,
            'pages' => ceil($total / $args['per_page']),
            'current_page' => $args['page']
        ];
        
        // Return just the courses array
        return $courses;
    }
    
    /**
     * Get course image URL
     */
    public function get_course_image($course_id) {
        global $wpdb;
        
        // First check if there's a cover_photo directly in the spaces table
        $cover_photo = $wpdb->get_var($wpdb->prepare(
            "SELECT cover_photo FROM {$wpdb->prefix}fcom_spaces
            WHERE id = %d AND type = 'course'",
            $course_id
        ));
        
        if (!empty($cover_photo)) {
            return $cover_photo;
        }
        
        // If no direct cover_photo, check for course image in meta
        $image_id = $wpdb->get_var($wpdb->prepare(
            "SELECT meta_value FROM {$wpdb->prefix}fcom_meta
            WHERE object_id = %d AND object_type = 'course' AND meta_key = 'featured_image'",
            $course_id
        ));
        
        if ($image_id) {
            // If it's a WordPress attachment ID
            if (is_numeric($image_id)) {
                $image_url = wp_get_attachment_image_url($image_id, 'medium');
                if ($image_url) {
                    return $image_url;
                }
            } else {
                // If it's a direct URL
                return $image_id;
            }
        }
        
        // Default image if none found
        $default_image = plugin_dir_url(dirname(__FILE__)) . 'assets/images/default-course.png';
        return apply_filters('fce_shortcodes_default_course_image', $default_image);
    }
    
    /**
     * Get course price
     */
    public function get_course_price($course_id) {
        global $wpdb;
        
        // Check for course price in meta
        $price = $wpdb->get_var($wpdb->prepare(
            "SELECT value FROM {$wpdb->prefix}fcom_meta
            WHERE object_id = %d AND object_type = 'course' AND meta_key = 'price'",
            $course_id
        ));
        
        if ($price !== null) {
            return floatval($price);
        }
        
        // Check if it's free
        $is_free = $wpdb->get_var($wpdb->prepare(
            "SELECT value FROM {$wpdb->prefix}fcom_meta
            WHERE object_id = %d AND object_type = 'course' AND meta_key = 'is_free'",
            $course_id
        ));
        
        if ($is_free === '1') {
            return 0;
        }
        
        // Default to null if no price info found
        return null;
    }
    
    /**
     * Check if course is free
     */
    public function is_course_free($course_id) {
        global $wpdb;
        
        // Check if it's explicitly marked as free
        $is_free = $wpdb->get_var($wpdb->prepare(
            "SELECT value FROM {$wpdb->prefix}fcom_meta
            WHERE object_id = %d AND object_type = 'course' AND meta_key = 'is_free'",
            $course_id
        ));
        
        if ($is_free === '1') {
            return true;
        }
        
        // Check if price is 0 or null
        $price = $this->get_course_price($course_id);
        return ($price === 0 || $price === null);
    }
    
    /**
     * Get course author
     */
    public function get_course_author($course_id) {
        global $wpdb;
        
        // Get author ID
        $author_id = $wpdb->get_var($wpdb->prepare(
            "SELECT user_id FROM {$wpdb->prefix}fcom_space_user
            WHERE space_id = %d AND role = 'admin'
            LIMIT 1",
            $course_id
        ));
        
        if (!$author_id) {
            return null;
        }
        
        // Get author data
        return get_userdata($author_id);
    }
    
    /**
     * Get course categories
     */
    public function get_course_categories($course_id) {
        global $wpdb;
        
        // Get all posts in this course
        $posts = $wpdb->get_col($wpdb->prepare(
            "SELECT id FROM {$wpdb->prefix}fcom_posts
            WHERE space_id = %d",
            $course_id
        ));
        
        if (empty($posts)) {
            return [];
        }
        
        // Get all terms associated with these posts
        $post_ids = implode(',', array_map('intval', $posts));
        $terms = $wpdb->get_results(
            "SELECT DISTINCT t.* FROM {$wpdb->prefix}fcom_terms t
            JOIN {$wpdb->prefix}fcom_term_feed tf ON t.id = tf.term_id
            WHERE tf.post_id IN ($post_ids) AND t.taxonomy_name = 'category'"
        );
        
        return $terms;
    }
    
    /**
     * Get course lesson count
     */
    public function get_course_lesson_count($course_id) {
        global $wpdb;
        
        return $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}fcom_posts
            WHERE space_id = %d AND type = 'course_lesson'",
            $course_id
        ));
    }
    
    /**
     * Get all course categories
     */
    public function get_all_categories() {
        global $wpdb;
        
        $categories = $wpdb->get_results(
            "SELECT DISTINCT t.* FROM {$wpdb->prefix}fcom_terms t
            JOIN {$wpdb->prefix}fcom_term_feed tf ON t.id = tf.term_id
            JOIN {$wpdb->prefix}fcom_posts p ON tf.post_id = p.id
            JOIN {$wpdb->prefix}fcom_spaces s ON p.space_id = s.id
            WHERE t.taxonomy_name = 'category' AND s.type = 'course'
            ORDER BY t.title ASC"
        );
        
        return $categories;
    }
    
    /**
     * Get categories (alias for get_all_categories)
     * 
     * @return array
     */
    public function get_categories() {
        return $this->get_all_categories();
    }
    
    /**
     * Get all course authors
     */
    public function get_all_authors() {
        global $wpdb;
        
        $authors = $wpdb->get_results(
            "SELECT DISTINCT u.ID, u.display_name, u.user_email
            FROM {$wpdb->users} u
            JOIN {$wpdb->prefix}fcom_space_user su ON u.ID = su.user_id
            JOIN {$wpdb->prefix}fcom_spaces s ON su.space_id = s.id
            WHERE s.type = 'course' AND su.role = 'admin'
            ORDER BY u.display_name ASC"
        );
        
        return $authors;
    }
    
    /**
     * AJAX handler for getting courses
     */
    public function handle_ajax_get_courses() {
        // Verify nonce
        check_ajax_referer('fce_shortcodes_nonce', 'nonce');
        
        $args = [
            'per_page' => isset($_POST['per_page']) ? absint($_POST['per_page']) : 10,
            'page' => isset($_POST['page']) ? absint($_POST['page']) : 1,
            'search' => isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '',
            'category' => isset($_POST['category']) ? sanitize_text_field($_POST['category']) : '',
            'author' => isset($_POST['author']) ? absint($_POST['author']) : '',
            'status' => isset($_POST['status']) ? sanitize_text_field($_POST['status']) : 'published',
            'orderby' => isset($_POST['orderby']) ? sanitize_text_field($_POST['orderby']) : 'title',
            'order' => isset($_POST['order']) ? sanitize_text_field($_POST['order']) : 'ASC'
        ];
        
        $courses = $this->get_courses($args);
        
        // Get pagination info
        global $fce_course_pagination;
        
        wp_send_json_success([
            'courses' => $courses,
            'pagination' => $fce_course_pagination
        ]);
    }
    
    /**
     * AJAX handler for getting course categories
     */
    public function handle_ajax_get_course_categories() {
        // Verify nonce
        check_ajax_referer('fce_shortcodes_nonce', 'nonce');
        
        $categories = $this->get_all_categories();
        
        wp_send_json_success($categories);
    }
    
    /**
     * AJAX handler for getting course authors
     */
    public function handle_ajax_get_course_authors() {
        // Verify nonce
        check_ajax_referer('fce_shortcodes_nonce', 'nonce');
        
        $authors = $this->get_all_authors();
        
        wp_send_json_success($authors);
    }
}

// Initialize the class
new FCE_Course_Search(); 