<?php

/**
 * FCE Community Stats
 * 
 * Provides functionality to display community statistics
 */
class FCE_Community_Stats {
    
    public function __construct() {
        // Register AJAX handlers if needed
        add_action('wp_ajax_fce_get_community_stats', [$this, 'handle_ajax_get_stats']);
        add_action('wp_ajax_nopriv_fce_get_community_stats', [$this, 'handle_ajax_get_stats']);
    }
    
    /**
     * Get community statistics
     * 
     * @param array $args Arguments for filtering stats
     * @return array Array of statistics
     */
    public function get_stats($args = []) {
        global $wpdb;
        
        // Default arguments
        $defaults = [
            'stat_type' => 'total_members',
            'space_id' => 0,
            'course_id' => 0,
            'time_period' => 'all', // all, today, week, month
        ];
        
        $args = wp_parse_args($args, $defaults);
        
        // Initialize result
        $result = [
            'value' => 0,
            'label' => $this->get_stat_label($args['stat_type']),
            'icon' => $this->get_stat_icon($args['stat_type'])
        ];
        
        // Get the appropriate statistic based on type
        switch ($args['stat_type']) {
            case 'total_members':
                $result['value'] = $this->get_total_members();
                break;
                
            case 'total_enrolled':
                $result['value'] = $this->get_total_enrolled($args['course_id']);
                break;
                
            case 'total_posts':
                $result['value'] = $this->get_total_posts($args['space_id']);
                break;
                
            case 'total_comments':
                $result['value'] = $this->get_total_comments($args['space_id']);
                break;
                
            case 'recent_posts':
                $result['value'] = $this->get_recent_posts($args['space_id'], $args['time_period']);
                break;
                
            case 'recent_comments':
                $result['value'] = $this->get_recent_comments($args['space_id'], $args['time_period']);
                break;
                
            case 'active_users':
                $result['value'] = $this->get_active_users($args['time_period']);
                break;
                
            case 'course_completion_rate':
                $result['value'] = $this->get_course_completion_rate($args['course_id']) . '%';
                break;
                
            case 'total_courses':
                $result['value'] = $this->get_total_courses();
                break;
                
            case 'total_completed':
                $result['value'] = $this->get_total_completed($args['course_id'], $args['time_period']);
                break;
        }
        
        return $result;
    }
    
    /**
     * Get total members count
     */
    private function get_total_members() {
        // Count users with subscriber role or higher
        $count = count_users();
        return isset($count['total_users']) ? $count['total_users'] : 0;
    }
    
    /**
     * Get total courses count
     */
    private function get_total_courses() {
        global $wpdb;
        
        // Debug log
        error_log('FCE Stats: Counting total courses');
        
        // First try to get courses from fcom_spaces table (this is how system-tools.php does it)
        $spaces_table = $wpdb->prefix . 'fcom_spaces';
        if ($wpdb->get_var("SHOW TABLES LIKE '{$spaces_table}'")) {
            error_log('FCE Stats: Found spaces table: ' . $spaces_table);
            
            $courses_count = $wpdb->get_var("SELECT COUNT(*) FROM {$spaces_table} WHERE type = 'course'");
            error_log('FCE Stats: Found ' . $courses_count . ' courses in spaces table');
            
            if ($courses_count > 0) {
                return (int) $courses_count;
            }
        }
        
        // Try alternative spaces table
        $alt_spaces_table = $wpdb->prefix . 'fc_spaces';
        if ($wpdb->get_var("SHOW TABLES LIKE '{$alt_spaces_table}'")) {
            error_log('FCE Stats: Found alternative spaces table: ' . $alt_spaces_table);
            
            $courses_count = $wpdb->get_var("SELECT COUNT(*) FROM {$alt_spaces_table} WHERE type = 'course'");
            error_log('FCE Stats: Found ' . $courses_count . ' courses in alternative spaces table');
            
            if ($courses_count > 0) {
                return (int) $courses_count;
            }
        }
        
        // Try dedicated course tables if spaces approach didn't work
        $possible_tables = [
            $wpdb->prefix . 'fcom_courses',
            $wpdb->prefix . 'fc_courses',
            $wpdb->prefix . 'fluentcourse_courses',
            $wpdb->prefix . 'fluentcourse_course'
        ];
        
        foreach ($possible_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'")) {
                error_log('FCE Stats: Found courses table: ' . $table);
                
                // Check table structure to determine correct column names
                $columns = $wpdb->get_results("SHOW COLUMNS FROM {$table}");
                $column_names = array_map(function($col) {
                    return $col->Field;
                }, $columns);
                
                // Determine status column name
                $status_column = in_array('status', $column_names) ? 'status' : 'post_status';
                $status_value = ($status_column === 'status') ? 'publish' : 'publish';
                
                // Log the query we're about to run
                $sql = "SELECT COUNT(*) FROM {$table} WHERE {$status_column} = '{$status_value}'";
                error_log('FCE Stats: Running query: ' . $sql);
                
                $count = (int) $wpdb->get_var($sql);
                error_log('FCE Stats: Found ' . $count . ' courses in ' . $table);
                
                if ($count > 0) {
                    return $count;
                }
            }
        }
        
        // Try to get courses from WordPress posts table with course post type
        $course_post_types = ['course', 'fluent_course', 'fc_course', 'fcom_course'];
        
        foreach ($course_post_types as $post_type) {
            error_log('FCE Stats: Checking WordPress post type: ' . $post_type);
            
            // Check if post type exists
            $post_type_exists = post_type_exists($post_type);
            if ($post_type_exists) {
                error_log('FCE Stats: Post type exists: ' . $post_type);
                
                $count_posts = wp_count_posts($post_type);
                $count = isset($count_posts->publish) ? (int) $count_posts->publish : 0;
                
                error_log('FCE Stats: Found ' . $count . ' courses with post type ' . $post_type);
                
                if ($count > 0) {
                    return $count;
                }
            }
        }
        
        // If we still don't have a count, try a direct query to the posts table
        $sql = "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type LIKE '%course%' AND post_status = 'publish'";
        error_log('FCE Stats: Running fallback query: ' . $sql);
        
        $count = (int) $wpdb->get_var($sql);
        error_log('FCE Stats: Found ' . $count . ' courses in fallback query');
        
        return $count > 0 ? $count : 0;
    }
    
    /**
     * Get total enrolled users count
     */
    private function get_total_enrolled($course_id = 0) {
        global $wpdb;
        
        // Debug log
        error_log('FCE Stats: Calculating enrollments. Course ID: ' . ($course_id ? $course_id : 'All courses'));
        
        // Try Fluent Community space_user table first (most common)
        $space_user_tables = [
            $wpdb->prefix . 'fcom_space_user',
            $wpdb->prefix . 'fc_space_user'
        ];
        
        foreach ($space_user_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'")) {
                error_log('FCE Stats: Found space_user table: ' . $table);
                
                // If course_id is specified, we need to find the space_id for this course
                if ($course_id > 0) {
                    // Try to find the space_id for this course from spaces table
                    $spaces_tables = [
                        $wpdb->prefix . 'fcom_spaces',
                        $wpdb->prefix . 'fc_spaces'
                    ];
                    
                    $space_id = null;
                    foreach ($spaces_tables as $spaces_table) {
                        if ($wpdb->get_var("SHOW TABLES LIKE '{$spaces_table}'")) {
                            // Check if the course exists in this table
                            $space_id_query = $wpdb->prepare(
                                "SELECT id FROM {$spaces_table} WHERE type = 'course' AND id = %d",
                                $course_id
                            );
                            $space_id = $wpdb->get_var($space_id_query);
                            
                            if ($space_id) {
                                error_log('FCE Stats: Found space_id ' . $space_id . ' for course_id ' . $course_id);
                                break;
                            }
                        }
                    }
                    
                    if ($space_id) {
                        $count_query = $wpdb->prepare(
                            "SELECT COUNT(DISTINCT user_id) FROM {$table} WHERE space_id = %d",
                            $space_id
                        );
                        $count = (int) $wpdb->get_var($count_query);
                        error_log('FCE Stats: Found ' . $count . ' enrollments for course ' . $course_id);
                        
                        if ($count > 0) {
                            return $count;
                        }
                    }
                } else {
                    // Count all unique users enrolled in any course
                    $count_query = "SELECT COUNT(DISTINCT user_id) FROM {$table} WHERE 1=1";
                    
                    // If we can determine which spaces are courses, filter by those
                    $spaces_tables = [
                        $wpdb->prefix . 'fcom_spaces',
                        $wpdb->prefix . 'fc_spaces'
                    ];
                    
                    foreach ($spaces_tables as $spaces_table) {
                        if ($wpdb->get_var("SHOW TABLES LIKE '{$spaces_table}'")) {
                            $course_spaces_query = "SELECT id FROM {$spaces_table} WHERE type = 'course'";
                            $course_spaces = $wpdb->get_col($course_spaces_query);
                            
                            if (!empty($course_spaces)) {
                                $course_spaces_str = implode(',', $course_spaces);
                                $count_query = "SELECT COUNT(DISTINCT user_id) FROM {$table} WHERE space_id IN ({$course_spaces_str})";
                                break;
                            }
                        }
                    }
                    
                    $count = (int) $wpdb->get_var($count_query);
                    error_log('FCE Stats: Found ' . $count . ' total enrollments across all courses');
                    
                    if ($count > 0) {
                        return $count;
                    }
                }
            }
        }
        
        // Try dedicated enrollment tables if space_user approach didn't work
        $enrollment_tables = [
            $wpdb->prefix . 'fcom_enrollments',
            $wpdb->prefix . 'fc_enrollments',
            $wpdb->prefix . 'fluentcourse_enrollments'
        ];
        
        foreach ($enrollment_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'")) {
                error_log('FCE Stats: Found enrollments table: ' . $table);
                
                $sql = "SELECT COUNT(DISTINCT user_id) FROM {$table} WHERE 1=1";
                
                if ($course_id > 0) {
                    $sql .= $wpdb->prepare(" AND course_id = %d", $course_id);
                }
                
                $count = (int) $wpdb->get_var($sql);
                error_log('FCE Stats: Found ' . $count . ' enrollments in ' . $table);
                
                if ($count > 0) {
                    return $count;
                }
            }
        }
        
        // If we still don't have enrollments, try WordPress user meta as a last resort
        if ($course_id > 0) {
            $meta_keys = [
                '_enrolled_in_course_' . $course_id,
                'enrolled_in_course_' . $course_id,
                'course_' . $course_id . '_enrolled'
            ];
            
            foreach ($meta_keys as $meta_key) {
                $count = (int) $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(DISTINCT user_id) FROM {$wpdb->usermeta} WHERE meta_key = %s AND meta_value = '1'",
                    $meta_key
                ));
                
                if ($count > 0) {
                    error_log('FCE Stats: Found ' . $count . ' enrollments via user meta: ' . $meta_key);
                    return $count;
                }
            }
        } else {
            // For all courses, try to find any enrollment meta
            $count = (int) $wpdb->get_var(
                "SELECT COUNT(DISTINCT user_id) FROM {$wpdb->usermeta} WHERE meta_key LIKE '%enrolled_in_course_%' AND meta_value = '1'"
            );
            
            if ($count > 0) {
                error_log('FCE Stats: Found ' . $count . ' total enrollments via user meta');
                return $count;
            }
        }
        
        // If all else fails, return 0
        error_log('FCE Stats: No enrollments found');
        return 0;
    }
    
    /**
     * Get total posts count
     */
    private function get_total_posts($space_id = 0) {
        global $wpdb;
        
        // Try all possible table names based on the schema
        $possible_tables = [
            $wpdb->prefix . 'fcom_posts',
            $wpdb->prefix . 'fc_posts'
        ];
        
        $table_name = null;
        foreach ($possible_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'")) {
                $table_name = $table;
                break;
            }
        }
        
        if (!$table_name) {
            // Fallback to WordPress posts
            $count_posts = wp_count_posts();
            return $count_posts->publish;
        }
        
        $sql = "SELECT COUNT(*) FROM {$table_name} WHERE status = 'published'";
        
        if ($space_id > 0) {
            $sql .= $wpdb->prepare(" AND space_id = %d", $space_id);
        }
        
        return (int) $wpdb->get_var($sql);
    }
    
    /**
     * Get total comments count
     */
    private function get_total_comments($space_id = 0) {
        global $wpdb;
        
        // First try WordPress comments as fallback
        $wp_comment_count = get_comment_count();
        $wp_comments = isset($wp_comment_count['approved']) ? $wp_comment_count['approved'] : 0;
        
        // Try all possible table names based on the schema
        $possible_tables = [
            $wpdb->prefix . 'fcom_post_comments',
            $wpdb->prefix . 'fc_comments'
        ];
        
        $comments_table = null;
        foreach ($possible_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'")) {
                $comments_table = $table;
                break;
            }
        }
        
        if (!$comments_table) {
            // Return WordPress comments count if no Fluent tables exist
            return $wp_comments;
        }
        
        // Get the posts table name
        $posts_table = null;
        $possible_post_tables = [
            $wpdb->prefix . 'fcom_posts',
            $wpdb->prefix . 'fc_posts'
        ];
        
        foreach ($possible_post_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'")) {
                $posts_table = $table;
                break;
            }
        }
        
        // Check table structure to determine correct column names
        $columns = $wpdb->get_results("SHOW COLUMNS FROM {$comments_table}");
        $column_names = array_map(function($col) {
            return $col->Field;
        }, $columns);
        
        // Determine post_id column name
        $post_id_column = in_array('post_id', $column_names) ? 'post_id' : 'post_parent';
        
        // Determine status column and value
        $status_column = in_array('status', $column_names) ? 'status' : 'type';
        $status_value = ($status_column === 'status') ? 'published' : 'comment';
        
        // Build the query
        $sql = "SELECT COUNT(*) FROM {$comments_table} WHERE {$status_column} = '{$status_value}'";
        
        if ($space_id > 0 && $posts_table) {
            $sql .= $wpdb->prepare(" AND {$post_id_column} IN (SELECT id FROM {$posts_table} WHERE space_id = %d)", $space_id);
        }
        
        $count = (int) $wpdb->get_var($sql);
        
        // If count is 0, try WordPress comments as fallback
        return $count > 0 ? $count : $wp_comments;
    }
    
    /**
     * Get recent posts count
     */
    private function get_recent_posts($space_id = 0, $time_period = 'all') {
        global $wpdb;
        
        // Try all possible table names based on the schema
        $possible_tables = [
            $wpdb->prefix . 'fcom_posts',
            $wpdb->prefix . 'fc_posts'
        ];
        
        $table_name = null;
        foreach ($possible_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'")) {
                $table_name = $table;
                break;
            }
        }
        
        if (!$table_name) {
            // Fallback to WordPress posts
            if ($time_period === 'all') {
                $count_posts = wp_count_posts();
                return $count_posts->publish;
            } else {
                $args = [
                    'post_type' => 'post',
                    'post_status' => 'publish',
                    'date_query' => $this->get_date_query($time_period),
                    'posts_per_page' => -1,
                    'fields' => 'ids',
                ];
                
                $query = new WP_Query($args);
                return $query->found_posts;
            }
        }
        
        // Check table structure to determine correct column names
        $columns = $wpdb->get_results("SHOW COLUMNS FROM {$table_name}");
        $column_names = array_map(function($col) {
            return $col->Field;
        }, $columns);
        
        // Determine date column
        $date_column = in_array('created_at', $column_names) ? 'created_at' : 'updated_at';
        
        $sql = "SELECT COUNT(*) FROM {$table_name} WHERE status = 'published'";
        
        if ($space_id > 0) {
            $sql .= $wpdb->prepare(" AND space_id = %d", $space_id);
        }
        
        // Add time period filter
        $date_condition = $this->get_sql_date_condition($date_column, $time_period);
        if ($date_condition) {
            $sql .= " AND " . $date_condition;
        }
        
        return (int) $wpdb->get_var($sql);
    }
    
    /**
     * Get recent comments count
     */
    private function get_recent_comments($space_id = 0, $time_period = 'all') {
        global $wpdb;
        
        // First try WordPress comments as fallback
        $wp_comments_count = 0;
        if ($time_period === 'all') {
            $count_comments = get_comment_count();
            $wp_comments_count = isset($count_comments['approved']) ? $count_comments['approved'] : 0;
        } else {
            $args = [
                'status' => 'approve',
                'date_query' => $this->get_date_query($time_period),
                'count' => true,
            ];
            
            $wp_comments_count = get_comments($args);
        }
        
        // Try all possible table names based on the schema
        $possible_tables = [
            $wpdb->prefix . 'fcom_post_comments',
            $wpdb->prefix . 'fc_comments'
        ];
        
        $comments_table = null;
        foreach ($possible_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'")) {
                $comments_table = $table;
                break;
            }
        }
        
        if (!$comments_table) {
            // Return WordPress comments count if no Fluent tables exist
            return $wp_comments_count;
        }
        
        // Get the posts table name
        $posts_table = null;
        $possible_post_tables = [
            $wpdb->prefix . 'fcom_posts',
            $wpdb->prefix . 'fc_posts'
        ];
        
        foreach ($possible_post_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'")) {
                $posts_table = $table;
                break;
            }
        }
        
        // Check table structure to determine correct column names
        $columns = $wpdb->get_results("SHOW COLUMNS FROM {$comments_table}");
        $column_names = array_map(function($col) {
            return $col->Field;
        }, $columns);
        
        // Determine post_id column name
        $post_id_column = in_array('post_id', $column_names) ? 'post_id' : 'post_parent';
        
        // Determine status column and value
        $status_column = in_array('status', $column_names) ? 'status' : 'type';
        $status_value = ($status_column === 'status') ? 'published' : 'comment';
        
        // Determine date column
        $date_column = in_array('created_at', $column_names) ? 'created_at' : 'updated_at';
        
        // Build the query
        $sql = "SELECT COUNT(*) FROM {$comments_table} WHERE {$status_column} = '{$status_value}'";
        
        if ($space_id > 0 && $posts_table) {
            $sql .= $wpdb->prepare(" AND {$post_id_column} IN (SELECT id FROM {$posts_table} WHERE space_id = %d)", $space_id);
        }
        
        // Add time period filter
        $date_condition = $this->get_sql_date_condition($date_column, $time_period);
        if ($date_condition) {
            $sql .= " AND " . $date_condition;
        }
        
        $count = (int) $wpdb->get_var($sql);
        
        // If count is 0, try WordPress comments as fallback
        return $count > 0 ? $count : $wp_comments_count;
    }
    
    /**
     * Get active users count
     */
    private function get_active_users($time_period = 'all') {
        global $wpdb;
        
        // Try all possible table names based on the schema
        $possible_post_tables = [
            $wpdb->prefix . 'fcom_posts',
            $wpdb->prefix . 'fc_posts'
        ];
        
        $posts_table = null;
        foreach ($possible_post_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'")) {
                $posts_table = $table;
                break;
            }
        }
        
        $possible_comment_tables = [
            $wpdb->prefix . 'fcom_post_comments',
            $wpdb->prefix . 'fc_comments'
        ];
        
        $comments_table = null;
        foreach ($possible_comment_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'")) {
                $comments_table = $table;
                break;
            }
        }
        
        if (!$posts_table || !$comments_table) {
            // Fallback to WordPress activity
            return $this->get_wp_active_users($time_period);
        }
        
        // Check posts table structure
        $posts_columns = $wpdb->get_results("SHOW COLUMNS FROM {$posts_table}");
        $posts_column_names = array_map(function($col) {
            return $col->Field;
        }, $posts_columns);
        
        // Check comments table structure
        $comments_columns = $wpdb->get_results("SHOW COLUMNS FROM {$comments_table}");
        $comments_column_names = array_map(function($col) {
            return $col->Field;
        }, $comments_columns);
        
        // Determine date columns
        $posts_date_column = in_array('created_at', $posts_column_names) ? 'created_at' : 'updated_at';
        $comments_date_column = in_array('created_at', $comments_column_names) ? 'created_at' : 'updated_at';
        
        // Determine status columns and values
        $posts_status_column = in_array('status', $posts_column_names) ? 'status' : 'type';
        $posts_status_value = ($posts_status_column === 'status') ? 'published' : 'post';
        
        $comments_status_column = in_array('status', $comments_column_names) ? 'status' : 'type';
        $comments_status_value = ($comments_status_column === 'status') ? 'published' : 'comment';
        
        // Get date conditions
        $date_condition_posts = $this->get_sql_date_condition($posts_date_column, $time_period);
        $date_condition_comments = $this->get_sql_date_condition($comments_date_column, $time_period);
        
        // Build the query
        $sql = "SELECT COUNT(DISTINCT user_id) FROM (
            SELECT user_id FROM {$posts_table} WHERE {$posts_status_column} = '{$posts_status_value}'";
        
        if ($date_condition_posts) {
            $sql .= " AND " . $date_condition_posts;
        }
        
        $sql .= " UNION 
            SELECT user_id FROM {$comments_table} WHERE {$comments_status_column} = '{$comments_status_value}'";
        
        if ($date_condition_comments) {
            $sql .= " AND " . $date_condition_comments;
        }
        
        $sql .= ") AS active_users";
        
        $count = (int) $wpdb->get_var($sql);
        
        // If count is 0, try WordPress users as fallback
        return $count > 0 ? $count : $this->get_wp_active_users($time_period);
    }
    
    /**
     * Get course completion rate
     */
    private function get_course_completion_rate($course_id) {
        global $wpdb;
        
        if (empty($course_id)) {
            return 0;
        }
        
        // Check if Fluent Course tables exist
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$wpdb->prefix}fc_enrollments'");
        
        if (!$table_exists) {
            return 0;
        }
        
        // Get total enrollments for this course
        $total_enrollments = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}fc_enrollments WHERE course_id = %d",
                $course_id
            )
        );
        
        if ($total_enrollments === 0) {
            return 0;
        }
        
        // Get completed enrollments for this course
        $completed_enrollments = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}fc_enrollments WHERE course_id = %d AND completed_at IS NOT NULL",
                $course_id
            )
        );
        
        // Calculate completion rate
        $completion_rate = ($completed_enrollments / $total_enrollments) * 100;
        
        return round($completion_rate, 1);
    }
    
    /**
     * Get active WordPress users
     * 
     * @param string $time_period Time period (all, today, week, month)
     * @return int Number of active users
     */
    private function get_wp_active_users($time_period = 'all') {
        global $wpdb;
        
        // Get date condition
        $date_condition = $this->get_sql_date_condition('post_date', $time_period);
        $date_condition_comments = $this->get_sql_date_condition('comment_date', $time_period);
        
        // Get users who have created posts or comments in the given time period
        $sql = "SELECT COUNT(DISTINCT user_id) FROM (
            SELECT post_author AS user_id FROM {$wpdb->posts} WHERE post_status = 'publish'";
        
        if ($date_condition) {
            $sql .= " AND " . $date_condition;
        }
        
        $sql .= " UNION 
            SELECT user_id FROM {$wpdb->comments} WHERE comment_approved = '1'";
        
        if ($date_condition_comments) {
            $sql .= " AND " . $date_condition_comments;
        }
        
        $sql .= ") AS active_users";
        
        return (int) $wpdb->get_var($sql);
    }
    
    /**
     * Get date query for WP_Query
     * 
     * @param string $time_period Time period (all, today, week, month)
     * @return array Date query
     */
    private function get_date_query($time_period) {
        if ($time_period === 'all') {
            return [];
        }
        
        $date_query = [];
        
        switch ($time_period) {
            case 'today':
                $date_query = [
                    'after' => date('Y-m-d 00:00:00', current_time('timestamp'))
                ];
                break;
                
            case 'week':
                $date_query = [
                    'after' => date('Y-m-d H:i:s', strtotime('-1 week', current_time('timestamp')))
                ];
                break;
                
            case 'month':
                $date_query = [
                    'after' => date('Y-m-d H:i:s', strtotime('-1 month', current_time('timestamp')))
                ];
                break;
        }
        
        return $date_query;
    }
    
    /**
     * Get SQL date condition
     * 
     * @param string $column Date column name
     * @param string $time_period Time period (all, today, week, month)
     * @return string SQL condition
     */
    private function get_sql_date_condition($column, $time_period) {
        if ($time_period === 'all') {
            return '';
        }
        
        switch ($time_period) {
            case 'today':
                return $column . " >= '" . date('Y-m-d 00:00:00', current_time('timestamp')) . "'";
                
            case 'week':
                return $column . " >= '" . date('Y-m-d H:i:s', strtotime('-1 week', current_time('timestamp'))) . "'";
                
            case 'month':
                return $column . " >= '" . date('Y-m-d H:i:s', strtotime('-1 month', current_time('timestamp'))) . "'";
                
            default:
                return '';
        }
    }
    
    /**
     * Get stat label
     * 
     * @param string $stat_type Stat type
     * @return string Stat label
     */
    public function get_stat_label($stat_type) {
        $labels = [
            'total_members' => 'Total Members',
            'total_enrolled' => 'Total Enrolled',
            'total_posts' => 'Total Posts',
            'total_comments' => 'Total Comments',
            'total_courses' => 'Total Courses',
            'total_completed' => 'Total Completed',
            'recent_posts' => 'Recent Posts',
            'recent_comments' => 'Recent Comments',
            'active_users' => 'Active Users',
            'course_completion_rate' => 'Completion Rate'
        ];
        
        return isset($labels[$stat_type]) ? $labels[$stat_type] : 'Statistic';
    }
    
    /**
     * Get stat icon
     * 
     * @param string $stat_type Stat type
     * @return string SVG icon markup
     */
    public function get_stat_icon($stat_type) {
        $icons = [
            'total_members' => '<svg viewBox="0 0 20 16"><path d="M17.3 13c.6 0 1.1-.4 1.6-.9.9-1.1-.6-2-1.2-2.5-.6-.4-1.2-.7-1.9-.8M15 7.2c1.2 0 2.1-.9 2.1-2.1 0-1.2-.9-2.1-2.1-2.1" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path><path d="M2.7 13c-.6 0-1.1-.4-1.6-.9-.9-1.1.6-2 1.2-2.5.6-.4 1.2-.7 1.9-.8M4.6 7.2c-1.2 0-2.1-.9-2.1-2.1 0-1.2.9-2.1 2.1-2.1" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path><path d="M6.7 10.6c-.9.5-3.1 1.6-1.7 2.9.7.7 1.4 1.1 2.3 1.1h5.3c.9 0 1.7-.5 2.3-1.1 1.4-1.3-.8-2.4-1.7-2.9-2-1.2-4.5-1.2-6.5 0z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path><path d="M12.9 4.2c0 1.6-1.3 3-3 3-1.6 0-3-1.3-3-3 0-1.6 1.3-3 3-3 1.6 0 3 1.3 3 3z" stroke="currentColor" stroke-width="1.5"></path></svg>',
            
            'total_enrolled' => '<svg viewBox="0 0 24 24"><path d="M22 5l-10-3-10 3v12l10 3 10-3V5zM12 4l7 2-7 2-7-2 7-2zM4 7l8 2v10l-8-2V7zm16 10l-6 2V9l6-2v10z" stroke="currentColor" fill="none" stroke-width="1.5"></path></svg>',
            
            'total_posts' => '<svg viewBox="0 0 24 24"><path d="M20 14.66V20a2 2 0 01-2 2H4a2 2 0 01-2-2V6a2 2 0 012-2h5.34" stroke="currentColor" fill="none" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path><path d="M18 2l4 4-10 10H8v-4L18 2z" stroke="currentColor" fill="none" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path></svg>',
            
            'total_comments' => '<svg viewBox="0 0 20 20"><path d="M6.7 11.2h6.6M6.7 7.1h3.3" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path><path d="M5.1 15.8c-1.1-.1-1.9-.4-2.4-1-1-1-1-2.5-1-5.7v-.4c0-3.1 0-4.7 1-5.7 1-1 2.5-1 5.7-1h3.3c3.1 0 4.7 0 5.7 1 1 1 1 2.5 1 5.7v.4c0 3.1 0 4.7-1 5.7-1 1-2.5 1-5.7 1-1.2 0-1.6.1-2 .2-1 .2-1.9.7-2.8 1.2-1.3.6-2 .9-2.4.7-.8-.6 0-2.4.2-3.2" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path></svg>',
            
            'total_courses' => '<svg viewBox="0 0 20 20"><path d="M10.7 5.8l4 1.1M9.9 9l2 .5M10 15l.8.2c2.2.6 3.4.9 4.3.4.9-.5 1.2-1.6 1.8-3.9l.9-3.2c.6-2.2.9-3.4.4-4.2-.5-.9-1.6-1.2-3.9-1.8l-.8-.2c-2.2-.6-3.4-.9-4.3-.4-.9.5-1.2 1.6-1.8 3.9l-.8 3.2c-.6 2.2-.9 3.4-.4 4.2.5.9 1.6 1.2 3.9 1.8z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path><path d="M10 17.5l-.8.2c-2.2.6-3.4.9-4.3.4-.9-.5-1.2-1.6-1.8-3.9l-.8-3.2c-.6-2.3-.9-3.4-.4-4.3.4-.8 1.4-.8 2.7-.8" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path></svg>',
            
            'total_completed' => '<svg viewBox="0 0 24 24"><path d="M5 7.2a2.2 2.2 0 012.2-2.2h1a2.2 2.2 0 001.55-.64l.7-.7a2.2 2.2 0 013.12 0l.7.7c.41.41.97.64 1.55.64h1a2.2 2.2 0 012.2 2.2v1c0 .58.23 1.14.64 1.55l.7.7a2.2 2.2 0 010 3.12l-.7.7a2.2 2.2 0 00-.64 1.55v1a2.2 2.2 0 01-2.2 2.2h-1a2.2 2.2 0 00-1.55.64l-.7.7a2.2 2.2 0 01-3.12 0l-.7-.7a2.2 2.2 0 00-1.55-.64h-1a2.2 2.2 0 01-2.2-2.2v-1a2.2 2.2 0 00-.64-1.55l-.7-.7a2.2 2.2 0 010-3.12l.7-.7A2.2 2.2 0 005 8.2v-1" stroke="currentColor" fill="none" stroke-width="1.5" stroke-linecap="round"></path><path d="M9 12l2 2 4-4" stroke="currentColor" fill="none" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path></svg>',
            
            'recent_posts' => '<svg viewBox="0 0 24 24"><path d="M20 14.66V20a2 2 0 01-2 2H4a2 2 0 01-2-2V6a2 2 0 012-2h5.34" stroke="currentColor" fill="none" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path><path d="M18 2l4 4-10 10H8v-4L18 2z" stroke="currentColor" fill="none" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path></svg>',
            
            'recent_comments' => '<svg viewBox="0 0 20 20"><path d="M6.7 11.2h6.6M6.7 7.1h3.3" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path><path d="M5.1 15.8c-1.1-.1-1.9-.4-2.4-1-1-1-1-2.5-1-5.7v-.4c0-3.1 0-4.7 1-5.7 1-1 2.5-1 5.7-1h3.3c3.1 0 4.7 0 5.7 1 1 1 1 2.5 1 5.7v.4c0 3.1 0 4.7-1 5.7-1 1-2.5 1-5.7 1-1.2 0-1.6.1-2 .2-1 .2-1.9.7-2.8 1.2-1.3.6-2 .9-2.4.7-.8-.6 0-2.4.2-3.2" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path></svg>',
            
            'active_users' => '<svg viewBox="0 0 24 24"><path d="M3 12h4.5l1.5 -6l4 12l2 -9l1.5 3h4.5"></path></svg>',
            
            'course_completion_rate' => '<svg viewBox="0 0 24 24"><path d="M20.2 7.8l-7.7 7.7-4-4-5.7 5.7" stroke="currentColor" fill="none" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path><path d="M15 7h6v6" stroke="currentColor" fill="none" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path></svg>'
        ];
        
        return isset($icons[$stat_type]) ? $icons[$stat_type] : '<svg viewBox="0 0 24 24"><circle cx="12" cy="12" r="10" stroke="currentColor" fill="none" stroke-width="1.5"></circle><line x1="12" y1="16" x2="12" y2="12" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></line><line x1="12" y1="8" x2="12.01" y2="8" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></line></svg>';
    }
    
    /**
     * AJAX handler for getting stats
     */
    public function handle_ajax_get_stats() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'fce_shortcodes_nonce')) {
            wp_send_json_error('Invalid nonce');
        }
        
        // Get parameters
        $stat_type = isset($_POST['stat_type']) ? sanitize_text_field($_POST['stat_type']) : 'total_members';
        $space_id = isset($_POST['space_id']) ? intval($_POST['space_id']) : 0;
        $course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
        $time_period = isset($_POST['time_period']) ? sanitize_text_field($_POST['time_period']) : 'all';
        
        // Get stats
        $stats = $this->get_stats([
            'stat_type' => $stat_type,
            'space_id' => $space_id,
            'course_id' => $course_id,
            'time_period' => $time_period,
        ]);
        
        wp_send_json_success($stats);
    }
    
    /**
     * Check if the Fluent Community plugin is active
     * 
     * @return bool
     */
    public function is_fluent_community_active() {
        return class_exists('FluentCommunity\\App\\App');
    }
    
    /**
     * Check if the Fluent Course plugin is active
     * 
     * @return bool
     */
    public function is_fluent_course_active() {
        return class_exists('FluentCourse\\App\\App');
    }
    
    /**
     * Check database structure and adapt queries
     * 
     * This method checks the actual database structure and adapts the queries accordingly
     */
    public function check_db_structure() {
        global $wpdb;
        
        // Check if comments table exists
        $comments_table = $wpdb->prefix . 'fce_comments';
        $comments_table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$comments_table}'");
        
        if (!$comments_table_exists) {
            // Try alternative table name
            $comments_table = $wpdb->prefix . 'fc_comments';
            $comments_table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$comments_table}'");
            
            if ($comments_table_exists) {
                // Log that we're using the alternative table
                error_log('FCE Stats: Using alternative comments table: ' . $comments_table);
            }
        }
        
        // Check comments table structure
        if ($comments_table_exists) {
            $columns = $wpdb->get_results("SHOW COLUMNS FROM {$comments_table}");
            $column_names = array_map(function($col) {
                return $col->Field;
            }, $columns);
            
            // Log the column names for debugging
            error_log('FCE Stats: Comments table columns: ' . implode(', ', $column_names));
            
            // Check if status column exists
            if (!in_array('status', $column_names)) {
                // Try alternative column name
                $status_column = in_array('comment_status', $column_names) ? 'comment_status' : 'status';
                error_log('FCE Stats: Using alternative status column: ' . $status_column);
            }
        }
        
        return [
            'comments_table' => $comments_table_exists ? $comments_table : false,
            'status_column' => isset($status_column) ? $status_column : 'status'
        ];
    }
    
    /**
     * Get total completed courses count
     * 
     * @param int $course_id Course ID (0 for all courses)
     * @param string $time_period Time period (all, today, week, month)
     * @return int Number of completed courses
     */
    private function get_total_completed($course_id = 0, $time_period = 'all') {
        global $wpdb;
        
        // Debug log
        error_log('FCE Stats: Counting course completions. Course ID: ' . ($course_id ? $course_id : 'All courses'));
        
        // First check the user_activities table - this is the primary source for course completions
        $activities_table = $wpdb->prefix . 'fcom_user_activities';
        
        if ($wpdb->get_var("SHOW TABLES LIKE '{$activities_table}'")) {
            error_log('FCE Stats: Found user activities table: ' . $activities_table);
            
            // Check table structure to determine correct column names
            $columns = $wpdb->get_results("SHOW COLUMNS FROM {$activities_table}");
            $column_names = array_map(function($col) {
                return $col->Field;
            }, $columns);
            
            error_log('FCE Stats: User activities table columns: ' . implode(', ', $column_names));
            
            // Build the query using action_name='course_completed' as shown in the Fluent Community code
            $sql = "SELECT COUNT(*) FROM {$activities_table} WHERE action_name = 'course_completed'";
            
            if ($course_id > 0) {
                $sql .= $wpdb->prepare(" AND feed_id = %d", $course_id);
            }
            
            // Add time period filter if needed
            if ($time_period !== 'all') {
                $date_column = in_array('created_at', $column_names) ? 'created_at' : 'updated_at';
                $date_condition = $this->get_sql_date_condition($date_column, $time_period);
                if ($date_condition) {
                    $sql .= " AND " . $date_condition;
                }
            }
            
            error_log('FCE Stats: Running query: ' . $sql);
            $count = (int) $wpdb->get_var($sql);
            error_log('FCE Stats: Found ' . $count . ' course completions in activities table');
            
            if ($count > 0) {
                return $count;
            }
        }
        
        // Try alternative activities table
        $alt_activities_table = $wpdb->prefix . 'fc_user_activities';
        
        if ($wpdb->get_var("SHOW TABLES LIKE '{$alt_activities_table}'")) {
            error_log('FCE Stats: Found alternative user activities table: ' . $alt_activities_table);
            
            // Check table structure to determine correct column names
            $columns = $wpdb->get_results("SHOW COLUMNS FROM {$alt_activities_table}");
            $column_names = array_map(function($col) {
                return $col->Field;
            }, $columns);
            
            error_log('FCE Stats: Alternative user activities table columns: ' . implode(', ', $column_names));
            
            // Build the query using action_name='course_completed'
            $sql = "SELECT COUNT(*) FROM {$alt_activities_table} WHERE action_name = 'course_completed'";
            
            if ($course_id > 0) {
                $sql .= $wpdb->prepare(" AND feed_id = %d", $course_id);
            }
            
            // Add time period filter if needed
            if ($time_period !== 'all') {
                $date_column = in_array('created_at', $column_names) ? 'created_at' : 'updated_at';
                $date_condition = $this->get_sql_date_condition($date_column, $time_period);
                if ($date_condition) {
                    $sql .= " AND " . $date_condition;
                }
            }
            
            error_log('FCE Stats: Running query: ' . $sql);
            $count = (int) $wpdb->get_var($sql);
            error_log('FCE Stats: Found ' . $count . ' course completions in alternative activities table');
            
            if ($count > 0) {
                return $count;
            }
        }
        
        // Try space_user table with completed_at field as fallback
        $space_user_tables = [
            $wpdb->prefix . 'fcom_space_user',
            $wpdb->prefix . 'fc_space_user'
        ];
        
        foreach ($space_user_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'")) {
                error_log('FCE Stats: Found space_user table: ' . $table);
                
                // Check if completed_at column exists
                $columns = $wpdb->get_results("SHOW COLUMNS FROM {$table}");
                $column_names = array_map(function($col) {
                    return $col->Field;
                }, $columns);
                
                if (in_array('completed_at', $column_names)) {
                    $sql = "SELECT COUNT(*) FROM {$table} WHERE completed_at IS NOT NULL";
                    
                    if ($course_id > 0) {
                        $sql .= $wpdb->prepare(" AND space_id = %d", $course_id);
                    } else {
                        // If no specific course, try to filter for course spaces only
                        $spaces_tables = [
                            $wpdb->prefix . 'fcom_spaces',
                            $wpdb->prefix . 'fc_spaces'
                        ];
                        
                        foreach ($spaces_tables as $spaces_table) {
                            if ($wpdb->get_var("SHOW TABLES LIKE '{$spaces_table}'")) {
                                $course_spaces_query = "SELECT id FROM {$spaces_table} WHERE type = 'course'";
                                $course_spaces = $wpdb->get_col($course_spaces_query);
                                
                                if (!empty($course_spaces)) {
                                    $course_spaces_str = implode(',', $course_spaces);
                                    $sql .= " AND space_id IN ({$course_spaces_str})";
                                    break;
                                }
                            }
                        }
                    }
                    
                    // Add time period filter if needed
                    if ($time_period !== 'all') {
                        $date_condition = $this->get_sql_date_condition('completed_at', $time_period);
                        if ($date_condition) {
                            $sql .= " AND " . $date_condition;
                        }
                    }
                    
                    error_log('FCE Stats: Running query: ' . $sql);
                    $count = (int) $wpdb->get_var($sql);
                    error_log('FCE Stats: Found ' . $count . ' course completions in space_user table');
                    
                    if ($count > 0) {
                        return $count;
                    }
                }
            }
        }
        
        // Try enrollments table as fallback
        $enrollment_tables = [
            $wpdb->prefix . 'fcom_enrollments',
            $wpdb->prefix . 'fc_enrollments',
            $wpdb->prefix . 'fluentcourse_enrollments'
        ];
        
        foreach ($enrollment_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'")) {
                error_log('FCE Stats: Found enrollments table: ' . $table);
                
                // Check if completed_at column exists
                $columns = $wpdb->get_results("SHOW COLUMNS FROM {$table}");
                $column_names = array_map(function($col) {
                    return $col->Field;
                }, $columns);
                
                if (in_array('completed_at', $column_names)) {
                    $sql = "SELECT COUNT(*) FROM {$table} WHERE completed_at IS NOT NULL";
                    
                    if ($course_id > 0) {
                        $sql .= $wpdb->prepare(" AND course_id = %d", $course_id);
                    }
                    
                    // Add time period filter if possible
                    if ($time_period !== 'all') {
                        $date_condition = $this->get_sql_date_condition('completed_at', $time_period);
                        if ($date_condition) {
                            $sql .= " AND " . $date_condition;
                        }
                    }
                    
                    error_log('FCE Stats: Running query: ' . $sql);
                    $count = (int) $wpdb->get_var($sql);
                    error_log('FCE Stats: Found ' . $count . ' course completions in enrollments table');
                    
                    if ($count > 0) {
                        return $count;
                    }
                }
            }
        }
        
        // If we still don't have completions, try WordPress user meta as a last resort
        if ($course_id > 0) {
            $meta_keys = [
                '_completed_course_' . $course_id,
                'completed_course_' . $course_id,
                'course_' . $course_id . '_completed'
            ];
            
            foreach ($meta_keys as $meta_key) {
                $count = (int) $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(DISTINCT user_id) FROM {$wpdb->usermeta} WHERE meta_key = %s AND meta_value = '1'",
                    $meta_key
                ));
                
                if ($count > 0) {
                    error_log('FCE Stats: Found ' . $count . ' completions via user meta: ' . $meta_key);
                    return $count;
                }
            }
        } else {
            // For all courses, try to find any completion meta
            $count = (int) $wpdb->get_var(
                "SELECT COUNT(DISTINCT user_id) FROM {$wpdb->usermeta} WHERE meta_key LIKE '%completed_course_%' AND meta_value = '1'"
            );
            
            if ($count > 0) {
                error_log('FCE Stats: Found ' . $count . ' total completions via user meta');
                return $count;
            }
        }
        
        // If all else fails, return 0
        error_log('FCE Stats: No course completions found');
        return 0;
    }
}

// Initialize the class
new FCE_Community_Stats(); 