<?php
/*
Plugin Name: FCE Shortcodes
Description: Shortcodes for Fluent Community content to be used anywhere in WordPress
Version: 1.0.0
Author: Acamedia
Author URI: https://acamedia.uk
*/

defined('ABSPATH') or die;

class FCE_Shortcodes {
    private $options;
    
    public function __construct() {
        // Initialize options with defaults
        $default_options = [
            'enable_course_search' => true,
            'courses_per_page' => 10,
            'show_course_description' => true,
            'show_course_image' => true,
            'show_course_price' => true,
            'show_course_author' => true,
            'show_course_categories' => true,
            'portal_url' => '',
            'portal_slug' => 'portal'
        ];
        
        $this->options = get_option('fce_shortcodes_settings', $default_options);
        
        // Ensure all default keys exist
        $this->options = wp_parse_args($this->options, $default_options);
        
        // Load required files
        $this->includes();
        
        // Initialize shortcodes
        add_action('init', [$this, 'register_shortcodes']);
        
        // Add admin menu
        add_action('admin_menu', [$this, 'add_admin_menu']);
        
        // Register settings
        add_action('admin_init', [$this, 'register_settings']);
        
        // Enqueue scripts and styles
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
    }
    
    /**
     * Include required files
     */
    private function includes() {
        // Include course search class
        require_once plugin_dir_path(__FILE__) . 'includes/class-course-search.php';
        
        // Include course links class
        require_once plugin_dir_path(__FILE__) . 'includes/class-fce-course-links.php';
        
        // Include demo importer class
        require_once plugin_dir_path(__FILE__) . 'includes/class-demo-importer.php';
        
        // Include community stats class
        require_once plugin_dir_path(__FILE__) . 'includes/class-community-stats.php';
        
        // Shortcodes are registered directly in this file, no need for a separate file
    }
    
    public function register_shortcodes() {
        add_shortcode('fce_course_search', [$this, 'course_search_shortcode']);
        add_shortcode('fce_community_stat', [$this, 'community_stat_shortcode']);
        add_shortcode('fce_community_stats', [$this, 'community_stats_grid_shortcode']);
    }
    
    public function course_search_shortcode($atts) {
        // Parse attributes
        $atts = shortcode_atts([
            'per_page' => $this->options['courses_per_page'],
            'show_description' => $this->options['show_course_description'],
            'show_image' => $this->options['show_course_image'],
            'show_price' => $this->options['show_course_price'],
            'show_author' => $this->options['show_course_author'],
            'show_categories' => $this->options['show_course_categories'],
            'category' => '',
            'author' => '',
            'status' => 'published'
        ], $atts);
        
        // Include the template
        ob_start();
        include plugin_dir_path(__FILE__) . 'templates/course-search.php';
        return ob_get_clean();
    }
    
    /**
     * Community stat shortcode
     */
    public function community_stat_shortcode($atts) {
        // Parse attributes
        $atts = shortcode_atts([
            'stat_type' => 'total_members', // total_members, total_enrolled, total_posts, total_comments, recent_posts, recent_comments, active_users, course_completion_rate
            'space_id' => 0,
            'course_id' => 0,
            'time_period' => 'all', // all, today, week, month
            'layout' => 'horizontal', // horizontal, vertical
            'size' => 'medium', // small, medium, large
            'background_color' => '',
            'text_color' => '',
            'accent_color' => '',
            'show_description' => false,
            'description' => ''
        ], $atts);
        
        // Convert string attributes to boolean
        $atts['show_description'] = filter_var($atts['show_description'], FILTER_VALIDATE_BOOLEAN);
        
        // Include the template
        ob_start();
        include plugin_dir_path(__FILE__) . 'templates/community-stats.php';
        return ob_get_clean();
    }
    
    /**
     * Community stats grid shortcode
     */
    public function community_stats_grid_shortcode($atts) {
        // Parse attributes
        $atts = shortcode_atts([
            'stats' => 'total_members,total_enrolled,total_posts,total_comments',
            'space_id' => 0,
            'course_id' => 0,
            'time_period' => 'all', // all, today, week, month
            'layout' => 'horizontal', // horizontal, vertical
            'size' => 'medium', // small, medium, large
            'columns' => 4
        ], $atts);
        
        // Include the template
        ob_start();
        include plugin_dir_path(__FILE__) . 'templates/community-stats-grid.php';
        return ob_get_clean();
    }
    
    public function add_admin_menu() {
        add_menu_page(
            'FCE Shortcodes',
            'FCE Shortcodes',
            'manage_options',
            'fce-shortcodes',
            [$this, 'render_settings_page'],
            'dashicons-shortcode',
            30
        );
        
        // Add the main settings page as a submenu
        add_submenu_page(
            'fce-shortcodes',
            'Settings',
            'Settings',
            'manage_options',
            'fce-shortcodes',
            [$this, 'render_settings_page']
        );
        
        // Add the demo importer page as a submenu
        $demo_importer = new FCE_Demo_Importer();
        add_submenu_page(
            'fce-shortcodes',
            'Import Demo Content',
            'Import Demo Content',
            'manage_options',
            'fce-demo-importer',
            [$demo_importer, 'render_importer_page']
        );
    }
    
    public function register_settings() {
        register_setting('fce_shortcodes_settings', 'fce_shortcodes_settings');
        
        add_settings_section(
            'fce_shortcodes_general_section',
            'General Settings',
            [$this, 'render_general_section'],
            'fce_shortcodes_settings'
        );
        
        add_settings_field(
            'enable_course_search',
            'Enable Course Search',
            [$this, 'render_enable_course_search_field'],
            'fce_shortcodes_settings',
            'fce_shortcodes_general_section'
        );
        
        add_settings_field(
            'courses_per_page',
            'Courses Per Page',
            [$this, 'render_courses_per_page_field'],
            'fce_shortcodes_settings',
            'fce_shortcodes_general_section'
        );
        
        add_settings_field(
            'show_course_description',
            'Show Course Description',
            [$this, 'render_show_course_description_field'],
            'fce_shortcodes_settings',
            'fce_shortcodes_general_section'
        );
        
        add_settings_field(
            'show_course_image',
            'Show Course Image',
            [$this, 'render_show_course_image_field'],
            'fce_shortcodes_settings',
            'fce_shortcodes_general_section'
        );
        
        add_settings_field(
            'show_course_price',
            'Show Course Price',
            [$this, 'render_show_course_price_field'],
            'fce_shortcodes_settings',
            'fce_shortcodes_general_section'
        );
        
        add_settings_field(
            'show_course_author',
            'Show Course Author',
            [$this, 'render_show_course_author_field'],
            'fce_shortcodes_settings',
            'fce_shortcodes_general_section'
        );
        
        add_settings_field(
            'show_course_categories',
            'Show Course Categories',
            [$this, 'render_show_course_categories_field'],
            'fce_shortcodes_settings',
            'fce_shortcodes_general_section'
        );
        
        add_settings_field(
            'portal_url',
            'Portal URL',
            [$this, 'render_portal_url_field'],
            'fce_shortcodes_settings',
            'fce_shortcodes_general_section'
        );
        
        add_settings_field(
            'portal_slug',
            'Portal Slug',
            [$this, 'render_portal_slug_field'],
            'fce_shortcodes_settings',
            'fce_shortcodes_general_section'
        );
    }
    
    public function render_general_section() {
        echo '<p>Configure the general settings for FCE Shortcodes.</p>';
    }
    
    public function render_enable_course_search_field() {
        $checked = isset($this->options['enable_course_search']) ? checked($this->options['enable_course_search'], true, false) : '';
        echo '<input type="checkbox" id="enable_course_search" name="fce_shortcodes_settings[enable_course_search]" value="1" ' . $checked . ' />';
        echo '<label for="enable_course_search">Enable the [fce_course_search] shortcode</label>';
    }
    
    public function render_courses_per_page_field() {
        $value = isset($this->options['courses_per_page']) ? esc_attr($this->options['courses_per_page']) : '10';
        echo '<input type="number" id="courses_per_page" name="fce_shortcodes_settings[courses_per_page]" value="' . $value . '" min="1" max="100" />';
        echo '<p class="description">Number of courses to display per page</p>';
    }
    
    public function render_show_course_description_field() {
        $checked = isset($this->options['show_course_description']) ? checked($this->options['show_course_description'], true, false) : '';
        echo '<input type="checkbox" id="show_course_description" name="fce_shortcodes_settings[show_course_description]" value="1" ' . $checked . ' />';
    }
    
    public function render_show_course_image_field() {
        $checked = isset($this->options['show_course_image']) ? checked($this->options['show_course_image'], true, false) : '';
        echo '<input type="checkbox" id="show_course_image" name="fce_shortcodes_settings[show_course_image]" value="1" ' . $checked . ' />';
    }
    
    public function render_show_course_price_field() {
        $checked = isset($this->options['show_course_price']) ? checked($this->options['show_course_price'], true, false) : '';
        echo '<input type="checkbox" id="show_course_price" name="fce_shortcodes_settings[show_course_price]" value="1" ' . $checked . ' />';
    }
    
    public function render_show_course_author_field() {
        $checked = isset($this->options['show_course_author']) ? checked($this->options['show_course_author'], true, false) : '';
        echo '<input type="checkbox" id="show_course_author" name="fce_shortcodes_settings[show_course_author]" value="1" ' . $checked . ' />';
    }
    
    public function render_show_course_categories_field() {
        $checked = isset($this->options['show_course_categories']) ? checked($this->options['show_course_categories'], true, false) : '';
        echo '<input type="checkbox" id="show_course_categories" name="fce_shortcodes_settings[show_course_categories]" value="1" ' . $checked . ' />';
    }
    
    public function render_portal_url_field() {
        $value = isset($this->options['portal_url']) ? esc_url($this->options['portal_url']) : '';
        echo '<input type="url" id="portal_url" name="fce_shortcodes_settings[portal_url]" value="' . $value . '" class="regular-text" />';
        echo '<p class="description">Enter the URL of your Fluent Community portal (e.g., https://example.com/portal). If left empty, the site URL will be used.</p>';
    }
    
    public function render_portal_slug_field() {
        $value = isset($this->options['portal_slug']) ? esc_attr($this->options['portal_slug']) : 'portal';
        echo '<input type="text" id="portal_slug" name="fce_shortcodes_settings[portal_slug]" value="' . $value . '" class="regular-text" />';
        echo '<p class="description">Enter the slug of your Fluent Community portal (e.g., "portal"). This is used to construct the correct URLs for courses and lessons.</p>';
    }
    
    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1>FCE Shortcodes Settings</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('fce_shortcodes_settings');
                do_settings_sections('fce_shortcodes_settings');
                submit_button();
                ?>
            </form>
            
            <div class="fce-shortcodes-help">
                <h2>Shortcode Builders</h2>
                
                <div class="fce-shortcode-builder-tabs">
                    <button class="fce-tab-button active" data-tab="course-search">Course Search</button>
                    <button class="fce-tab-button" data-tab="community-stat">Community Stat</button>
                    <button class="fce-tab-button" data-tab="community-stats-grid">Community Stats Grid</button>
                </div>
                
                <div class="fce-shortcode-builder-content">
                    <!-- Course Search Builder -->
                    <div class="fce-tab-content active" id="course-search-tab">
                        <h3>Course Search Shortcode Builder</h3>
                        <p>Build a customized course search shortcode by selecting options below:</p>
                        
                        <div class="fce-builder-form">
                            <div class="fce-builder-field">
                                <label for="cs-per-page">Courses Per Page:</label>
                                <input type="number" id="cs-per-page" value="<?php echo esc_attr($this->options['courses_per_page']); ?>" min="1" max="100">
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-show-description">Show Description:</label>
                                <select id="cs-show-description">
                                    <option value="true" <?php selected($this->options['show_course_description'], true); ?>>Yes</option>
                                    <option value="false" <?php selected($this->options['show_course_description'], false); ?>>No</option>
                                </select>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-show-image">Show Image:</label>
                                <select id="cs-show-image">
                                    <option value="true" <?php selected($this->options['show_course_image'], true); ?>>Yes</option>
                                    <option value="false" <?php selected($this->options['show_course_image'], false); ?>>No</option>
                                </select>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-show-price">Show Price:</label>
                                <select id="cs-show-price">
                                    <option value="true" <?php selected($this->options['show_course_price'], true); ?>>Yes</option>
                                    <option value="false" <?php selected($this->options['show_course_price'], false); ?>>No</option>
                                </select>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-show-author">Show Author:</label>
                                <select id="cs-show-author">
                                    <option value="true" <?php selected($this->options['show_course_author'], true); ?>>Yes</option>
                                    <option value="false" <?php selected($this->options['show_course_author'], false); ?>>No</option>
                                </select>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-show-categories">Show Categories:</label>
                                <select id="cs-show-categories">
                                    <option value="true" <?php selected($this->options['show_course_categories'], true); ?>>Yes</option>
                                    <option value="false" <?php selected($this->options['show_course_categories'], false); ?>>No</option>
                                </select>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-category">Filter by Category (slug):</label>
                                <input type="text" id="cs-category" placeholder="Leave empty for all categories">
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-author">Filter by Author ID:</label>
                                <input type="number" id="cs-author" placeholder="Leave empty for all authors" min="1">
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-status">Filter by Status:</label>
                                <select id="cs-status">
                                    <option value="published">Published</option>
                                    <option value="draft">Draft</option>
                                    <option value="all">All</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="fce-builder-preview">
                            <h4>Generated Shortcode:</h4>
                            <div class="fce-shortcode-preview" id="cs-shortcode-preview">[fce_course_search]</div>
                            <button class="button fce-copy-shortcode" data-shortcode="[fce_course_search]">Copy Shortcode</button>
                        </div>
                    </div>
                    
                    <!-- Community Stat Builder -->
                    <div class="fce-tab-content" id="community-stat-tab">
                        <h3>Community Stat Shortcode Builder</h3>
                        <p>Build a customized community stat shortcode by selecting options below:</p>
                        
                        <div class="fce-builder-form">
                            <div class="fce-builder-field">
                                <label for="fce-stat-type">Statistic Type:</label>
                                <select id="fce-stat-type" name="stat_type">
                                    <option value="total_members">Total Members</option>
                                    <option value="total_enrolled">Total Enrolled</option>
                                    <option value="total_completed">Total Completed</option>
                                    <option value="total_posts">Total Posts</option>
                                    <option value="total_comments">Total Comments</option>
                                    <option value="total_courses">Total Courses</option>
                                    <option value="recent_posts">Recent Posts</option>
                                    <option value="recent_comments">Recent Comments</option>
                                    <option value="active_users">Active Users</option>
                                    <option value="course_completion_rate">Course Completion Rate</option>
                                </select>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-space-id">Space ID:</label>
                                <input type="number" id="cs-space-id" value="0" min="0" placeholder="0 for all spaces">
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-course-id">Course ID:</label>
                                <input type="number" id="cs-course-id" value="0" min="0" placeholder="0 for all courses">
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-time-period">Time Period:</label>
                                <select id="cs-time-period">
                                    <option value="all">All Time</option>
                                    <option value="today">Today</option>
                                    <option value="week">Last 7 Days</option>
                                    <option value="month">Last 30 Days</option>
                                </select>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-layout">Layout:</label>
                                <select id="cs-layout">
                                    <option value="horizontal">Horizontal</option>
                                    <option value="vertical">Vertical</option>
                                </select>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-size">Size:</label>
                                <select id="cs-size">
                                    <option value="small">Small</option>
                                    <option value="medium" selected>Medium</option>
                                    <option value="large">Large</option>
                                </select>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-bg-color">Background Color:</label>
                                <input type="color" id="cs-bg-color" value="#ffffff">
                                <button class="button button-small cs-clear-color" data-target="cs-bg-color">Clear</button>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-text-color">Text Color:</label>
                                <input type="color" id="cs-text-color" value="#333333">
                                <button class="button button-small cs-clear-color" data-target="cs-text-color">Clear</button>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-accent-color">Accent Color:</label>
                                <input type="color" id="cs-accent-color" value="#0073aa">
                                <button class="button button-small cs-clear-color" data-target="cs-accent-color">Clear</button>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="cs-show-desc">Show Description:</label>
                                <select id="cs-show-desc">
                                    <option value="false" selected>No</option>
                                    <option value="true">Yes</option>
                                </select>
                            </div>
                            
                            <div class="fce-builder-field cs-description-field" style="display: none;">
                                <label for="cs-description">Description:</label>
                                <input type="text" id="cs-description" placeholder="Enter a custom description">
                            </div>
                        </div>
                        
                        <div class="fce-builder-preview">
                            <h4>Generated Shortcode:</h4>
                            <div class="fce-shortcode-preview" id="stat-shortcode-preview">[fce_community_stat stat_type="total_members" layout="horizontal" size="medium"]</div>
                            <button class="button fce-copy-shortcode" data-shortcode="[fce_community_stat stat_type=&quot;total_members&quot; layout=&quot;horizontal&quot; size=&quot;medium&quot;]">Copy Shortcode</button>
                        </div>
                    </div>
                    
                    <!-- Community Stats Grid Builder -->
                    <div class="fce-tab-content" id="community-stats-grid-tab">
                        <h3>Community Stats Grid Shortcode Builder</h3>
                        <p>Build a customized community stats grid shortcode by selecting options below:</p>
                        
                        <div class="fce-builder-form">
                            <div class="fce-builder-field">
                                <label>Select Stats to Display:</label>
                                <div class="fce-checkbox-group">
                                    <label><input type="checkbox" name="stats[]" value="total_members"> Total Members</label>
                                    <label><input type="checkbox" name="stats[]" value="total_enrolled"> Total Enrolled</label>
                                    <label><input type="checkbox" name="stats[]" value="total_completed"> Total Completed</label>
                                    <label><input type="checkbox" name="stats[]" value="total_posts"> Total Posts</label>
                                    <label><input type="checkbox" name="stats[]" value="total_comments"> Total Comments</label>
                                    <label><input type="checkbox" name="stats[]" value="total_courses"> Total Courses</label>
                                    <label><input type="checkbox" name="stats[]" value="recent_posts"> Recent Posts</label>
                                    <label><input type="checkbox" name="stats[]" value="recent_comments"> Recent Comments</label>
                                    <label><input type="checkbox" name="stats[]" value="active_users"> Active Users</label>
                                    <label><input type="checkbox" name="stats[]" value="course_completion_rate"> Course Completion Rate</label>
                                </div>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="csg-space-id">Space ID:</label>
                                <input type="number" id="csg-space-id" value="0" min="0" placeholder="0 for all spaces">
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="csg-course-id">Course ID:</label>
                                <input type="number" id="csg-course-id" value="0" min="0" placeholder="0 for all courses">
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="csg-time-period">Time Period:</label>
                                <select id="csg-time-period">
                                    <option value="all">All Time</option>
                                    <option value="today">Today</option>
                                    <option value="week">Last 7 Days</option>
                                    <option value="month">Last 30 Days</option>
                                </select>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="csg-layout">Layout:</label>
                                <select id="csg-layout">
                                    <option value="horizontal">Horizontal</option>
                                    <option value="vertical">Vertical</option>
                                </select>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="csg-size">Size:</label>
                                <select id="csg-size">
                                    <option value="small">Small</option>
                                    <option value="medium" selected>Medium</option>
                                    <option value="large">Large</option>
                                </select>
                            </div>
                            
                            <div class="fce-builder-field">
                                <label for="csg-columns">Columns:</label>
                                <select id="csg-columns">
                                    <option value="1">1</option>
                                    <option value="2">2</option>
                                    <option value="3">3</option>
                                    <option value="4" selected>4</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="fce-builder-preview">
                            <h4>Generated Shortcode:</h4>
                            <div class="fce-shortcode-preview" id="grid-shortcode-preview">[fce_community_stats stats="total_members,total_enrolled,total_posts,total_comments" layout="horizontal" size="medium" columns="4"]</div>
                            <button class="button fce-copy-shortcode" data-shortcode="[fce_community_stats stats=&quot;total_members,total_enrolled,total_posts,total_comments&quot; layout=&quot;horizontal&quot; size=&quot;medium&quot; columns=&quot;4&quot;]">Copy Shortcode</button>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="fce-shortcodes-help">
                <h2>Available Shortcodes</h2>
                <div class="fce-shortcode-help-item">
                    <h3>[fce_course_search]</h3>
                    <p>Displays a course search feature with filters and pagination.</p>
                    <h4>Attributes:</h4>
                    <ul>
                        <li><code>per_page</code> - Number of courses per page (default: <?php echo esc_html($this->options['courses_per_page']); ?>)</li>
                        <li><code>show_description</code> - Show course description (default: <?php echo $this->options['show_course_description'] ? 'true' : 'false'; ?>)</li>
                        <li><code>show_image</code> - Show course image (default: <?php echo $this->options['show_course_image'] ? 'true' : 'false'; ?>)</li>
                        <li><code>show_price</code> - Show course price (default: <?php echo $this->options['show_course_price'] ? 'true' : 'false'; ?>)</li>
                        <li><code>show_author</code> - Show course author (default: <?php echo $this->options['show_course_author'] ? 'true' : 'false'; ?>)</li>
                        <li><code>show_categories</code> - Show course categories (default: <?php echo $this->options['show_course_categories'] ? 'true' : 'false'; ?>)</li>
                        <li><code>category</code> - Filter by category slug (default: empty)</li>
                        <li><code>author</code> - Filter by author ID (default: empty)</li>
                        <li><code>status</code> - Filter by status (default: published)</li>
                    </ul>
                </div>
                
                <div class="fce-shortcode-help-item">
                    <h3>[fce_community_stat]</h3>
                    <p>Displays a single community statistic in an attractive format.</p>
                    <h4>Attributes:</h4>
                    <ul>
                        <li><code>stat_type</code> - Type of statistic to display (default: total_members)</li>
                        <ul>
                            <li><code>total_members</code> - Total number of members</li>
                            <li><code>total_enrolled</code> - Total number of enrolled users</li>
                            <li><code>total_completed</code> - Total number of course completions</li>
                            <li><code>total_posts</code> - Total number of posts</li>
                            <li><code>total_comments</code> - Total number of comments</li>
                            <li><code>total_courses</code> - Total number of courses</li>
                            <li><code>recent_posts</code> - Number of recent posts</li>
                            <li><code>recent_comments</code> - Number of recent comments</li>
                            <li><code>active_users</code> - Number of active users</li>
                            <li><code>course_completion_rate</code> - Course completion rate</li>
                        </ul>
                        <li><code>space_id</code> - Filter by space ID (default: 0, all spaces)</li>
                        <li><code>course_id</code> - Filter by course ID (default: 0, all courses)</li>
                        <li><code>time_period</code> - Time period for stats (default: all)</li>
                        <ul>
                            <li>all - All time</li>
                            <li>today - Today only</li>
                            <li>week - Last 7 days</li>
                            <li>month - Last 30 days</li>
                        </ul>
                        <li><code>layout</code> - Layout style (default: horizontal)</li>
                        <ul>
                            <li>horizontal - Icon and text side by side</li>
                            <li>vertical - Icon above text</li>
                        </ul>
                        <li><code>size</code> - Size of the stat display (default: medium)</li>
                        <ul>
                            <li>small - Compact size</li>
                            <li>medium - Standard size</li>
                            <li>large - Larger size</li>
                        </ul>
                        <li><code>background_color</code> - Custom background color (default: white)</li>
                        <li><code>text_color</code> - Custom text color (default: dark gray)</li>
                        <li><code>accent_color</code> - Custom accent color for icon and value (default: blue)</li>
                        <li><code>show_description</code> - Show custom description (default: false)</li>
                        <li><code>description</code> - Custom description text (default: empty)</li>
                    </ul>
                </div>
                
                <div class="fce-shortcode-help-item">
                    <h3>[fce_community_stats]</h3>
                    <p>Displays multiple community statistics in a grid layout.</p>
                    <h4>Attributes:</h4>
                    <ul>
                        <li><code>stats</code> - Comma-separated list of stats to display (default: total_members,total_enrolled,total_posts,total_comments)</li>
                        <li><code>space_id</code> - Filter by space ID (default: 0, all spaces)</li>
                        <li><code>course_id</code> - Filter by course ID (default: 0, all courses)</li>
                        <li><code>time_period</code> - Time period for stats (default: all)</li>
                        <li><code>layout</code> - Layout style (default: horizontal)</li>
                        <li><code>size</code> - Size of the stat displays (default: medium)</li>
                        <li><code>columns</code> - Number of columns in the grid (default: 4)</li>
                    </ul>
                </div>
            </div>
            
            <div class="fce-shortcodes-help">
                <h2>Demo Content</h2>
                <p>Need some content to test with? You can import demo courses and lessons.</p>
                <p><a href="<?php echo admin_url('admin.php?page=fce-demo-importer'); ?>" class="button button-secondary">Import Demo Content</a></p>
            </div>
        </div>
        <?php
    }
    
    public function enqueue_scripts() {
        wp_enqueue_style(
            'fce-shortcodes-styles',
            plugin_dir_url(__FILE__) . 'assets/css/fce-shortcodes.css',
            [],
            '1.0.0'
        );
        
        wp_enqueue_script(
            'fce-shortcodes-scripts',
            plugin_dir_url(__FILE__) . 'assets/js/fce-shortcodes.js',
            ['jquery'],
            '1.0.0',
            true
        );
        
        wp_localize_script('fce-shortcodes-scripts', 'fceShortcodes', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('fce_shortcodes_nonce'),
            'pluginUrl' => plugin_dir_url(__FILE__)
        ]);
    }
    
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'toplevel_page_fce-shortcodes' && $hook !== 'fce-shortcodes_page_fce-demo-importer') {
            return;
        }
        
        wp_enqueue_style(
            'fce-shortcodes-admin-styles',
            plugin_dir_url(__FILE__) . 'assets/css/fce-shortcodes-admin.css',
            [],
            '1.0.0'
        );
        
        wp_enqueue_script(
            'fce-shortcodes-admin-scripts',
            plugin_dir_url(__FILE__) . 'assets/js/fce-shortcodes-admin.js',
            ['jquery'],
            '1.0.0',
            true
        );
    }
}

/**
 * Initialize the plugin
 */
function fce_shortcodes_init() {
    new FCE_Shortcodes();
    
    // Ensure placeholder image exists
    fce_ensure_placeholder_image();
}
add_action('plugins_loaded', 'fce_shortcodes_init');

/**
 * Ensure placeholder image exists
 */
function fce_ensure_placeholder_image() {
    $placeholder_dir = plugin_dir_path(__FILE__) . 'assets/images';
    $placeholder_file = $placeholder_dir . '/course-placeholder.png';
    
    // Create directory if it doesn't exist
    if (!file_exists($placeholder_dir)) {
        wp_mkdir_p($placeholder_dir);
    }
    
    // Create placeholder image if it doesn't exist
    if (!file_exists($placeholder_file)) {
        // Simple placeholder image data (1x1 transparent PNG)
        $placeholder_data = base64_decode('iVBORw0KGgoAAAANSUhEUgAAAQAAAAEACAMAAABrrFhUAAAAA1BMVEXm5uTvAyFKAAAASElEQVR4nO3BMQEAAADCoPVPbQhfoAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAICXAcTgAAG6EJuyAAAAAElFTkSuQmCC');
        file_put_contents($placeholder_file, $placeholder_data);
    }
}

/**
 * Register AJAX handlers
 */
function fce_register_ajax_handlers() {
    // Course search AJAX handler
    add_action('wp_ajax_fce_search_courses', 'fce_ajax_search_courses');
    add_action('wp_ajax_nopriv_fce_search_courses', 'fce_ajax_search_courses');
    
    // Community stats AJAX handler
    add_action('wp_ajax_fce_get_stats', 'fce_ajax_get_stats');
    add_action('wp_ajax_nopriv_fce_get_stats', 'fce_ajax_get_stats');
}
add_action('init', 'fce_register_ajax_handlers');

/**
 * AJAX handler for course search
 */
function fce_ajax_search_courses() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'fce_shortcodes_nonce')) {
        wp_send_json_error('Invalid nonce');
    }
    
    // Include the course search class
    require_once plugin_dir_path(__FILE__) . 'includes/class-course-search.php';
    
    // Create instance
    $course_search = new FCE_Course_Search();
    
    // Get search parameters
    $search = isset($_POST['search']) ? sanitize_text_field($_POST['search']) : '';
    $category = isset($_POST['category']) ? sanitize_text_field($_POST['category']) : '';
    $price = isset($_POST['price']) ? sanitize_text_field($_POST['price']) : '';
    $sort = isset($_POST['sort']) ? sanitize_text_field($_POST['sort']) : 'title';
    $page = isset($_POST['page']) ? intval($_POST['page']) : 1;
    $per_page = isset($_POST['per_page']) ? intval($_POST['per_page']) : 9;
    
    // Search courses
    $results = $course_search->search_courses([
        'search' => $search,
        'category' => $category,
        'price' => $price,
        'sort' => $sort,
        'page' => $page,
        'per_page' => $per_page
    ]);
    
    wp_send_json_success($results);
}

/**
 * AJAX handler for community stats
 */
function fce_ajax_get_stats() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'fce_shortcodes_nonce')) {
        wp_send_json_error('Invalid nonce');
    }
    
    // Include the community stats class
    require_once plugin_dir_path(__FILE__) . 'includes/class-community-stats.php';
    
    // Create instance
    $stats = new FCE_Community_Stats();
    
    // Get parameters
    $stat_type = isset($_POST['stat_type']) ? sanitize_text_field($_POST['stat_type']) : 'total_members';
    $space_id = isset($_POST['space_id']) ? intval($_POST['space_id']) : 0;
    $course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
    $time_period = isset($_POST['time_period']) ? sanitize_text_field($_POST['time_period']) : 'all';
    
    // Get stats
    $result = $stats->get_stats([
        'stat_type' => $stat_type,
        'space_id' => $space_id,
        'course_id' => $course_id,
        'time_period' => $time_period,
    ]);
    
    // Debug log
    error_log('FCE Stats AJAX Response: ' . print_r($result, true));
    
    wp_send_json_success($result);
} 