/**
 * FCE Shortcodes Frontend Scripts
 */
(function($) {
    'use strict';
    
    // Course Search functionality
    function initCourseSearch() {
        // Check if course search exists on the page
        if ($('.fce-course-search').length === 0) {
            return;
        }
        
        // Elements
        var $courseSearch = $('.fce-course-search');
        var $searchInput = $('.fce-course-search-input');
        var $categoryFilter = $('.fce-course-category-filter');
        var $priceFilter = $('.fce-course-price-filter');
        var $sortFilter = $('.fce-course-sort-filter');
        var $courseGrid = $('.fce-course-grid');
        var $prevButton = $('.fce-course-pagination-prev');
        var $nextButton = $('.fce-course-pagination-next');
        var $paginationInfo = $('.fce-course-pagination-info');
        
        // State
        var state = {
            search: '',
            category: '',
            price: '',
            sort: 'title',
            page: 1,
            perPage: parseInt($courseSearch.data('per-page')) || 9,
            totalPages: 1,
            loading: false
        };
        
        // Initialize event listeners
        function initEventListeners() {
            // Search input
            $searchInput.on('input', function() {
                state.search = $(this).val();
                state.page = 1;
                fetchCourses();
            });
            
            // Category filter
            $categoryFilter.on('change', function() {
                state.category = $(this).val();
                state.page = 1;
                fetchCourses();
            });
            
            // Price filter
            $priceFilter.on('change', function() {
                state.price = $(this).val();
                state.page = 1;
                fetchCourses();
            });
            
            // Sort filter
            $sortFilter.on('change', function() {
                state.sort = $(this).val();
                state.page = 1;
                fetchCourses();
            });
            
            // Pagination
            $prevButton.on('click', function() {
                if (state.page > 1) {
                    state.page--;
                    fetchCourses();
                }
            });
            
            $nextButton.on('click', function() {
                if (state.page < state.totalPages) {
                    state.page++;
                    fetchCourses();
                }
            });
        }
        
        // Fetch courses via AJAX
        function fetchCourses() {
            if (state.loading) {
                return;
            }
            
            state.loading = true;
            $courseSearch.addClass('loading');
            
            $.ajax({
                url: fceShortcodes.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'fce_search_courses',
                    nonce: fceShortcodes.nonce,
                    search: state.search,
                    category: state.category,
                    price: state.price,
                    sort: state.sort,
                    page: state.page,
                    per_page: state.perPage
                },
                success: function(response) {
                    if (response.success) {
                        renderCourses(response.data.courses);
                        state.totalPages = response.data.total_pages;
                        updatePagination();
                    } else {
                        console.error('Error fetching courses:', response.data);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX error:', error);
                },
                complete: function() {
                    state.loading = false;
                    $courseSearch.removeClass('loading');
                }
            });
        }
        
        // Render courses
        function renderCourses(courses) {
            if (courses.length === 0) {
                $courseGrid.html('<div class="fce-course-search-no-results">No courses found matching your criteria.</div>');
                return;
            }
            
            var html = '';
            
            courses.forEach(function(course) {
                html += '<div class="fce-course-card">';
                
                // Course image
                html += '<div class="fce-course-image">';
                html += '<a href="' + course.url + '">';
                html += '<img src="' + course.image + '" alt="' + course.title + '" onerror="this.src=\'' + fceShortcodes.pluginUrl + 'assets/images/course-placeholder.png\'" />';
                html += '</a>';
                html += '</div>';
                
                // Course content
                html += '<div class="fce-course-content">';
                html += '<h3 class="fce-course-title"><a href="' + course.url + '">' + course.title + '</a></h3>';
                html += '<div class="fce-course-excerpt">' + course.excerpt + '</div>';
                html += '</div>';
                
                // Course footer
                html += '<div class="fce-course-footer">';
                html += '<div class="fce-course-lessons">' + course.lessons_count + ' Lessons</div>';
                
                // Price
                if (course.is_free) {
                    html += '<div class="fce-course-price fce-course-price-free">Free</div>';
                } else {
                    html += '<div class="fce-course-price fce-course-price-paid">' + course.price + '</div>';
                }
                
                html += '</div>';
                
                // Categories
                if (course.categories.length > 0) {
                    html += '<div class="fce-course-categories">Categories: ';
                    
                    course.categories.forEach(function(category, index) {
                        html += '<a href="' + category.url + '" class="fce-course-category">' + category.name + '</a>';
                        
                        if (index < course.categories.length - 1) {
                            html += ', ';
                        }
                    });
                    
                    html += '</div>';
                }
                
                html += '</div>';
            });
            
            $courseGrid.html(html);
            
            // Handle image loading errors
            $('.fce-course-image img').on('error', function() {
                var placeholderUrl = fceShortcodes.pluginUrl + 'assets/images/course-placeholder.png';
                $(this).attr('src', placeholderUrl);
            });
        }
        
        // Update pagination buttons and info
        function updatePagination() {
            $prevButton.prop('disabled', state.page === 1);
            $nextButton.prop('disabled', state.page === state.totalPages);
            $paginationInfo.text(`Page ${state.page} of ${state.totalPages}`);
        }
        
        // Initialize
        initEventListeners();
        
        // Initial fetch of courses
        fetchCourses();
    }
    
    // Handle image loading errors
    $('.fce-course-image img').on('error', function() {
        var placeholderUrl = fceShortcodes.pluginUrl + 'assets/images/course-placeholder.png';
        $(this).attr('src', placeholderUrl);
    });
    
    // Initialize animated counting for stats
    function initStatCounters() {
        // Check if stats exist on the page
        if ($('.fce-community-stat').length === 0) {
            return;
        }
        
        // Check if CountUp.js is loaded, if not, load it
        if (typeof CountUp === 'undefined') {
            // Simple version of CountUp.js
            var CountUp = function(target, startVal, endVal, decimals, duration, options) {
                this.target = target;
                this.startVal = startVal;
                this.endVal = endVal;
                this.decimals = decimals || 0;
                this.duration = duration || 2;
                this.options = options || {};
                
                this.start = function() {
                    var _this = this;
                    var startTime = null;
                    var currentValue = this.startVal;
                    
                    function updateValue(timestamp) {
                        if (!startTime) startTime = timestamp;
                        var progress = Math.min((timestamp - startTime) / (_this.duration * 1000), 1);
                        var value = _this.startVal + progress * (_this.endVal - _this.startVal);
                        
                        // Format the value
                        var formattedValue = _this.formatNumber(value);
                        
                        // Update the target
                        _this.target.innerHTML = formattedValue;
                        
                        if (progress < 1) {
                            requestAnimationFrame(updateValue);
                        }
                    }
                    
                    requestAnimationFrame(updateValue);
                };
                
                this.formatNumber = function(num) {
                    var neg = num < 0;
                    num = Math.abs(num);
                    var x = Math.pow(10, this.decimals);
                    num = Math.round(num * x) / x;
                    num = num.toFixed(this.decimals);
                    
                    // Add commas for thousands separators
                    if (this.options.useGrouping) {
                        var parts = num.toString().split('.');
                        parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, this.options.separator || ',');
                        num = parts.join(this.options.decimal || '.');
                    }
                    
                    // Add prefix and suffix
                    if (this.options.prefix) num = this.options.prefix + num;
                    if (this.options.suffix) num = num + this.options.suffix;
                    
                    return (neg ? '-' : '') + num;
                };
            };
        }
        
        // Helper function to check if element is in viewport
        function isElementInViewport(el) {
            var rect = el.getBoundingClientRect();
            return (
                rect.top >= 0 &&
                rect.left >= 0 &&
                rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) &&
                rect.right <= (window.innerWidth || document.documentElement.clientWidth)
            );
        }
        
        // Process each stat
        $('.fce-community-stat').each(function() {
            var $stat = $(this);
            var $value = $stat.find('.fce-stat-value');
            var valueText = $value.text();
            
            // Check if the value is a percentage
            var isPercentage = valueText.indexOf('%') !== -1;
            var numericValue = parseFloat(valueText.replace(/[^0-9.-]+/g, ''));
            
            // Store the end value for later
            $stat.attr('data-end-value', numericValue);
            
            // If it's a percentage, store that info
            if (isPercentage) {
                $value.attr('data-suffix', '%');
            }
            
            // Debug log
            console.log('Stat initialized:', {
                type: $stat.data('stat-type'),
                value: numericValue,
                isPercentage: isPercentage
            });
        });
        
        // Function to start counter animation
        function startCounter($stat) {
            if ($stat.attr('data-counted') === 'true') {
                return;
            }
            
            var $value = $stat.find('.fce-stat-value');
            var endValue = parseFloat($stat.attr('data-end-value'));
            var suffix = $value.attr('data-suffix') || '';
            
            // Debug log
            console.log('Starting counter for:', {
                type: $stat.data('stat-type'),
                endValue: endValue,
                suffix: suffix
            });
            
            // Create options for CountUp
            var options = {
                useEasing: true,
                useGrouping: true,
                separator: ',',
                decimal: '.',
                prefix: '',
                suffix: suffix
            };
            
            // Initialize CountUp
            var counter = new CountUp($value[0], 0, endValue, 0, 2.5, options);
            counter.start();
            
            // Mark as counted
            $stat.attr('data-counted', 'true');
        }
        
        // Check if stats are in viewport on scroll
        function checkStatsInViewport() {
            $('.fce-community-stat').each(function() {
                if (isElementInViewport(this)) {
                    startCounter($(this));
                }
            });
        }
        
        // Initial check
        checkStatsInViewport();
        
        // Check on scroll
        $(window).on('scroll', checkStatsInViewport);
    }
    
    // Initialize when DOM is ready
    $(document).ready(function() {
        // Debug log
        console.log('FCE Shortcodes initialized');
        
        // Initialize components
        initCourseSearch();
        initStatCounters();
    });
    
})(jQuery); 